
#include "tokenizer.h"

/* Default parse algorithm:
 *
 * Numbers: start with + or -, and continues to have digits or .
 * ( and ) as brackets
 * Alpha atom: First char isalpha, and after that isalnum.
 * Oper atom:  all chars have to be symbolic
 *
 */

#include "lispassert.h"
#include "lisperror.h"

static LispBoolean IsDigit(LispChar c)
{
    return ((c>='0' && c<='9') || c == '.');
}

static LispBoolean IsAlpha(LispChar c)
{
    return ( (c>='a' && c<='z') || (c>='A' && c<='Z') );
}

static LispBoolean IsAlNum(LispChar c)
{
    return (IsAlpha(c) || IsDigit(c));
}

static const char symbolics[] = "~`!@#$%^&*-_=+:<>?/\\|";

static LispBoolean IsSymbolic(LispChar c)
{
    const char *ptr=symbolics;
    while (*ptr)
    {
        if (*ptr++ == c)
            return 1;
    }
    return 0;
}


LispStringPtr LispTokenizer::NextToken(LispInput& aInput,
                                       LispHashTable& aHashTable)
{
    LispChar c;
    LispInt firstpos;
    LispCharPtr start;
    
REDO:
    firstpos = aInput.Position();
    start = &aInput.StartPtr()[firstpos];

    // End of stream: return empty string
    if (aInput.EndOfStream())
        goto FINISH;

    c = aInput.Next();

    //Parse brackets
    if (c == '(')
    {
    }
    else if (c == ')')
    {
    }
    else if (c == '{')
    {
    }
    else if (c == '}')
    {
    }
    else if (c == ',')
    {
    }
    else if (c == ';')
    {
    }
    // parse comments
    else if (c == '/' && aInput.Peek() == '*')
    {
        aInput.Next(); //consume *
    FALSEALARM:
        while (aInput.Next() != '*' && !aInput.EndOfStream());
        Check(!aInput.EndOfStream(),KLispErrCommentToEndOfFile);
        if (aInput.Peek() == '/')
        {
            aInput.Next();  // consume /
            goto REDO;
        }
        goto FALSEALARM;
    }
    // parse literal strings
    else if (c == '\"')
    {
        LispString aResult;
        aResult.SetNrItems(0);
        aResult.Append(c);
        while (aInput.Peek() != '\"')
        {
            if (aInput.Peek() == '\\')
            {
                aInput.Next();
                Check(!aInput.EndOfStream(),KLispErrParsingInput);
            }
            aResult.Append(aInput.Next());
            Check(!aInput.EndOfStream(),KLispErrParsingInput);
        }
        aResult.Append(aInput.Next()); // consume the close quote
        aResult.Append('\0');
        return aHashTable.LookUp(aResult.String());
    }
    //parse atoms
    else if (IsAlpha(c))
    {
        while (IsAlNum( aInput.Peek()))
        {
            aInput.Next();
        }
    }

    else if (IsSymbolic(c))
    {
        while (IsSymbolic( aInput.Peek()))
        {
            aInput.Next();
        }
    }
    else if (c == '[')
    {
        while (aInput.Peek() == '[')
        {
            aInput.Next();
        }
    }
    else if (c == ']')
    {
        while (aInput.Peek() == ']')
        {
            aInput.Next();
        }
    }
    else if (IsDigit(c))
    {
        while (IsDigit( aInput.Peek()))
        {
            aInput.Next();
        }
    }

    // Treat the char as a space.
    else
    {
        goto REDO;
    }

FINISH:
    return aHashTable.LookUpCounted(start,aInput.Position()-firstpos);
}



