/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define CHANNEL_INTERNAL_ACCESS
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#include <errno.h>
#include "compat/strerror.h"
#include "connection.h"
#include "packet.h"
#include "queue.h"
#include "eventlog.h"
#include "list.h"
#include "message.h"
#include "account.h"
#include "util.h"
#include "prefs.h"
#include "channel.h"


static int channellist_load_permanent(char const * filename);


static t_channel * channellist_head=NULL;
static t_channel * channellist_curr=NULL;
static t_channelmember * memberlist_curr=NULL;
static t_list const * const * banlist_curr=NULL;


extern t_channel * channel_create(char const * fullname, char const * shortname, char const * clienttag, int permflag, int botflag, int operflag)
{
    t_channel * channel;
    
    if (!fullname)
    {
        eventlog(eventlog_level_error,"channel_create","got NULL fullname");
	return NULL;
    }
    if (fullname[0]=='\0')
    {
        eventlog(eventlog_level_error,"channel_create","got empty fullname");
	return NULL;
    }
    if (shortname && shortname[0]=='\0')
    {
        eventlog(eventlog_level_error,"channel_create","got empty shortname");
	return NULL;
    }
    if (clienttag && strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"channel_create","client tag has bad length (%u chars)",strlen(clienttag));
	return NULL;
    }
    
    /* non-permanent checks this in channel_add_connection */
    if (permflag)
    {
	if (channellist_find_channel(fullname))
	{
	    eventlog(eventlog_level_error,"channel_create","could not create duplicate permanent channel (fullname \"%s\")",fullname);
	    return NULL;
	}
	if (shortname && channellist_find_channel(shortname))
	{
	    eventlog(eventlog_level_error,"channel_create","could not create duplicate permanent channel (shortname \"%s\")",shortname);
	    return NULL;
	}
    }
    
    if (!(channel = malloc(sizeof(t_channel))))
    {
        eventlog(eventlog_level_error,"channel_create","could not allocate memory for channel");
        return NULL;
    }
    
    eventlog(eventlog_level_info,"channel_create","creating new channel \"%s\" shortname=%s%s%s clienttag=%s%s%s",fullname,
	     shortname?"\"":"(", /* all this is doing is printing the name in quotes else "none" in parens */
	     shortname?shortname:"none",
	     shortname?"\"":")",
	     clienttag?"\"":"(",
	     clienttag?clienttag:"none",
	     clienttag?"\"":")");
    
    if (!(channel->name = strdup(fullname)))
    {
        eventlog(eventlog_level_info,"channel_create","unable to allocate memory for channel->name");
	pfree(channel,sizeof(t_channel));
	return NULL;
    }
    
    if (!shortname)
	channel->shortname = NULL;
    else
	if (!(channel->shortname = strdup(shortname)))
	{
	    eventlog(eventlog_level_info,"channel_create","unable to allocate memory for channel->shortname");
	    pfree((void *)channel->name,strlen(channel->name)+1); /* avoid warning */
	    pfree(channel,sizeof(t_channel));
	    return NULL;
	}
    
    if (clienttag)
    {
	if (!(channel->clienttag = strdup(clienttag)))
	{
	    eventlog(eventlog_level_error,"channel_create","could not allocate memory for channel->clienttag");
	    if (channel->shortname)
		pfree((void *)channel->shortname,strlen(channel->shortname)+1); /* avoid warning */
	    pfree((void *)channel->name,strlen(channel->name)+1); /* avoid warning */
	    pfree(channel,sizeof(t_channel));
	    return NULL;
	}
    }
    else
	channel->clienttag = NULL;
    
    channel->permanent  = permflag;
    channel->allowbots  = botflag;
    channel->allowopers = operflag;
    
    channel->memberlist = NULL;
    channel->banlist    = NULL;
    channel->opr        = NULL;
    channel->next_opr   = NULL;
    
    channel->next       = channellist_head;
    channellist_head = channel;
    
    eventlog(eventlog_level_debug,"channel_create","channel created successfully");
    return channel;
}


extern int channel_destroy(t_channel * channel)
{
    t_channel * curr;
    t_list * *  ban;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_destroy","got NULL channel");
	return -1;
    }
    
    if (channel->memberlist)
    {
	eventlog(eventlog_level_debug,"channel_destroy","channel is not empty, deferring");
	channel->permanent = 0; /* make it go away when the last person leaves */
	return -1;
    }
    
    curr = channellist_head;
    if (curr==channel)
	channellist_head = channellist_head->next;
    else
    {
        while (curr->next && curr->next!=channel)
            curr = curr->next;
        
        if (!curr->next)
	{
	    eventlog(eventlog_level_error,"channel_destroy","could not find channel \"%s\"",channel->name);
	    return -1;
	}
	
        curr->next = curr->next->next;
    }
    
    eventlog(eventlog_level_info,"channel_destroy","destroying channel \"%s\"",channel->name);
    
    for (ban=&channel->banlist; *ban; )
    {
	char const * banned;
	
	if (!(banned = list_get_item(*ban)))
	    eventlog(eventlog_level_error,"channel_destroy","found NULL name in banlist");
	else
	    pfree((void *)banned,strlen(banned)+1); /* avoid warning */
	if (list_remove(ban)<0)
	{
	    eventlog(eventlog_level_error,"channel_destroy","unable to remove item from list");
	    ban = list_get_next(*ban);
	}
    }
    
    if (channel->clienttag)
	pfree((void *)channel->clienttag,strlen(channel->clienttag)+1); /* avoid warning */

    if (channel->shortname)
	pfree((void *)channel->shortname,strlen(channel->shortname)+1); /* avoid warning */

    pfree((void *)channel->name,strlen(channel->name)+1); /* avoid warning */

    pfree(channel,sizeof(t_channel));
    
    return 0;
}


extern char const * channel_get_name(t_channel const * channel)
{
    if (!channel)
    {
        eventlog(eventlog_level_warn,"channel_get_name","got NULL channel");
	return "";
    }
    
    return channel->name;
}


extern char const * channel_get_clienttag(t_channel const * channel)
{
    if (!channel)
    {
        eventlog(eventlog_level_error,"channel_get_clienttag","got NULL channel");
	return "";
    }
    
    return channel->clienttag;
}


extern int channel_get_permanent(t_channel const * channel)
{
    if (!channel)
    {
        eventlog(eventlog_level_error,"channel_get_permanent","got NULL channel");
	return 0;
    }
    
    return channel->permanent;
}


extern int channel_add_connection(t_channel * channel, t_connection * connection)
{
    t_channelmember * member;
    t_connection *    user;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_add_connection","got NULL channel");
	return -1;
    }
    if (!connection)
    {
	eventlog(eventlog_level_error,"channel_add_connection","got NULL connection");
	return -1;
    }
    
eventlog(eventlog_level_debug,"channel_create","trying to join...");
    if (channel_check_banning(channel,connection))
	return -1;
    
    if (!(member = malloc(sizeof(t_channelmember))))
    {
	eventlog(eventlog_level_error,"channel_add_connection","could not allocate memory for channelmember");
	return -1;
    }
    member->connection = connection;
    member->next = channel->memberlist;
    channel->memberlist = member;
    
eventlog(eventlog_level_error,"channel_add_connection","joining...");
    message_send(connection,MT_JOINING,connection,channel_get_name(channel));
    for (user=channel_get_first(channel); user; user=channel_get_next())
    {
	message_send(connection,MT_ADD,user,NULL);
	if (user!=connection)
	    message_send(user,MT_JOIN,connection,NULL);
    }
    
eventlog(eventlog_level_debug,"channel_create","choosing operator");
    if (!channel->opr)
	channel_choose_operator(channel,connection); /* try using this account */
    
    return 0;
}


extern int channel_del_connection(t_channel * channel, t_connection * connection)
{
    t_channelmember * curr;
    t_channelmember * temp;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_del_connection","got NULL channel");
        return -1;
    }
    if (!connection)
    {
	eventlog(eventlog_level_error,"channel_del_connection","got NULL connection");
        return -1;
    }
    
    channel_message_send(channel,MT_PART,connection,NULL);
    
    curr = channel->memberlist;
    if (curr->connection==connection)
    {
        channel->memberlist = channel->memberlist->next;
        pfree(curr,sizeof(t_channelmember));
    }
    else
    {
        while (curr->next && curr->next->connection!=connection)
            curr = curr->next;
        
        if (curr->next)
        {
            temp = curr->next;
            curr->next = curr->next->next;
            pfree(temp,sizeof(t_channelmember));
        }
    }
    
    if (!channel->memberlist && !channel->permanent) /* if channel is empty, delete it unless it's a permanent channel */
    {
        conn_del_flags(connection,MF_GAVEL);
        channel_destroy(channel);
    }
    else
    {
	if (channel->next_opr==connection)
	    channel_set_next_operator(channel,NULL);
	if (channel->opr==connection)
	    channel_choose_operator(channel,NULL);
    }
    
    return 0; /* FIXME: return error if not in channel */
}


extern void channel_update_latency(t_connection * me)
{
    t_channel *    channel;
    t_packet *     packet;
    t_connection * c;
    
    if (!me)
    {
	eventlog(eventlog_level_error,"channel_update_latency","got NULL connection");
        return;
    }
    if (!(channel = conn_get_channel(me)))
    {
	eventlog(eventlog_level_error,"channel_update_latency","connection has no channel");
        return;
    }
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	eventlog(eventlog_level_error,"channel_update_latency","could not create packet");
	return;
    }
    
    message_normal_format(packet,MT_USERFLAGS,me,NULL); /* handles NULL text */
    
    for (c=channel_get_first(channel); c; c=channel_get_next())
        if (conn_get_class(c)==conn_class_normal)
            queue_push_packet(conn_get_out_queue(c),packet);
    
    packet_del_ref(packet);
}


extern void channel_update_flags(t_connection * me)
{
    t_channel *    channel;
    t_packet *     packet;
    t_packet *     bpacket;
    t_connection * c;
    
    if (!me)
    {
	eventlog(eventlog_level_error,"channel_update_flags","got NULL connection");
        return;
    }
    if (!(channel = conn_get_channel(me)))
    {
	eventlog(eventlog_level_error,"channel_update_flags","connection has no channel");
        return;
    }
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	eventlog(eventlog_level_error,"channel_update_flags","could not create packet");
	return;
    }
    if (!(bpacket = packet_create(packet_class_raw)))
    {
	eventlog(eventlog_level_error,"channel_update_flags","could not create bpacket");
	packet_del_ref(packet);
	return;
    }
    
    message_normal_format(packet,MT_USERFLAGS,me,NULL); /* handles NULL text */
    message_bot_format(bpacket,MT_USERFLAGS,me,NULL); /* handles NULL text */
    
    for (c=channel_get_first(channel); c; c=channel_get_next())
	switch (conn_get_class(c))
	{
	case conn_class_bot:
	    queue_push_packet(conn_get_out_queue(c),bpacket);
	    break;
	case conn_class_normal:
            queue_push_packet(conn_get_out_queue(c),packet);
	    break;
	default:
	    eventlog(eventlog_level_error,"channel_update_flags","bad connection class %d in channel",(int)conn_get_class(c));
	}
    
    packet_del_ref(bpacket);
    packet_del_ref(packet);
}


extern void channel_message_send(t_channel const * channel, unsigned int type, t_connection * me, char const * text)
{
    t_packet *     packet;
    t_packet *     bpacket;
    t_connection * c;
    unsigned int   flag;
    char const *   tname;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_message_send","got NULL channel");
        return;
    }
    if (!me)
    {
	eventlog(eventlog_level_error,"channel_message_send","got NULL connection");
        return;
    }
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	eventlog(eventlog_level_error,"channel_message_send","could not create packet");
	return;
    }
    if (!(bpacket = packet_create(packet_class_raw)))
    {
	eventlog(eventlog_level_error,"channel_message_send","could not create bpacket");
	packet_del_ref(packet);
	return;
    }
    
    message_normal_format(packet,type,me,text); /* handles NULL text */
    message_bot_format(bpacket,type,me,text); /* handles NULL text */
    
    flag = 0;
    tname = conn_get_username(me);
    for (c=channel_get_first(channel); c; c=channel_get_next())
        if (!(c==me && (type==MT_MESSAGE || type==MT_JOIN)) &&
	    !(conn_check_ignoring(c,tname)==1 &&
	      (type==MT_MESSAGE || type==MT_WHISPER || type==MT_EMOTE || type==MT_BROADCAST)))
	    switch (conn_get_class(c))
	    {
	    case conn_class_bot:
		queue_push_packet(conn_get_out_queue(c),bpacket);
		if (c!=me)
		    flag = 1;
		break;
	    case conn_class_normal:
        	queue_push_packet(conn_get_out_queue(c),packet);
		if (c!=me)
		    flag = 1;
		break;
	    default:
		eventlog(eventlog_level_error,"channel_message_send","bad connection class %d in channel",(int)conn_get_class(c));
	    }
    conn_unget_username(tname);
    
    packet_del_ref(bpacket);
    packet_del_ref(packet);
    
    if (!flag && (type==MT_MESSAGE || type==MT_EMOTE))
	message_send(me,MT_WARN,me,"No one hears you.");
}


extern int channel_choose_operator(t_channel * channel, t_connection * tryop)
{
    t_connection *          oldop;
    t_channelmember const * curr;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_choose_operator","got NULL channel");
	return -1;
    }
    
    oldop = channel->opr;
    
    if (channel->next_opr) /* someone was designated */
    {
	if (conn_get_channel(channel->next_opr)!=channel)
	{
	    eventlog(eventlog_level_error,"channel_choose_operator","designated next operator not in this channel");
	    return -1;
	}
	channel->opr = channel->next_opr;
	channel->next_opr = NULL;
    }
    else
    {
        t_connection * newop;
	t_account *    account;
        
	newop = channel->opr;
	
	/* choose the designee, or the last (oldest) member of the channel
           that is eligible */
	for (curr=channel->memberlist; curr; curr=curr->next)
	    if (curr->connection!=channel->opr)
	    {
		if (!(account = conn_get_account(curr->connection)))
		{
		    eventlog(eventlog_level_error,"channel_choose_operator","connection without account in channel \"%s\"",channel->name);
		    continue;
		}
		/* designation overrides everything else... */
		if (tryop && curr->connection!=tryop)
		    continue;
		/* specific account setting overrides other settings... */
		if (account_get_auth_operator(account,channel->name)==1) /* default to false */
		{
        	    newop = curr->connection;
		    continue;
		}
		/* channel overrides general account settings... */
		if (!channel->allowopers)
		    continue;
		/* finally, check general account settings */
		if (account_get_auth_operator(account,NULL)==1) /* default to false */
        	    newop = curr->connection;
	    }
	
	channel->opr = newop;
    }
    
    /* the requested connection was not chosen */
    if (tryop && channel->opr!=tryop)
	return -1;
    
    /* there is nobody else who can be an op */
    if (oldop==channel->opr)
        channel->opr = NULL;
    
    /* update flags */
    if (oldop)
        conn_del_flags(oldop,MF_GAVEL);
    if (channel->opr)
    {
	conn_add_flags(channel->opr,MF_GAVEL);
	message_send(channel->opr,MT_WARN,channel->opr,"You are now the operator for this channel.");
    }
    
    return 0;
}


extern t_connection * channel_get_operator(t_channel const * channel)
{
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_get_operator","got NULL channel");
	return NULL;
    }
    
    return channel->opr;
}


extern int channel_set_next_operator(t_channel * channel, t_connection * conn)
{
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_set_next_operator","got NULL channel");
	return -1;
    }
    
    if (conn && channel->opr==conn)
	return -1;
    
    channel->next_opr = conn;
    return 0;
}


extern int channel_ban_user(t_channel * channel, char const * user)
{
    t_list const * const * curr;
    char *                 temp;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_ban_user","got NULL channel");
	return -1;
    }
    if (!user)
    {
	eventlog(eventlog_level_error,"channel_ban_user","got NULL user");
	return -1;
    }
    if (!channel->name)
    {
	eventlog(eventlog_level_error,"channel_ban_user","got channel with NULL name");
	return -1;
    }
    
    if (strcasecmp(channel->name,CHANNEL_NAME_BANNED)==0 ||
	strcasecmp(channel->name,CHANNEL_NAME_KICKED)==0)
        return -1;
    
    for (curr=(t_list const * const *)&channel->banlist; *curr; curr=list_get_next_const(*curr)) /* avoid warning */
        if (strcasecmp(list_get_item(*curr),user)==0)
            return 0;
    
    if (!(temp = strdup(user)))
    {
        eventlog(eventlog_level_error,"channel_ban_user","could not allocate memory for temp");
        return -1;
    }
    if (list_append_item(&channel->banlist,temp)<0)
    {
	pfree(temp,strlen(user)+1);
        eventlog(eventlog_level_error,"channel_ban_user","unable to append to list");
        return -1;
    }
    return 0;
}


extern int channel_unban_user(t_channel * channel, char const * user)
{
    t_list * * curr;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_unban_user","got NULL channel");
	return -1;
    }
    if (!user)
    {
	eventlog(eventlog_level_error,"channel_unban_user","got NULL user");
	return -1;
    }
    
    for (curr=&channel->banlist; *curr; curr=list_get_next(*curr))
        if (strcasecmp(list_get_item(*curr),user)==0)
        {
	    char const * banned;
	    
	    if (!(banned = list_get_item(*curr)))
		eventlog(eventlog_level_error,"channel_destroy","found NULL name in banlist");
	    else
        	pfree((void *)banned,strlen(banned)+1); /* avoid warning */
            if (list_remove(curr)<0)
            {
                eventlog(eventlog_level_error,"channel_unban_user","unable to remove item from list");
                return -1;
            }
            return 0;
        }
    
    return -1;
}


extern int channel_check_banning(t_channel const * channel, t_connection const * user)
{
    t_list const * const * curr;
    
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_check_banning","got NULL channel");
	return -1;
    }
    if (!user)
    {
	eventlog(eventlog_level_error,"channel_check_banning","got NULL user");
	return -1;
    }
    
    if (!channel->allowbots && conn_get_class(user)==conn_class_bot)
	return 1;
    
    for (curr=(t_list const * const *)&channel->banlist; *curr; curr=list_get_next_const(*curr)) /* avoid warning */
        if (conn_match(user,list_get_item(*curr))==1)
            return 1;
    
    return 0;
}


extern int channel_get_length(t_channel const * channel)
{
    t_channelmember const * curr;
    int                     count;
    
    for (curr=channel->memberlist,count=0; curr; curr=curr->next,count++);
    
    return count;
}


extern t_connection * channel_get_first(t_channel const * channel)
{
    if (!channel)
    {
	eventlog(eventlog_level_error,"channel_get_first","got NULL channel");
        return NULL;
    }
    
    memberlist_curr = channel->memberlist;
    
    return channel_get_next();
}


extern t_connection * channel_get_next(void)
{
    t_channelmember * member;
    
    if (memberlist_curr)
    {
        member = memberlist_curr;
        memberlist_curr = memberlist_curr->next;
        
        return member->connection;
    }
    return NULL;
}


extern char const * channel_banlist_get_first(t_channel const * channel)
{
    banlist_curr = (t_list const * const *)&channel->banlist; /* avoid warning */
    if (*banlist_curr)
        return list_get_item(*banlist_curr);
    
    return NULL;
}


extern char const * channel_banlist_get_next(void)
{
    if (banlist_curr)
    {
	if (*banlist_curr)
	    banlist_curr = list_get_next_const(*banlist_curr);
	
	if (*banlist_curr)
	    return list_get_item(*banlist_curr);
    }
    
    return NULL;
}


static int channellist_load_permanent(char const * filename)
{
    FILE *       fp;
    unsigned int line;
    unsigned int pos;
    unsigned int len;
    unsigned int endpos;
    int          botflag;
    int          operflag;
    char *       buff;
    char *       temp;
    char *       name;
    char *       sname;
    char *       tag;
    char *       bot;
    char *       oper;
    
    if (!filename)
    {
	eventlog(eventlog_level_error,"channellist_load_permanent","got NULL filename");
	return -1;
    }
    if (!(fp = fopen(filename,"r")))
    {
	eventlog(eventlog_level_error,"channellist_load_permanent","could not open file \"%s\" for reading (fopen: %s)",filename,strerror(errno));
	return -1;
    }
    
    for (line=1; (buff = file_get_line(fp)); line++)
    {
	for (pos=0; buff[pos]=='\t' || buff[pos]==' '; pos++);
	len = strlen(buff)+1;
	if (buff[pos]=='\0' || buff[pos]=='#')
        {
	    pfree(buff,len);
	    continue;
        }
        if ((temp = strrchr(buff,'#')))
	{
	    *temp = '\0';
	    for (endpos=strlen(buff)-1;  buff[endpos]=='\t' || buff[endpos]==' '; endpos--);
	    buff[endpos+1] = '\0';
	}
	
	if (!(name = malloc(len)))
	{
	    eventlog(eventlog_level_error,"channellist_load_permanent","could not allocate memory for name");
	    pfree(buff,len);
	    continue;
	}
	if (!(sname = malloc(len)))
	{
	    eventlog(eventlog_level_error,"channellist_load_permanent","could not allocate memory for sname");
	    pfree(name,len);
	    pfree(buff,len);
	    continue;
	}
	if (!(tag = malloc(len)))
	{
	    eventlog(eventlog_level_error,"channellist_load_permanent","could not allocate memory for tag");
	    pfree(sname,len);
	    pfree(name,len);
	    pfree(buff,len);
	    continue;
	}
	if (!(bot = malloc(len)))
	{
	    eventlog(eventlog_level_error,"channellist_load_permanent","could not allocate memory for bot");
	    pfree(tag,len);
	    pfree(sname,len);
	    pfree(name,len);
	    pfree(buff,len);
	    continue;
	}
	if (!(oper = malloc(len)))
	{
	    eventlog(eventlog_level_error,"channellist_load_permanent","could not allocate memory for oper");
	    pfree(bot,len);
	    pfree(tag,len);
	    pfree(sname,len);
	    pfree(name,len);
	    pfree(buff,len);
	    continue;
	}
	
	if (sscanf(buff,"\"%[^\"]\" \"%[^\"]\" \"%[^\"]\" %s %s",name,sname,tag,bot,oper)!=5)
	    if (sscanf(buff,"\"%[^\"]\" \"%[^\"]\" NULL %s %s",name,sname,bot,oper)==4)
	    {
		pfree(tag,len);
		tag = NULL;
	    }
	    else if (sscanf(buff,"\"%[^\"]\" NULL \"%[^\"]\" %s %s",name,tag,bot,oper)==4)
	    {
		pfree(sname,len);
		sname = NULL;
	    }
	    else if (sscanf(buff,"\"%[^\"]\" NULL NULL %s %s",name,bot,oper)==3)
	    {
		pfree(tag,len);
		pfree(sname,len);
		tag=sname = NULL;
	    }
	    else
	    {
		eventlog(eventlog_level_error,"channellist_load_permanent","malformed line %u in file \"%s\"",line,filename);
		pfree(oper,len);
		pfree(bot,len);
		pfree(tag,len);
		pfree(sname,len);
		pfree(name,len);
		pfree(buff,len);
		continue;
	    }
	
	if (strcasecmp(bot,"true")==0 ||
	    strcasecmp(bot,"yes")==0 ||
	    strcasecmp(bot,"on")==0 ||
	    strcmp(bot,"1")==0)
	    botflag = 1;
	else if (strcasecmp(bot,"false")==0 ||
		 strcasecmp(bot,"no")==0 ||
		 strcasecmp(bot,"off")==0 ||
		 strcmp(bot,"0")==0)
	    botflag = 0;
	else
	{
	    eventlog(eventlog_level_error,"channellist_load_permanent","invalid boolean value for field 4 on line %u",line);
	    pfree(oper,len);
	    pfree(bot,len);
	    pfree(tag,len);
	    pfree(sname,len);
	    pfree(name,len);
	    pfree(buff,len);
	    continue;
        }
	if (strcasecmp(oper,"true")==0 || strcasecmp(oper,"yes")==0 || strcmp(oper,"1")==0)
	    operflag = 1;
	else if (strcasecmp(oper,"false")==0 || strcasecmp(oper,"no")==0 || strcmp(oper,"0")==0)
	    operflag = 0;
	else
	{
	    eventlog(eventlog_level_error,"channellist_load_permanent","invalid boolean value for field 5 on line %u",line);
	    pfree(oper,len);
	    pfree(bot,len);
	    pfree(tag,len);
	    pfree(sname,len);
	    pfree(name,len);
	    pfree(buff,len);
	    continue;
        }
	
	pfree(oper,len);
	pfree(bot,len);
	
	channel_create(name,sname,tag,1,botflag,operflag);
	/* FIXME: call channel_delete() on current perm channels and do a
	   channellist_find_channel() and set the long name, perm flag, etc,
	   otherwise call channel_create(). This will make restarting handle
	   re-reading this file correctly. */
	
	if (tag)
	    pfree(tag,len);
	if (sname)
	    pfree(sname,len);
	pfree(name,len);
	pfree(buff,len);
    }
    
    fclose(fp);
    return 0;
}


extern void channellist_init(void)
{
    channellist_head = NULL;
    
    channellist_load_permanent(prefs_get_channelfile());
}


extern void channellist_unload(void)
{
    while (channellist_head)
	channel_destroy(channellist_head);
}


extern int channellist_get_length(void)
{
    t_channel const * curr;
    int               count=0;
    
    for (curr=channellist_head; curr; curr=curr->next)
	count++;
    
    return count;
}


extern t_channel * channellist_find_channel(char const * name)
{
    t_channel * temp;
    
    if (!name)
    {
	eventlog(eventlog_level_error,"channellist_find_channel","got NULL name");
	return NULL;
    }
    
    for (temp=channellist_head; temp; temp=temp->next)
    {
	if (!temp->name)
	{
	    eventlog(eventlog_level_error,"channellist_find_channel","found channel with NULL name");
	    continue;
	}
	if (strcasecmp(temp->name,name)==0 ||
	    (temp->shortname && strcasecmp(temp->shortname,name)==0))
	    return temp;
    }
    
    return NULL;
}


extern t_channel * channellist_get_first(void)
{
    channellist_curr = channellist_head;
    
    return channellist_curr;
}


extern t_channel * channellist_get_next(void)
{
    if (channellist_curr)
        channellist_curr = channellist_curr->next;
    
    return channellist_curr;
}
