/* Copyright (c) 1998 John E. Davis (davis@space.mit.edu)
 *
 * This file is part of slrn.
 *
 * Slrn is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * Slrn is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Slrn; see the file COPYING.  If not, write to the Free
 * Software Foundation, 59 Temple Place - Suite 330, 
 * Boston, MA  02111-1307, USA.
 */

/* MIME handling routines.
 *
 * Author: Michael Elkins <elkins@aero.org>
 * Modified by John E. Davis <davis@space.mit.edu>
 * 
 * Change Log:
 * Aug 20, 1997 patch from "Byrial Jensen" <byrial@post3.tele.dk>
 *   added.  Apparantly RFC2047 requires the whitespace separating
 *   multiple encoded words in headers to be ignored.
 *   Status: unchecked
 */

#include "config.h"
#include "slrnfeat.h"

#include <stdio.h>
#include <string.h>


#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include <ctype.h>

#if defined(__os2__) || defined(__NT__)
# include <process.h>
#endif

#include <slang.h>
#include "jdmacros.h"

#ifdef KANJI
#include "khenkan.h"
#endif
#include "server.h"
#include "slrn.h"
#include "misc.h"
#ifndef KANJI
#include "slrn.h"
#endif
#include "group.h"
#include "art.h"
#include "util.h"

#if SLRN_HAS_MIME
/* rest of file in this ifdef */

#include "mime.h"

int Slrn_Use_Mime = 1;
int Slrn_Use_Meta_Mail = 1;
char *Slrn_MetaMail_Cmd;

int Slrn_Mime_Was_Parsed;
int Slrn_Mime_Was_Modified;
int Slrn_Mime_Needs_Metamail;

char *Slrn_Mime_Display_Charset;

/* These are all supersets of US-ASCII.  Only the first N characters are 
 * matched, where N is the length of the table entry.
 */
static char *Compatable_Charsets[] =
{
   "US-ASCII",			       /* This MUST be zeroth element */
#ifdef KANJI
   "ISO-8859-1",
   "ISO-8859-2",
   "ISO-8859-3",
   "ISO-8859-4",
   "ISO-8859-5",
   "ISO-8859-6",
   "ISO-8859-7",
   "ISO-8859-8",
   "ISO-8859-9",
   "KOI8-R",
   "ISO-2022-JP",
   "ISO-2022-JP-2",
   "X-ISO-2022-JP-2",
   "Shift_JIS",
   "utf-8",
   "utf-7",
#else
   "ISO-8859-",
   "iso-latin1",		       /* knews adds his one */
   "KOI8-R",
#endif
   NULL
};

#ifdef KANJI
Charset_Type Article_Charset;
typedef struct {
  char *buf;
  int len;
} Mime_Field;
static Mime_Field Multipart_Boundary;
static Mime_Field Content_Type_String;
#else
static char *Char_Set;
#endif
static int Content_Type;
#define CONTENT_TYPE_TEXT		0x01
#define CONTENT_TYPE_MESSAGE		0x02
#define CONTENT_TYPE_MULTIPART		0x03
#define CONTENT_TYPE_UNSUPPORTED	0x10

static int Content_Subtype;
#define CONTENT_SUBTYPE_PLAIN		0x01
#ifdef KANJI
#define CONTENT_SUBTYPE_HTML		0x02
#define CONTENT_SUBTYPE_PARTIAL		0x03
#define CONTENT_SUBTYPE_UNKNOWN		0x04
#else
#define CONTENT_SUBTYPE_UNKNOWN		0x02
#endif
#define CONTENT_SUBTYPE_UNSUPPORTED	0x10

static int Encoding_Method;
#define ENCODED_7BIT			1
#define ENCODED_8BIT			2
#define ENCODED_QUOTED			3
#define ENCODED_BASE64			4
#define ENCODED_BINARY			5
#define ENCODED_UNSUPPORTED		6

#ifndef isalnum
#define isalnum(x) \
  ((((x) <= 'Z') && ((x) >= 'A')) \
   || (((x) <= 'z') && ((x) >= 'a')) \
   || (((x) <= '9') && ((x) >= '0')))
#endif

#ifdef KANJI
static Slrn_Article_Line_Type *find_header_line (char *header, Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end)
#else
static Slrn_Article_Line_Type *find_header_line (char *header)
#endif
{
#ifdef KANJI
   Slrn_Article_Line_Type *line = beg;
#else
   Slrn_Article_Line_Type *line = Slrn_Article_Lines;
#endif
   unsigned char ch = (unsigned char) UPPER_CASE(*header);
   unsigned int len = strlen (header);
   
#ifdef KANJI
   while ((line != NULL) && (line != end) && (line->flags & HEADER_LINE))
#else
   while ((line != NULL) && (line->flags & HEADER_LINE))
#endif
     {
	unsigned char ch1 = (unsigned char) *line->buf;
	if ((ch == UPPER_CASE(ch1))
	    && (0 == slrn_case_strncmp ((unsigned char *)header,
					(unsigned char *)line->buf,
					len)))
	  return line;
	line = line->next;
     }
   return NULL;
}


#ifdef KANJI
static int find_compatable_charset (char *cs, unsigned int len)
#else
static char *find_compatable_charset (char *cs, unsigned int len)
#endif
{
   char **compat_charset;
   
   compat_charset = Compatable_Charsets;
   while (*compat_charset != NULL)
     {
#ifdef KANJI
	if ((0 == slrn_case_strncmp ((unsigned char *) cs,
				     (unsigned char *) *compat_charset,
				     len))
	    && (len == strlen(*compat_charset)))
	  return (compat_charset - Compatable_Charsets);
#else
	unsigned int len1;
	
	len1 = strlen (*compat_charset);
	if (len1 <= len) 
	  {
	     if (0 == slrn_case_strncmp ((unsigned char *) cs,
					 (unsigned char *) *compat_charset,
					 len1))
	       return *compat_charset;
	  }
#endif
	compat_charset++;
     }
#ifdef KANJI
   return -1;
#else
   return NULL;
#endif
}

#ifdef KANJI
Charset_Type slrn_find_charset(char *s)
{
  int i;

  i = find_compatable_charset(s, strlen(s));
  if (i == -1) {
    i = 0;
  }
  return i;
}

static char *MIME_Field;
static int MIME_Field_Length;

static int search_field(char *field, Slrn_Article_Line_Type *line, char *b) {
  int len = strlen(field);

  MIME_Field = NULL;
  MIME_Field_Length = 0;

  do {
    while (b == line->buf || NULL != (b = slrn_strchr (b, ';')))
      {
	b = slrn_skip_whitespace (b + 1);

	if (0 != slrn_case_strncmp ((unsigned char *)b,
				    (unsigned char *)field,
				    len))
	  continue;

	b = slrn_skip_whitespace (b + len);
	while (*b == 0) {
	  line = line->next;
	  if ((line == NULL)
	      || ((line->flags & HEADER_LINE) == 0)
	      || ((*(b = line->buf) != ' ') && (*b == '\t')))
	    return 0;
	  b = slrn_skip_whitespace (b);
	}

	if (*b != '=') continue;
	b++;
	if (*b == '"') b++;
	MIME_Field = b;
	while (*b && (*b != ';')
	       && /* (*b != ' ') && */ (*b != '\t') && (*b != '\n')
	       && (*b != '"'))
	  b++;
	MIME_Field_Length = b - MIME_Field;
	return MIME_Field_Length;
      }
    line = line->next;
  } while ((line != NULL)
	   && (line->flags & HEADER_LINE)
	   && ((*(b = line->buf) == ' ') || (*b == '\t')));
  return 0;
}

static int parse_content_type_line (Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end)
#else
static int parse_content_type_line (void)
#endif
{
   Slrn_Article_Line_Type *line;
   char *b;
#ifdef KANJI
   char *sep;
#endif
   
   /* Use default: text/plain; charset=us-ascii */
   Content_Type = CONTENT_TYPE_TEXT;
   Content_Subtype = CONTENT_SUBTYPE_PLAIN;
#ifndef KANJI
   Char_Set = Compatable_Charsets[0];
#endif
   
#ifdef KANJI
   if (NULL == (line = find_header_line ("Content-Type:", beg, end)))
#else
   if (NULL == (line = find_header_line ("Content-Type:")))
#endif
     return 0;
   
   b = slrn_skip_whitespace (line->buf + 13);
#ifdef KANJI
   Content_Type_String.buf = b;
   sep = slrn_strchr (b, ';');
   if (sep != NULL) {
     Content_Type_String.len = sep - b;
   } else { /* "Content-Type:" not terminated ';' */
     if (slrn_strchr (b, ' ') == NULL) {
       Content_Type_String.len = strlen(b);
     } else {
       Content_Type_String.len = 0;
     }
   }
#endif
   
   if (0 == slrn_case_strncmp ((unsigned char *)b,
			       (unsigned char *) "text/",
			       5))
     {
	b += 5;
	if (0 != slrn_case_strncmp ((unsigned char *)b,
				    (unsigned char *) "plain",
				    5))
	  {
#ifdef KANJI
	    if (0 != slrn_case_strncmp ((unsigned char *)b,
					(unsigned char *) "html",
					4))
	      {
		Content_Subtype = CONTENT_SUBTYPE_UNKNOWN;
	      } else {
		Content_Subtype = CONTENT_SUBTYPE_HTML;
		return 0;
	      }
#else
	     Content_Subtype = CONTENT_SUBTYPE_UNSUPPORTED;
	     return -1;
#endif
	  }
	b += 5;
     }
   else if (0 == slrn_case_strncmp ((unsigned char *)b,
				    (unsigned char *) "message/",
				    5))
     {
	Content_Type = CONTENT_TYPE_MESSAGE;
#ifndef KANJI
	Content_Subtype = CONTENT_SUBTYPE_UNKNOWN;
#endif
	b += 8;
#ifdef KANJI
	if (0 != slrn_case_strncmp ((unsigned char *)b,
				    (unsigned char *) "partial",
				    7))
	  {
	    Content_Subtype = CONTENT_SUBTYPE_UNKNOWN;
	  } else {
	    Content_Subtype = CONTENT_SUBTYPE_PARTIAL;
	    Multipart_Boundary.len = search_field("id", line, b);
	    if (Multipart_Boundary.len > 0) {
	      Multipart_Boundary.buf = MIME_Field;
	      return 0;
	    } else {
	      return -1;
	    }
	  }
#endif
     }
   else if (0 == slrn_case_strncmp ((unsigned char *)b,
				    (unsigned char *) "multipart/",
				    5))
     {
	Content_Type = CONTENT_TYPE_MULTIPART;
	Content_Subtype = CONTENT_SUBTYPE_UNKNOWN;
	b += 10;
#ifdef KANJI
	Multipart_Boundary.len = search_field("boundary", line, b);
	if (Multipart_Boundary.len > 0) {
	  Multipart_Boundary.buf = MIME_Field;
	  return 0;
	} else {
	  return -1;
	}
#endif
     }
   else
     {
	Content_Type = CONTENT_TYPE_UNSUPPORTED;
#ifdef KANJI
	Content_Subtype = CONTENT_SUBTYPE_UNSUPPORTED;
#endif
	return -1;
     }
   
#ifdef KANJI
   Article_Charset = USASCII;
   if (search_field("charset", line, b)) {
     Article_Charset = find_compatable_charset (MIME_Field, MIME_Field_Length);
     if (Article_Charset == -1) {
       Article_Charset = USASCII;
       /* return -1; */ /* window-1250 ʤѤʥ󥳡ǥ󥰤Ǥ̤ */
     }
   }
#else
   do
     {
	while (NULL != (b = slrn_strchr (b, ';')))
	  {
	     char *charset;
	     unsigned int len;
	     
	     b = slrn_skip_whitespace (b + 1);
	     
	     if (0 != slrn_case_strncmp ((unsigned char *)b,
					 (unsigned char *)"charset",
					 7))
	       continue;
	     
	     b = slrn_skip_whitespace (b + 7);
	     while (*b == 0)
	       {
		  line = line->next;
		  if ((line == NULL)
		      || ((line->flags & HEADER_LINE) == 0)
		      || ((*(b = line->buf) != ' ') && (*b == '\t')))
		    return -1;
		  b = slrn_skip_whitespace (b);
	       }
	     
	     if (*b != '=') continue;
	     b++;
	     if (*b == '"') b++;
	     charset = b;
	     while (*b && (*b != ';')
		    && (*b != ' ') && (*b != '\t') && (*b != '\n')
		    && (*b != '"'))
	       b++;
	     len = b - charset;
	     
	     Char_Set = find_compatable_charset (charset, len);
	     if (Char_Set == NULL) return -1;
	     return 0;
	  }
	line = line->next;
     }
   while ((line != NULL)
	  && (line->flags & HEADER_LINE)
	  && ((*(b = line->buf) == ' ') || (*b == '\t')));
   
#endif
   return 0;
}

#ifdef KANJI
static int parse_content_transfer_encoding_line (Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end)
#else
static int parse_content_transfer_encoding_line (void)
#endif
{
   Slrn_Article_Line_Type *line;
   unsigned char *buf;
   
   Encoding_Method = ENCODED_7BIT;
#ifdef KANJI
   line = find_header_line ("Content-Transfer-Encoding:", beg, end);
#else
   line = find_header_line ("Content-Transfer-Encoding:");
#endif
   if (line == NULL) return 0;
   
   buf = (unsigned char *) slrn_skip_whitespace (line->buf + 26);
   if (*buf == '"') buf++;
   
   if (0 == slrn_case_strncmp (buf, (unsigned char *) "7bit", 4))
     Encoding_Method = ENCODED_7BIT;
   else if (0 == slrn_case_strncmp (buf, (unsigned char *) "8bit", 4))
     Encoding_Method = ENCODED_8BIT;
   else if (0 == slrn_case_strncmp (buf, (unsigned char *) "base64", 6))
     Encoding_Method = ENCODED_BASE64;
   else if (0 == slrn_case_strncmp (buf, (unsigned char *) "quoted-printable", 16))
     Encoding_Method = ENCODED_QUOTED;
   else if (0 == slrn_case_strncmp (buf, (unsigned char *) "binary", 6))
     Encoding_Method = ENCODED_BINARY;
   else
     {
	Encoding_Method = ENCODED_UNSUPPORTED;
	return -1;
     }
   return 0;
}

static int Index_Hex[128] =
{
   -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     0, 1, 2, 3,  4, 5, 6, 7,  8, 9,-1,-1, -1,-1,-1,-1,
     -1,10,11,12, 13,14,15,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     -1,10,11,12, 13,14,15,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1
};
#define HEX(c) (Index_Hex[(unsigned char)(c) & 0x7F])

static char *decode_quoted_printable (char *dest,
				      char *src, char *srcmax,
				      int treat_underscore_as_space)
{
   char *allowed_in_qp = "0123456789ABCDEFabcdef";
   char ch;
   while (src < srcmax)
     {
	ch = *src++;
	if ((ch == '=') && (src + 1 < srcmax)
	    && (NULL != slrn_strchr (allowed_in_qp, src[0]))
	    && (NULL != slrn_strchr (allowed_in_qp, src[1])))
	  {
	     *dest++ = (16 * HEX(src[0])) + HEX(src[1]);
	     src += 2;
	  }
	else if ((ch == '_') && treat_underscore_as_space)
	  {
	     *dest++ = ' ';
	  }
	else *dest++ = ch;
     }
   return dest;
}

static int Index_64[128] =
{
   -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
     -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
     52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,-1,-1,-1,
     -1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
     15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
     -1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
     41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1
};

#define BASE64(c) (Index_64[(unsigned char)(c) & 0x7F])

static char *decode_base64 (char *dest, char *src, char *srcmax)
{
   while (src + 3 < srcmax)
     {
	*dest++ = (BASE64(src[0]) << 2) | (BASE64(src[1]) >> 4);
	
	if (src[2] == '=') break;
	*dest++ = ((BASE64(src[1]) & 0xf) << 4) | (BASE64(src[2]) >> 2);
	
	if (src[3] == '=') break;
	*dest++ = ((BASE64(src[2]) & 0x3) << 6) | BASE64(src[3]);
	src += 4;
     }
   return dest;
}

int slrn_rfc1522_decode_string (char *s)
{
   char *s1, *s2, ch;
   char *charset, method, *txt;
   char *after_last_encoded_word;
   char *after_whitespace;
   unsigned int count;
   unsigned int len;

   count = 0;
   after_whitespace = NULL;
   after_last_encoded_word = NULL;

   while (1)
     {
	while ((NULL != (s = slrn_strchr (s, '=')))
	       && (s[1] != '?')) s++;
	if (s == NULL) break;
	
	s1 = s;
	charset = s = s1 + 2;
	while (((ch = *s) != 0)
	       && (ch != '?') && (ch != ' ') && (ch != '\t') && (ch != '\n'))
	  s++;
	
	if (ch != '?')
	  {
	     s = s1 + 2;
	     continue;
	  }
	
	charset = s1 + 2;
	len = s - charset;
	
#ifdef KANJI
	Article_Charset = find_compatable_charset (charset, len);
	if (Article_Charset == -1) {
	  Article_Charset = USASCII;
	  charset = NULL;
	}
#else
	charset = find_compatable_charset (charset, len);
#endif
	s++;			       /* skip ? */
	method = *s++;		       /* skip B,Q */
	method = UPPER_CASE(method);
	
	if ((charset == NULL) || ((method != 'B') && (method != 'Q'))
	    || (*s != '?'))
	  {
	     s = s1 + 2;
	     continue;
	  }
	
	/* Now look for the final ?= after encoded test */
	s++;			       /* skip ? */
	txt = s;
       
	while ((ch = *s) != 0)
	  {
	     /* Appararantly some programs do not convert ' ' to '_' in 
	      * quoted printable.  Sigh.
	      */
	     if (((ch == ' ') && (method != 'Q'))
		 || (ch == '\t') || (ch == '\n'))
	       break;
	     if ((ch == '?') && (s[1] == '='))
	       break;
	     
	     s++;
	  }
	
	if ((ch != '?') || (s[1] != '='))
	  {
	     s = s1 + 2;
	     continue;
	  }
	
	if (s1 == after_whitespace)
	  s1 = after_last_encoded_word;

	/* Note: these functions return a pointer to the END of the decoded
	 * text.
	 */
#ifdef KANJI
	s2 = s1;
#endif
	if (method == 'B')
	  s1 = decode_base64 (s1, txt, s);
	else s1 = decode_quoted_printable (s1, txt, s, 1);
	
	/* Now move everything over */
#ifdef KANJI
	/* ǥɤƲԤʤɤޤޤƤ϶֤ */
	while(s2 < s1) {
	  ch = *s2;
	  if (ch == '\t' || ch == '\012' || ch == '\015') {
	    *s2 = ' ';
	  }
	  s2++;
	}
	s1 = s2;

	/* mime֤ζʸ̵ */
	s2 = slrn_skip_whitespace (s + 2);
	/* ʬmime encodeɤå */
	if (*s2 != '=' || *(s2 + 1) != '?')
#endif
	s2 = s + 2;		       /* skip final ?= */
	s = s1;			       /* start from here next loop */
	while ((ch = *s2++) != 0) *s1++ = ch;
	*s1 = 0;
	
	count++;
	
	after_last_encoded_word = s;
	s = slrn_skip_whitespace (s);
	after_whitespace = s;
     }
   return count;
}





#ifdef KANJI
static void rfc1522_decode_headers (Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end)
#else
static void rfc1522_decode_headers (void)
#endif
{
#ifdef KANJI
   Slrn_Article_Line_Type *line = beg;
#else
   Slrn_Article_Line_Type *line = Slrn_Article_Lines;
#endif
   
#ifdef KANJI
   while ((line != NULL) && (line != end) && (line->flags & HEADER_LINE))
#else
   while ((line != NULL) && (line->flags & HEADER_LINE))
#endif
     {
	if (slrn_rfc1522_decode_string (line->buf))
	  {
	     Slrn_Mime_Was_Modified = 1;
	  }
	line = line->next;
     }
}


#ifdef KANJI
static void decode_mime_base64 (Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end)
#else
static void decode_mime_base64 (void)
#endif
{
#ifdef KANJI
   Slrn_Article_Line_Type *line = beg, *new_l;
   int continued = 0, decoded = 0;
   
   while (line != NULL && line != end) {
     char *b, ch;
     unsigned int len;

     if ((line->flags & HEADER_LINE) || (line->buf[0] == '\0')) {
       line = line->next;               /* skip header and separator */
       continue;
     }

     /* decode base64 */
     if (!decoded) {
       b = line->buf;
       b = decode_base64 (b, b, b + strlen(b));
       if (b != NULL) {
	 *b = '\0';
       } else {
	 return;
       }
     }
     /* join with previous line */
     if (continued) {
       b = line->prev->buf;
       len = strlen(b);
       b = slrn_realloc(b, len + strlen(line->buf) + 1, 1);
       if (b == NULL) {
	 return;
       }
       strcpy(b + len, line->buf);
       new_l = line->prev;
       new_l->next = line->next;
       if (line->next != NULL) {
	 line->next->prev = new_l;
       }
       slrn_free(line->buf);
       slrn_free((char *)line);
       line = new_l;
       line->buf = b;
       continued = 0;
     }

     decoded = 0;
     b = line->buf;
     while((ch = *b) != '\012' && ch != '\015' && ch != '\0') b++;
     if (ch != '\0') { /* \012, \015, \015\012 Τ줫λ */
       if (ch == '\015' && *(b + 1)  == '\012') {
	 *b++ = '\0';
       }
       *b++ = '\0';
       if (*b != '\0') {
	 /* split line */
	 new_l = (Slrn_Article_Line_Type *) slrn_malloc(sizeof (Slrn_Article_Line_Type), 0, 1);
	 if (new_l == NULL)
	   return;
	 new_l->buf = slrn_safe_strmalloc(b);

	 new_l->next = line->next;
	 if (new_l->next != NULL) {
	   new_l->next->prev = new_l;
	 }
	 line->next = new_l;
	 new_l->prev = line;
	 new_l->flags = line->flags;
	 decoded = 1;
       }
     } else {
       continued = 1;
     }
     line = line->next;
   }
   Slrn_Mime_Was_Modified = 1;
#else
   Slrn_Mime_Needs_Metamail = 1;
#endif
}

/* This function checks if the last character on curr_line is an = and 
 * if it is, then it merges curr_line and curr_line->next. See RFC1341,
 * section 5.1 (Quoted-Printable Content-Transfer-Encoding) rule #5.
 * [csp@ohm.york.ac.uk]
 */
static int merge_if_soft_linebreak (Slrn_Article_Line_Type *curr_line)
{
   Slrn_Article_Line_Type *next_line;

   while ((next_line = curr_line->next) != NULL)
     {
	char *b = curr_line->buf;
	unsigned int len;

	len = strlen (b);
	if (len == 0) return 0;
   
	len--;
	if (b[len] != '=') return 0;
	
	/* Remove the excess = character... */
	b[len] = '\0';
	
#ifdef KANJI
	b = slrn_realloc(b, 1 + len + strlen (next_line->buf), 1);
	if (b == NULL)
#else
	if (NULL == (b = (char *) SLREALLOC (b, 1 + len + strlen (next_line->buf))))
#endif
	  return -1;

	curr_line->buf = b;
	
	strcpy (b + len, next_line->buf);
	
	/* Unlink next_line from the linked list of lines in the article... */
	curr_line->next = next_line->next;
	if (next_line->next != NULL)
	  next_line->next->prev = curr_line;
	
	SLFREE (next_line->buf);
	SLFREE (next_line);
     }
   return 0;
}

#ifdef KANJI
static void decode_mime_quoted_printable (Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end)
#else
static void decode_mime_quoted_printable (void)
#endif
{
#ifdef KANJI
   Slrn_Article_Line_Type *line = beg;
#else
   Slrn_Article_Line_Type *line = Slrn_Article_Lines;
#endif
   
   /* skip to body */
   while ((line != NULL) && (line->flags & HEADER_LINE))
     line = line->next;
   if (line == NULL) return;
   
#ifdef KANJI
   while (line != NULL && line != end)
#else
   while (line != NULL)
#endif
     {
	char *b;
	unsigned int len;
	
	b = line->buf;
	len = strlen (b);
	if (len && (b[len - 1] == '=') 
	    && (line->next != NULL))
	  {
	     (void) merge_if_soft_linebreak (line);
	     b = line->buf;
	     len = strlen (b);
	  }

	b = decode_quoted_printable (b, b, b + len, 0);
	if (b < line->buf + len)
	  {
	     *b = 0;
	     Slrn_Mime_Was_Modified = 1;
	  }
	
	line = line->next;
     }
}



void slrn_mime_article_init (void)
{
   Slrn_Mime_Was_Modified = 0;
   Slrn_Mime_Was_Parsed = 0;
   Slrn_Mime_Needs_Metamail = 0;
}

#ifdef KANJI
int Slrn_Is_Content_Subtype_Partial() {
  return(Content_Subtype == CONTENT_SUBTYPE_PARTIAL);
}

static int Section[8];  /* multipart section number */
static Slrn_Article_Line_Type *set_boundary(Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end, Mime_Field boundary, int level) {
  Slrn_Article_Line_Type *line = beg;
  char *b;
  int body = 0;

  if (line == NULL || line == end
      || (line->buf != NULL && line->buf[0] == '\0')) {
    return (NULL);
  }

  while((line != NULL) && (line != end)) {
    b = line->buf;
    if (!body) {
      line->flags |= HEADER_LINE;
      if (Section[level] == 0 && level > 0) { /* nested multipart */
	line->flags |= HIDDEN_LINE;
      }
      if (*b == '\0') { /* end of header */
	body = 1;
	if (Section[level] == 0 && level > 0) {
	  line->flags &= ~HIDDEN_LINE;
	}
      }
    } else {
      if (Section[level] == 0) { /* hide comments "This is a multi...." */
	line->flags |= HIDDEN_LINE;
      }
      if (strncmp(b, "--", 2) == 0
	  && strncmp(b + 2, boundary.buf, boundary.len) == 0) {
	int len = strlen(b + 2);
	if (len == boundary.len
	    || (len == boundary.len + 2 && strncmp(b + len, "--", 2) == 0)) {
	  return (line);
	}
      }
    }
    if (line->next == NULL && body && beg->prev != NULL) {
      b = beg->prev->buf;
      SLFREE(line->buf);
      line->buf = slrn_safe_strmalloc(b);
      return (line);
    }
    line = line->next;
  }
  return (NULL);
}

static void write_boundary(Slrn_Article_Line_Type *bound, int level) {
  char *b;                       /* level: -1 if last boundary line */
  int len, i, j = 2;

  b = bound->buf;
  len = strlen(b);
  if (level >= 0) { /* not last boundary line */
    b[j] = '[';
    j++;
    for(i = 0; i <= level; i++) {
      sprintf(b + j, "%d.", Section[i]);
      j = strlen(b);
    }
    if (len < j + Content_Type_String.len + 2) {
      len = j + Content_Type_String.len + 10;
      b = slrn_realloc(b, len + 1, 1);
      bound->buf = b;
      b[len] = '\0';
    }
    strncpy(b + j, Content_Type_String.buf, Content_Type_String.len);
    j += Content_Type_String.len;
    b[j] = ']';
    j++;
  }
  for(i = j; i < len; i++) {
    b[i] = '-';
  }
  bound->flags |= HEADER_LINE;
  bound->flags &= ~HIDDEN_LINE;
}

static void hide_body(Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end) {
  Slrn_Article_Line_Type *line = beg->prev;

  while((line = line->next) != NULL && line != end) {
    if (line->flags & HEADER_LINE) {
      if (Content_Type != CONTENT_TYPE_TEXT
	  && Content_Type != CONTENT_TYPE_MESSAGE) {
	continue;
      }
    } else if (Content_Type == CONTENT_TYPE_TEXT
	       || Content_Type == CONTENT_TYPE_MESSAGE) {
      continue;
    }
    line->flags |= HIDDEN_LINE;    
  }
}

static void mime_process_single_part (Slrn_Article_Line_Type *beg, Slrn_Article_Line_Type *end, int level) {

  rfc1522_decode_headers (beg, end);

  Article_Charset = USASCII;
  if ((-1 == parse_content_type_line (beg, end))
      || (-1 == parse_content_transfer_encoding_line (beg, end)))
#else
void slrn_mime_process_article (void)
{
   if ((Slrn_Use_Mime == 0)
       || Slrn_Mime_Was_Parsed
       || (Slrn_Article_Lines == NULL))
     return;
   
   Slrn_Mime_Was_Parsed = 1;	       /* or will be */
   
   rfc1522_decode_headers ();

   if (NULL == find_header_line ("Mime-Version:")) return;
   if ((-1 == parse_content_type_line ())
       || (-1 == parse_content_transfer_encoding_line ()))
#endif
     {
	Slrn_Mime_Needs_Metamail = 1;
	return;
     }
   
#ifdef KANJI
  if (Content_Type == CONTENT_TYPE_MULTIPART) { /* process multipart */
    Slrn_Article_Line_Type *newend, *bline = NULL;
    Mime_Field boundary = Multipart_Boundary;
    Mime_Field type_string = Content_Type_String;
    int type = Content_Type;
    int subtype = Content_Subtype;

    if (level > 7) {
      level = 7;
    }
    Section[level] = 0;
    while((newend = set_boundary(beg, end, boundary, level)) != NULL) {
      if (Section[level] > 0) {
	mime_process_single_part(beg, newend, level + 1);
	write_boundary(bline, level);
	if (Content_Type != CONTENT_TYPE_MULTIPART) {
	  hide_body(beg, newend);
	}
      }
      Section[level]++;
      bline = newend;
      beg = newend->next;
    }
    write_boundary(bline, -1);  /* write last boundary line */
    Content_Type = type;
    Content_Subtype = subtype;
    Content_Type_String = type_string;
    return;
  }
#ifdef PARTIAL /* not completed yet */
  if (Content_Subtype == CONTENT_SUBTYPE_PARTIAL) { /* partial message */
    slrn_mime_retrieve_partial_messages();
    return;
  }
#endif
#endif
   switch (Encoding_Method)
     {
      case ENCODED_7BIT:
      case ENCODED_8BIT:
      case ENCODED_BINARY:
	/* Already done. */
	return;
	
      case ENCODED_BASE64:
#ifdef KANJI
	decode_mime_base64 (beg, end);
#else
	decode_mime_base64 ();
#endif
	break;
	
      case ENCODED_QUOTED:
#ifdef KANJI
	decode_mime_quoted_printable (beg, end);
#else
	decode_mime_quoted_printable ();
#endif
	break;
	
      default:
	Slrn_Mime_Needs_Metamail = 1;
	return;
     }
#ifdef KANJI
   if (Article_Charset == UTF8 && level > 0
       && Content_Type == CONTENT_TYPE_TEXT) {
     while (beg != end) {
       slrn_convert_utf82int((unsigned char *)beg->buf);
       beg = beg->next;
     }
     Article_Charset = ISO2022JP;
   }
#endif
}

#ifdef KANJI
void slrn_mime_process_article (void)
{
   if ((Slrn_Use_Mime == 0)
       || Slrn_Mime_Was_Parsed
       || (Slrn_Article_Lines == NULL))
     return;
   
   Slrn_Mime_Was_Parsed = 1;	       /* or will be */
   
   /*  Mime-Version إåʤ MIME ̵ޤˤĤƤʤơ
      Content-Type إåĤƤΤΤǤߺѤ */
   if (NULL == find_header_line ("Mime-Version:", Slrn_Article_Lines, NULL)
       && NULL == find_header_line ("Content-Type:", Slrn_Article_Lines, NULL)) {
     rfc1522_decode_headers (Slrn_Article_Lines, NULL);
     return;
   }
   mime_process_single_part(Slrn_Article_Lines, NULL, 0);
}
#endif
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

int slrn_mime_call_metamail (void)
{
#ifdef VMS
   return 0;
#else
   int init = Slrn_TT_Initialized;
   char cmd [2 * SLRN_MAX_PATH_LEN];
   char tempfile [SLRN_MAX_PATH_LEN];
   Slrn_Article_Line_Type *ptr = Slrn_Article_Lines;
   FILE *fp;
   char *tmp, *mm;
   
   if ((Slrn_Use_Meta_Mail == 0)
       || Slrn_Batch
#ifdef KANJI
       || (slrn_get_yesno (1, Slrn_Japanese_Messages
			   ? "MIMEmetamailǽޤ"
			   : "Process this MIME article with metamail") <= 0))
#else
       || (slrn_get_yesno (1, "Process this MIME article with metamail") <= 0))
#endif
     return 0;

# if defined(__os2__) || defined(__NT__)
   if (NULL == (tmp = getenv ("TMP")))
     tmp = ".";
# else
   tmp = "/tmp";
# endif
   
   fp = slrn_open_tmpfile_in_dir (tmp, tempfile, "w");

   if (fp == NULL)
     {
	slrn_error ("Unable to open tmp file for metamail.");
	return 0;
     }
   
#ifdef KANJI
     fputs(Slrn_Orig_Article, fp);
#else
   while (ptr) 
     {
	fputs(ptr->buf, fp);
	putc('\n', fp);
	ptr = ptr->next;
     }
#endif
   slrn_fclose(fp);

   mm = Slrn_MetaMail_Cmd;
   
   if ((mm == NULL)
       || (*mm == 0)
       || (strlen (mm) > SLRN_MAX_PATH_LEN))
     mm = "metamail";

   sprintf (cmd, "%s %s", mm, tempfile);
   
   /* Make sure that metamail has a normal environment */
   slrn_set_display_state (0);
   
   slrn_posix_system(cmd, 0);
   slrn_delete_file (tempfile);
   
   printf("Press return to continue ...");
   getchar();
   fflush(stdin); /* get rid of any pending input! */
   
   slrn_set_display_state (init);
   return 1;
#endif  /* NOT VMS */
}


/* -------------------------------------------------------------------------
 * MIME encoding routines.
 * -------------------------------------------------------------------------*/

static char *Mime_Posting_Charset;
static int Mime_Posting_Encoding;

int slrn_mime_scan_file (FILE *fp)
{
   /* This routine scans the article to determine what CTE should be used */
   unsigned int linelen = 0;
   unsigned int maxlinelen = 0;
   int ch;
   int cr = 0;
   unsigned int hibin = 0;
   
   
   /* Skip the header.  8-bit characters in the header are taken care of
    * elsewhere since they ALWAYS need to be encoded.
    */
   while ((ch = getc(fp)) != EOF)
     {
	if (ch == '\n')
	  {
	     ch = getc(fp);
	     if (ch == '\n')
	       break;
	  }
     }
   
   if (ch == EOF)
     {
	rewind (fp);
	return -1;
     }
   
   while ((ch = getc(fp)) != EOF)
     {
	linelen++;
	if (ch & 0x80) hibin = 1; /* 8-bit character */
	
	if (ch == '\n')
	  {
	     if (linelen > maxlinelen)	maxlinelen = linelen;
	     linelen = 0;
	  }
	else if (((unsigned char)ch < 32) && (ch != '\t') && (ch != 0xC))
	  cr = 1;		       /* not tab or formfeed */
     }
   if (linelen > maxlinelen) maxlinelen = linelen;
   
   if (hibin > 0)
     {
	/* 8-bit data.  US-ASCII is NOT a valid charset, so use ISO-8859-1 */
	if (slrn_case_strcmp((unsigned char *)"us-ascii",
			     (unsigned char *)Slrn_Mime_Display_Charset) == 0)
	  Mime_Posting_Charset = "iso-8859-1";
	else
	  Mime_Posting_Charset = Slrn_Mime_Display_Charset;
     }
#ifdef KANJI
   else if (!cr &&
	    -1 != find_compatable_charset (Slrn_Mime_Display_Charset,
					   strlen (Slrn_Mime_Display_Charset)))
#else
   else if (NULL != find_compatable_charset (Slrn_Mime_Display_Charset,
					     strlen (Slrn_Mime_Display_Charset)))
#endif
     /* 7-bit data.  Check to make sure that this display supports US-ASCII */
     Mime_Posting_Charset = "us-ascii";
   else
     Mime_Posting_Charset = Slrn_Mime_Display_Charset;

#if 0
   if ((maxlinelen > 990) || (cr > 0))
     {
	Mime_Posting_Encoding = ENCODED_QUOTED;
     }
   else
#endif
     if (hibin > 0)
     Mime_Posting_Encoding = ENCODED_8BIT;
   else
     Mime_Posting_Encoding = ENCODED_7BIT;
   
   rewind(fp);
   return 0;
}

#define IS_RFC850_SPECIAL(c) \
 (((c) == '(') || ((c) == ')') || ((c) == '<') || ((c) == '>') || ((c) == '"'))

#ifdef KANJI
/*
 * For slrn_convert_utf72int()
 *   utf-7 ǻѤƤ Modified MIME base64 Ѵ
 */
int slrn_base64(int c)
{
  return BASE64(c);
}

/* 
 * slrnΥꥸʥmime encoderϡѥ󥳡ǥ󥰤
 * ݡȤƤʤܤbase64̤ʤɲá
 */
static char basis_64[] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
/* src must less than srcmax */
static unsigned char *encode_base64(unsigned char *dest,
				    unsigned char *src, unsigned char *srcmax)
{
  int c1, c2, c3, pad = 0;

  while (src < srcmax) {
    c1 = *src++;
    *dest++ = basis_64[c1>>2];
    if (src < srcmax) {
      c2 = *src++;
    } else {
      c2 = 0;
      pad = 1;
    }
    *dest++ = basis_64[((c1 & 0x3)<< 4) | ((c2 & 0xf0) >> 4)];
    if (src < srcmax) {
      c3 = *src++;
    } else {
      c3 = 0;
      pad++;
    }
    if (pad < 2) {
      *dest++ = basis_64[((c2 & 0xf) << 2) | ((c3 & 0xc0) >>6)];
    } else {
      *dest++ = '=';
    }
    if (pad == 0) {
      *dest++ = basis_64[c3 & 0x3f];
    } else {
      *dest++ = '=';
    }
  }
  return (dest);
}

/* 
 * output64chunkto64Ϻƣ@mimekit1.1Υɤ
 * ͤˤƤޤΥ󥳡Τϡڡ
 * θƤʤƤʤΤǤ
 *
 * RFC2047 ˸̩бˤϡ
 * 1. "a" ξ硢"" Τߤ encode  decode  " a" Ȥʤä
 *    ޤ-> "a" ޤƤ٤ encode 뤷ʤ
 *
 * 2. " " ξ硢""  "" ̡ encode ȡdecode 
 *    "" ȤʤäƤޤ-> space ޤư encode 뤫""
 *      "" Τɤ餫 space ޤ롣
 *
 * 3. encoded word  space ǤʤФʤʤĤޤꡢܸ
 * äĤƤȾʸ⤹٤ƤҤȤĤ encoded word ˤʤФ
 * ʤǤ encode ʸβΤǡ
 *  encode ¸ߤ뤬RFC2047 ˸̩ǤȤϸʤ
 *
 * : 2. space ޤư encode ñζڤ꤬ʤʤä
 * ޤΤǡRFC ˸̩ǤȤϸʤɤĹʤ 
 * folding Ƥޤʬ encode ΤĹʤäƤ
 * ޤΤǡǤϰ encode 뤳Ȥˤ롣
 *
 */
/* 르ꥺ
 * 1. ޤܸλϤޤ ESC sequence õ
   2. ޤǸ롣
   3. ܸν ESC sequence õ
   4. ǡμܸޤʤСܸν ESC sequence
      õ
   5. 4 򷫤֤ơμʸܸޤޤʤõ
   6. 2 ΰ֤鸽Ϥޤ base64 󥳡ɤ롣

    75 ʸʲ folding Τȡstructured field бʤ
   Фʤʤ
 */
/* only for ISO-2022-JP */
int Slrn_Mime_Strict_Rfc = 0; /* RFC2047 ˸̩ encode Ԥ(
				 ASCII ʸ encode Ƥޤ) */
#define IS_TO_KANJI(p) (*(p) == ESC && *((p) + 1) == '$' \
			&& (*((p) + 2) == 'B' || *((p) + 2) == '@'))
#define IS_TO_ASCII(p) (*(p) == ESC && *((p) + 1) == '(' \
			&& (*((p) + 2) == 'B' || *((p) + 2) == 'J'))
static int encode_base64_header(unsigned char *d, unsigned char *dmax,
				unsigned char *s, char *charset,
				unsigned int c_len, int ignore_specials)
{
  char *p, ch, *beg, *end, *to_ascii, *to_kanji, *beg_line;
  int len = 0, count, beg_kanji, end_kanji;

  /* check if buffer 'd' has a enough length */
  if (dmax - d < strlen((char *)s) * 4 / 3) return (-1);

  p = slrn_strchr((char *)s, ':');  /* skip header name */
  if (p != NULL) {
    while(*(++p) != ' '); /* skip next space */
    p--;
    beg = (char *)s;
    while(beg != p && d < dmax) { /* copy header */
      *d++ = *beg++;
    }
  } else {
    p = (char *)s;
  }

  while(*p != '\0' && d < dmax) {
    beg_line = beg = p;
    while (((ch = *p) != 0) && d < dmax && !IS_TO_KANJI(p)
	   && (len < 76 || beg != beg_line)) {
      *d++ = ch;
      p++;
      len++;
      if (ch == ' ' || (ignore_specials == 0 && IS_RFC850_SPECIAL(ch))) {
	beg = p;   /* remember the beginning of the current word */
      }
    }
    if (ch == 0) break;
    if (d == dmax) return (-1);

    if (len > 75) { /* line break(in text) */
      p = beg;  /* search previous word's end */
      while(*(beg - 1) == ' ') beg--; /* skip white space backward */
      d -= p - beg;
      *d++ = '\n';
      *d++ = ' ';
      len = 0;
      continue;
    }

    /* found to_kanji sequence and the beginning of encoding */
    to_kanji = p;
    if (Slrn_Mime_Strict_Rfc) {
      d -= p - beg;
      len -= p - beg;
      p = beg;
    } else {
      beg = p;
    }

    /* search to_ascii sequence and the end of encoding */
    while((ch = *p) != 0) {
      if (IS_TO_ASCII(p)) {            /* skip to space or end of line */
	to_ascii = p;
	p += 3;
	if (!Slrn_Mime_Strict_Rfc) {
	  end = p;
	  break;
	}
	while((ch = *p) != 0 && ch != ' ' && ch != '\n'
	      && (ignore_specials || !IS_RFC850_SPECIAL(ch))) p++;
	end = p;
	if (ch == 0) break;
	p = slrn_skip_whitespace(p);
	while((ch = *p) != 0 && ch != ' ' && ch != '\n' && !IS_TO_KANJI(p)
	      && (ignore_specials || !IS_RFC850_SPECIAL(ch))) {
	  p++;
	}
	if (!IS_TO_KANJI(p)) break;
      }
      p++;
    }

    /* encode between beg and end */
    p = beg;   /* p must less than end */
    beg_kanji = 0;  /* flag whether last string ended in kanji or ascii*/
    while (p < end && d < dmax) {
      unsigned char *new_d;

      if (d + c_len >= dmax) return (-1);

      if (len + c_len + 14 > 75) { /* 14 = to_kanji(4) + one char(4) */
	*d++ = '\n';		   /*	   + to_ascii(4) + "?="(2)   */
	*d++ = ' ';
	len = 0;
      } else if (!Slrn_Mime_Strict_Rfc
		 && p != (char *)s && *(p - 1) != ' '
		 && (ignore_specials || !IS_RFC850_SPECIAL(*(p - 1)))) {
	*d++ = ' ';
	len++;
      }
      strcpy((char *)d, charset);
      d += c_len;
      len += c_len + 2; /* 2 = "?=" */

      /* count length of encoded word */
      count = beg_kanji * 4;                /* 8 = one char(4) + to_ascii(4) */
      while(p < end && ((ch = *p) != 0) && len + count + 8 < 76) {
	if (IS_TO_KANJI(p)) {
	  if (Slrn_Mime_Strict_Rfc && len  + count > 70) {
	    len = 76;
	    break;
	  } else {   /* flag whether current string ended in kanji or ascii*/
	    end_kanji = 1;
	  }
	  p += 3;
	} else if (IS_TO_ASCII(p)) {
	  end_kanji = 0;
	  p += 3;
	} else if (*p > ' ' && end_kanji) { /* 2 byte character */
	  p += 2;
	} else { /* 1 byte character */
	  p++;
	}
	count = (20 - (beg + 60 - p) / 3) * 4;
      }

      if (len + count + 8 < 76) {
	p = end;
      }
      if (beg_kanji) {
	strncpy((char *)jtmp_str, to_kanji, 3);
	count = 3;
      } else {
	count = 0;
      }

      strncpy((char *)jtmp_str + count, beg, p - beg);
      count += p - beg;
      if (end_kanji) {
	strncpy((char *)jtmp_str + count, to_ascii, 3);
	count += 3;
      }
      /* add trailing space */
      if (!Slrn_Mime_Strict_Rfc && *p == ' ') {
	jtmp_str[count] = ' ';
	count++;
	p++;
      }

      new_d = encode_base64(d, jtmp_str, jtmp_str + count);
      len += new_d - d;
      d = new_d;

      beg_kanji = end_kanji;
      beg = p;

      if (d + 1 >= dmax) return (-1);
      d[0] = '?';
      d[1] = '=';
      d += 2;
      if (!Slrn_Mime_Strict_Rfc && *p != ' ') {
	*d++ = ' ';
	len++;
      }
    }
  }

  *d = 0;
  return (0);
}
#endif

/* This routine returns -1 if d is not big enough to encode s.
 * Technically, this routine is incorrect.  It should encode whitespace
 * separated words and not arbitrary sequences of text.
 */
static int rfc1522_encode (unsigned char *d, unsigned int len,
			   unsigned char *s, int ignore_specials)
{
   /* This routine encodes the a string containing 8-bit characters according
    * to the conventions of RFC1522.  The strategy here is to print up to
    * the first word which contains 8-bit chars before starting the "Q"
    * encoding.  When it has been determined that all of the 8-bit
    * characters have been encoded, switch back to normal mode.  This
    * approach should generate relatively short encoded words and should be
    * more generally readable.
    */
   
   unsigned int hibit;		   /* How many 8-bit characters are left? */
#ifdef KANJI
   unsigned int esc = 0;
#endif
   unsigned char *p, ch, *dmax;
   char charset[256];
   unsigned int charset_len;

   /* First scan the string to see if there are 8-bit characters
    * Count them for later use.
    */
   p = s;
   hibit = 0;
   while ((ch = *p) != 0)
     {
	if (ch & 0x80) hibit++;
#ifdef KANJI
	else if (ch == ESC) esc++;
#endif
	p++;
     }
   
#ifdef KANJI
   if (hibit == 0 && esc == 0)
#else
   if (hibit == 0)
#endif
     {
	if (s + len < p) return -1;
	
	strcpy((char *)d, (char *)s);
	return 0;
     }
   
   if (0 == slrn_case_strcmp ((unsigned char *)Slrn_Mime_Display_Charset,
			      (unsigned char *)"us-ascii"))
     {
#ifdef KANJI
       if (esc)
	 strcpy (charset, "=?ISO-2022-JP?B?");
       else
#endif
	strcpy (charset, "=?iso-8859-1?Q?");
     }
#ifdef KANJI
   else if (esc)
     sprintf (charset, "=?%s?B?", Slrn_Mime_Display_Charset);
#endif
   else sprintf (charset, "=?%s?Q?", Slrn_Mime_Display_Charset);
   charset_len = strlen (charset);
   
   p = s;
   dmax = d + len;
   /* leave room for the 0 character */
   dmax--;
#ifdef KANJI
   if (esc) return encode_base64_header(d, dmax, s, charset, charset_len,
					ignore_specials);
#endif
   
   /* algorithm:
    *  1. copy up to first word that contains one or more eight bit chars.
    *  2. encode to last char of last word that contains eight bit chars,
    *     or until a RFC822 special is encountered: <>()"
    *  3. Repeat
    * The rationale behind this is that 7 bit characters are allowed only
    * in certain portions of the header where an 8 bit character would never
    * appear.  Thus by encoding only in regions of 8 bit charcters delimited
    * by the RFC822 specials, we are sure not to decode into forbidden regions.
    *
    * RFC1522 forbids strarting or ending a =?..?..?..?= sequence in mid-word.
    */
   while (1)
     {
	/* 1. Copy to first 8 bit character */
	while (((ch = *p) != 0) && (d < dmax) && (0 == (ch & 0x80)))
	  {
	     *d++ = ch;
	     p++;
	  }
	
	if (ch == 0)
	  break;
	
	if (d == dmax) 
	  {
	     *d = 0;
	     return -1;
	  }
	
	/* walk back until we find a word boundary or begin-of-line */
	do
	  {
	     p--;
	     ch = *p;
	     if ((ch == ' ') || (ch == '\t')
		 || ((ignore_specials == 0) && IS_RFC850_SPECIAL(ch)))
	       {
		  p++;
		  break;
	       }
 	     d--;
	  }
	while (p != s);

 	
	/* 2. Now start encoding.  Work up to the first occurance of an RFC822
	 *    special character or until there are no more eight bit characters
	 *    left.  RFC1522 
	 */
	if (d + charset_len >= dmax)
	  return -1;
	
	strcpy ((char *)d, charset);
	d += charset_len;

	/* Note: rfc1522 places a limit on the length of the encoded word.  
	 * That limit is ignored here.  Perhaps someday I will recode this to 
	 * take the limit into account.
	 */
	while ((d < dmax) 
	       && ((ch = *p) != 0)
	       && ((ignore_specials != 0) || (0 == IS_RFC850_SPECIAL(ch)))
	       && ((hibit != 0) 
		   || ((ch != ' ') && (ch != '\t'))))
	  {
	     if (ch == ' ') *d++ = '_';
	     else if ((ch & 0x80) 
		      || ((ch < 32) && (ch != '\n'))
		      || IS_RFC850_SPECIAL(ch)
		      || (ch == '=')  /* = cannot occur as part of encoded word */
		      || (ch == '?')  /* ? cannot occur as part of encoded word */
		      || (ch == '_'))  /* encode this since SPACE gets mapped to it */
	       {
		  /* We need 3 characters to encode this. */
		  if (d + 3 >= dmax)
		    {
		       *d = 0;
		       return -1;
		    }
		  sprintf ((char *)d, "=%02X", (int) ch);
		  d += 3;
		  if (ch & 0x80) hibit--;
	       }
	     else *d++ = ch;
	     p++;
	  }
	
	/* Remove encoded spaces at the end (' '--> '_') */
	while ((p > s) && (*(p-1) == ' '))
	  {
	     d--; 
	     p--;
	  }

	/* Now turn off encoding.  We need two characters */
	if (d + 1 >= dmax) 
	  {
	     *d = 0;
	     return -1;
	  }
	
	d[0] = '?';
	d[1] = '=';
	d += 2;
     }
   *d = 0;
   return 0;
}

void slrn_mime_header_encode (char *s, unsigned int bytes)
{
   char buf[1024];
   unsigned int len = strlen (s);
   
   if (len < sizeof (buf))
     {
	int ignore_specials;
	
	/* Perhaps this test should be extended to include other headers. */
	ignore_specials = !slrn_case_strncmp ((unsigned char *)s, 
					      (unsigned char *)"Subject: ", 9);
	
	strcpy (buf, s);
	if (0 == rfc1522_encode ((unsigned char *)s, bytes, 
				 (unsigned char *)buf,
				 ignore_specials))
	    return;
     }
   
   /* Cannot do it so strip it to 8 bits. */
   while (*s)
     {
	*s = *s & 0x7F;
	s++;
     }
}

void slrn_mime_add_headers (FILE *fp)
{
   char *encoding;
   
   if (Mime_Posting_Charset == NULL)
     Mime_Posting_Charset = "us-ascii";

   switch (Mime_Posting_Encoding)
     {
      default:
      case ENCODED_8BIT:
	encoding = "8bit";
	break;
	
      case ENCODED_7BIT:
	if (!strcmp ("us-ascii", Mime_Posting_Charset))
	  return;
	encoding = "7bit";
	break;
	
      case ENCODED_QUOTED:
	encoding = "quoted-printable";
     }
   
   if (fp != NULL)
     {
	fprintf (fp, "\
Mime-Version: 1.0\n\
Content-Type: text/plain; charset=%s\n\
Content-Transfer-Encoding: %s\n",
		 Mime_Posting_Charset,
		 encoding);
     }
   else
     {
	Slrn_Post_Obj->po_printf ("\
Mime-Version: 1.0\n\
Content-Type: text/plain; charset=%s\n\
Content-Transfer-Encoding: %s\n",
		 Mime_Posting_Charset,
		 encoding);
     }
}

FILE *slrn_mime_encode (FILE *fp)
{
   if ((Mime_Posting_Encoding == ENCODED_7BIT)
       || (Mime_Posting_Encoding == ENCODED_8BIT))
     return fp;
   
   /* Add encoding later. */
   return fp;
}




#endif  /* SLRN_HAS_MIME */


