/*							hyperg.c
 *
 *	Confluent hypergeometric function
 *
 *
 *
 * SYNOPSIS:
 *
 * float a, b, x, y, hyperg();
 *
 * y = hyperg( a, b, x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Computes the confluent hypergeometric function
 *
 *                          1           2
 *                       a x    a(a+1) x
 *   F ( a,b;x )  =  1 + ---- + --------- + ...
 *  1 1                  b 1!   b(b+1) 2!
 *
 * Many higher transcendental functions are special cases of
 * this power series.
 *
 * As is evident from the formula, b must not be a negative
 * integer or zero unless a is an integer with 0 >= a > b.
 *
 * The routine attempts both a direct summation of the series
 * and an asymptotic expansion.  In each case error due to
 * roundoff, cancellation, and nonconvergence is estimated.
 * The result with smaller estimated error is returned.
 *
 *
 *
 * ACCURACY:
 *
 * Tested at random points (a, b, x), all three variables
 * ranging from 0 to 30.
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      0,5         10000       6.6e-7      1.3e-7
 *    IEEE      0,30        30000       1.1e-5      6.5e-7
 *
 * Larger errors can be observed when b is near a negative
 * integer or zero.  Certain combinations of arguments yield
 * serious cancellation error in the power series summation
 * and also are not in the region of near convergence of the
 * asymptotic series.  An error message is printed if the
 * self-estimated relative error is greater than 1.0e-3.
 *
 */

/*							hyperg.c */


/*
Cephes Math Library Release 2.1:  November, 1988
Copyright 1984, 1987, 1988 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

#include "mconf.h"

extern float MAXNUMF, MACHEPF;

#define fabsf(x) ( (x) < 0 ? -(x) : (x) )

#ifdef ANSIC
float exp(float);
float hyp2f0(float, float, float, int, float *);
static float hy11af(float, float, float, float *);
static float hy11pf(float, float, float, float *);
float log(float), gamma(float), lgam(float);
#else
float exp(), hyp2f0();
float log(), gamma(), lgam();
static float hy11pf(), hy11af();
#endif

#ifdef ANSIC
float hyperg( float aa, float bb, float xx )
#else
float hyperg( aa, bb, xx)
double aa, bb, xx;
#endif
{
float a, b, x, asum, psum, acanc, pcanc;


a = aa;
b = bb;
x = xx;
/* See if a Kummer transformation will help */
/*
temp = b - a;
if( fabsf(temp) < fabsf(b) )
	return( exp(x) * hyperg( temp, b, -x )  );
*/

psum = hy11pf( a, b, x, &pcanc );
if( pcanc < 1.0e-6f )
	goto done;


/* try asymptotic series */

asum = hy11af( a, b, x, &acanc );


/* Pick the result with less estimated error */

if( acanc < pcanc )
	{
	pcanc = acanc;
	psum = asum;
	}

done:
if( pcanc > 1.0e-3f )
	mtherr( "hyperg", PLOSS );

return( psum );
}




/* Power series summation for confluent hypergeometric function		*/


#ifdef ANSIC
static float hy11pf( float aa, float bb, float xx, float *err )
#else
static float hy11pf( aa, bb, xx, err )
double aa, bb, xx;
float *err;
#endif
{
float a, b, x, n, a0, sum, t, u, temp;
float an, bn, maxt, pcanc;

a = aa;
b = bb;
x = xx;
/* set up for power series summation */
an = a;
bn = b;
a0 = 1.0f;
sum = 1.0f;
n = 1.0f;
t = 1.0f;
maxt = 0.0f;


while( t > MACHEPF )
	{
	if( bn == 0 )			/* check bn first since if both	*/
		{
		mtherr( "hyperg", SING );
		return( MAXNUMF );	/* an and bn are zero it is	*/
		}
	if( an == 0 )			/* a singularity		*/
		return( sum );
	if( n > 200 )
		goto pdone;
	u = x * ( an / (bn * n) );

	/* check for blowup */
	temp = fabsf(u);
	if( (temp > 1.0f ) && (maxt > (MAXNUMF/temp)) )
		{
		pcanc = 1.0f;	/* estimate 100% error */
		goto blowup;
		}

	a0 *= u;
	sum += a0;
	t = fabsf(a0);
	if( t > maxt )
		maxt = t;
/*
	if( (maxt/fabsf(sum)) > 1.0e17f )
		{
		pcanc = 1.0f;
		goto blowup;
		}
*/
	an += 1.0f;
	bn += 1.0f;
	n += 1.0f;
	}

pdone:

/* estimate error due to roundoff and cancellation */
if( sum != 0.0f )
	maxt /= fabsf(sum);
maxt *= MACHEPF; 	/* this way avoids multiply overflow */
pcanc = fabsf( MACHEPF * n  +  maxt );

blowup:

*err = pcanc;

return( sum );
}


/*							hy11a()	*/
/* asymptotic formula for hypergeometric function:
 *
 *        (    -a                         
 *  --    ( |z|                           
 * |  (b) ( -------- 2f0( a, 1+a-b, -1/x )
 *        (  --                           
 *        ( |  (b-a)                      
 *
 *
 *                                x    a-b                     )
 *                               e  |x|                        )
 *                             + -------- 2f0( b-a, 1-a, 1/x ) )
 *                                --                           )
 *                               |  (a)                        )
 */

#ifdef ANSIC
static float hy11af( float aa, float bb, float xx, float *err )
#else
static float hy11af( aa, bb, xx, err )
double aa, bb, xx;
float *err;
#endif
{
float a, b, x, h1, h2, t, u, temp, acanc, asum, err1, err2;

a = aa;
b = bb;
x = xx;
if( x == 0 )
	{
	acanc = 1.0f;
	asum = MAXNUMF;
	goto adone;
	}
temp = log( fabsf(x) );
t = x + temp * (a-b);
u = -temp * a;

if( b > 0 )
	{
	temp = lgam(b);
	t += temp;
	u += temp;
	}

h1 = hyp2f0( a, a-b+1, -1.0f/x, 1, &err1 );

temp = exp(u) / gamma(b-a);
h1 *= temp;
err1 *= temp;

h2 = hyp2f0( b-a, 1.0f-a, 1.0f/x, 2, &err2 );

if( a < 0 )
	temp = exp(t) / gamma(a);
else
	temp = exp( t - lgam(a) );

h2 *= temp;
err2 *= temp;

if( x < 0.0f )
	asum = h1;
else
	asum = h2;

acanc = fabsf(err1) + fabsf(err2);


if( b < 0 )
	{
	temp = gamma(b);
	asum *= temp;
	acanc *= fabsf(temp);
	}


if( asum != 0.0f )
	acanc /= fabsf(asum);

acanc *= 30.0f;	/* fudge factor, since error of asymptotic formula
		 * often seems this much larger than advertised */

adone:


*err = acanc;
return( asum );
}

/*							hyp2f0()	*/

#ifdef ANSIC
float hyp2f0(float aa, float bb, float xx, int type, float *err)
#else
float hyp2f0( aa, bb, xx, type, err )
double aa, bb, xx;
int type;	/* determines what converging factor to use */
float *err;
#endif
{
float a, b, x, a0, alast, t, tlast, maxt;
float n, an, bn, u, sum, temp;

a = aa;
b = bb;
x = xx;
an = a;
bn = b;
a0 = 1.0f;
alast = 1.0f;
sum = 0.0f;
n = 1.0f;
t = 1.0f;
tlast = 1.0e9f;
maxt = 0.0f;

do
	{
	if( an == 0 )
		goto pdone;
	if( bn == 0 )
		goto pdone;

	u = an * (bn * x / n);

	/* check for blowup */
	temp = fabsf(u);
	if( (temp > 1.0f ) && (maxt > (MAXNUMF/temp)) )
		goto error;

	a0 *= u;
	t = fabsf(a0);

	/* terminating condition for asymptotic series */
	if( t > tlast )
		goto ndone;

	tlast = t;
	sum += alast;	/* the sum is one term behind */
	alast = a0;

	if( n > 200 )
		goto ndone;

	an += 1.0f;
	bn += 1.0f;
	n += 1.0f;
	if( t > maxt )
		maxt = t;
	}
while( t > MACHEPF );


pdone:	/* series converged! */

/* estimate error due to roundoff and cancellation */
*err = fabsf(  MACHEPF * (n + maxt)  );

alast = a0;
goto done;

ndone:	/* series did not converge */

/* The following "Converging factors" are supposed to improve accuracy,
 * but do not actually seem to accomplish very much. */

n -= 1.0f;
x = 1.0f/x;

switch( type )	/* "type" given as subroutine argument */
{
case 1:
	alast *= ( 0.5f + (0.125f + 0.25f*b - 0.5f*a + 0.25f*x - 0.25f*n)/x );
	break;

case 2:
	alast *= 2.0f/3.0f - b + 2.0f*a + x - n;
	break;

default:
	;
}

/* estimate error due to roundoff, cancellation, and nonconvergence */
*err = MACHEPF * (n + maxt)  +  fabsf( a0 );


done:
sum += alast;
return( sum );

/* series blew up: */
error:
*err = MAXNUMF;
mtherr( "hyperg", TLOSS );
return( sum );
}
