#!/bin/sh
# edit the line below if you need to use
# a specific wish on your system
# the next line restarts using wish \
                exec wish "$0" "$@"

#!/usr/local/bin/wish
##
## This is the second attempt at creating a
## Macintosh-like network chooser in tcl.
## The idea for this version is to make a generic
## interface that accepts plug-in for protocol-specific
## functionality
##
## Ethan Gold <etgold@cs.columbia.edu> 2/24/98 -> present
##
######## plugin API description: ########
## see the API_Description file that came
## with the distribution for a more complete
## description of the API.
## - A listbox for listing entities into
## - A frame for posting any kind of buttons
##   or other GUI elements into
## - Paths to bin directores and stuff as
##   supplied by major protocol plugins or config files
## - A current "zone" or "workgroup" variable
## - A simple interface to the relavent local selection,
##   hopefully more robust than direct X selection
## - A function to call on initialization to register
##   the plugin with the main chooser
##
#### In turn, the plugins must supply the following
#### procedures to the main chooser in the form of
#### "pluginname.procname"
## - An invocation procedure to be called when
##   the plugin's icon is clicked in the service browser
## - An "unloading" procedure to be called when another
##   service is selected
######## end plugin API description ########

######## global defines area ########
## set this to point to your library installation directory
#set libdir "/usr/local/home/etgold/code/tcl-scripts/tkchooser2"
set libdir "/usr/local/lib/tkchooser2"
set version "0.65.1"

set plugindir "plugins"
set icondir "icons"
set genericicon "generic.pnm"
set configfilename "chooser.cfg"
if ![info exists env(HOME)] {set env(HOME) "~"}
set userconfdir "$env(HOME)/.tkchooser"

## tcl default
set propfont "Helvetica -12 bold"
set fixedfont "fixed"

## set error counter
set errno 1
set debug 0

## variable containing name of listbox
## available to plugins
set pluglist ".rightside.top.pluglist"
set pluglistlabel ".rightside.top.pluglabel"
set plugframe ".rightside.bot"

## space for plugins to store state information
## accessed thru functions
set plug_globals(dummy) dum
unset plug_globals(dummy)

## kludge for plugin textentry variables
set plug_globalflag ""

set curr_pluglist_item ""
set currplug ""
set currproto ""

## set up major protocol service arrays
set appletalkservices(dummy) "dum"
unset appletalkservices(dummy)
set smbservices(dummy) "dum"
unset smbservices(dummy)
set ipservices(dummy) "dum"
unset ipservices(dummy)
set miscservices(dummy) "dum"
unset miscservices(dummy)
set appletalkflag 0
set smbflag 0
set ipflag 0
set miscflag 0

######## end global defines ########


######## get configuration options from filesystem ###########
## load global config file from the etc or the library directory
## and then load the major protocol
## handlers which will define the high-level layer-independant
## network API for each major protocol
##
if { [file readable [lindex $argv 0] ]} {
    puts "using supplied config file: [lindex $argv 0]"
    source [lindex $argv 0]
} elseif { [file exists "/etc/$configfilename"] } {
    source "/etc/$configfilename"
    puts "using config in /etc"
} elseif { [file exists "$libdir/$configfilename"] } {
    source "$libdir/$configfilename"
    puts "using config in $libdir"
} else {
    puts "unable to find global any configfile: $configfilename."
    puts "flying by the seat of our pants - expect a crash"
}


if { ![file exists $userconfdir] } {
    ## make sure the ~/.tkchooser directory exists
    catch {exec mkdir $userconfdir}
}

## source's the user's options
if { [file exists $userconfdir/$configfilename] } {
    source $userconfdir/$configfilename
}

puts "using libdir $libdir"

######## end configuration options from filesystem ###########


######## Build main Chooser window ########

## frame for left side
frame .leftside
## frame for right side
frame .rightside
## frame for the middle
frame .middle

########### build leftside frame
## a frame for the top left half of the window
frame .leftside.top
## a main scrolled canvas listbox
label .leftside.top.servlabel -text "Services" -border 2
canvas .leftside.top.services -relief sunken -border 2 \
	-width 175 -height 200 -background white \
	-yscrollcommand ".leftside.top.servscroll set" \
	-takefocus 0

scrollbar .leftside.top.servscroll -command ".leftside.top.services yview" \
	-takefocus 0

pack .leftside.top.servlabel -side top
pack .leftside.top.services -side left -fill both
pack .leftside.top.servscroll -side left -fill y

## a frame for the bottom left half of the window
frame .leftside.bot
## a zone selection listbox
label .leftside.bot.zonelabel -text "Zones/Workgroups:" -border 2
listbox .leftside.bot.zones -width 25 -exportselection 0 \
	-yscrollcommand ".leftside.bot.zonescroll set" -background white
scrollbar .leftside.bot.zonescroll -command ".leftside.bot.zones yview" \
	-takefocus 0

pack .leftside.bot.zonelabel -side top
pack .leftside.bot.zones -side left -fill both -expand 1
pack .leftside.bot.zonescroll -side left -fill y

## pack top and bottom halves of leftside
pack .leftside.top -side top -fill y -expand 1
pack .leftside.bot -side bottom -fill y -expand 1

############## build middle frame
button .middle.close -width 10 -text "Close" -command {exit}
## create the menubutton and leave the menu building for later
menubutton .middle.protocols -text "Protocols:" -relief raised
menu .middle.protocols.protomenu
label .middle.label -text " "

pack .middle.close -side bottom -anchor s
pack .middle.label -side top
pack .middle.protocols

############## build rightside frame
## a frame for the top right half of the window
frame .rightside.top

## a listbox to be made available to plugins
## as defined in the global section
label $pluglistlabel -text "Entities:" -border 2
listbox $pluglist -width 25 -height 12 -exportselection 0 \
	-yscrollcommand [list $pluglist"scroll" set] -background white
scrollbar $pluglist"scroll" -command "$pluglist yview" \
	-takefocus 0

pack $pluglistlabel -side top
pack $pluglist -side left -fill both -expand 1
pack $pluglist"scroll" -side left -fill y

## an empty frame for use by plugins
## on the bottom right section of the main window
frame $plugframe -width 200 -height 195 -relief groove -border 2

## a status widget in the plugin frame
message $plugframe.status -width 200 -text "ready."
pack $plugframe.status -side bottom -anchor s -fill x

## pack top and bottom halves of rightside
pack .rightside.top -side top -anchor n
pack $plugframe -side bottom -fill both

## pack all the frames into the main window
pack .leftside -side left -anchor n -fill y -expand 1
pack .middle -side left -anchor n -fill y
pack .rightside -side right -anchor n

update
######## End Build main window ########

############# begin procedures and functions ##################

## procedure to load in plugins from plugin directory
proc load_plugins {} {
    global libdir plugindir userconfdir
    
    if {[debug]} {puts "loading global plugins..."}
    set pluglist [glob -nocomplain "$libdir/$plugindir/*.tcl"]
    foreach name $pluglist {
	if {[file readable $name]} {
	    if {[debug]} {puts "found plugin: $name"}
	    source $name
	}
    }
    if {[debug]} {puts "finished loading global plugins."}

    if {[debug]} {puts "loading local plugins..."}
    set pluglist [glob -nocomplain "$userconfdir/$plugindir/*.tcl"]
    foreach name $pluglist {
	if {[file readable $name]} {
	    if {[debug]} {puts "found plugin: $name"}
	    source $name
	}
    }
    if {[debug]} {puts "finished loading local plugins"}
}

################ Plugin API ######################

## functions to return the names of GUI
## elements that the plugins are allowed to manipulate
proc plug_frame {} {
    global plugframe
    return $plugframe
}
proc plug_list {} {
    global pluglist
    return $pluglist
}
proc plug_list_label {} {
    global pluglistlabel
    return $pluglistlabel
}

proc get_curr_item {} {
    global curr_pluglist_item
    return $curr_pluglist_item
}

proc get_glob {index} {
    global plug_globals
    if {[info exists plug_globals($index)]} {
	return $plug_globals($index)
    } else {
	return ""
    }
}

proc set_glob {index value} {
    global plug_globals
    set plug_globals($index) $value
}

proc get_propfont {} {
    global propfont
    return $propfont
}

proc get_fixedfont {} {
    global fixedfont
    return $fixedfont
}

proc debug {} {
    global debug
    return $debug
}

## function to do tilde completion on a filename
proc tilde {name} {
    global env
    ## do tilde expansion
    regsub "~/" $name "$env(HOME)/" newfile
    ## substitute - assume bash variables if USER isn't there
    if {[info exists env(USER)]} {
	regsub $env(USER) $env(HOME) "" tophome
    } else {
	regsub $env(LOGNAME) $env(HOME) "" tophome
    }
    regsub "~" $newfile $tophome file_name
    return $file_name
}

## display an error message
proc error {caller errormessage} {
    global errno
    toplevel .error_$errno -class Dialog
    label .error_$errno.label -relief groove -text "$caller error:"
    message .error_$errno.message -width 300 -text "$errormessage"
    button .error_$errno.ok -text "Ok" -command [list destroy .error_$errno] \
	    -default active
    pack .error_$errno.label -fill x
    pack .error_$errno.message -fill both -expand 1
    pack .error_$errno.ok
    set mx [winfo pointerx .]
    set my [winfo pointery .]
    wm geometry .error_$errno "+[expr $mx-50]+[expr $my-50]"
    wm title .error_$errno "tkchooser error #$errno"
    incr errno
    update
    #bind .error_$errno.message <Return> ".error_$errno.ok invoke"
}

## function plugin calls to register itself
proc register_plugin {major plugname plugpubname} {
    global appletalkservices smbservices ipservices miscservices
    global appletalkflag smbflag ipflag miscflag
    upvar $major protocol

    if {[debug]} {
	puts "register_plugin called with $major $plugname $plugpubname"
    }
    
    ## this is uglee
    if {[string compare $major "appletalk"] == 0} {
	## if the major protocol is not running
	## then just return
	if {!$appletalkflag} {
	    puts "appletalk services not available"
	    return
	}
	
	## otherwise, register the plugin
	if {[debug]} {
	    puts "registering plugin: $plugname, $plugpubname"
	}
	set appletalkservices($plugname) $plugpubname	
    } elseif {[string compare $major "smb"] == 0} {
	## if the major protocol is not running
	## then just return
	if {!$smbflag} {
	    puts "smb services not available"
	    return
	}

	## otherwise, register the plugin
	if {[debug]} {
	    puts "registering plugin: $plugname, $plugpubname"
	}
	set smbservices($plugname) $plugpubname	
	
    } elseif {[string compare $major "ip"] == 0} {
	## if the major protocol is not running
	## then just return
	if {!$ipflag} {
	    puts "ip services not available"
	    return
	}

	## otherwise, register the plugin
	if {[debug]} {
	    puts "registering plugin: $plugname, $plugpubname"
	}
	set ipservices($plugname) $plugpubname	

    } elseif {[string compare $major "misc"] == 0} {
	## if the major protocol is not running
	## then just return
	if {!$miscflag} {
	    puts "misc services not available"
	    return
	}

	## otherwise, register the plugin
	if {[debug]} {
	    puts "registering plugin: $plugname, $plugpubname"
	}
	set miscservices($plugname) $plugpubname		

    } else {
	puts "unsuppored protocol: $major"
    }
    
    ## all other function names can be deduced
    ## from the plugin name

}

## procedure to test depenancies
proc check_deps {deps} {
    foreach program $deps {
	set loc ""
	if {[debug]} {puts "checking for $program"}
	if {![file executable $program]} {
	    catch {exec which $program} result
	    if {[debug]} {puts "checking in path for $program: $result"}
	    ## trivial test
	    if {[llength $result] == 1} {
		set loc $result
	    }
	}
	lappend locs $loc
    }
    return $locs
}

## procedure to toggle the status of the login entry widgets
## assumes widgets are of the form:
## $w.nameframe.name and $w.passframe.pass
proc toggleentries {w flag} {
    global $flag
    upvar 0 $flag checkflag

    if {$checkflag} {
	$w.nameframe.name configure -state disabled -foreground grey75
	$w.passframe.pass configure -state disabled -foreground grey75
    } else {
	$w.nameframe.name configure -state normal -foreground black
	$w.passframe.pass configure -state normal -foreground black
    }
}
################ End Plugin API ######################

## internal function to build services canvas
proc build_services {major} {
    global $major libdir icondir genericicon userconfdir
    upvar $major protocol
    
    if {[debug]} {
	puts "building $major services: [array names protocol]..."
    }

    .leftside.top.services delete all

    ## foreach plugin defined for this protocol
    ## create it's icon in the canvas and bind
    ## single-click to it's startup function

    set spacing 45
    set rownum 1
    set leftside 1
    foreach plugin [array names protocol] {
	set icon [$plugin.geticon]
	set iconpathglob "$libdir/$icondir"
	set iconpathloc "$userconfdir/$icondir"
	if {[debug]} {
	    puts "activating $protocol($plugin) with icon $iconpathglob/$icon"
	}
	
	if {!$leftside} { 
	    set X [expr $spacing * 3]
	    set leftside 1
	} else {
	    set X $spacing
	    set leftside 0
	}
	
	if {$rownum == 1} {
	    set Y [expr $spacing/1.5]
	} else {
	    set Y [expr $spacing * $rownum]
	}

	## put in a check for the icon and make a generic one
	if {[file readable $iconpathloc/$icon]} {
	    image create photo $plugin -file $iconpathloc/$icon
	} elseif {[file readable $iconpathglob/$icon]} {
	    image create photo $plugin -file $iconpathglob/$icon
	} else {
	    puts "tkchooser2: could not find or read $icon for $plugin plugin"
	    image create photo $plugin -file $iconpathglob/$genericicon
	}

	.leftside.top.services create image \
		$X $Y -anchor center -image "$plugin" -tag "$plugin"
	
	.leftside.top.services create text \
		[expr $X] [expr $Y + $spacing/2] \
		-text [$plugin.getpubname] \
		-font fixed -fill black \
		-tag "$plugin"

	.leftside.top.services bind $plugin <ButtonPress> \
		[list startplug $plugin]

	## make sure the scroll region for the canvas will let
	## us see all the icons
	if {$Y >= 150} {
	    .leftside.top.services configure \
		    -scrollregion "0 0 300 [expr $Y + $Y/2]"
	}

	if {$leftside} {set rownum [expr $rownum + 1]}
    }
    update
    if {[debug]} {
	puts "finished building $major services"
    }
}

## procedure to remove services
proc remove_services {} {
    .leftside.top.services delete all
}

## procedure to startup plugins when their
## icons are clicked
proc startplug {plugin} {
    global currplug curr_pluglist_item plug_globals plug_globalflag

    ## Do the background-reversal feedback thingie.
    ## If there's an active selection indicator, kill it.
    .leftside.top.services delete "bg_selected"
    
    ## create a selection indicator box around the new plugin
    set boxlist [.leftside.top.services bbox $plugin]
    set x1 [lindex $boxlist 0]
    set x1 [expr $x1 - 2]
    set y1 [lindex $boxlist 1]
    set y1 [expr $y1 - 2]
    set x2 [lindex $boxlist 2]
    set x2 [expr $x2 + 2]
    set y2 [lindex $boxlist 3]
    set y2 [expr $y2 + 2]
    .leftside.top.services create rectangle $x1 $y1 $x2 $y2 \
	    -outline black -tag "bg_selected"
    .leftside.top.services lower "bg_selected" $plugin

    ## if there's something running, then shut it off
    ## and clear the plugin globals
    if {[string compare $currplug ""] != 0} {
	$currplug.stop
	foreach index [array names plug_globals] {
	    unset plug_globals($index)
	}
	set plug_globalflag ""
    }
    
    ## set up and start the new plugin
    set currplug $plugin
    set curr_pluglist_item ""
    $currplug.start
}

## procedure called from doubleclick on
## plugin list window to call the current
## plugin's doubleclick function
proc doubleclick_currplug {} {
    global currplug
    $currplug.doubleclick
}

## procedure to start up a new protocol
proc startproto {protocol} {
    global currproto menuproto currplug pluglistlabel plugframe
    if {[debug]} {
	puts "starting $protocol module"
    }
    $pluglistlabel configure -text "Entities:"

    ## stop any running plugins
    if {[string compare $currplug ""] != 0 } {
	$currplug.stop
    }
    ## stop the running protocol
    if {[string compare $currproto ""] != 0 } {
	$currproto.stop
    }
    
    ## reset and start the new protocol
    set currproto $protocol
    set menuproto $protocol
    $plugframe.status configure -text "starting $protocol"
    update
    $protocol.start
    .leftside.top.servlabel configure -text "$protocol Services:"
    $plugframe.status configure -text "ready."
    update
}


######### End Procedures and Functions ############

######### Resume startup execution ############

## now that we have the config file loaded, the main
## window built, and the functions defined, check for
## protocol module files.

if [debug] {puts "TkChooser version $version"}

## check for appletalk module
if { [file exists $libdir/appletalk.tcl] && $appletalkflag } {
    if {[debug]} {puts "checking for Appletalk module"}
    source $libdir/appletalk.tcl
}

## check for SMB module
if { [file exists $libdir/smb.tcl] && $smbflag } {
    if {[debug]} {puts "checking for SMB module"}
    source $libdir/smb.tcl
}

## check for IP module
if { [file exists $libdir/ip.tcl] && $ipflag } {
    if {[debug]} {puts "checking for IP module"}
    source $libdir/ip.tcl
}

## check for Misc module
if { [file exists $libdir/misc.tcl] && $miscflag } {
    if {[debug]} {puts "checking for Misc module"}
    source $libdir/misc.tcl
}

## source the fileselction box code so everyone can use it
source $libdir/fileselect.tcl

## search the plugin directory and register 'em
load_plugins

## build a protocol selection menu based on what protocols
## are defined as active

if {$appletalkflag} {
    .middle.protocols.protomenu add radiobutton \
	    -label "Appletalk" -variable menuproto -value "appletalk" \
	    -command "build_services appletalkservices; startproto appletalk"
    #-command "startproto appletalk; build_services appletalkservices"
}
if {$smbflag} {
    .middle.protocols.protomenu add radiobutton \
	    -label "SMB" -variable menuproto -value "smb" \
	    -command "build_services smbservices; startproto smb"
    #-command "startproto smb; build_services smbservices"
}
if {$ipflag} {
    .middle.protocols.protomenu add radiobutton \
	    -label "IP" -variable menuproto -value "ip" \
	    -command "build_services ipservices; startproto ip"
    #-command "startproto smb; build_services ipservices"
}
if {$miscflag} {
    .middle.protocols.protomenu add radiobutton \
	    -label "Misc" -variable menuproto -value "misc" \
	    -command "build_services miscservices; startproto misc"
    #-command "startproto misc; build_services miscservices"
}
.middle.protocols configure -menu .middle.protocols.protomenu

## select and activate the default protocol
set testflag "flag"
set testflag \$$defaultprotocol$testflag
if {[eval list $testflag]} {
    set service "services"
    set service $defaultprotocol$service
    if {[debug]} {
        puts "debug: trying to build $service"
    }
    build_services $service
    startproto $defaultprotocol
} else {
    error "Startup" "default protocol has been disabled. proceed with caution"
    update
}

wm title . "Tkchooser $version"
wm iconname . "Tkchooser"

############## Event Bindings ################
## plugin listbox
bind $pluglist <ButtonRelease> {
    if {[debug]} {
	puts "[$pluglist curselection]"
    }
    if {[string compare [$pluglist curselection] ""] != 0} {
	set curr_pluglist_item [$pluglist get [$pluglist curselection]]
    }
}

## module zone/workgroup listbox
bind .leftside.bot.zones <ButtonRelease> {
    #$currproto.setcurrzone [selection get]
    set newzone  [.leftside.bot.zones get [.leftside.bot.zones curselection]]
    $currproto.setcurrzone "$newzone"
    if {[string compare $currplug ""] != 0} {
	$currplug.newzone
    }
}

bind .leftside.bot.zones <Return> {
    #$currproto.setcurrzone [selection get]
    set newzone  [.leftside.bot.zones get [.leftside.bot.zones curselection]]
    $currproto.setcurrzone "$newzone"
    if {[string compare $currplug ""] != 0} {
	$currplug.newzone
    }
}


bind $pluglist <Double-1> {
    #set curr_pluglist_item [selection get]
    if {[string compare [$pluglist curselection] ""] != 0} {
	set curr_pluglist_item [$pluglist get [$pluglist curselection]]
    }
    doubleclick_currplug
}

bind $pluglist <Return> {
    if {[string compare [$pluglist curselection] ""] != 0} {
	set curr_pluglist_item [$pluglist get [$pluglist curselection]]
    }
    doubleclick_currplug    
}

bind .middle.close <Return> {
    .middle.close invoke
}

############# End Event Bindings ###############

