/*
 * Copyright (c) 1997, 1998  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_TYPE_SIZE_T
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, unistd.h)
 *   AC_CHECK_FUNCS(strchr, memcpy)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "hostname.h"

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#ifdef __STDC__
void *memchr(const void *, int, size_t);
int memcmp(const void *, const void *, size_t);
void *memmove(void *, const void *, size_t);
void *memset(void *, int, size_t);
#else /* not __STDC__ */
char *memchr();
int memcmp();
char *memmove();
char *memset();
#endif /* not __STDC__ */
#endif /* not HAVE_MEMCPY */


/*
 * Get my canonical hostname and its address.
 *
 * The hostname and address are put into `hostname' and `address'.
 * If unknown, UNKNOWN_HOST is put into it.
 * If succeeded, 0 is returned.  Otherwise -1 is returned.
 */
int
identify_my_host(hostname, address)
    char *hostname;
    char *address;
{
    struct hostent *byname;
    struct in_addr addr;
    char *strp;
    char *ntoa;

    /*
     * Set hostname and address to UNKNOWN_HOST.
     */  
    strncpy(hostname, UNKNOWN_HOST, MAXLEN_HOSTNAME);
    *(hostname + MAXLEN_HOSTNAME) = '\0';
    strncpy(address, UNKNOWN_HOST, MAXLEN_HOSTNAME);
    *(address + MAXLEN_HOSTNAME) = '\0';

    /*
     * Get my hostname by gethostname() and gethostbyname().
     */
    if (gethostname(hostname, MAXLEN_HOSTNAME + 1) < 0)
	return -1;

    byname = gethostbyname(hostname);
    if (byname == NULL)
	return -1;
    if (MAXLEN_HOSTNAME < strlen(byname->h_name)) {
	strncpy(hostname, byname->h_name, MAXLEN_HOSTNAME);
	*(hostname + MAXLEN_HOSTNAME) = '\0';
	syslog(LOG_ERR, "too long hostname: %s...", hostname);
	return -1;
    }
    strcpy(hostname, byname->h_name);

    /*
     * Get an address of the hostname.
     */
    memcpy(&addr, byname->h_addr, sizeof(addr));
    ntoa = inet_ntoa(addr);
    if (MAXLEN_HOSTNAME < strlen(ntoa)) {
	strncpy(address, ntoa, MAXLEN_HOSTNAME);
	*(address + MAXLEN_HOSTNAME) = '\0';
	syslog(LOG_ERR, "too long address: %s...", address);
	return -1;
    }
    strcpy(address, ntoa);

    /*
     * Change the hostname to lower cases.
     */
    for (strp = hostname; *strp != '\0'; strp++) {
	if (isupper(*strp))
	    *strp = tolower(*strp);
    }

    return 0;
}


/*
 * Get remote hostname and address of the socket `fd'.
 *
 * The hostname and address are put into `hostname' and `address'.
 * If unknown, UNKNOWN_HOST is put into it.
 * If succeed, 0 is returned.  Otherwise -1 is returned.
 */
int
identify_remote_host(fd, hostname, address)
    int fd;
    char *hostname;
    char *address;
{
    struct sockaddr_in peer;
    struct hostent *byname, *byaddr;
    char **addrp;
    char *strp;
    char *ntoa;
    int peerlen;

    /*
     * Set hostname and address to UNKNOWN_HOST.
     */  
    strncpy(hostname, UNKNOWN_HOST, MAXLEN_HOSTNAME);
    *(hostname + MAXLEN_HOSTNAME) = '\0';
    strncpy(address, UNKNOWN_HOST, MAXLEN_HOSTNAME);
    *(address + MAXLEN_HOSTNAME) = '\0';

    /*
     * Get an IP address of the remote host by getpeername().
     */
    peerlen = sizeof(peer);
    if (getpeername(fd, (struct sockaddr *)&peer, &peerlen) < 0) {
	syslog(LOG_ERR, "getpeername() failed, %m");
	return -1;
    }
    ntoa = inet_ntoa(peer.sin_addr);
    if (MAXLEN_HOSTNAME < strlen(ntoa)) {
	strncpy(address, ntoa, MAXLEN_HOSTNAME);
	*(address + MAXLEN_HOSTNAME) = '\0';
	syslog(LOG_ERR, "too long address: %s...", address);
	return -1;
    }
    strcpy(address, ntoa);

    /*
     * Get a hostname of the remote host by gethostbyaddr().
     */
    byaddr = gethostbyaddr((char *)&peer.sin_addr, sizeof(peer.sin_addr),
	AF_INET);
    if (byaddr == NULL) {
	syslog(LOG_ERR, "cannot get a hostname: %s", address);
	return 0;
    }
    if (MAXLEN_HOSTNAME < strlen(byaddr->h_name)) {
	strncpy(hostname, byaddr->h_name, MAXLEN_HOSTNAME);
	*(hostname + MAXLEN_HOSTNAME) = '\0';
	syslog(LOG_ERR, "too long hostname: %s...", hostname);
	return -1;
    }
    strcpy(hostname, byaddr->h_name);
    
    /*
     * Verify the hostname by reversed lookup.
     */
    byname = gethostbyname(hostname);
    if (byname == NULL) {
	syslog(LOG_ERR, "cannot get a hostname: %s", address);
	return 0;
    }
    for (addrp = byname->h_addr_list; *addrp != NULL; addrp++) {
	if (memcmp(&peer.sin_addr, *addrp, byname->h_length) == 0)
	    break;
    }
    if (*addrp == NULL) {
	syslog(LOG_ERR, "reversed lookup mismatch: %s -> %s",
	    address, hostname);
	strncpy(hostname, UNKNOWN_HOST, MAXLEN_HOSTNAME);
	*(hostname + MAXLEN_HOSTNAME) = '\0';
	return 0;
    }

    /*
     * Change the hostname to lower cases.
     */
    for (strp = hostname; *strp != '\0'; strp++) {
	if (isupper(*strp))
	    *strp = tolower(*strp);
    }

    return 0;
}
