/*
 * Copyright (c) 1997, 1998  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_C_CONST
 *   AC_TYPE_SIZE_T
 *   AC_HEADER_STDC
 *   AC_CHECK_HEADERS(string.h, memory.h, unistd.h)
 *   AC_CHECK_FUNCS(memcpy)
 *   AC_HEADER_TIME
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else /* not TIME_WITH_SYS_TIME */
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else /* not HAVE_SYS_TIME_H */
#include <time.h>
#endif /* not HAVE_SYS_TIME_H */
#endif /* not TIME_WITH_SYS_TIME */

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#ifdef __STDC__
void *memchr(const void *, int, size_t);
int memcmp(const void *, const void *, size_t);
void *memmove(void *, const void *, size_t);
void *memset(void *, int, size_t);
#else /* not __STDC__ */
char *memchr();
int memcmp();
char *memmove();
char *memset();
#endif /* not __STDC__ */
#endif /* not HAVE_MEMCPY */


/*
 * Bind and listen to a TCP port.
 *
 * `port' is a port number to which the server listens.
 * `backlog' is a maximum queue length of pending connections.
 *
 * If successful, the file descriptor of the socket is returned.
 * Otherwise, -1 is returned.
 */
int
set_server_port(port, backlog)
    int port;
    int backlog;
{
    int file;
    struct sockaddr_in sin;
    int opt;

    /*
     * Create a socket.
     */
    file = socket(AF_INET, SOCK_STREAM, 0);
    if (file < 0) {
	syslog(LOG_ERR, "socket() failed, %m");
	goto failed;
    }

#ifdef SO_REUSEADDR
    opt = 1;
    if (setsockopt(file, SOL_SOCKET, SO_REUSEADDR, (char *)&opt,
	sizeof(opt)) < 0) {
	syslog(LOG_ERR, "setsockopt() failed, %m");
	goto failed;
    }
#endif

    /*
     * Bind the socket.
     */
    memset(&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = htons(port);
    sin.sin_addr.s_addr = htonl(INADDR_ANY);
    if (bind(file, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
	syslog(LOG_ERR, "bind() failed, %m");
	goto failed;
    }

    /*
     * Listen to the port.
     */
    if (listen(file, backlog) < 0) {
	syslog(LOG_ERR, "listen() failed, %m");
	goto failed;
    }

    syslog(LOG_DEBUG, "debug: set server port: %d/tcp", port);
    return file;

    /*
     * An error occurs...
     */
  failed:
    syslog(LOG_ERR, "cannot listen to the port: %d/tcp", port);
    return -1;
}


/*
 * Wait until a new connection comes.
 *
 * `file' is a socket to wait a connection.
 * The system call listen() must have been called for `file'.
 * (A file created by set_server_port() is suitable for `file'.)
 *
 * The function never returns until a new connection comes, or an error
 * occurs.  If an error occurs, -1 is returned.  Otherwise 0 is returned.
 */
int
wait_server_port(file)
    int file;
{
    fd_set mask;
    int status;

    for (;;) {
	/*
	 * Wait until a new connection comes.
	 */
	FD_ZERO(&mask);
	FD_SET(file, &mask);
	status = select(file + 1, &mask, NULL, NULL, NULL);
	if (0 < status && FD_ISSET(file, &mask))
	    break;
	else if (errno == EINTR) {
	    /*
	     * When a child process exits, select() returns -1 and `errno'
	     * is set to EINTR.
	     */ 
	    return -1;
	} else {
	    syslog(LOG_ERR, "select() failed, %m");
	    goto failed;
	}
    }

    syslog(LOG_DEBUG, "debug: catch a new connection");
    return 0;

    /*
     * An error occurs...
     */
  failed:
    syslog(LOG_ERR, "failed to catch a new connection");
    return -1;
}


