/******************************************************************************\
 gnofin/view.c   $Revision: 1.36 $
 Copyright (C) 1999 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "gnofin.h"
#include <gnome.h>
#include <errno.h>
#include "account.h"
#include "record.h"
#include "file.h"
#include "view.h"
#include "imp-qif.h"

#define enable_file_save(view)	           (set_enable_file_save(view,TRUE))
#define enable_file_save_as(view)	   (set_enable_file_save_as(view,TRUE))
#define enable_edit_undo(view)	           (set_enable_edit_undo(view,TRUE))
#define enable_edit_redo(view)	           (set_enable_edit_redo(view,TRUE))
#define enable_edit_cut(view)	           (set_enable_edit_cut(view,TRUE))
#define enable_edit_copy(view)	           (set_enable_edit_copy(view,TRUE))
#define enable_edit_paste(view)	           (set_enable_edit_paste(view,TRUE))
#define enable_edit_select_all(view)       (set_enable_edit_select_all(view,TRUE))
#define enable_edit_find(view)	           (set_enable_edit_find(view,TRUE))
#define enable_edit_find_again(view)       (set_enable_edit_find_again(view,TRUE))
#define enable_account_delete(view)	   (set_enable_account_delete(view,TRUE))
#define enable_account_props(view)	   (set_enable_account_props(view,TRUE))
#define enable_record_delete(view)	   (set_enable_record_delete(view,TRUE))
#define enable_record_view(view)	   (set_enable_record_view(view,TRUE))
#define enable_record_update(view)	   (set_enable_record_update(view,TRUE))
#define enable_record_toggle_status(view)  (set_enable_record_toggle_status(view,TRUE))
 
#define disable_file_save(view)	           (set_enable_file_save(view,FALSE))
#define disable_file_save_as(view)	   (set_enable_file_save_as(view,FALSE))
#define disable_edit_undo(view)	           (set_enable_edit_undo(view,FALSE))
#define disable_edit_redo(view)	           (set_enable_edit_redo(view,FALSE))
#define disable_edit_cut(view)	           (set_enable_edit_cut(view,FALSE))
#define disable_edit_copy(view)	           (set_enable_edit_copy(view,FALSE))
#define disable_edit_paste(view)	   (set_enable_edit_paste(view,FALSE))
#define disable_edit_select_all(view)      (set_enable_edit_select_all(view,FALSE))
#define disable_edit_find(view)	           (set_enable_edit_find(view,FALSE))
#define disable_edit_find_again(view)      (set_enable_edit_find_again(view,FALSE))
#define disable_account_delete(view)       (set_enable_account_delete(view,FALSE))
#define disable_account_props(view)	   (set_enable_account_props(view,FALSE))
#define disable_record_delete(view)	   (set_enable_record_delete(view,FALSE))
#define disable_record_view(view)	   (set_enable_record_view(view,FALSE))
#define disable_record_update(view)	   (set_enable_record_update(view,FALSE))
#define disable_record_toggle_status(view) (set_enable_record_toggle_status(view,FALSE))

/* helper functions */
static void 	   fin_on_nop			(FinView     * view) {printf("not implemented\n");}
static void	   close_it			(FinView     * view);
static gint	   save_it			(FinView     * view);

static FinRecord * del_record			(FinView     * view, 
						 FinAccount  * account,   /* NULL uses active */
						 FinRecord   * record,    /* NULL uses position */
						 gint          position); /* -1 uses record */
static gint	   put_record			(FinView     * view,
						 FinAccount  * account,   /* NULL uses active */
						 FinRecord   * record,    /* NULL invalid */
						 gboolean      insert);

static void	   del_link_if_exists		(FinView     * view,
						 FinRecord   * record);
static void	   put_link_if_exists		(FinView     * view,
						 FinRecord   * record,
						 int	       insert);

static void        create_window		(FinView     * view);
static GtkCList *  create_list			(FinView     * view,
						 const gchar * name,
						 gint          position);
//static void        create_account_info_dialog	(FinView     * view);
static void        create_calendar_dialog	(FinView     * view);

static void	   refresh_list			(FinView     * view, 
						 GtkCList    * clist,   /* if NULL, view->clist */
						 FinAccount  * account, /* if NULL, view->account */
						 gboolean      recalc_bal);
static void	   refresh_info			(FinView     * view);
static void	   refresh_account_names	(FinView     * view);
static void	   refresh_check_no		(FinView     * view,
						 gboolean      scan);
static void	   refresh_transfer_accts	(FinView     * view);
static void	   refresh_type_list		(FinView     * view);
static void	   refresh_cleared_bal		(FinView     * view);

static void	   get_toolbar_buttons		(FinView     * view);

static void	   set_dirty			(FinView     * view,
						 gboolean      value);
static void	   put_filename			(FinView     * view,
						 const gchar * filename);
static void	   set_filename			(FinView     * view,
						 const gchar * filename);

static void	   set_enable_file_save		(FinView     * view, gboolean);
static void	   set_enable_edit_undo		(FinView     * view, gboolean);
static void	   set_enable_edit_redo		(FinView     * view, gboolean);
static void	   set_enable_edit_cut		(FinView     * view, gboolean);
static void	   set_enable_edit_copy		(FinView     * view, gboolean);
static void	   set_enable_edit_paste	(FinView     * view, gboolean);
static void	   set_enable_edit_select_all	(FinView     * view, gboolean);
static void	   set_enable_edit_find		(FinView     * view, gboolean);
static void	   set_enable_edit_find_again	(FinView     * view, gboolean);
static void	   set_enable_account_delete	(FinView     * view, gboolean);
static void	   set_enable_account_props	(FinView     * view, gboolean);
static void	   set_enable_record_delete	(FinView     * view, gboolean);
static void	   set_enable_record_update	(FinView     * view, gboolean);
static void	   set_enable_record_toggle_status (FinView  * view, gboolean);
static void	   set_enable_record_view	(FinView     * view, gboolean);

/* signal functions */
static void fin_on_new			(GtkWidget * widget, FinView * view);
static void fin_on_open			(GtkWidget * widget, FinView * view);
static void fin_on_save			(GtkWidget * widget, FinView * view);
static void fin_on_save_as		(GtkWidget * widget, FinView * view);
static void fin_on_about		(GtkWidget * widget, FinView * view);
static gint fin_on_close		(GtkWidget * widget, FinView * view);
static void fin_on_exit			(GtkWidget * widget, FinView * view);
static void fin_on_account_new		(GtkWidget * widget, FinView * view);
static void fin_on_account_delete	(GtkWidget * widget, FinView * view);
static void fin_on_account_props	(GtkWidget * widget, FinView * view);
static void fin_on_insert_record	(GtkWidget * widget, FinView * view);
static void fin_on_update_record	(GtkWidget * widget, FinView * view);
static void fin_on_delete_record	(GtkWidget * widget, FinView * view);
static void fin_on_record_type_changed 	(GtkWidget * widget, FinView * view);
static void fin_on_show_calendar	(GtkWidget * widget, FinView * view);
static void fin_on_undo			(GtkWidget * widget, FinView * view);
static void fin_on_redo			(GtkWidget * widget, FinView * view);

static void fin_on_hide_calendar	(GtkWidget * widget,
					 GdkEvent  * event,
					 FinView   * view);
static void fin_on_entry_date_changed   (GtkWidget * widget,
					 FinView   * view);

static void fin_on_select_record	(GtkWidget       * clist, 
					 gint		   row, 
					 gint		   column, 
					 GdkEventButton  * event,
					 FinView         * view);
static void fin_on_unselect_record	(GtkWidget       * clist,
					 gint		   row,
					 gint		   column,
					 GdkEventButton  * event,
					 FinView         * view);
static void fin_on_switch_account	(GtkWidget	 * notebook,
					 GtkNotebookPage * page,
					 guint		   page_num,
					 FinView         * view);

static void create_account_info_dialog  (FinView     * view);

#ifdef FIN_DUMP_ENABLE
static void fin_on_dump_state		(GtkWidget * widget, FinView * view);
#endif

static gchar * clist_titles[] = 
{
  "Date", "Type", "Description", "S", "Amount", "Balance"
};

static gint clist_def_widths[] = 
{
  70, 80, 300, 10, 50, 50
};

static const FinRecordField clist_types[] =
{
  FIN_RECORD_FIELD_DATE,
  FIN_RECORD_FIELD_TYPE_FULL,
  FIN_RECORD_FIELD_INFO,
  FIN_RECORD_FIELD_STATUS,
  FIN_RECORD_FIELD_AMOUNT,
  FIN_RECORD_FIELD_BALANCE,
};


#define FIN_HAVE_FILENAME(view)  ((gint)((view)->filename != NULL))
#define FIN_NULL_FILENAME        NULL


/* Declare some data types for the history */

typedef enum _FinOperationType FinOperationType;
typedef struct _FinHistoryData FinHistoryData;

enum _FinOperationType
{
  FIN_OP_INSERT_RECORD,
  FIN_OP_UPDATE_RECORD,
  FIN_OP_DELETE_RECORD,
  FIN_OP_TOGGLE_RECORD_STATUS,
  FIN_OP_INSERT_ACCOUNT,
  FIN_OP_UPDATE_ACCOUNT,
  FIN_OP_DELETE_ACCOUNT,
};

struct _FinHistoryData
{
  FinOperationType type;

  FinAccount  * account;
  FinAccount  * account_old;
  FinRecord   * record;
  FinRecord   * record_old;
};

#define FIN_IS_RECORD_OP(type)  (((type) >= FIN_OP_INSERT_RECORD) && \
				 ((type) <= FIN_OP_DELETE_RECORD))
#define FIN_IS_ACCOUNT_OP(type) (((type) >= FIN_OP_INSERT_ACCOUNT) && \
				 ((type) <= FIN_OP_DELETE_ACCOUNT))


/* BASIC HANDLERS
 *
 * close_it
 * save_it
 * fin_on_new
 * fin_on_open
 * fin_on_save
 * fin_on_save_as
 * fin_on_about
 * fin_on_save_changes_before_closing
 * fin_on_close
 * fin_on_exit
 * fin_on_delete_window
 */

static void
close_it (FinView * view)
{
  g_return_if_fail(view != NULL);

  fin_trace("");

  if (view->set)
  {
    int i, num;

    num = g_list_length(view->set->accounts);

    fin_trace("num = %d", num);

    /* block switch_page signal, to prevent redrawing as we remove pages */
    gtk_signal_handler_block(GTK_OBJECT(view->notebook), 
    			     view->notebook_switch_page_id);

    /* remove all notebook pages */
    for (i=num-1; i>=0; --i)
      gtk_notebook_remove_page(GTK_NOTEBOOK(view->notebook), i);

    gtk_signal_handler_unblock(GTK_OBJECT(view->notebook), 
    			       view->notebook_switch_page_id);

    /* okay to close data */
    fin_account_set_unref(view->set);

    view->clist = NULL;
    view->set = NULL;
    view->account = NULL;
    view->selection = -1;

    set_dirty(view, -1);
  }

  if (view->history)
    fin_history_clear(FIN_HISTORY(view->history));

  disable_record_view(view);
  disable_account_delete(view);
  disable_account_props(view);
  disable_edit_undo(view);
  disable_edit_redo(view);
  disable_record_delete(view);
  disable_record_toggle_status(view);
}

static gint
save_it (FinView * view)
{
  const gchar * failed = NULL;

  fin_trace("");

  g_return_val_if_fail(view, FALSE);
  g_return_val_if_fail(view->set, FALSE);
  g_return_val_if_fail(FIN_HAVE_FILENAME(view), FALSE);

  /* backup existing file */
  if (fin_file_exists(view->filename))
    fin_file_backup(view->filename);

  /* write file */ 
  if ((failed = fin_file_write(view->filename, view->set)) != NULL) 
  {
    fin_error_dialog(view->window, "Error saving file: %s\n[%s]", view->filename, failed);
    return FALSE;
  }

  /* reset dirty flag */
  set_dirty(view, -1);

  return TRUE;
}

static void
fin_on_new (GtkWidget * widget,
	    FinView   * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);

  if (view->set)
    fin_on_close(NULL, view);

  view->set = fin_account_set_new();
  set_filename(view, FIN_NULL_FILENAME);
  refresh_info(view);
  refresh_cleared_bal(view);
}

static void
fin_on_open (GtkWidget * widget,
	     FinView   * view)
{
  gchar * name;

  fin_trace("");

  if ((name = fin_get_filename_dialog(view->window, "Open", view->filename, FALSE)) != NULL)
  {
    fin_view_load_file(view, name);
    g_free(name);
  }
}

static void
fin_on_save (GtkWidget * widget,
	     FinView   * view)
{
  fin_trace("");

  g_return_if_fail(view);

  if (FIN_HAVE_FILENAME(view) == FALSE)
    fin_on_save_as(NULL, view);
  else
    save_it(view);
}

static void
fin_on_save_as (GtkWidget * widget,
		FinView   * view)
{
  gchar * name;

  fin_trace("");

  g_return_if_fail(view);

  if ((name = fin_get_filename_dialog(view->window, "Save", view->filename, TRUE)) != NULL)
  {
    /* FIXME: what if we have problems saving the file?? */
    set_filename(view, name);
    save_it(view);
    g_free(name);
  }
}

static void
fin_on_about (GtkWidget * widget,
	      FinView * view)
{
  GtkWidget * about;
  GtkWindow * window = GTK_WINDOW(view->window);
  const gchar *authors[] = {"Darin Fisher, original author.", "Martin Fisher, maintainer.", "Torsten Landschoff, Debian packages.", NULL};

  fin_trace("");
  
  about = gnome_about_new(FIN_TITLE, FIN_VERSION, "(C) 1999 Darin Fisher",
    (const char **)authors, "A GNOME financial manager.\nhttp://jagger.berkeley.edu/~dfisher/gnofin", NULL);
  gnome_dialog_set_parent(GNOME_DIALOG(about), GTK_WINDOW(window));
  gtk_window_set_modal(GTK_WINDOW(about), TRUE);

  gtk_widget_show(about);
}

static gint
fin_on_close (GtkWidget * widget,
	      FinView   * view)
{
  fin_trace("");

  g_return_val_if_fail(view, FALSE);

  if (view->dirty)
  {
    switch (fin_question_dialog(view->window, "Save modifications before closing?"))
    {
    case FIN_YES:
      if (!save_it(view))
        return FALSE;
      break;
    case FIN_NO:
      fin_trace("discarding modifications per user request...");
      break;
    case -1:
    case FIN_CANCEL:
      return FALSE;
    }
  }
  close_it(view);
  return TRUE;
}

static void
fin_on_exit (GtkWidget * widget,
	     FinView   * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);

  if (fin_on_close(NULL, view))
  {
    fin_view_save_config(view);
    fin_exit();
  }
}

static gint
fin_on_delete_window (GtkWidget * widget,
		      GdkEvent  * event,
		      FinView   * view)
{
  fin_trace("");

  if (fin_on_close(NULL, view))
  {
    fin_view_save_config(view);
    fin_exit();
    return FALSE;
  }
  else
    return TRUE;  /* do not close window */
}

static void
fin_on_configure_window (GtkWidget         * widget,
			 GdkEventConfigure * event,
			 FinView           * view)
{
  fin_trace("");

  view->config.width = event->width;
  view->config.height = event->height;
}


/* ACCOUNT HANDLERS
 *
 * fin_put_account
 * fin_del_account
 * fin_on_real_account_new
 * fin_on_account_new
 * fin_on_account_delete
 * fin_on_real_account_props
 * fin_on_account_props
 * fin_on_switch_account
 * refresh_account_names
 * fin_find_account_page
 */

static void
fin_put_account (FinView * view, FinAccount * account, gint enable)
{
  GtkCList * clist;
  gint position;

  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(account);

  /* create new account set if necessary */
  if (view->set == NULL)
    view->set = fin_account_set_new();

  view->set->accounts =
    g_list_insert_sorted(view->set->accounts, account, (GCompareFunc) fin_account_sort_fcn);

  position = g_list_index(view->set->accounts, account);

  if (view->account == NULL)
  {
    view->account = account;
    enable_account_delete(view);
    enable_account_props(view);
    enable_record_view(view);
  }
  refresh_account_names(view);
  refresh_type_list(view);

  clist = create_list(view, account->name, position);
  refresh_list(view, clist, account, TRUE);

  if (enable)
  {
    /* switch to this account -- emits switch_page */
    gtk_notebook_set_page(GTK_NOTEBOOK(view->notebook), position);
  }
}

static FinAccount *
fin_del_account (FinView * view, FinAccount * account, gint position)
{
  GtkNotebook * notebook;
  GList * link;

  fin_trace("");

  g_return_val_if_fail(view, NULL);

  if (account == NULL)
  {
    if (position < 0)
    {
      account = view->account;
      position = view->page;
      link = g_list_nth(view->set->accounts, position);
    }
    else
    {
      /* get account at specified position */
      link = g_list_nth(view->set->accounts, position);
      account = LIST_GET(FinAccount, link);
    }
    g_return_val_if_fail(account, NULL);
    g_return_val_if_fail(link, NULL);
  }
  else
  {
    for (link=view->set->accounts, position=0; link; link=link->next, ++position)
    {
      if (LIST_GET(FinAccount, link) == account)
        break;
    }
    g_return_val_if_fail(link, NULL);
  }

  notebook = GTK_NOTEBOOK(view->notebook);

  /* remove list view page */
  {
    /* block switch_page signal, so we do not get out of sync */
    gtk_signal_handler_block(GTK_OBJECT(notebook), 
			     view->notebook_switch_page_id);

    /* this changes view->account */
    gtk_notebook_remove_page(notebook, position);

    gtk_signal_handler_unblock(GTK_OBJECT(notebook), 
			       view->notebook_switch_page_id);
  }

  /* remove account from account set */
  view->set->accounts = g_list_remove_link(view->set->accounts, link);

  if (view->set->accounts == NULL)
  {
    fin_trace("removing last account");
    view->clist = NULL;
    view->account = NULL;
    view->page = -1;

    disable_record_view(view);
    disable_account_delete(view);
    disable_account_props(view);
    disable_record_delete(view);
  }
  else
  {
    GtkBin * bin;

    /* figure out which account is visible */
    view->page = gtk_notebook_get_current_page(notebook);

    fin_trace("current_page = %d", view->page);

    /* set current account accordingly */
    view->account = 
      LIST_GET(FinAccount, g_list_nth(view->set->accounts, view->page));

    /* update list view */
    bin = GTK_BIN(gtk_notebook_get_nth_page(notebook, view->page));
    g_assert(GTK_IS_CLIST(bin->child));
    view->clist = GTK_CLIST(bin->child);
    refresh_list(view, NULL, NULL, TRUE);

    /* update edit view */
    refresh_check_no(view, TRUE);
    refresh_transfer_accts(view);
  }

  refresh_type_list(view);

  return account;
}

static gboolean
fin_verify_account_name (FinView     * view, 
			 const gchar * name, 
			 FinAccount  * mask)  /* if !NULL, do not compare against this */
{
  fin_trace("");

  g_return_val_if_fail(view, FALSE);

  fin_trace ("Name: %s", name);

  /* check if name not empty */
  if (name[0] == '\0')
  {
    fin_error_dialog(view->window,
    		     "The account name you have entered is invalid.\n"
                     "Please enter a non-empty account name.");
    return FALSE;
  }
 
  /* check if name already in use */
  if (view->set)
  {
    GList * it;

    for (it=view->set->accounts; it; it=it->next)
    {
      FinAccount * account = LIST_GET(FinAccount, it);

      if (account != mask && strcasecmp(account->name, name) == 0)
      {
        fin_error_dialog(view->window,
			 "The account name you have entered is invalid.\n"
	                 "Please enter an account name that is not already in use.");
        return FALSE;
      }
    }
  }

  fin_trace ("fin_verify_account_name return");

  return TRUE;
}

static gboolean
fin_on_real_account_new (GtkWidget * widget, FinView * view)
{
  FinAccount * account;
  gchar * name;
  gchar * info;

  fin_trace("");

  g_return_val_if_fail(view, FALSE);

  name = gtk_entry_get_text(GTK_ENTRY(view->account_name));
  info = gtk_entry_get_text(GTK_ENTRY(view->account_info));

  if (!fin_verify_account_name(view,name,NULL))
    return FALSE;

  account = fin_account_new(name, info);

  fin_put_account(view, account, TRUE);
  set_dirty(view, TRUE);

  /* add operation to history */
  {
    FinHistoryData * data = g_new0(FinHistoryData, 1);

    data->type = FIN_OP_INSERT_ACCOUNT;
    data->account = account;

    fin_account_ref(account);
    fin_history_remember(FIN_HISTORY(view->history), data);

    enable_edit_undo(view);
    disable_edit_redo(view);
  }

  return TRUE;
}

static void
fin_on_account_new (GtkWidget * widget,
		    FinView   * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);

  if (view->account_dialog == NULL)
    create_account_info_dialog(view);

  gtk_entry_set_text(GTK_ENTRY(view->account_name), "");
  gtk_entry_set_text(GTK_ENTRY(view->account_info), "");

  while (gnome_dialog_run(GNOME_DIALOG(view->account_dialog)) == 0 &&
         fin_on_real_account_new(widget,view) == FALSE) ;

  gtk_widget_hide(view->account_dialog);
}

static void
fin_on_account_delete (GtkWidget   * widget,
		       FinView * view)
{
  FinAccount * account;

  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(view->account);

  if (view->set == NULL)
    return;

  account = view->account;

  fin_del_account(view, account, -1);
  set_dirty(view, TRUE);

  /* add operation to history */
  {
    FinHistoryData * data = g_new0(FinHistoryData, 1);

    data->type = FIN_OP_DELETE_ACCOUNT;
    data->account = account;

    fin_history_remember(FIN_HISTORY(view->history), data);

    enable_edit_undo(view);
    disable_edit_redo(view);
  }
}

static gboolean
fin_on_real_account_props (GtkWidget * widget,
			   FinView   * view)
{
  gchar * name;
  gchar * info;

  fin_trace("");

  g_return_val_if_fail(view, FALSE);

  name = gtk_entry_get_text(GTK_ENTRY(view->account_name));
  info = gtk_entry_get_text(GTK_ENTRY(view->account_info));

  if (!fin_verify_account_name(view,name,view->account))
    return FALSE;

  /* remember operation */
  {
    FinHistoryData * data = g_new0(FinHistoryData, 1);
    
    data->type = FIN_OP_UPDATE_ACCOUNT;
    data->account = view->account;

    /* allocate a place-holder for account name + info */
    data->account_old = g_new0(FinAccount, 1);
    data->account_old->name = view->account->name;
    data->account_old->info = view->account->info;
    
    fin_history_remember(view->history, data);

    enable_edit_undo(view);
    disable_edit_redo(view);
  }

  view->account->name = g_strdup(gtk_entry_get_text(GTK_ENTRY(view->account_name)));
  view->account->info = g_strdup(gtk_entry_get_text(GTK_ENTRY(view->account_info)));

  set_dirty(view, TRUE);

  gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(view->notebook),
  				  gtk_notebook_get_nth_page(GTK_NOTEBOOK(view->notebook), view->page),
				  view->account->name);

  return TRUE;
}

static void
fin_on_account_props (GtkWidget * widget,
		      FinView   * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->account != NULL);

  if (view->account_dialog == NULL)
    create_account_info_dialog(view);
  
  gtk_entry_set_text(GTK_ENTRY(view->account_name), view->account->name);
  gtk_entry_set_text(GTK_ENTRY(view->account_info), view->account->info);

  while (gnome_dialog_run(GNOME_DIALOG(view->account_dialog)) == 0 &&
         fin_on_real_account_props(widget,view) == FALSE) ;

  gtk_widget_hide(view->account_dialog);
}

static void
fin_on_switch_account (GtkWidget       * notebook,
		       GtkNotebookPage * page,
		       guint		 page_num,
		       FinView         * view)
{
  fin_trace("page_num = %d", page_num);

  g_return_if_fail(view != NULL);

  view->page = page_num;
  view->account = LIST_GET(FinAccount, 
  			   g_list_nth(view->set->accounts, page_num));
  view->clist = GTK_CLIST(GTK_BIN(page->child)->child);

  refresh_list(view, NULL, NULL, TRUE);
  refresh_check_no(view, TRUE);
  refresh_transfer_accts(view);
  refresh_cleared_bal(view);

  set_enable_record_delete(view, view->account->records != NULL);
  set_enable_record_update(view, view->account->records != NULL);
}

static void
refresh_account_names (FinView * view)
{
  GList * it;
  int i, num;

  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->set != NULL);

  if (view->account_names)
    g_free(view->account_names);

  num = g_list_length(view->set->accounts);
  view->account_names = g_new0(gchar *, num + 1);

  for (i=0, it=view->set->accounts; i<num; ++i, it=it->next)
    view->account_names[i] = LIST_GET(FinAccount, it)->name;

  refresh_transfer_accts(view);
}

static GtkWidget *
fin_find_account_page (FinView    * view,
		       FinAccount * account)
{
  fin_trace("");

  g_return_val_if_fail(view, NULL);
  g_return_val_if_fail(account, NULL);

  return gtk_notebook_get_nth_page(GTK_NOTEBOOK(view->notebook), 
  				   g_list_index(view->set->accounts, account));
}


/* LIST HANDLERS
 *
 * create_negative_balance_style
 * refresh_list
 * fin_update_status_display
 * fin_on_key_press
 * fin_on_select_record
 * fin_on_unselect_record
 */

static void
create_negative_balance_style (FinView * view)
{
  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(view->clist);

  /* indicate negative balance with red text */
  if (view->negative_balance_style == NULL)
  {
    GdkColor col;

    col.red = 0xffff;
    col.green = 0;
    col.blue = 0;

    view->negative_balance_style = gtk_style_copy(GTK_WIDGET(view->clist)->style);
    view->negative_balance_style->fg[GTK_STATE_NORMAL] = col;
    view->negative_balance_style->fg[GTK_STATE_SELECTED] = col;
  }
}

static void
refresh_list (FinView    * view,
	      GtkCList   * clist,
	      FinAccount * account,
	      gboolean     recalc_bal)
{
  fin_trace("");

  g_return_if_fail(view);

  if (clist == NULL)
    clist = view->clist;
  if (account == NULL)
    account = view->account;

  if (clist)
  {
    GtkScrolledWindow * scrollwin;
    gint selection = -1;
    GtkAdjustment * vadjust, * hadjust;
    gfloat vval, hval;

    if (clist->selection)
      selection = GPOINTER_TO_INT(clist->selection->data);

    scrollwin = GTK_SCROLLED_WINDOW(GTK_WIDGET(clist)->parent);
    vadjust = gtk_clist_get_vadjustment(clist);
    hadjust = gtk_clist_get_hadjustment(clist);

    vval = vadjust->value;
    hval = hadjust->value;

    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);

    if (account)
    {
      FinRecord * last_record = NULL;
      GList * it;

      for (it=account->records; it; it=it->next)
      {
	gint row;
	char ** text;
	FinRecord * record = LIST_GET(FinRecord, it);

	if (recalc_bal)
	{
	  if (last_record != NULL)
	    record->balance = last_record->balance + record->amount;
	  else
	    record->balance = record->amount;
	}

	text = fin_stringized_record_new(record, clist_types, sizeof_array(clist_types));
	row = gtk_clist_append(clist, text);

	if (record->balance < 0)
	{
	  if (view->negative_balance_style == NULL)
	    create_negative_balance_style(view);

	  gtk_clist_set_cell_style(clist, row, 5, view->negative_balance_style);
	}

	last_record = record;
      }
    }

    gtk_clist_thaw(clist);

    gtk_adjustment_set_value(vadjust, vval);
    gtk_adjustment_set_value(hadjust, hval);

    gtk_clist_set_vadjustment(clist, vadjust);
    gtk_clist_set_hadjustment(clist, hadjust);

    if (selection != -1)
    {
      gtk_clist_select_row(clist, selection, -1);
      gtk_widget_queue_resize(GTK_WIDGET(clist));
    }
  }
}

static void
fin_update_status_display (FinView    * view,
			   FinAccount * account,
			   FinRecord  * record)
{
  FinRecordField type = FIN_RECORD_FIELD_STATUS;
  gchar ** text;

  text = fin_stringized_record_new(record, &type, 1);

  gtk_clist_set_text(GTK_CLIST(view->clist),
		     view->selection,
		     3, text[0]);

  fin_stringized_record_free(text);
}

static void
fin_on_toggle_status (GtkWidget * widget,
		      FinView   * view)
{
  GList * it;
  FinRecord * record;

  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(view->selection != -1);
  
  it = g_list_nth(view->account->records, view->selection);
  record = LIST_GET(FinRecord, it);

  /* fin_record_toggle_status(record); */
  record->cleared = !record->cleared;

  fin_update_status_display(view, view->account, record);
  set_dirty(view, TRUE);

  /* remember record */
  {
    FinHistoryData * data = g_new0(FinHistoryData, 1);

    data->type = FIN_OP_TOGGLE_RECORD_STATUS;
    data->account = view->account;
    data->record = record;

    fin_history_remember(view->history, data);

    enable_edit_undo(view);
    disable_edit_redo(view);
  }

  refresh_cleared_bal(view);
}

static void
fin_on_key_press (GtkWidget   * clist,
		  GdkEventKey * event,
		  FinView     * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);

  if (view->selection != -1)
  {
    switch (event->keyval)
    {
    case GDK_C:
    case GDK_c:
      fin_on_toggle_status(NULL,view);
      break;
    case GDK_Delete:
    case GDK_BackSpace:
      fin_on_delete_record(NULL, view);
      break;
    }
  }
}

static void
fin_on_select_record (GtkWidget *	clist,
		      gint		row,
		      gint		column,
		      GdkEventButton *	event,
		      FinView *	view)
{
  g_return_if_fail(view);

  fin_trace("selecting record (%i,%i)", row, column);

  /* load selection into edit panel, provided... */
  if (clist == GTK_WIDGET(view->clist))
  {
    static const FinRecordField types[] =
    {
      FIN_RECORD_FIELD_DATE,
      FIN_RECORD_FIELD_TYPE,
      FIN_RECORD_FIELD_TYPE_DATA,
      FIN_RECORD_FIELD_INFO,
      FIN_RECORD_FIELD_AMOUNT,
    };
    FinRecord * record;
    gchar ** text;

    fin_trace("row = %d", row);

    view->selection = row;

    enable_record_update(view);
    enable_record_delete(view);
    enable_record_toggle_status(view);

    record = LIST_GET(FinRecord, g_list_nth(view->account->records, row));

    text = fin_stringized_record_new(record, types, sizeof_array(types));

    gtk_entry_set_text(GTK_ENTRY(view->record_date), text[0]);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(view->record_source)->entry), text[1]);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(view->record_info)->entry), text[3]);
    gtk_entry_set_text(GTK_ENTRY(view->record_amount), text[4]);

    fin_on_record_type_changed(GTK_COMBO(view->record_source)->entry, view);

    switch (view->current_record_type)
    {
    case FIN_RECORD_TYPE_CHK:
      gtk_entry_set_text(GTK_ENTRY(view->record_check_no), text[2]);
      break;
    case FIN_RECORD_TYPE_XFR:
      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(view->record_transfer_acct)->entry), text[2]);
      break;
    default:
    }
  }
}

static void
fin_on_unselect_record (GtkWidget *	 clist,
		        gint		 row,
		        gint		 column,
		        GdkEventButton * event,
		        FinView *	 view)
{
  g_return_if_fail(view != NULL);

  if (view->selection == row)
  {
    view->selection = -1;
    disable_record_delete(view);
    disable_record_toggle_status(view);
  }
  
  fin_trace("unselecting record %i", row);

  disable_record_update(view);
}

static void
fin_on_resize_column (GtkWidget * clist,
		      gint	  column,
		      gint	  width,
		      FinView   * view)
{
  fin_trace("");

  view->config.col_widths[column] = width;
}


/* EDIT VIEW HANDLERS
 *
 * fin_on_record_type_changed
 * fin_on_info_key_press
 * fin_on_insert_info_text 
 * fin_update_status_display 
 * refresh_info
 * refresh_check_no 
 * refresh_transfer_accts 
 */

static void
fin_on_record_type_changed (GtkWidget * entry,
		            FinView   * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);

  view->current_record_type =
    fin_record_parse_type(
      gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(view->record_source)->entry)));

  switch (view->current_record_type)
  {
  case FIN_RECORD_TYPE_XFR:
    gtk_widget_hide(view->record_check_no);
    gtk_widget_show(view->record_transfer_acct);

    gtk_widget_set_sensitive(view->record_check_no, FALSE);
    gtk_widget_set_sensitive(view->record_transfer_acct, TRUE);
    break;
  case FIN_RECORD_TYPE_CHK:
    gtk_widget_hide(view->record_transfer_acct);
    gtk_widget_show(view->record_check_no);

    gtk_widget_set_sensitive(view->record_check_no, TRUE);
    gtk_widget_set_sensitive(view->record_transfer_acct, FALSE);

    refresh_check_no(view, FALSE);
    break;
  default:
    gtk_widget_hide(view->record_transfer_acct);
    gtk_widget_show(view->record_check_no);

    gtk_widget_set_sensitive(view->record_check_no, FALSE);
    gtk_widget_set_sensitive(view->record_transfer_acct, FALSE);

    gtk_entry_set_text(GTK_ENTRY(view->record_check_no), "");
  }
}

static void
fin_on_info_key_press (GtkWidget   * widget,
		       GdkEventKey * event,
		       FinView     * view)
{
  GtkEntry * entry = GTK_ENTRY(widget);
  gchar * text;

  if (event->keyval == GDK_BackSpace || event->keyval == GDK_Delete)
    return; /* dont pattern match */

  if (view->info_key_press_skip)
  {
    fin_trace("%x", event->keyval);
    if (event->keyval == GDK_Tab || event->keyval == GDK_ISO_Left_Tab)
    {
      fin_trace("switching focus");
      if (event->state & GDK_SHIFT_MASK)
      {
        /* focus left */
	if (view->record_check_no->state != GTK_STATE_INSENSITIVE)
	  gtk_widget_grab_focus(view->record_check_no);
	else if (view->record_transfer_acct->state != GTK_STATE_INSENSITIVE)
	  gtk_widget_grab_focus(GTK_COMBO(view->record_transfer_acct)->entry);
	else
	  gtk_widget_grab_focus(GTK_COMBO(view->record_source)->entry);
      }
      else
      {
        /* focus right */
	gtk_widget_grab_focus(view->record_amount);
      }
      
      /* clear selection */
      gtk_entry_select_region(GTK_ENTRY(GTK_COMBO(view->record_info)->entry), 0, 0);
    }
    else
      fin_trace("quick return");
    return;
  }
  else
    view->info_key_press_skip = 1;

  text = gtk_entry_get_text(entry);
  fin_trace("text=%s",text);

  /* try to complete info string */
  if (view->set->info_cache != NULL && strlen(text) > 0)
  {
    GList * it;
    gint length = strlen(text);

    for (it=view->set->info_cache; it; it=it->next)
    {
      FinRecordInfo * info = LIST_GET(FinRecordInfo, it);
      int n = strlen(info->string);
      int match = (strncmp(text, info->string, length)==0);

      fin_trace("comparing against %s", info->string);

      /* case sensitive comparison */
      if (n==length && match)
        return;
      else if (n>length && match)
      {
        /* found first alphabetical match */
	gtk_signal_handler_block(GTK_OBJECT(widget), view->info_insert_text_id);
	gtk_entry_set_text(entry, info->string);
	gtk_signal_handler_unblock(GTK_OBJECT(widget), view->info_insert_text_id);
	gtk_entry_select_region(entry, length, -1);
	return;
      }
    }
  }
}

static void
fin_on_insert_info_text (GtkWidget   * widget,
			 const gchar * text,
			 gint          length,
			 gint        * position,
			 FinView     * view)
{
  g_return_if_fail(view != NULL);
  g_return_if_fail(view->set != NULL);

  view->info_key_press_skip = 0;
}

static void
refresh_info(FinView * view)
{
  GList * strings = NULL;

  fin_trace("");

  g_return_if_fail(view);

  if (view->set->info_cache != NULL)
  {
    strings = fin_record_info_stringize(view->set->info_cache);
    gtk_combo_set_popdown_strings(GTK_COMBO(view->record_info), strings);
  }
  else
    gtk_list_clear_items(GTK_LIST(GTK_COMBO(view->record_info)->list), 0, -1);

  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(view->record_info)->entry), "");
}

static void
refresh_check_no (FinView * view, gboolean scan)
{

  fin_trace("");

  g_return_if_fail(view);

  if (view->account == NULL)
  {
    gtk_entry_set_text(GTK_ENTRY(view->record_check_no), "");
    return;
  }

  if (scan)
  {
    GList * it;

    g_return_if_fail(view->account != NULL);

    view->next_check_no = 0;
    for (it=view->account->records; it; it=it->next)
    {
      FinRecord * record = LIST_GET(FinRecord, it);

      if (record->type == FIN_RECORD_TYPE_CHK)
      {
        if (FIN_RECORD_CHECK_NO(record) >= view->next_check_no)
	  view->next_check_no = FIN_RECORD_CHECK_NO(record) + 1;
      }
    }
  }

  if (view->current_record_type == FIN_RECORD_TYPE_CHK)
  {
    gchar buf[FIN_RECORD_FIELD_MAXLEN_TYPE_DATA];
    snprintf(buf, FIN_RECORD_FIELD_MAXLEN_TYPE_DATA, "%d", view->next_check_no);
    gtk_entry_set_text(GTK_ENTRY(view->record_check_no), buf);
  }
  else
    gtk_entry_set_text(GTK_ENTRY(view->record_check_no), "");
}

static void
refresh_transfer_accts (FinView * view)
{
  GList * it;
  GList * strings = NULL;

  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->set != NULL);

  for (it=view->set->accounts; it; it=it->next)
  {
    FinAccount * account = LIST_GET(FinAccount, it);

    g_return_if_fail(view->account != NULL);

    if (account != view->account)
      strings = g_list_append(strings, account->name);
  }

  if (strings == NULL)
  {
    gtk_list_clear_items(GTK_LIST(GTK_COMBO(view->record_transfer_acct)->list), 0, -1);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(view->record_transfer_acct)->entry), "<none>");
    gtk_widget_set_sensitive(GTK_COMBO(view->record_transfer_acct)->entry, FALSE);
    gtk_widget_set_sensitive(GTK_COMBO(view->record_transfer_acct)->button, FALSE);
    gtk_widget_hide(GTK_COMBO(view->record_transfer_acct)->button);
  }
  else
  {
    gtk_combo_set_popdown_strings(GTK_COMBO(view->record_transfer_acct), strings);
    gtk_widget_set_sensitive(GTK_COMBO(view->record_transfer_acct)->entry, TRUE);
    gtk_widget_set_sensitive(GTK_COMBO(view->record_transfer_acct)->button, TRUE);
    
    if (g_list_length(strings) == 1)
      gtk_widget_hide(GTK_COMBO(view->record_transfer_acct)->button);
    else
      gtk_widget_show(GTK_COMBO(view->record_transfer_acct)->button);
  }
}

static void
refresh_type_list (FinView * view)
{
  GList * types;
  gint multi_accounts;

  fin_trace("");

  g_return_if_fail(view);

  multi_accounts = g_list_length(view->set->accounts) > 1;
  types = fin_record_stringize_types(multi_accounts);
  gtk_combo_set_popdown_strings(GTK_COMBO(view->record_source), types);
  g_list_free(types);
}

void
refresh_cleared_bal (FinView * view)
{
  fin_trace("");

  g_return_if_fail(view);

  if (view->account)
  {
    gchar buf[80], bal[89];

    fin_account_refresh_cleared_bal(view->account); /* FIXME: this is not efficient */
    fin_money_stringize(bal, sizeof(bal), view->account->cleared_bal);
    snprintf(buf, sizeof(buf), "cleared  %s", bal);
    gtk_label_set_text(GTK_LABEL(view->cleared_bal), buf);
  }
  else
    gtk_label_set_text(GTK_LABEL(view->cleared_bal), "cleared  0");
}


/* HISTORY HANDLERS
 *
 * fin_on_undo
 * fin_on_redo
 * fin_on_history_free_item
 * fin_on_history_undo_item 
 * fin_on_history_redo_item 
 */

static void
fin_on_undo (GtkWidget * widget, FinView * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->history != NULL);
  g_return_if_fail(FIN_IS_HISTORY(view->history));
  g_return_if_fail(view->history->can_undo);

  fin_history_undo(view->history);

  if (!FIN_HISTORY_CAN_UNDO(view->history))
    disable_edit_undo(view);
  enable_edit_redo(view);
}

static void
fin_on_redo (GtkWidget * widget, FinView * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->history != NULL);
  g_return_if_fail(FIN_IS_HISTORY(view->history));
  g_return_if_fail(view->history->can_redo);

  fin_history_redo(view->history);

  if (!FIN_HISTORY_CAN_REDO(view->history))
    disable_edit_redo(view);
  enable_edit_undo(view);
}

static void
fin_on_history_free_item (FinHistory * history, 
			  gpointer     item,
			  FinView    * view)
{
  FinHistoryData * data;

  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(item != NULL);

  data = (FinHistoryData *) item;

  if (FIN_IS_RECORD_OP(data->type))
  {
    fin_record_unref(data->record);

    if (data->type == FIN_OP_UPDATE_RECORD)
      fin_record_unref(data->record_old);
  }
  else
    fin_account_unref(data->account);

  g_free(data);
}

static void
fin_on_history_undo_item (FinHistory * history, 
			  gpointer     item,
			  FinView    * view)
{
  FinHistoryData * data;

  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(item != NULL);

  data = (FinHistoryData *) item;

  switch (data->type)
  {
  case FIN_OP_INSERT_RECORD:
    /* uninsert record --> simple delete */
    del_record(view, data->account, data->record, -1);
    fin_record_unref(data->record);
    del_link_if_exists(view, data->record);
    break;
  case FIN_OP_UPDATE_RECORD:
    fin_record_swap(data->record, data->record_old);
    put_record(view, data->account, data->record, FALSE);
    /* 3 cases: already an XFR, no longer an XFR, now an XFR */
    if (FIN_RECORD_IS_XFR(data->record) && FIN_RECORD_IS_XFR(data->record_old))
    {
      del_link_if_exists(view, data->record_old);
      put_link_if_exists(view, data->record, FALSE);
    }
    else if (FIN_RECORD_IS_XFR(data->record_old))
    {
      del_link_if_exists(view, data->record_old);
    }
    else if (FIN_RECORD_IS_XFR(data->record))
    {
      put_link_if_exists(view, data->record, TRUE);
    }
    break;
  case FIN_OP_DELETE_RECORD:
    /* undelete record --> simple insert */
    put_record(view, data->account, data->record, TRUE);
    fin_record_ref(data->record);
    put_link_if_exists(view, data->record, TRUE);
    break;
  case FIN_OP_TOGGLE_RECORD_STATUS:
    data->record->cleared = !data->record->cleared;
    put_record(view, data->account, data->record, FALSE);
    break;
  case FIN_OP_INSERT_ACCOUNT:
    fin_del_account(view, data->account, -1);
    fin_account_unref(data->account);
    break;
  case FIN_OP_UPDATE_ACCOUNT:
    {
      GtkWidget * page;
      gchar * temp;

      page = fin_find_account_page(view, data->account);
      g_assert(page);

      temp = data->account->name;
      data->account->name = data->account_old->name;
      data->account_old->name = temp;

      temp = data->account->info;
      data->account->info = data->account_old->info;
      data->account_old->info = temp;

      gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(view->notebook),
      				      page, data->account->name);
    }
    break;
  case FIN_OP_DELETE_ACCOUNT:
    fin_put_account(view, data->account, FALSE);
    fin_account_ref(data->account);
    break;
  }

  set_dirty(view, FALSE);
}

static void
fin_on_history_redo_item (FinHistory * history, 
			  gpointer     item,
			  FinView    * view)
{
  FinHistoryData * data;

  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(item != NULL);

  data = (FinHistoryData *) item;

  switch (data->type)
  {
  case FIN_OP_INSERT_RECORD:
    put_record(view, data->account, data->record, TRUE);
    fin_record_ref(data->record);
    put_link_if_exists(view, data->record, TRUE);
    break;
  case FIN_OP_UPDATE_RECORD:
    fin_record_swap(data->record, data->record_old);
    put_record(view, data->account, data->record, FALSE);
    /* 3 cases: already an XFR, no longer an XFR, now an XFR */
    if (FIN_RECORD_IS_XFR(data->record) && FIN_RECORD_IS_XFR(data->record_old))
    {
      del_link_if_exists(view, data->record_old);
      put_link_if_exists(view, data->record, FALSE);
    }
    else if (FIN_RECORD_IS_XFR(data->record_old))
    {
      del_link_if_exists(view, data->record_old);
    }
    else if (FIN_RECORD_IS_XFR(data->record))
    {
      put_link_if_exists(view, data->record, TRUE);
    }
    break;
  case FIN_OP_DELETE_RECORD:
    /* re-delete record */
    del_record(view, data->account, data->record, -1);
    fin_record_unref(data->record); /* since record is no longer being displayed */
    del_link_if_exists(view, data->record);
    break;
  case FIN_OP_TOGGLE_RECORD_STATUS:
    data->record->cleared = !data->record->cleared;
    put_record(view, data->account, data->record, FALSE);
    break;
  case FIN_OP_INSERT_ACCOUNT:
    fin_put_account(view, data->account, FALSE);
    fin_account_ref(data->account);
    break;
  case FIN_OP_UPDATE_ACCOUNT:
    {
      GtkWidget * page;
      gchar * temp;

      page = fin_find_account_page(view, data->account);
      g_assert(page);

      temp = data->account->name;
      data->account->name = data->account_old->name;
      data->account_old->name = temp;

      temp = data->account->info;
      data->account->info = data->account_old->info;
      data->account_old->info = temp;

      gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(view->notebook),
      				      page, data->account->name);
    }
    break;
  case FIN_OP_DELETE_ACCOUNT:
    fin_del_account(view, data->account, -1);
    break;
  }

  set_dirty(view, TRUE);
}


/* MISCELLANEOUS
 *
 * set_dirty
 * put_filename
 * set_filename
 * fin_on_dump_state
 */

static void
set_dirty (FinView * view, gint dirty)
{
  fin_trace("%d", dirty);

  g_return_if_fail(view);

  if (dirty < 0)
    view->dirty = 0;  /* reset */
  else
  { 
    if (dirty)
      view->dirty++;
    else
      view->dirty--;
  }

  fin_trace("view->dirty = %d", view->dirty);

  set_enable_file_save(view, view->dirty != 0);
  put_filename(view, view->filename);
}

static void
put_filename (FinView * view, const gchar * filename)
{
  gchar * title = NULL;
  gchar dirty_indicator = ' ';

  fin_trace("");

  g_return_if_fail(view);

  if (view->dirty != 0)
    dirty_indicator = '*';

  if (filename)
    title = g_strdup_printf("%s - %s%c", FIN_TITLE, filename, dirty_indicator);
  else
    title = g_strdup_printf("%s%c", FIN_TITLE, dirty_indicator);

  gtk_window_set_title(GTK_WINDOW(view->window), title);
  g_free(title);
}

static void
set_filename (FinView * view, const gchar * filename)
{
  fin_trace("filename = %s", filename);

  g_return_if_fail(view);

  if (FIN_HAVE_FILENAME(view))
  {
    g_free(view->filename);
    view->filename = NULL;
  }
  if (filename)
    view->filename = g_strdup(filename);

  put_filename(view, filename);
}

#ifdef FIN_DUMP_ENABLE

static void
fin_on_dump_state (GtkWidget * widget,
		   FinView   * view)
{
  fin_trace("");

  fin_account_set_dump(view->set, stdout);

  printf("account  \t%p\n", view->account);
  printf("filename \t\"%s\"\n", view->filename);
  printf("selection\t%d\n", view->selection);
  printf("page     \t%d\n", view->page);
  printf("dirty    \t%d\n", view->dirty);

  fin_history_dump(view->history);

#ifdef FIN_DUMP_TOOLBAR
  /* probe toolbar */
  {
    GnomeDockItem * item;
    int i;

    item = gnome_dock_get_item_by_name(GNOME_DOCK(GNOME_APP(view->window)->dock),
    				       GNOME_APP_TOOLBAR_NAME,
				       NULL, NULL, NULL, NULL);
    printf("\ntoolbar probe\n");

    i = GNOME_IS_DOCK_ITEM(item);
    printf("GNOME_IS_DOCK_ITEM(item)\t%d\n", i);

    if (i)
    {
      GList * it;
      GtkWidget * toolbar = gnome_dock_item_get_child(GNOME_DOCK_ITEM(item));
      
      i = GTK_IS_TOOLBAR(toolbar);
      printf("GTK_IS_TOOLBAR(toolbar)\t\t%d\n", i);

      if (i)
      {
        printf("toolbar has %d children\n", GTK_TOOLBAR(toolbar)->num_children);
	for (it=GTK_TOOLBAR(toolbar)->children; it; it=it->next)
	{
	  GtkToolbarChild * child = LIST_GET(GtkToolbarChild, it);
	  printf("  child type = %d\n", child->type);
	}
      }
    }
  }
#endif /* ! FIN_DUMP_TOOLBAR */

#ifdef FIN_DUMP_MENUS
  /* probe menus */
  {
    GtkWidget * menushell = GNOME_APP(view->window)->menubar;
    int i;

    i = GTK_IS_MENU_SHELL(menushell);
    printf("GTK_IS_MENU_SHELL(menushell)\t\t%d\n", i); 

    if (i)
    {
      GList * it;
      
      for (it=GTK_MENU_SHELL(menushell)->children; it; it=it->next)
      {
        i = GTK_IS_MENU_ITEM(LIST_GET(GtkWidget, it));
	printf("  menushell->child is a menuitem\t%d\n", i);

	if (i)
	{
	  GtkMenuItem * menuitem = LIST_GET(GtkMenuItem, it);

	  i = GTK_IS_MENU_SHELL(menuitem->submenu);
	  printf("  menuitem->submenu is a menu shell\t%d\n", i);

	  if (i)
	  {
	    GList * it2 = NULL;
	    GtkMenuShell * ms = GTK_MENU_SHELL(menuitem->submenu);

	    if (ms != NULL)
	    {
	      for (it2=ms->children; it2; it2=it2->next)
	      {
		GtkWidget * ww;
		GtkMenuItem * ii = LIST_GET(GtkMenuItem, it2);

		if (ii != NULL)
		{
		  i = GTK_IS_MENU_ITEM(ii);
		  printf("    is menu item: %i\t(%s)\n", i, gtk_type_name(GTK_OBJECT_TYPE(ii)));

		  if (i)
		  {
		    ww = GTK_BIN(ii)->child;
		    if (ww != NULL)
		      printf("      child type\t%s\n", gtk_type_name(GTK_OBJECT_TYPE(ww)));
		  }
		} 
	      }
	    }
	  }
	}
      }
    }
  }
#endif /* FIN_DUMP_MENUS */
}

#endif /* FIN_DUMP_ENABLE */


/* GUI CREATION
 *
 * create_list
 * create_window
 * create_account_info_dialog 
 */

static GtkCList *
create_list (FinView     * view,
             const gchar * name,
	     gint	   position)
{
  GtkWidget * scrolled_win;
  GtkWidget * clist;

  fin_trace("");

  /* create scrolled window */
  {
    scrolled_win = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
				   GTK_POLICY_AUTOMATIC, 
				   GTK_POLICY_AUTOMATIC);
    gtk_widget_show(scrolled_win);

    view->scroll = scrolled_win;
  }

  /* create list window */
  {
    int i;

    clist = gtk_clist_new_with_titles(FIN_VIEW_NUMCOLS, clist_titles);
    for (i=0; i<FIN_VIEW_NUMCOLS; ++i)
      gtk_clist_set_column_width(GTK_CLIST(clist), i, view->config.col_widths[i]);
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_clist_set_column_justification(GTK_CLIST(clist), 4, GTK_JUSTIFY_RIGHT);
    gtk_clist_set_column_justification(GTK_CLIST(clist), 5, GTK_JUSTIFY_RIGHT);
    gtk_clist_column_titles_passive(GTK_CLIST(clist));
    gtk_container_add(GTK_CONTAINER(scrolled_win), clist);
    gtk_widget_show(clist);

    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
		       GTK_SIGNAL_FUNC(fin_on_select_record), view);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
		       GTK_SIGNAL_FUNC(fin_on_unselect_record), view);
    gtk_signal_connect(GTK_OBJECT(clist), "key_press_event",
    		       GTK_SIGNAL_FUNC(fin_on_key_press), view);
    gtk_signal_connect(GTK_OBJECT(clist), "resize_column",
    		       GTK_SIGNAL_FUNC(fin_on_resize_column), view);
  }

  /* append to notebook */
  {
    gtk_notebook_insert_page(GTK_NOTEBOOK(view->notebook),
      scrolled_win, gtk_label_new(name), position);
  }

  return GTK_CLIST(clist);
}

static void
create_window (FinView * view)
{
  GtkWidget * window;
  GtkWidget * vbox;
  GtkWidget * notebook;

  fin_trace("");

  /* create main window */
  {
    window = gnome_app_new(prog, FIN_TITLE);
    gtk_window_set_policy(GTK_WINDOW(window), 1, 1, 0);
    gtk_window_set_default_size(GTK_WINDOW(window), view->config.width, view->config.height);

    //gtk_widget_realize(window);
    gtk_signal_connect(GTK_OBJECT(window), "delete_event",
		       GTK_SIGNAL_FUNC(fin_on_delete_window), view);
    gtk_signal_connect(GTK_OBJECT(window), "configure_event",
		       GTK_SIGNAL_FUNC(fin_on_configure_window), view);
    gtk_widget_set_usize(window, 100, 40);

    gtk_widget_show(window);

    view->window = window;
  }

  /* create menus */
  {
    GnomeUIInfo imports_menu[] =
    {
      GNOMEUIINFO_END
    };

    GnomeUIInfo exports_menu[] =
    {
      GNOMEUIINFO_END
    };
    GnomeUIInfo file_menu[] =
    {
      GNOMEUIINFO_MENU_NEW_ITEM("_New ", "Begin new register", fin_on_new, view),
      GNOMEUIINFO_MENU_OPEN_ITEM(fin_on_open, view),
      GNOMEUIINFO_MENU_SAVE_ITEM(fin_on_save, view),
      GNOMEUIINFO_MENU_SAVE_AS_ITEM(fin_on_save_as, view),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_SUBTREE(N_("_Import"), imports_menu),
      GNOMEUIINFO_SUBTREE(N_("_Export"), exports_menu),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_MENU_EXIT_ITEM(fin_on_exit, view),
      GNOMEUIINFO_END
    };
    enum file_indices {
      file_new_idx,
      file_open_idx,
      file_save_idx,
      file_save_as_idx,
      file_imports_idx=5,
      file_exports_idx,
    };
    GnomeUIInfo edit_menu[] = 
    {
      GNOMEUIINFO_MENU_UNDO_ITEM(fin_on_undo, view),
      GNOMEUIINFO_MENU_REDO_ITEM(fin_on_redo, view),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_MENU_CUT_ITEM(fin_on_nop, view),
      GNOMEUIINFO_MENU_COPY_ITEM(fin_on_nop, view),
      GNOMEUIINFO_MENU_PASTE_ITEM(fin_on_nop, view),
      GNOMEUIINFO_MENU_SELECT_ALL_ITEM(fin_on_nop, view),
      GNOMEUIINFO_SEPARATOR,
      GNOMEUIINFO_MENU_FIND_ITEM(fin_on_nop, view),
      GNOMEUIINFO_MENU_FIND_AGAIN_ITEM(fin_on_nop, view),
      GNOMEUIINFO_END
    };
    enum edit_indices {
      edit_undo_idx,
      edit_redo_idx,
      edit_cut_idx = 3,
      edit_copy_idx,
      edit_paste_idx,
      edit_select_all_idx,
      edit_find_idx = 8,
      edit_find_again_idx
    };
    GnomeUIInfo account_menu[] =
    {
      { GNOME_APP_UI_ITEM, N_("_New"), NULL,
        fin_on_account_new, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
      { GNOME_APP_UI_ITEM, N_("_Delete"), NULL,
        fin_on_account_delete, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("_Properties"), NULL,
        fin_on_account_props, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP, 0, 0, NULL },
      GNOMEUIINFO_END
    };
    enum account_indices {
      account_new_idx,
      account_delete_idx,
      account_props_idx = 3
    };
    GnomeUIInfo help_menu[] =
    {
      GNOMEUIINFO_MENU_ABOUT_ITEM(fin_on_about, view),
#ifdef FIN_DUMP_ENABLE
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, N_("Dump _State"), NULL,
        fin_on_dump_state, view, NULL,
        GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
#endif
      GNOMEUIINFO_END
    };
    GnomeUIInfo main_menu[] =
    {
      GNOMEUIINFO_MENU_FILE_TREE(file_menu),
      GNOMEUIINFO_MENU_EDIT_TREE(edit_menu),
      GNOMEUIINFO_SUBTREE(N_("_Account"), account_menu),
      GNOMEUIINFO_MENU_HELP_TREE(help_menu),
      GNOMEUIINFO_END
    };
    const struct
    {
      GtkWidget ** ptr;
      GnomeUIInfo * tree;
      gint index;
    }
    menus[] =
    {
      { &view->menu_file_save,         file_menu,      file_save_idx },
      { &view->menu_file_save_as,      file_menu,      file_save_as_idx },
      { &view->menu_file_imports,      file_menu,      file_imports_idx },
      { &view->menu_file_exports,      file_menu,      file_exports_idx },
      { &view->menu_edit_undo,         edit_menu,      edit_undo_idx },
      { &view->menu_edit_redo,         edit_menu,      edit_redo_idx },
      { &view->menu_edit_cut,          edit_menu,      edit_cut_idx },
      { &view->menu_edit_copy,         edit_menu,      edit_copy_idx },
      { &view->menu_edit_paste,        edit_menu,      edit_paste_idx },
      { &view->menu_edit_select_all,   edit_menu,      edit_select_all_idx },
      { &view->menu_edit_find,         edit_menu,      edit_find_idx },
      { &view->menu_edit_find_again,   edit_menu,      edit_find_again_idx },
      { &view->menu_account_delete,    account_menu,   account_delete_idx },
      { &view->menu_account_props,     account_menu,   account_props_idx }
    };
    int i;

    gnome_app_create_menus(GNOME_APP(window), main_menu);

    for (i=0; i<sizeof_array(menus); ++i)
      *menus[i].ptr = menus[i].tree[menus[i].index].widget;
  }

  /* create toolbar */
  {
    GnomeUIInfo toolbar[] =
    {
      { GNOME_APP_UI_ITEM, NULL, N_("Save Accounts [ctl-s]"),
        fin_on_save, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SAVE, 0, 0, NULL },
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, NULL, N_("Undo Modification [ctl-z]"),
        fin_on_undo, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_UNDO, 0, 0, NULL },
      { GNOME_APP_UI_ITEM, NULL, N_("Redo Modification [ctl-r]"),
        fin_on_redo, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REDO, 0, 0, NULL },
      { GNOME_APP_UI_ITEM, NULL, N_("Delete Transaction [del]"),
        fin_on_delete_record, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CLOSE, 0, 0, NULL },
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, NULL, N_("Toggle Transaction Status [c]"),
        fin_on_toggle_status, view, NULL,
	GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_CLEAR, 0, 0, NULL },
      GNOMEUIINFO_SEPARATOR,
      { GNOME_APP_UI_ITEM, NULL, N_("Exit Program [ctl-q]"),
        fin_on_exit, view, NULL,
        GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_EXIT, 0, 0, NULL },
      GNOMEUIINFO_END
    };
    gnome_app_create_toolbar(GNOME_APP(window), toolbar);
  }

  /* create vertical packing box */
  {
    vbox = gtk_vbox_new(FALSE, 0);
    gnome_app_set_contents(GNOME_APP(window), vbox);
    gtk_widget_show(vbox);
  }

  /* create notebook window */
  {
    notebook = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_BOTTOM);
    gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
    view->notebook_switch_page_id =
      gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
			 GTK_SIGNAL_FUNC(fin_on_switch_account), view);
    gtk_widget_show(notebook);

    view->notebook = notebook;
  }

  /* create other */
  {
    GtkWidget * table;
    GtkWidget * combo;
    GtkWidget * entry;
    GtkWidget * button;
    GtkWidget * statusbar;
    GtkWidget * hbox;
    GtkWidget * label;

    /* Table */
    {
      table = gtk_table_new(2, 60, TRUE);
      gtk_box_pack_end(GTK_BOX(vbox), table, FALSE, FALSE, 0);
      gtk_widget_show(table);
    }

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 9, 0, 1);
    gtk_widget_show(hbox);

    /* Date */
    {
      entry = gtk_entry_new();
      gtk_entry_set_max_length(GTK_ENTRY(entry), FIN_RECORD_FIELD_MAXLEN_DATE-1);
      gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
      gtk_widget_show(entry);

      view->entry_date_changed_id = 
	gtk_signal_connect(GTK_OBJECT(entry), "changed",
			   GTK_SIGNAL_FUNC(fin_on_entry_date_changed), view);

      view->record_date = entry;
    }

    /* Calendar popup button */
    {
      GtkWidget * arrow;

      button = gtk_button_new();
      GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
      gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
      gtk_widget_show(button);

      view->calendar_button = button;

      arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
      gtk_container_add(GTK_CONTAINER(button), arrow);
      gtk_widget_show(arrow);

      create_calendar_dialog(view);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
			 GTK_SIGNAL_FUNC(fin_on_show_calendar), view);
    }

    /* Source */
    {
      GList * types;

      combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table), combo, 9, 15, 0, 1);
      gtk_widget_show(combo);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
      types = fin_record_stringize_types(0);
      gtk_combo_set_popdown_strings(GTK_COMBO(combo), types);
      g_list_free(types);
      gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed",
			 GTK_SIGNAL_FUNC(fin_on_record_type_changed), view);

      view->record_source = combo;
    }

    /* Check No */
    {
      entry = gtk_entry_new();
      gtk_entry_set_max_length(GTK_ENTRY(entry), FIN_RECORD_FIELD_MAXLEN_TYPE_DATA-1);
      gtk_table_attach_defaults(GTK_TABLE(table), entry, 15, 22, 0, 1);
      gtk_widget_show(entry);

      view->record_check_no = entry;
    }

    /* Transfer Account List */
    {
      combo = gtk_combo_new();
      gtk_table_attach_defaults(GTK_TABLE(table), combo, 15, 22, 0, 1);
      gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
      gtk_widget_hide(combo);

      view->record_transfer_acct = combo;
    }

    /* Info List */
    {
      combo = gtk_combo_new();
      gtk_combo_set_case_sensitive(GTK_COMBO(combo), TRUE);
      entry = GTK_COMBO(combo)->entry;
      gtk_entry_set_max_length(GTK_ENTRY(entry), FIN_RECORD_FIELD_MAXLEN_INFO-1);
      gtk_table_attach_defaults(GTK_TABLE(table), combo, 22, 54, 0, 1);
      gtk_widget_show(combo);

      view->info_key_press_id = 
	gtk_signal_connect_after(GTK_OBJECT(entry), "key_press_event",
				 GTK_SIGNAL_FUNC(fin_on_info_key_press), view);

      view->info_insert_text_id = 
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "insert_text",
			   GTK_SIGNAL_FUNC(fin_on_insert_info_text), view);

      view->record_info = combo;
    }

    /* Amount */
    {
      entry = gtk_entry_new();
      gtk_entry_set_max_length(GTK_ENTRY(entry), FIN_RECORD_FIELD_MAXLEN_AMOUNT-1);
      gtk_table_attach_defaults(GTK_TABLE(table), entry, 54, 60, 0, 1);
      gtk_widget_show(entry);

      view->record_amount = entry;
    }

/*
    printf("sleeping.."); fflush(stdout);
    sleep(2);
    printf("done\n");
    */

    /* Status Bar */
    {
      statusbar = gtk_statusbar_new();
      gtk_table_attach_defaults(GTK_TABLE(table), statusbar, 0, 36, 1, 2);
      //gnome_app_set_statusbar(GNOME_APP(view->window), statusbar);
      gtk_widget_show(statusbar);

      view->statusbar = statusbar;
    }

    /* Cleared Balance Indicator */
    {
      GtkWidget * frame;

      frame = gtk_frame_new(NULL);
      gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
      gtk_table_attach_defaults(GTK_TABLE(table), frame, 36, 48, 1, 2);
      gtk_widget_show(frame);

      label = gtk_label_new("");
      gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
      gtk_misc_set_padding(GTK_MISC(label), 2, 2);
      gtk_container_add(GTK_CONTAINER(frame), label);
      gtk_widget_show(label);

      view->cleared_bal = label;
    }

    /* Insert button */
    {
      button = gtk_button_new_with_label("Insert");
      gtk_table_attach_defaults(GTK_TABLE(table), button, 48, 54, 1, 2);
      gtk_widget_show(button);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
			 GTK_SIGNAL_FUNC(fin_on_insert_record), view);

      view->insert_button = button;
    }

    /* Update button */
    {
      button = gtk_button_new_with_label("Update");
      gtk_table_attach_defaults(GTK_TABLE(table), button, 54, 60, 1, 2);
      gtk_widget_show(button);
      gtk_signal_connect(GTK_OBJECT(button), "clicked",
			 GTK_SIGNAL_FUNC(fin_on_update_record), view);

      view->update_button = button;
    }
  }
}

static void
create_account_info_dialog (FinView * view)
{
  GtkWidget * dialog;
  GtkWidget * entry;
  GtkWidget * label;
  GtkWidget * table;
  GtkWidget * hbox;

  fin_trace("");

  g_return_if_fail(view != NULL);

  /* Dialog */
  {
    dialog = gnome_dialog_new("Account Properties",
    			      GNOME_STOCK_BUTTON_OK,
			      GNOME_STOCK_BUTTON_CANCEL,
			      NULL);
    gnome_dialog_close_hides(GNOME_DIALOG(dialog), TRUE);
    gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(view->window));

    view->account_dialog = dialog;
  }
  
  /* Entry fields */
  {
    hbox = gtk_hbox_new(FALSE, 10);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), hbox, FALSE, FALSE, 10);
    gtk_widget_show(hbox);

    table = gtk_table_new(2, 10, TRUE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table), 5);
    gtk_box_pack_start(GTK_BOX(hbox), table, FALSE, FALSE, 10);
    gtk_widget_show(table);

    label = gtk_label_new("Account Name:");
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 0, 1);
    gtk_widget_show(label);

    label = gtk_label_new("Account Info:");
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 1, 2);
    gtk_widget_show(label);

    entry = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(entry), FIN_ACCOUNT_NAME_MAX-1);
    gtk_table_attach_defaults(GTK_TABLE(table), entry, 3, 10, 0, 1);
    gtk_widget_show(entry);

    view->account_name = entry;

    entry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(table), entry, 3, 10, 1, 2);
    gtk_widget_show(entry);

    view->account_info = entry;
  }
}


/* EXTERNAL INTERFACE
 *
 * fin_view_new
 * fin_view_load_file
 * fin_view_load_config
 * fin_view_save_config
 * fin_on_import
 * fin_view_load_import_filter
 * fin_on_export
 * fin_view_load_export_filter
 */

FinView * fin_view_new()
{
  FinView * view;

  fin_trace("");

  /* allocate data structure and assign fields */
  view = g_new0(FinView, 1);
  view->selection = -1;

  fin_view_load_config(view);

  /* create history manager */
  {
    view->history = fin_history_new();

    gtk_signal_connect(GTK_OBJECT(view->history), "free_item",
    		       GTK_SIGNAL_FUNC(fin_on_history_free_item), view);
    gtk_signal_connect(GTK_OBJECT(view->history), "undo_item",
    		       GTK_SIGNAL_FUNC(fin_on_history_undo_item), view);
    gtk_signal_connect(GTK_OBJECT(view->history), "redo_item",
    		       GTK_SIGNAL_FUNC(fin_on_history_redo_item), view);
  }

  /* create window + initialize */
  create_window(view);
  get_toolbar_buttons(view);

  fin_on_new(NULL, view);

  /* disable buttons */
  disable_file_save(view);
  disable_record_view(view);
  disable_edit_undo(view);
  disable_edit_redo(view);
  disable_edit_cut(view);
  disable_edit_copy(view);
  disable_edit_paste(view);
  disable_edit_select_all(view);
  disable_edit_find(view);
  disable_edit_find_again(view);
  disable_account_delete(view);
  disable_account_props(view);
  disable_record_delete(view);
  disable_record_toggle_status(view);

  return view;
}

gboolean
fin_view_load_file (FinView * view, const gchar * filename)
{
  FinAccountSet * set = NULL;

  fin_trace("");

  /* load database file */
  {
    const gchar * failed = NULL;
    fin_trace("loading database: %s", filename);
    if ((failed = fin_file_read(filename, &set)) != NULL)
    {
      fin_error_dialog(view->window, "Error loading file: %s\n[%s]", filename, failed);
      return FALSE;
    }
  }

  fin_trace("got new account set: %p", set);
  
  /* discard existing datafile if open, set new */
  {
    if (!fin_on_close(NULL, view))
    {
      fin_account_set_unref(set);
      return FALSE;
    }

    view->set = set;
    if (view->set->accounts)
      view->account = LIST_GET(FinAccount, view->set->accounts);
    else
      view->account = NULL;
    set_filename(view, filename);
    refresh_account_names(view);
  }

  fin_trace("adding entries to list...");

  /* add entries to list */
  {
    GtkCList * clist = NULL;
    GList * it;
    gint i;

    fin_trace("view->set->accounts: %p", view->set->accounts);

    /* remove existing notebook pages */
    while ((i=gtk_notebook_get_current_page(GTK_NOTEBOOK(view->notebook))) != -1)
      gtk_notebook_remove_page(GTK_NOTEBOOK(view->notebook), i);

    view->clist = NULL;

    for (it=view->set->accounts, i=0; it; it=it->next, ++i)
    {
      FinAccount * account = LIST_GET(FinAccount, it);

      clist = create_list(view, account->name, i);

      /* first account is initially active */
      if (i == 0)
        view->clist = clist;

      refresh_list(view, clist, account, TRUE);

      /* select and scroll to bottom-most record */
      gtk_clist_select_row(clist, clist->rows-1, -1);
      gtk_adjustment_set_value(clist->vadjustment, clist->vadjustment->upper);
    }
  }

  refresh_type_list(view);
  refresh_info(view);
  refresh_check_no(view, TRUE);

  if (view->account)
  {
    /* reselect first account -- hack, b/c we need the type_list to be refreshed */
    gtk_clist_select_row(view->clist, view->clist->rows-1, -1);
    gtk_adjustment_set_value(view->clist->vadjustment, view->clist->vadjustment->upper);

    enable_account_delete(view);
    enable_account_props(view);
    enable_record_view(view);
  }

  set_dirty(view, -1);

  return TRUE;
}

void
fin_view_load_config (FinView * view)
{
  gchar path[256];
  int i;

  fin_trace("");

  view->config.width = gnome_config_get_int("/gnofin/Geometry/Width=640");
  view->config.height = gnome_config_get_int("/gnofin/Geometry/Height=400");

  for (i=0; i<FIN_VIEW_NUMCOLS; ++i)
  {
    snprintf(path, sizeof(path), "/gnofin/Geometry/ColWidth%d=%d", i, clist_def_widths[i]);
    view->config.col_widths[i] = gnome_config_get_int(path);
  }
}

void
fin_view_save_config (FinView * view)
{
  gchar path[256];
  int i;

  fin_trace("");

  gnome_config_set_int("/gnofin/Geometry/Width", view->config.width);
  gnome_config_set_int("/gnofin/Geometry/Height", view->config.height);

  for (i=0; i<FIN_VIEW_NUMCOLS; ++i)
  {
    snprintf(path, sizeof(path), "/gnofin/Geometry/ColWidth%d", i);
    gnome_config_set_int(path, view->config.col_widths[i]);
  }

  gnome_config_sync();
}


typedef struct _ImpArgs {
  FinView * view;
  FinImportFilter * imp;
} ImpArgs;

typedef struct _ExpArgs {
  FinView * view;
  FinExportFilter * exp;
} ExpArgs;


static void
fin_on_import (GtkWidget * widget,
	       FinView   * view)
{
  FILE * file;
  gchar * filename;

  fin_trace("");

  if ((filename = fin_get_filename_dialog(view->window, "Import", view->filename, FALSE)) != NULL)
  {

    if (filename != NULL)
    {
      file = fopen(filename, "r");
      if (file != NULL)
      {
        if (qif_import((void*)view, file, filename)) {
          fin_view_load_file (view, "/tmp/importqif");
          remove ("/tmp/importqif");
        }
	    g_free(filename);
	    fclose(file);
      }
      else
	    fin_error_dialog("Error importing file: %s\n", strerror(errno));
    }
  }
}


void
fin_view_load_import_filter (FinView         * view,
			     FinImportFilter * imp)
{
  GtkWidget * submenu, * menuitem;
  ImpArgs * args;

  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(imp);
  g_return_if_fail(imp->name);
  g_return_if_fail(imp->import);

  args = g_new0(ImpArgs, 1);
  args->view = view;
  args->imp = imp;

  view->imports = g_list_append(view->imports, imp);

  submenu = GTK_MENU_ITEM(view->menu_file_imports)->submenu;
  menuitem = gtk_menu_item_new_with_label(imp->name); 
  gtk_widget_show(menuitem);
  gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
  		     GTK_SIGNAL_FUNC(fin_on_import), view);
  gtk_menu_append(GTK_MENU(submenu), menuitem);
}


void
fin_on_export (GtkWidget * menuitem, ExpArgs * args)
{
  FILE * file;
  gchar * filename;

  fin_trace("");

  filename = fin_get_filename_dialog(args->view->window, "Specify Output File", NULL, TRUE);

  if (filename != NULL)
  {
    file = fopen(filename, "wt");
    if (file != NULL)
    {
      args->exp->export(file, filename, args->view->set);
      g_free(filename);
      fclose(file);
    }
    else
      fin_error_dialog("Error exporting file: %s\n", strerror(errno));
  }
}

void
fin_view_load_export_filter (FinView         * view,
			     FinExportFilter * exp)
{
  GtkWidget * submenu, * menuitem;
  ExpArgs * args;

  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(exp);
  g_return_if_fail(exp->name);
  g_return_if_fail(exp->export);

  args = g_new0(ExpArgs, 1);
  args->view = view;
  args->exp = exp;

  view->exports = g_list_append(view->exports, exp);

  submenu = GTK_MENU_ITEM(view->menu_file_exports)->submenu;
  menuitem = gtk_menu_item_new_with_label(exp->name); 
  gtk_widget_show(menuitem);
  gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
  		     GTK_SIGNAL_FUNC(fin_on_export), args);
  gtk_menu_append(GTK_MENU(submenu), menuitem);
}


/* TOOLBAR/MENU
 *
 * get_toolbar_buttons
 * set_enable_???
 */

static void
get_toolbar_buttons (FinView * view)
{
  /* get pointers to toolbar buttons */

  GnomeDockItem * item;
  GtkWidget * toolbar;
  gint i;

  const struct 
  {
    GtkWidget ** ptr;
    gint position;
  }
  tools[] =
  {
    { &view->tool_file_save,		0 },   /* be careful here !! */
    { &view->tool_edit_undo,		2 },
    { &view->tool_edit_redo,		3 },
    { &view->tool_record_delete,	4 },
    { &view->tool_record_toggle_status,	6 },
  };

  item = gnome_dock_get_item_by_name(GNOME_DOCK(GNOME_APP(view->window)->dock),
				     GNOME_APP_TOOLBAR_NAME,
				     NULL, NULL, NULL, NULL);
  toolbar = gnome_dock_item_get_child(GNOME_DOCK_ITEM(item));

  for (i=0; i<sizeof_array(tools); ++i)
  {
    GtkToolbarChild * child;

    child = LIST_GET(GtkToolbarChild, g_list_nth(GTK_TOOLBAR(toolbar)->children,tools[i].position));
    g_assert(child->type == GTK_TOOLBAR_CHILD_BUTTON);
    tools[i].ptr[0] = child->widget;
  }
}

static void
set_enable_file_save (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->tool_file_save, enable);
  gtk_widget_set_sensitive(view->menu_file_save, enable);
}

static void
set_enable_edit_undo (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->tool_edit_undo, enable);
  gtk_widget_set_sensitive(view->menu_edit_undo, enable);
}

static void
set_enable_edit_redo (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->tool_edit_redo, enable);
  gtk_widget_set_sensitive(view->menu_edit_redo, enable);
}

static void
set_enable_edit_cut (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_edit_cut, enable);
}

static void
set_enable_edit_copy (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_edit_copy, enable);
}

static void
set_enable_edit_paste (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_edit_paste, enable);
}

static void
set_enable_edit_select_all (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_edit_select_all, enable);
}

static void
set_enable_edit_find (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_edit_find, enable);
}

static void
set_enable_edit_find_again (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_edit_find_again, enable);
}

static void
set_enable_account_delete (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_account_delete, enable);
}

static void
set_enable_account_props (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->menu_account_props, enable);
}

static void
set_enable_record_delete (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->tool_record_delete, enable);
}

static void
set_enable_record_toggle_status (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->tool_record_toggle_status, enable);
}

static void
set_enable_record_update (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  view->can_update = enable;
  gtk_widget_set_sensitive(view->update_button, enable);
}

static void
set_enable_record_view (FinView * view, gboolean enable)
{
  fin_trace("%d",enable);

  gtk_widget_set_sensitive(view->record_date, enable);
  gtk_widget_set_sensitive(view->record_source, enable);
  gtk_widget_set_sensitive(view->record_info, enable);
  gtk_widget_set_sensitive(view->record_amount, enable);
  gtk_widget_set_sensitive(view->insert_button, enable);
  gtk_widget_set_sensitive(view->update_button, enable & view->can_update);

  gtk_widget_set_sensitive(view->calendar_button, enable);

  fin_on_record_type_changed(GTK_COMBO(view->record_source)->entry, view);

  if (enable == FALSE)
  {
    gtk_widget_set_sensitive(view->record_check_no, FALSE);
    gtk_widget_set_sensitive(view->record_transfer_acct, FALSE);
  }
}


/* RECORD HANDLERS
 *
 * del_record
 * put_record
 * fin_parse_record
 * fin_on_insert_record
 * fin_on_update_record 
 * fin_on_delete_record
 */

static void
del_link_if_exists(FinView * view, FinRecord * record)
{
  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(record);

  /* we are only interested in making sure that the linked
     record is no longer displayed */
  if (record->type == FIN_RECORD_TYPE_XFR && record->have_linked_XFR)
  {
    FinTransferLink * link = FIN_RECORD_XFR_LINK(record);
    del_record(view, link->account, link->record, -1);
    fin_record_unref(link->record);
  }
}

static void
put_link_if_exists(FinView * view, FinRecord * record, int insert)
{
  fin_trace("");

  g_return_if_fail(view);
  g_return_if_fail(record);

  /* we are only interested in making sure that the linked
     record is displayed */
  if (record->type == FIN_RECORD_TYPE_XFR && record->have_linked_XFR)
  {
    FinTransferLink * link = FIN_RECORD_XFR_LINK(record);
    put_record(view, link->account, link->record, insert);
    fin_record_ref(link->record);
  }
}

static FinRecord *
del_record (FinView    * view,
	    FinAccount * account,
	    FinRecord  * record,
	    gint         position)
{
  GList * it, * tail;

  fin_trace("");

  /* this should be okay, or we wouldn't have a selection */
  g_return_val_if_fail(view->set, NULL);
  g_return_val_if_fail(view->set->accounts, NULL);
  g_return_val_if_fail(view->set->accounts->data, NULL);
  g_return_val_if_fail(view->account, NULL);
  g_return_val_if_fail(view->history, NULL);

  if (account == NULL)
    account = view->account;

  if (record == NULL)
  {
    g_return_val_if_fail(position >= 0, NULL);
    it = g_list_nth(account->records, position);
    record = LIST_GET(FinRecord, it);
  }
  else
  {
    for (it=account->records, position=0; it; it=it->next, ++position)
      if (it->data == record)
        break;
    g_return_val_if_fail(it, NULL);
  }

  /* remove record */
  {
    tail = it->next;
    account->records = g_list_remove_link(account->records, it);
    it = tail;
  }
  
  /* update clist */
  {
    GtkCList * clist;
    int i = position;

    if (account == view->account)
      clist = GTK_CLIST(view->clist);
    else
    {
      int page = g_list_index(view->set->accounts, account);
      clist = GTK_CLIST(GTK_BIN(gtk_notebook_get_nth_page(GTK_NOTEBOOK(view->notebook), page))->child);
    }

    gtk_clist_freeze(clist);
    gtk_clist_remove(clist, i);

    for (; it; it=it->next, ++i)
    {
      FinRecord * r = LIST_GET(FinRecord, it);

      if (it->prev)
	r->balance = r->amount + LIST_GET(FinRecord, it->prev)->balance;
      else
	r->balance = r->amount;

      /* update balance text */
      {
	gchar ** text;

	FinRecordField type = FIN_RECORD_FIELD_BALANCE;
	text = fin_stringized_record_new(r, &type, 1);

	gtk_clist_set_text(clist, i, 5, text[0]);

	if (r->balance < 0)
	{
	  if (view->negative_balance_style == NULL)
	    create_negative_balance_style(view);

	  gtk_clist_set_cell_style(clist, i, 5,
				   view->negative_balance_style);
	}
	else
	  gtk_clist_set_cell_style(clist, i, 5, GTK_WIDGET(view->clist)->style);

	fin_stringized_record_free(text);
      }
    }
    gtk_clist_thaw(clist);
  }

  if (view->account->records == NULL)
  {
    view->selection = -1;
    disable_record_delete(view);
    disable_record_toggle_status(view);
  }

  refresh_cleared_bal(view);

  return record;
}

static gint
put_record (FinView    * view,
		FinAccount * account,
		FinRecord  * record,
		gint insert)
{
  GtkCList * clist = NULL;
  gint position = -1;
  GList * it;
  gint i;

  fin_trace("");

  g_return_val_if_fail(view, -1);
  g_return_val_if_fail(record, -1);

  if (account == NULL)
    account = view->account;

  /* store new record in account, update info cache (if necessary), set dirty flag */
  if (insert)
  {
    account->records = g_list_insert_sorted(account->records, record,
					    (GCompareFunc) fin_record_sort_fcn);
    /* search record list for this record and remember position */
    for (i=0, it=account->records; it; it=it->next, ++i)
      if (it->data == record) break;
    g_assert(it != NULL);
    position = i;
  }
  else
  {
    gint i1, i2;

    /* for updates, we want to make sure we refresh the list from the 
     * earliest record, so we:
     *
     *  - locate record in list
     *  - resort list
     *  - relocate record in list
     *  - set (it,position) equal to earliest location
     */

    i1 = g_list_index(account->records, record);

    account->records = g_list_sort(account->records,
				   (GCompareFunc) fin_record_sort_fcn);

    i2 = g_list_index(account->records, record);

    if (i1 < i2)
      i = i1;
    else
      i = i2;

    it = g_list_nth(account->records, i);

    fin_trace("i1=%d, i2=%d", i1, i2);
    position = i2;
  }

  fin_trace("i=%d, position=%d",i,position);

  if (view->set->info_cache == NULL)
    view->set->info_cache = record->info;
  else
    view->set->info_cache = g_list_first(record->info);

  {
    gint first_time = 1;

    if (account == view->account)
      clist = view->clist;
    else
    {
      int page = g_list_index(view->set->accounts, account);
      clist = GTK_CLIST(GTK_BIN(gtk_notebook_get_nth_page(GTK_NOTEBOOK(view->notebook), page))->child);
    }

    gtk_clist_freeze(clist);

    for (; it; it=it->next, ++i)
    {
      FinRecord * r = LIST_GET(FinRecord, it);

      fin_trace("i=%d, it->prev=%p",i,it->prev);

      if (it->prev)
	r->balance = r->amount + LIST_GET(FinRecord, it->prev)->balance;
      else
	r->balance = r->amount;
 

      /* update balance text */
      {
	gchar ** text;
	gint k;

	text = fin_stringized_record_new(r, clist_types, sizeof_array(clist_types));

	if (first_time && insert)
	{
	  fin_trace("ddd");
	  gtk_clist_append(clist, text);
	  first_time = 0;
	}

	g_assert(sizeof_array(clist_types) == 6);
	for (k=0; k<6; ++k)
	  gtk_clist_set_text(clist, i, k, text[k]);
	fin_stringized_record_free(text);

	if (r->balance < 0)
	{
	  if (view->negative_balance_style == NULL)
	    create_negative_balance_style(view);

	  gtk_clist_set_cell_style(clist, i, 5, view->negative_balance_style);
	}
	else
	  gtk_clist_set_cell_style(clist, i, 5, GTK_WIDGET(view->clist)->style);
      }
    }
    gtk_clist_thaw(clist);
  }

  if (account == view->account)
  {
    /* refresh info strings in info combo box */
    refresh_info(view);

    /* select the record just inserted */
    gtk_clist_select_row(clist, position, -1);

    if (record->type == FIN_RECORD_TYPE_CHK && FIN_RECORD_CHECK_NO(record) >= view->next_check_no)
    {
      view->next_check_no = FIN_RECORD_CHECK_NO(record) + 1;
      refresh_check_no(view, FALSE);
    }
  }

  refresh_cleared_bal(view);

  return position;
}

static gboolean
fin_parse_record (FinView * view, FinRecord * record)
{
  gchar * text[5] = {0};
  int k, result;
  FinRecordField fields[] = {
    FIN_RECORD_FIELD_DATE,
    FIN_RECORD_FIELD_TYPE,
    FIN_RECORD_FIELD_TYPE_DATA,
    FIN_RECORD_FIELD_INFO,
    FIN_RECORD_FIELD_AMOUNT,
  };
  GtkWidget * entry[] = {
    view->record_date,
    GTK_COMBO(view->record_source)->entry,
    NULL, 
    GTK_COMBO(view->record_info)->entry,
    view->record_amount,
  };

  fin_trace("");

  g_return_val_if_fail(record != NULL, FALSE);

  switch (view->current_record_type)
  {
  case FIN_RECORD_TYPE_XFR:
    entry[2] = GTK_COMBO(view->record_transfer_acct)->entry;
    break;
  case FIN_RECORD_TYPE_CHK:
    entry[2] = view->record_check_no;
    break;
  default:
  }

  for (k=0; k<sizeof_array(fields); ++k)
  {
    if (entry[k] && GTK_IS_ENTRY(entry[k]))
      text[k] = gtk_entry_get_text(GTK_ENTRY(entry[k]));
    fin_trace("text[%d] = \"%s\"",k,text[k]);
  }

  result = fin_record_parse_fields(record, fields, sizeof_array(fields), text);

  fin_trace("fin_record_parse_fields() returned %d", result);
  if (result > 0)
  {
    const gchar * field;

    /* should only worry about parse errors 
     * for the date and amount fields */
    switch (result)
    {
    case 1:
      field = "date field";
      break;
    case 3:
      switch (view->current_record_type)
      {
      case FIN_RECORD_TYPE_CHK:
	field = "check number";
	break;
      case FIN_RECORD_TYPE_XFR:
        field = "transfer account";
        break;
      default:
	field = "<unexpected>";
        g_warning("Unexpected parse error: problem with field %d\n", result);
      }
      break;
    case 5:
      field = "amount field";
      break;
    default:
      field = "<unexpected>";
      g_warning("Unexpected parse error: problem with field %d\n", result);
    }

    fin_error_dialog(view->window,"Error parsing transaction: problem with the %s.", field);
  }

  return (result == 0);
}

static gint
fin_verify_check_no (FinView * view, FinRecord * record)
{
  FinAccount * account;
  GList * it;

  fin_trace("");

  g_return_val_if_fail(view, FALSE);

  if (record->type != FIN_RECORD_TYPE_CHK)
    return TRUE;

  account = view->account;

  for (it=account->records; it; it=it->next)
  {
    FinRecord * other = LIST_GET(FinRecord, it);

    if (other != record && other->type == FIN_RECORD_TYPE_CHK)
    {
      if (FIN_RECORD_CHECK_NO(other) == FIN_RECORD_CHECK_NO(record))
      {
        fin_error_dialog(view->window, "The check number you have entered is already in use !!");
        return FALSE;
      }
    }
  }

  return TRUE;
}

static void
fin_on_insert_record (GtkWidget * widget, FinView * view)
{
  FinRecord * record;

  fin_trace("");

  g_return_if_fail(view);

  record = fin_record_new(view->set->info_cache);

  if (!(fin_parse_record(view, record) && fin_verify_check_no(view, record)))
    fin_record_unref(record);
  else
  {
    /* handle transfer */
    if (record->type == FIN_RECORD_TYPE_XFR)
    {
      FinAccount * d_acc;
      FinRecord * d_rec;
      gchar * ac_name;

      g_assert(record->have_linked_XFR == 0);

      ac_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(view->record_transfer_acct)->entry));
      d_acc = fin_account_set_lookup_account(view->set, ac_name);
      d_rec = fin_account_set_transfer_funds(view->set, view->account, record, d_acc);

      fin_record_ref(d_rec);
      put_record(view, d_acc, d_rec, FALSE); // just do an update on new record
    }

    put_record(view, view->account, record, TRUE);
    set_dirty(view, TRUE);

    gtk_clist_moveto(GTK_CLIST(view->clist), 
    		     g_list_index(view->account->records, record),
    		     -1, 0.5, 0);

    /* remember record */
    {
      FinHistoryData * data = g_new0(FinHistoryData, 1);

      fin_record_ref(record);

      data->type = FIN_OP_INSERT_RECORD;
      data->account = view->account;
      data->record = record;

      fin_history_remember(view->history, data);

      enable_edit_undo(view);
      disable_edit_redo(view);
    }
  }
}

static void
fin_on_update_record (GtkWidget * widget, FinView * view)
{
  GList * it;
  FinRecord * record, * record_old;

  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->selection != -1);
  g_return_if_fail(view->account != NULL);

  it = g_list_nth(view->account->records, view->selection);
  record = LIST_GET(FinRecord, it);

  record_old = fin_record_dup(record);

  if (!(fin_parse_record(view, record) && fin_verify_check_no(view, record)))
    fin_record_unref(record_old);
  else
  {
    /* handle transfer */
    if (record->type == FIN_RECORD_TYPE_XFR)
    {
      int insert = 0;
      fin_trace("handling transfer request...");
      /* 2 cases: this was already an XFR _or_ this is now an XFR */
      if (record_old->type == FIN_RECORD_TYPE_XFR)
      {
        FinTransferLink * link = FIN_RECORD_XFR_LINK(record);

        /* 2 cases: same account _or_ new account */
	if (FIN_RECORD_XFR_LINK(record_old)->account == link->account)
	{
	  fin_trace("removing existing link...");
	  /* remove existing link.. then insert */
	  del_record(view, link->account, link->record, -1);
	  fin_record_unref(link->record);  /* no longer displayed */
	  insert = 1;
	}
	else
	{
	  fin_trace("updating existing link...");
	  /* synchronize linked record */
	  fin_record_sync_link(link->record, record);

	  put_record(view, link->account, link->record, FALSE);
	}
      }
      else
        insert = 1;
      if (insert)
      {
        /* same code as for insert */
	FinAccount * d_acc;
	FinRecord * d_rec;
	gchar * ac_name;

	fin_trace("inserting link...");

	g_assert(record->have_linked_XFR == 0);

	ac_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(view->record_transfer_acct)->entry));
	d_acc = fin_account_set_lookup_account(view->set, ac_name);
	d_rec = fin_account_set_transfer_funds(view->set, view->account, record, d_acc);

	fin_record_ref(d_rec); // add reference for being displayed
	put_record(view, d_acc, d_rec, FALSE); // just do an update on new record
      }
    }
    else if (record_old->type == FIN_RECORD_TYPE_XFR)
    {
      /* remove existing link */
      fin_trace("removing existing link...");
      del_link_if_exists(view, record_old);
    }

    put_record(view, view->account, record, FALSE);
    set_dirty(view, TRUE);

    /* remember record */
    {
      FinHistoryData * data = g_new0(FinHistoryData, 1);

      data->type = FIN_OP_UPDATE_RECORD;
      data->account = view->account;
      data->record = record;
      data->record_old = record_old;

      fin_history_remember(view->history, data);

      enable_edit_undo(view);
      disable_edit_redo(view);
    }
  }
}

static void
fin_on_delete_record (GtkWidget * widget,
		      FinView   * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);

  if (view->selection == -1)
    fin_error_dialog(view->window, "You need to first select a transaction to delete!");
  else
  {
    FinRecord * record = del_record(view, NULL, NULL, view->selection); 

    g_return_if_fail(record != NULL);

    /* handle linked transfer record */
    del_link_if_exists(view, record);

    set_dirty(view, TRUE);

    /* remember discarded record */
    {
      FinHistoryData * data = g_new0(FinHistoryData, 1);

      data->type = FIN_OP_DELETE_RECORD;
      data->account = view->account;
      data->record = record;

      fin_record_ref(record);

      /* this may result in items being removed, so 
       * we ref/unref to protect it */
      fin_history_remember(view->history, data);

      fin_record_unref(record);

      enable_edit_undo(view);
      disable_edit_redo(view);
    }
  }
}


/* CALENDAR
 *
 * fin_on_entry_date_changed
 * fin_on_calendar_date_changed
 * fin_on_show_calendar
 * fin_on_hide_calendar
 * fin_on_create_calendar_dialog
 */
 
static void
fin_on_entry_date_changed (GtkWidget * widget, FinView * view)
{
  GDate date;

  fin_trace("");

  g_return_if_fail(view);

  if (view->calendar)
  {
    g_date_clear(&date, 1);
    g_date_set_parse(&date, gtk_entry_get_text(GTK_ENTRY(view->record_date)));

    if (g_date_valid(&date))
    {
      GtkCalendar * calendar = GTK_CALENDAR(view->calendar);

      g_date_set_month(&date, g_date_month(&date) - 1);

      gtk_signal_handler_block(GTK_OBJECT(view->calendar), view->calendar_month_changed_id);      
      gtk_signal_handler_block(GTK_OBJECT(view->calendar), view->calendar_day_changed_id);      

      gtk_calendar_freeze(calendar);
      gtk_calendar_select_month(calendar, g_date_month(&date), g_date_year(&date));
      gtk_calendar_select_day(calendar, g_date_day(&date));
      gtk_calendar_thaw(calendar);

      gtk_signal_handler_unblock(GTK_OBJECT(view->calendar), view->calendar_month_changed_id);      
      gtk_signal_handler_unblock(GTK_OBJECT(view->calendar), view->calendar_day_changed_id);      
    }
  }
}

static void
fin_on_calendar_date_changed (GtkCalendar * calendar, FinView * view)
{
  gchar buf[FIN_RECORD_FIELD_MAXLEN_DATE];
  guint year;
  guint month;
  guint day;

  fin_trace("");

  g_return_if_fail(view != NULL);

  gtk_calendar_get_date(calendar, &year, &month, &day);
  month++;

  fin_trace("%d/%d/%d",month,day,year);

  /* generate date string */
  {
    GDate date;
    g_date_clear(&date, 1);
    g_date_set_dmy(&date, day, month, year);

    fin_date_stringize(buf, sizeof(buf), &date);
  }

  gtk_signal_handler_block(GTK_OBJECT(view->record_date), view->entry_date_changed_id);

  gtk_entry_set_text(GTK_ENTRY(view->record_date), buf);

  gtk_signal_handler_unblock(GTK_OBJECT(view->record_date), view->entry_date_changed_id);
}

static void
fin_on_show_calendar (GtkWidget * widget, FinView * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->calendar_dialog != NULL);
  g_return_if_fail(view->calendar_button != NULL);

  if (GTK_WIDGET_VISIBLE(view->calendar_dialog))
    gtk_widget_hide(view->calendar_dialog);
  else
  {
    /* force the calendar to be updated every time it is displayed */
    fin_on_entry_date_changed(NULL, view);
    gtk_widget_show(view->calendar_dialog);
  }
}

static void
fin_on_hide_calendar (GtkWidget * widget, GdkEvent * event, FinView * view)
{
  fin_trace("");

  g_return_if_fail(view != NULL);
  g_return_if_fail(view->calendar_dialog != NULL);
  g_return_if_fail(view->calendar_button != NULL);

  gtk_widget_hide(view->calendar_dialog);
}

static void
create_calendar_dialog (FinView * view)
{
  GtkWidget * dialog;
  GtkWidget * calendar;

  /* Dialog */
  {
    dialog = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(dialog),
    				 GTK_WINDOW(view->window));
    gtk_window_set_title(GTK_WINDOW(dialog), "Calendar");
    gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
    gtk_window_set_policy(GTK_WINDOW(dialog), 0, 0, 0);

    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
    		       GTK_SIGNAL_FUNC(fin_on_hide_calendar), view);

    view->calendar_dialog = dialog;
  }

  /* Calendar */
  {
    calendar = gtk_calendar_new();
    gtk_calendar_display_options(GTK_CALENDAR(calendar), 
    				 GTK_CALENDAR_SHOW_HEADING |
				 GTK_CALENDAR_SHOW_DAY_NAMES);
    gtk_container_add(GTK_CONTAINER(dialog), calendar);
    gtk_widget_show(calendar);

    view->calendar_month_changed_id =
      gtk_signal_connect(GTK_OBJECT(calendar), "month_changed",
			 GTK_SIGNAL_FUNC(fin_on_calendar_date_changed), view);

    view->calendar_day_changed_id =
      gtk_signal_connect(GTK_OBJECT(calendar), "day_selected",
			 GTK_SIGNAL_FUNC(fin_on_calendar_date_changed), view);

    view->calendar = calendar;
  }
}

/* THE END */
