/*
 *
 * (c) Vladi Belperchinov-Shabanski "Cade" <cade@biscom.net> 1996-1999
 *
 * SEE `README',`LICENSE' OR `COPYING' FILE FOR LICENSE AND OTHER DETAILS!
 *
 */

#include "vfu.h"
#include "vfudir.h"
#include "vfumenu.h"
#include "vfufiles.h"
#include "vfuview.h"
#include "vfusys.h"
#include "vfucopy.h"

#define CopyBuffSize  512*1024 // it's a 1/2 meg -- not too much :)

char *cmDESC[] = { "COPY", "MOVE", "SymLINK" };
char *copybuff = NULL;

/////////////////////////////////////////////
//
// some utilities...
//

int IsSameFile( const char *src, const char *dst ) // 0 for ok
{
  #ifdef _TARGET_GO32_
  char _f1[MAX_PATH];
  char _f2[MAX_PATH];
  _fixpath( src, _f1 );
  _fixpath( dst, _f2 );
  ASSERT( _f1[1] == ':' && _f2[1] == ':' );
  return (strcasecmp( _f1, _f2 ) != 0);
  #else
  struct stat st1;
  struct stat st2;
  int r1 = stat( src, &st1 );
  int r2 = stat( dst, &st2 );
  if (r1==0 && r2==0 && st1.st_dev==st2.st_dev && st1.st_ino==st2.st_ino ) return 0; else return 1;
  #endif
};

int IsSameDevice( const char *src, const char *dst ) // 0 for ok
{
  #ifdef _TARGET_GO32_
  char _f1[MAX_PATH];
  char _f2[MAX_PATH];
  _fixpath( src, _f1 );
  _fixpath( dst, _f2 );
  ASSERT( _f1[1] == ':' && _f2[1] == ':' );
  return ( _f1[0] != _f2[0] );
  #else
  char *ch;
  struct stat st1;
  struct stat st2;
  char _f1[MAX_PATH];
  char _f2[MAX_PATH];
  strcpy( _f1, src );
  ch = strrchr( _f1, '/' ); if (ch == NULL) _f1[0] = 0; else ch[1] = 0;
  strcat( _f1, "." );
  strcpy( _f2, dst );
  ch = strrchr( _f2, '/' ); if (ch == NULL) _f2[0] = 0; else ch[1] = 0;
  strcat( _f2, "." );
  int r1 = stat( _f1, &st1 );
  int r2 = stat( _f2, &st2 );
  if (r1==0 && r2==0 && st1.st_dev==st2.st_dev ) return 0; else return 1;
  #endif
};


int fast_stat( const char* s, struct stat *st )
{
  #ifdef _TARGET_GO32_
  _djstat_flags = _STAT_INODE|_STAT_EXEC_EXT|_STAT_EXEC_MAGIC|_STAT_DIRSIZE|_STAT_ROOT_TIME;
  int res = stat( s, st );
  _djstat_flags = 0;
  return res;
  #else
  return stat( s, st );
  #endif
};


/////////////////////////////////////////////
//
// globals for copy/move
//
fsize_t cSize = 0; // current size
fsize_t aSize = 0; // all size

void ShowCopyPos( fsize_t ccSize, fsize_t aaSize ) // current file's `currSize and allSize'
{
  char tmp[16];
  if (  aSize < 1 )  aSize = 1; // avoid DBZero
  if ( aaSize < 1 ) aaSize = 1; // avoid DBZero
  if ( ccSize == aaSize ) // kind'a hack? :) every single 100% each is not meaningfull really
    sprintf( tmp, "     %%%5.1f", (100.0*(cSize+ccSize))/aSize);
  else
    sprintf( tmp, "%5.1f%%%5.1f", (100.0*ccSize)/aaSize, (100.0*(cSize+ccSize))/aSize);
  ConOut( MAXX - 12, STATLINE2+2, tmp, cMESSAGE );
}

/////////////////////////////////////////////
//
// file/dir copy
//
int OverMode = 0;
int CopyFreeSpaceCheck = 0;

int OverwriteIfExist( const char* _fr, const char *_to )
{
if (OverMode == omAsk || OverMode == omNever)
  if ( access(_to, F_OK) == 0)
    {
    if (OverMode == omNever) return 0;

    struct stat stat_src;
    struct stat stat_dst;
    fast_stat( _fr, &stat_src );
    fast_stat( _to, &stat_dst );

    if (OverMode == omAIfNewer)
      return ( stat_src.st_mtime >= stat_dst.st_mtime );
    Beep();

    String str;
    char sttime[32];

    char s_t = (OptTime(stat_src) == OptTime(stat_dst))?'*':' '; // same time
    char s_s = (stat_src.st_size == stat_dst.st_size)?'*':' '; // same size

    TimeStr(OptTime(stat_src), sttime); str = (long int)stat_src.st_size; StrComma(str);
    sprintf(sss, "SRC: %s%c %11s%c %s", sttime, s_t, str.asis(), s_s, _fr );
    say1(sss);

    TimeStr(OptTime(stat_dst), sttime); str = (long int)stat_dst.st_size; StrComma(str);
    sprintf(sss, "DST: %s%c %11s%c %s", sttime, s_t, str.asis(), s_s, _to );
    say2(sss);

    MenuBox( "Overwrite", "Y Yes,N No,A Always overwrite,V Never overwrite,I If SRC is newer (MODIFY),W Always if newer (MODIFY)", -1 );
    int ch = MenuBoxExitCh;

    say1( "" );
    say2( "" );

    switch (ch)
      {
      case 'Y' : return 1;
      case 'N' : return 0;
      case 'I' : return ( stat_src.st_mtime >= stat_dst.st_mtime ); break;
      case 'A' : OverMode = omAlways; return 1;
      case 'V' : OverMode = omNever; return 0;
      case 'W' : OverMode = omAIfNewer; return 0;
      case 27  : OverMode = omAbort; return 0;
      }
    }
return 1;
}

int __CopyFile( const char* src, const char* dst )
{
  struct stat sta;
  memset( &sta, 0, sizeof(sta) );
  fast_stat( src, &sta );

  fsize_t size = (fsize_t)(sta.st_size);

  if (CopyFreeSpaceCheck)
    {
    struct statfs stafs;
    memset( &stafs, 0, sizeof(stafs) );
    FilePath( dst, sss );
    statfs( sss, &stafs );
    fsize_t bfree = ((fsize_t)(stafs.f_bsize)) * (opt.RealFreeSpace?stafs.f_bfree:stafs.f_bavail);
    if (size > bfree )
      {
      Beep();
      sprintf(sss, "Insufficient disk space on destination path!  Free: %d, Need: %d", bfree, aSize ); say1( sss );
      say2( dst );
      MenuBox( "Error prompt", "C Continue anyway,S Skip file,N Do NOT check free space,A Abort operation", -1 );
      switch (MenuBoxExitCh)
        {
        case 'C' : break;
        case 'S' : return 0; break;
        case 'N' : CopyFreeSpaceCheck = 0; break;
        case 'A' :
        case 27  : OverMode = omAbort; return 1; break;
        }
      }
    }

  ASSERT( copybuff != NULL );
  if (copybuff == NULL) return 1;

  FILE *f1 = NULL;
  FILE *f2 = NULL;

  f1 = fopen( src, "rb" );
  if (!f1) return 1;

  fsetattr_s( dst, ATTR_WRITE_ON ); // allow write on the target
  f2 = fopen( dst, "wb" );
  if (!f2)
    {
    fclose(f1);
    return 1;
    }

  long z = 0;
  fsize_t cp = 0; // current pos...

  do
    {
    z = fread(copybuff, 1, CopyBuffSize, f1);
    if (z > 0) z = fwrite(copybuff, 1, z, f2);
    if (z == -1) { fclose(f1); fclose(f2); return 1; }
    cp += z;
    ShowCopyPos( cp, size );
    }
  while (z == CopyBuffSize);

  fclose(f1);
  fclose(f2);

  if ( cp != size )
    {
    unlink(dst);
    return 1;
    }
  //  ASSERT( cp == size );

  // preserve attribs
  attrs_t attrs;
  fgetattr_s( src, attrs );
  fsetattr_s( dst, attrs );

  // preserve timestamps
  utimbuf utb = { 0, 0 };
  utb.actime  = sta.st_atime;
  utb.modtime = sta.st_mtime;
  utime( dst, &utb );

  // preserve owner/group
  if (opt.FileCopyOwner) chown( dst, sta.st_uid, sta.st_gid );

  return 0;
}

int __CopyLink( const char* src, const char* dst )
{
  #ifdef _TARGET_UNIX_
  char tmp[1024];
  int z = readlink( src, tmp, sizeof(tmp)-1);
  if (z < 1) return 1;
  tmp[z] = 0;
  if (symlink( tmp, dst ) == -1) return 1;
  attrs_t attrs;
  fgetattr_s( src, attrs );
  fsetattr_s( dst, attrs );
  return 0;
  #endif // _TARGET_UNIX_
}

int CopyFile(const char* src, const char*dst,
             const char* name1,
             const char* name2,
             fsize_t fsize, int mode )
{
  char _fr[MAX_PATH];
  char _to[MAX_PATH];
  strcpy( _fr, src ); strcat( _fr, name1 );
  strcpy( _to, dst ); strcat( _to, name2 );

  if ( IsSameFile( _fr, _to ) == 0 ) return 1;

  if (!OverwriteIfExist( _fr, _to )) return 0; // not error really // omSkip;
  if (access( _to, F_OK ) == 0)
    {
    fsetattr_s( _to, ATTR_WRITE_ON ); // allow write on the target
    if (unlink( _to )) return 1;
    }

  //---progress report----------------------
  String str;
  str = cmDESC[mode]; str += ": "; str += name2/*_fr*/;
  if (str.len() > 20) { StrSLeft( str, 27 ); str += "..."; }
  StrPad( str, -20 );
  str += " => "; str += _to;
  if ( str.len() < MAXX )
    StrPad( str, -MAXX );
  else
    StrSLeft( str, MAXX );
  ConOut( 1, STATLINE2+1, str, cSTATUS );
  //----------------------------------------

  int res = 255;
  while(res)
    {
      if ( mode == cmCOPY )
        {
        if ( fsize == -2 ) // -2 means link
          res = __CopyLink( _fr, _to );
        else
          res = __CopyFile( _fr, _to );
        } else
      if ( mode == cmMOVE )
        {
        if ( IsSameDevice( _fr, _to ) != 0 )
          { // cross-filesystem move
          if ( fsize == -2 ) // -2 means link
            res = __CopyLink( _fr, _to );
          else
            res = __CopyFile( _fr, _to );
          if (res == 0)
            res = (unlink( _fr ) != 0);
          }
        else
          { // same filesystem move
          res = rename(_fr, _to);
          }
        } else
      if ( mode == cmLINK )
        {
        res = symlink( _fr, _to ) != 0;
        }
      if (res != 0)
        {
        sprintf( sss, "%s: %s -> %s", cmDESC[mode], _fr, _to );
        say1( sss );
        DescribeErrno();
        MenuBox( "Error occured", "T Try again,S Skip file,A Abort action", -1 );
        switch (MenuBoxExitCh)
          {
          case 'T' : res = 255; break;
          case 'S' : res = 0; break;
          case 'A' :
          case 27  : res = 0; OverMode = omAbort; break;
          }
        }
    }
  cSize += fsize;
//  ShowCopyPos( 1, 1 );
  return res;
};

int CopyDir (const char* src, const char*dst,
             const char* name1,
             const char* name2,
             fsize_t fsize, int mode )
{
  char newdname[MAX_PATH];
  char _fr[MAX_PATH];
  char _to[MAX_PATH];
  strcpy( _fr, src ); strcat( _fr, name1 );
  strcpy( _to, dst ); strcat( _to, name2 );

  if (IsSameFile( src, dst ) == 0) return 1;

  int z;
  if (mode == cmMOVE)
    if ( (z = IsSameDevice( _fr, _to )) == 0 && access( _to, F_OK ) != 0 )
      { // same filesystem and not existing target -- do rename()
      cSize += fsize;
      return (rename( _fr, _to ) != 0);
      }

  int res = 0;
  DIR *dir;
  dirent *de;
  if(MakePath( _to ))
    {
    say2( _to );
    if(tolower(Ask( "Cannot create path! ( ESC=cancel, C=continue )", "\033Cc" )) == 27)
      {
      OverMode = omAbort;
      return 0;
      }
    }

  // get directory stats
  attrs_t dir_attrs;
  fgetattr_s( _fr, dir_attrs );
  struct stat sta;
  fast_stat( _fr, &sta );
  utimbuf dir_utb = { 0, 0 };
  dir_utb.actime  = sta.st_atime;
  dir_utb.modtime = sta.st_mtime;

  strcat( _fr, "/" );
  strcat( _to, "/" );

  dir = opendir( _fr );
  if (!dir)
    {
    sprintf( sss, "Cannot read dir: %s", _fr );
    say1( sss );
    DescribeErrno();
    ConGetch();
    return 1;
    }
  while( (de = readdir( dir )) )
    {
    if (strcmp( de->d_name, ".") == 0) continue;
    if (strcmp( de->d_name, "..") == 0) continue;
    sprintf( newdname, "%s%s", _fr, de->d_name );
    struct stat st;
    struct stat lst;
    lstat( newdname, &lst );
    #ifdef _TARGET_GO32_
    dosstat(dir, &st);
    #else
    stat( newdname, &st );
    #endif
    if (S_ISDIR(st.st_mode) && !S_ISLNK(lst.st_mode))
      res += CopyDir( _fr, _to, de->d_name, de->d_name, st.st_size, mode );
    else
      if (S_ISLNK(lst.st_mode))
        res += CopyFile( _fr, _to, de->d_name, de->d_name, -2, mode );
      else
        res += CopyFile( _fr, _to, de->d_name, de->d_name, st.st_size, mode );
    if (BreakOp()) OverMode = omAbort;
    if (OverMode == omAbort) break;
    }
  closedir( dir );

  // restore directory stats
  fsetattr_s( _to, dir_attrs );
  utime( _to, &dir_utb );

  if (mode == cmMOVE) rmdir( _fr );

  cSize += fsize;
  ShowCopyPos( 1, 1 );
  return res;
};


void CopyMoveFiles( int one, int mode )
{
  #ifdef _TARGET_GO32_
  if ( mode == cmLINK )
    {
    say1( "SymLINK: SymLinks are NOT supported on this platform!" );
    return;
    }
  #endif

  char temp[255];
  OverMode = omAsk;

  ASSERT( mode == cmCOPY || mode == cmMOVE || mode == cmLINK );

  if ( SelCount == 0 && one == 0 ) one = 1;

  if ( one == 0) sprintf(temp, "%s SELECTION to: ", cmDESC[mode] );
  else
  if ( one == 1 )
    sprintf(temp, "%s `%s' to:", cmDESC[mode], Files[FLI]->fname);
  else
    return;
  GetDirName(temp, opt.LastCopyPath[mode] );
  if (TargetDir[0] == 0) return;

  SimplifyPath( NULL, TargetDir );

  strcpy(opt.LastCopyPath[mode], TargetDir);
  say1("");

  int z;

  int err = 0;

  UpdateStats(); // to get right AllSize/SelSize/etc. ... JTBS

  copybuff = new char[CopyBuffSize];
  ASSERT(copybuff != NULL);
  if (copybuff == NULL)
    {
    say1( "ENORM.ERROR: cannot allocate copy buffer :( ..." );
    return;
    }

  cSize = 0;
  aSize = CalcSelSize( one );
  if (aSize == 0) aSize++; // JTBS avoid division by 0 :)
  OverMode = omAsk; // twice--
  int _go_ = 1;
  CopyFreeSpaceCheck = opt.FreeSpaceCheck;
  if (opt.FreeSpaceCheck)
    {
      if ( (mode == cmMOVE && IsSameDevice( CPath, TargetDir ) != 0 ) || (mode == cmCOPY))
        {
        struct statfs stafs;
        FilePath( TargetDir, sss );
        statfs( sss, &stafs );
        fsize_t bfree = ((fsize_t)(stafs.f_bsize)) * (opt.RealFreeSpace?stafs.f_bfree:stafs.f_bavail);
        if (aSize > bfree )
          {
          Beep();
          sprintf(sss, "Insufficient disk space on destination path!  Free: %d, Need: %d", bfree, aSize );
          say1( sss );
          sprintf( sss, "Press ENTER to continue anyway or ESC to abort" );
          say2( sss );
          int ch = 0;
          while(ch != 13 && ch != 27) ch = ConGetch();
          _go_ = (ch == 13);
          }
        }
    }
  if (_go_)
  for(z = 0; z < FilesCount; z++ )
    {
    Files[z]->x = 0; // used to find which ones are for removing after move
    if ((Files[z]->sel && one == 0) || (z == FLI && one == 1))
      {
      int rres = 0;
      TF *fi = Files[z];
      if ( fi->is_dir && !fi->is_link && mode != cmLINK )
        {
        rres += CopyDir( CPath, TargetDir, fi->name, fi->fname, fi->size, mode );
        }
      else
        {
        if (fi->is_link)
          rres = CopyFile( CPath, TargetDir, fi->name, fi->fname, -2, mode );
        else
          rres = CopyFile( CPath, TargetDir, fi->name, fi->fname, fi->size, mode );
        }
      if (mode == cmMOVE && rres == 0)
        {
        // check if file not exist -- should be removed
        if (access(Files[z]->name, F_OK)) Files[z]->x = 1;
        /*
        Files[z] = NULL;
        delete Files[z];
        */
        }
      if (rres == 100) rres = 0; // copy/move aborted -- no error
      err += rres;
      if (BreakOp()) OverMode = omAbort;
      if (OverMode == omAbort) break;
      }
    }
  if (mode == cmMOVE)
    {
    for(z = 0; z < FilesCount; z++ )
      if ( Files[z]->x )
        {
        delete Files[z];
        Files[z] = NULL;
        }
    Pack(); // differs from Move
    StatFS();
    ReDrawSta();
    draw = 1;
    }

  if (copybuff != NULL) delete copybuff;

  if (err)
    {
    sprintf(sss, "%s errors: %d", cmDESC[mode], err );
    say1(sss);
    DescribeErrno();
    }
  else
    {
    sprintf(sss, "%s ok.", cmDESC[mode] );
    say1(sss);
    }
  OverMode = omAsk; // JTBS
  say2( "" );
}

