/*  Screem:  Todo.c,
 *  Handles todo tasks for individual sites
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <glade/glade.h>

#include "site.h"
#include "todo.h"

extern Site *current_site;

static void screem_todo_item_set( Todo_item *item, gchar *t, gchar *a, 
				  gchar *p, gchar *l, gchar *d );

static void todo_insert_tasks( GtkWidget *task_list, Site *site );
void todo_select_task( GtkCList *task_list, gint row, gint column,
		       GdkEventButton *event );
void todo_dialog_closed( GtkWidget *widget, GdkEvent *event );
void todo_dialog_clicked( GtkWidget *widget, gint button );

static void edit_todo_item( GtkWidget *task_list );
static void add_todo_item( GtkWidget *task_list );

static GtkWidget* create_dialog( void );
static GtkWidget* create_edit_dialog( void );

static GladeXML *xml;
static GtkWidget *dialog = NULL;

Todo_item* screem_todo_item_new()
{
	Todo_item *item;

	item = (Todo_item*)g_malloc( sizeof( Todo_item ) );

	item->task = NULL;
	item->assigned = NULL;
	item->priority = NULL;
	item->linked_to = NULL;
	item->description = NULL;

	return item;
}

void screem_todo_item_destroy( Todo_item *item )
{
	g_return_if_fail( item != NULL );

	g_free( item->task );
	g_free( item->assigned );
	g_free( item->priority );
	g_free( item->linked_to );
	g_free( item->description );

	g_free( item );
}

static void screem_todo_item_set( Todo_item *item, gchar *t, gchar *a, 
				  gchar *p, gchar *l, gchar *d )
{
	g_return_if_fail( item != NULL );
	
	if( t ) {
		g_free( item->task );
		item->task = g_strdup( t );
	}
	if( a ) {
		g_free( item->assigned );
		item->assigned = g_strdup( a );
	}
	if( p ) {
		g_free( item->priority );
		item->priority = g_strdup( p );
	}
	if( l ) {
		g_free( item->linked_to );
		item->linked_to = g_strdup( l );
	}
	if( d ) {
		g_free( item->description );
		item->description = g_strdup( d );
	}
}

void todo_view()
{
	GtkWidget *list = NULL;
	Site *site;

	site = current_site;

	g_return_if_fail( site != NULL );

	if( dialog ) {
		gdk_window_raise( dialog->window );
                gdk_window_show( dialog->window );
                return;
	}

	xml = glade_xml_new( GLADE_PATH"/todo.glade", "todo_dialog" );
	glade_xml_signal_autoconnect( xml );
	
	dialog = glade_xml_get_widget( xml, "todo_dialog" );
	list = glade_xml_get_widget( xml, "list" );

	todo_insert_tasks( list, site );
}

static void todo_insert_tasks( GtkWidget *task_list, Site *site )
{
	GList *list;
	Todo_item *todo_item;
	gchar *entry[ 5 ];

	g_return_if_fail( site != NULL );

	for( list = screem_site_get_tasks( site ); list; list = list->next ) {
		if( ( todo_item = (Todo_item*) list->data ) ) {
			entry[ 0 ] = todo_item->task;
			entry[ 1 ] = todo_item->assigned;
			entry[ 2 ] = todo_item->priority;
			entry[ 3 ] = todo_item->linked_to;
			entry[ 4 ] = todo_item->description;
			gtk_clist_freeze( GTK_CLIST( task_list ) );
                        gtk_clist_append( GTK_CLIST( task_list ), entry );
                        gtk_clist_thaw( GTK_CLIST( task_list ) );
		}
	}
}

void todo_select_task( GtkCList *task_list, gint row, gint column,
			      GdkEventButton *event )
{
	gtk_object_set_data( GTK_OBJECT( task_list ), "row", (gpointer)row );
}

void todo_dialog_closed( GtkWidget *widget, GdkEvent *event )
{
	dialog = NULL;
}

void todo_dialog_clicked( GtkWidget *widget, gint button )
{
	GtkWidget *task_list = glade_xml_get_widget( xml, "list" );
	gint row;
	gchar *task;
	
	switch( button ) {
		/* Edit button */
	case 0:
		edit_todo_item( task_list );
		break;
		/* Complete button */
	case 1:
		row = (gint)gtk_object_get_data( GTK_OBJECT(task_list),"row" );
		gtk_clist_get_text( GTK_CLIST( task_list ), row, 0, &task );
		screem_site_remove_task( current_site, task );
		gtk_clist_remove( GTK_CLIST( task_list ), row );
		break;
		/* Add button */
	case 2:
		add_todo_item( task_list );
		break;
		/* Close button */
	case 3:
		dialog = NULL;
		gtk_widget_destroy( widget );
		break;
		/* Help button */
	case 4:
		break;
	}
}

static void edit_todo_item( GtkWidget *task_list )
{
    	GtkWidget *dialog;
	gint button;
	Todo_item *item;

	GladeXML *xml_ = glade_xml_new( GLADE_PATH"/todo.glade", 
					"todo_edit_dialog" );

	GtkWidget *task = glade_xml_get_widget( xml_, "task" );
	GtkWidget *assigned = glade_xml_get_widget( xml_, "assigned" );
	GtkWidget *priority = glade_xml_get_widget( xml_, "priority" );
	GtkWidget *linked = glade_xml_get_widget( xml_, "linked" );
	GtkWidget *description = glade_xml_get_widget( xml_, "description" );

	gchar *t;
	gchar *a;
	gchar *p;
	gchar *l;
	gchar *d;

	gint row = (gint)gtk_object_get_data( GTK_OBJECT( task_list ), "row" );

	glade_xml_signal_autoconnect( xml_ );
	dialog = glade_xml_get_widget( xml_, "todo_edit_dialog" );

	gtk_entry_set_editable( GTK_ENTRY( task ), FALSE );

	gtk_clist_get_text( GTK_CLIST( task_list ), row, 0, &t );
	gtk_clist_get_text( GTK_CLIST( task_list ), row, 1, &a );
	gtk_clist_get_text( GTK_CLIST( task_list ), row, 2, &p );
	gtk_clist_get_text( GTK_CLIST( task_list ), row, 3, &l );
	gtk_clist_get_text( GTK_CLIST( task_list ), row, 4, &d );

	gtk_entry_set_text( GTK_ENTRY( task ), t );
	gtk_entry_set_text( GTK_ENTRY( assigned ), a );
	gtk_entry_set_text( GTK_ENTRY( priority ), p );
	gtk_entry_set_text( GTK_ENTRY( linked ), l );
	gtk_entry_set_text( GTK_ENTRY( description ), d );

	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	a = g_strdup( gtk_entry_get_text( GTK_ENTRY( assigned ) ) );
	p = g_strdup( gtk_entry_get_text( GTK_ENTRY( priority ) ) );
	l = g_strdup( gtk_entry_get_text( GTK_ENTRY( linked ) ) );
	d = g_strdup( gtk_entry_get_text( GTK_ENTRY( description ) ) );

	item = screem_site_locate_task( current_site, t );

	gtk_widget_destroy( dialog );
	if( button == 1 ) {
		g_free( a );
		g_free( p );
		g_free( l );
		g_free( d );
		return;
	}
	/* ok clicked */

	gtk_clist_set_text( GTK_CLIST( task_list ), row, 1, a );
	gtk_clist_set_text( GTK_CLIST( task_list ), row, 2, p );
	gtk_clist_set_text( GTK_CLIST( task_list ), row, 3, l );
	gtk_clist_set_text( GTK_CLIST( task_list ), row, 4, d );

	screem_todo_item_set( item, NULL, a, p, l, d );

	g_free( a );
	g_free( p );
	g_free( l );
	g_free( d );
}

static void add_todo_item( GtkWidget *task_list )
{
	GtkWidget *dialog;
	gint button;
	Todo_item *item;
	gchar *entry[ 5 ];

	GladeXML *xml_ = glade_xml_new( GLADE_PATH"/todo.glade", 
					"todo_edit_dialog" );

	GtkWidget *task = glade_xml_get_widget( xml_, "task" );
	GtkWidget *assigned = glade_xml_get_widget( xml_, "assigned" );
	GtkWidget *priority = glade_xml_get_widget( xml_, "priority" );
	GtkWidget *linked = glade_xml_get_widget( xml_, "linked" );
	GtkWidget *description = glade_xml_get_widget( xml_, "description" );

	gchar *t;
	gchar *a;
	gchar *p;
	gchar *l;
	gchar *d;

	glade_xml_signal_autoconnect( xml_ );
	dialog = glade_xml_get_widget( xml_, "todo_edit_dialog" );

	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	t = g_strdup( gtk_entry_get_text( GTK_ENTRY( task ) ) );
	a = g_strdup( gtk_entry_get_text( GTK_ENTRY( assigned ) ) );
	p = g_strdup( gtk_entry_get_text( GTK_ENTRY( priority ) ) );
	l = g_strdup( gtk_entry_get_text( GTK_ENTRY( linked ) ) );
	d = g_strdup( gtk_entry_get_text( GTK_ENTRY( description ) ) );

	gtk_widget_destroy( dialog );
	if( button == 1 ) {
		g_free( t );
		g_free( a );
		g_free( p );
		g_free( l );
		g_free( d );
		return;
	}

	/* ok clicked */
	item = screem_todo_item_new();
	screem_todo_item_set( item, t, a, p, l, d );

	/* add it to the clist */
	entry[ 0 ] = t;
	entry[ 1 ] = a;
	entry[ 2 ] = p;
	entry[ 3 ] = l;
	entry[ 4 ] = d;
	gtk_clist_append( GTK_CLIST( task_list ), entry );


	g_free( t );
	g_free( a );
	g_free( p );
	g_free( l );
	g_free( d );
	
	screem_site_add_task( current_site, item );
}
