/*  Screem:  siteTreeUI.c,
 *  The UI side of the site tree, handling user interaction such as clicking,
 *  and drag and drop
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <gnome.h>
#include <gnome-xml/debugXML.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>

#include <sys/types.h>

#include "cvs.h"
#include "editor.h"
#include "fileops.h"
#include "page.h"
#include "pageUI.h"
#include "preview.h"
#include "site.h"
#include "siteTree.h"
#include "siteTreeUI.h"
#include "siteUI.h"
#include "support.h"
#include "xml.h"

#include "dir-open.xpm"
#include "dir-close.xpm"

extern Site *current_site;
extern GtkWidget *app;

extern GList *icon_list;

static void build_file_tree( GtkWidget *tree, Site *site, xmlDocPtr doc );
static void add_node( Site *site, GtkCTree *t, xmlDocPtr doc, xmlNodePtr node,
		      GtkCTreeNode *parent );

static void tree_item_clicked(GtkCTree *tree, GtkCTreeNode *row, gint column);

static void tree_open_file( void );
static void tree_delete_file( void );
static void tree_rename_file( void );

static GdkPixmap *folder_open = NULL;
static GdkPixmap *folder_close = NULL;
static GdkBitmap *folder_open_mask = NULL;
static GdkBitmap *folder_close_mask = NULL;

static GnomeUIInfo file_tree_menu[] = {
        GNOMEUIINFO_MENU_NEW_ITEM( N_("_New Page..."), 
                                   N_("Add a new page to the site"), 
                                   GTK_SIGNAL_FUNC( screem_page_create_page ),
				   NULL ),
	GNOMEUIINFO_MENU_NEW_ITEM( N_( "_New Blank Page" ),
				   N_("Create a blank page (add if working on a site)" ),
				   GTK_SIGNAL_FUNC( screem_page_create_blank_page ),
				   NULL ),
        { GNOME_APP_UI_ITEM, N_("_Delete File"), 
          N_( "Delete File" ),
          GTK_SIGNAL_FUNC( tree_delete_file ), NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("_Rename File"), 
          N_( "Rename File" ),
          GTK_SIGNAL_FUNC( tree_rename_file ), NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("_Open"), N_( "Opens the file" ),
          GTK_SIGNAL_FUNC( tree_open_file ), NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 'O',
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("Refresh"), N_( "Refresh the tree" ),
          GTK_SIGNAL_FUNC( refresh_file_tree ), NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

static GnomeUIInfo tree_dnd_menu[] = {
        { GNOME_APP_UI_ITEM, N_("_Move here"), N_("Move file"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("_Copy here"), N_("Copy file"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_SEPARATOR,
        { GNOME_APP_UI_ITEM, N_("Cancel drag"), N_("Cancel drag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};

static void load_folder_icons()
{
	GdkImlibImage *image;

	if( ! folder_open ) {
		/* we haven't created the dir icon the file tree yet */
		image = gdk_imlib_create_image_from_xpm_data( DIRECTORY_OPEN_XPM );
		gdk_imlib_render( image, image->rgb_width, image->rgb_height );
		folder_open = gdk_imlib_move_image( image );
		folder_open_mask = gdk_imlib_move_mask( image );
		
		image = gdk_imlib_create_image_from_xpm_data( DIRECTORY_CLOSE_XPM );
		gdk_imlib_render( image, image->rgb_width, image->rgb_height );
		folder_close = gdk_imlib_move_image( image );
		folder_close_mask = gdk_imlib_move_mask( image );
	}
}

static void build_file_tree( GtkWidget *tree, Site *site, xmlDocPtr doc )
{
	xmlNodePtr node;
 	GtkCTreeNode *tnode;

	g_return_if_fail( tree != NULL );
        g_return_if_fail( doc != NULL );

	load_folder_icons();

	node = doc->root;
	/* not interested in the xml version element */
        node = node->childs;

	gtk_clist_freeze( GTK_CLIST( tree ) );
	add_node( site, GTK_CTREE( tree ), doc, node, NULL );

	/* now sort the tree */
	tnode = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );
        gtk_ctree_sort_recursive( GTK_CTREE( tree ), tnode );
        gtk_ctree_expand( GTK_CTREE( tree ), tnode );

	gtk_clist_thaw( GTK_CLIST( tree ) );
}


/*
 * refresh_file_tree:
 *
 * Rebuilds the file tree
 *
 * return values: none
 */
void refresh_file_tree()
{
	Site *site;
	xmlDocPtr doc;
	GtkWidget *tree;

	const gchar *site_path;

	site = current_site;

	g_return_if_fail( site != NULL );

	site_path = screem_site_get_pathname( site );

	doc = build_directory_tree( site, site_path, NULL );

	g_return_if_fail( doc != NULL );

	identify_pages( site, doc, doc->root->childs );

	tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );
	gtk_clist_clear( GTK_CLIST( tree ) );
	build_file_tree( tree, site, doc );
}
static void add_node( Site *site, GtkCTree *t, xmlDocPtr doc, xmlNodePtr node,
		      GtkCTreeNode *parent )
{
        gchar *text[ 2 ] = { NULL, NULL };
        GtkCTreeNode *this = parent;
	gchar *entry;
	gchar *temp;

	const gchar *mime_type;
	const gchar *site_path;

	GdkPixmap *open = folder_open;
        GdkBitmap *open_mask = folder_open_mask;
        GdkPixmap *close = folder_close;
        GdkBitmap *close_mask = folder_close_mask;

	gboolean is_dir;

	GList *list;
	Icons *icon = NULL;
	const gchar *file;

    	g_return_if_fail( t != NULL );

	if( site )
		site_path = screem_site_get_pathname( site );
	else
		site_path = RESOURCE_PATH;

        /* add the node */
        entry = xmlNodeListGetString( doc, node->childs, 1 );

	mime_type = xml_get_value( node, "type" );

	if( entry ) {
		temp = entry + strlen( site_path );
		is_dir = g_file_test( entry, G_FILE_TEST_ISDIR );
		if( ! is_dir )
			entry = g_strdup( g_filename_pointer( temp ) );
		else {
			if( strlen( temp ) > 0 ) {
				entry = temp + strlen( temp ) - 1;
				while( *(--entry) != G_DIR_SEPARATOR ){}
				entry = g_strndup( entry, strlen( entry ) -1 );
			} else {
				entry = g_strdup( G_DIR_SEPARATOR_S );
			}
		}
		text[ 0 ] = entry;
		temp -= strlen( site_path );

		if( ! is_dir ) {
			icon = icon_from_mime_type( mime_type );
			
			open = icon->open;
			open_mask = icon->open_mask;
			close = open;
			close_mask = open_mask;
		}

		if( strcmp( mime_type, "application/x-screem" ) ) {
			this = gtk_ctree_insert_node( t, parent, NULL, text,
						      3, close, close_mask, 
						      open, open_mask, 
						      (gboolean)parent, FALSE);
			/* set up its row data */
			gtk_ctree_node_set_row_data( GTK_CTREE( t ), this,
						     temp );
		} else {
			g_free( temp );
		}
		g_free( entry );
	}

        /* has children? */
        if( node->childs )
                add_node( site, t, doc, node->childs, this );

        /* move on to the next sibling */
        if( node->next )
                add_node( site, t, doc, node->next, parent );
}

void tree_clicked( GtkWidget *widget, GdkEventButton *event,
		   gpointer data )
{
	GtkCTreeNode *node = NULL;
        GtkWidget *menu;
        gint row = -1;
        gint col;

	static guint32 time = 0;

	if( ! current_site )
		return;

	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );

	if( row != - 1 )
                node = gtk_ctree_node_nth( GTK_CTREE( widget ), ( guint )row );

	/* set the node that was clicked on */
        if( node ) {
                gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", GINT_TO_POINTER( row ) );
        } else {
                gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", NULL);
	}

	/* what button was clicked? */
	if( event->button == 1 ) {
                if( time == event->time ) /* double click */
                        tree_item_clicked( GTK_CTREE( widget ), node, 1 );
                time = event->time;
        } else if( event->button == 3 ) {
                /* select the line underneath the mouse */
                if( node )
                        gtk_ctree_select( GTK_CTREE( widget ), node );
                menu = gnome_popup_menu_new( file_tree_menu );
                gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0 );
                gtk_widget_destroy( menu );
        }
}

static void tree_item_clicked( GtkCTree *tree, GtkCTreeNode *row, gint column )
{
	Page *page = NULL;
	static Page *prev;
	gchar *path = NULL;
	const gchar *mime_type;
	const gchar *mime_exec;
	gchar **params;
	gint num = 0;

	const gchar *page_path;
	gchar *pname;

	GtkWidget *box = gtk_object_get_data( GTK_OBJECT( app ), "tab_box" );
	GtkWidget *tab;

	if( ! current_site )
		return;

	path = gtk_ctree_node_get_row_data( tree, row );

	if( ! path )
		return;

	if( g_file_test( path, G_FILE_TEST_ISDIR ) )
		return;

	/* locate the page with the matching path */
	page = screem_site_locate_page( current_site, path );

	mime_type = gnome_mime_type( path );

	/* if the path wasn't a page then we open it in the program
	   determined by its mime type */
	if( ! page ) {
		mime_exec = gnome_mime_program( mime_type );
		if( ! mime_exec )
			return;
		/* replace an occurance of %f in mime_exec with the
		   filename to open */
		params = g_strsplit( mime_exec, " ", 255 );
                while( params[ num ] ) {
                        if( ! strcmp( params[ num ], "%f" ) )
                                params[ num ] = g_strdup( path );
                        num ++;
                }
                path = g_strjoinv( " ", params );
                g_strfreev( params );
		
		/* now execite the progra, */
		gnome_execute_shell( NULL, path );

		g_free( path );

		prev = NULL;

		return;
	}

	if( page == prev )
		return;



	screem_page_insert( page );
}


void tree_set_dnd_data( GtkWidget *widget, GdkDragContext *context,
			GtkSelectionData *selectionData, guint info,
			guint time, gpointer data)
{
	GtkCTreeNode *node;
        gchar *path = NULL;
        gchar *text = NULL;
	gint row;

        if( ! GTK_CLIST( widget )->selection )
                return;
	
	if( GTK_IS_CTREE( widget ) ) {
		node = ((GList*)(GTK_CLIST(widget)->selection))->data;
		if( node )
			path = gtk_ctree_node_get_row_data( GTK_CTREE(widget),
							    node );
	} else {
		row = (gint)gtk_object_get_data(GTK_OBJECT(widget), "drag_start");
		path = gtk_clist_get_row_data( GTK_CLIST( widget ), row );
	}
	
        if( path ) {
                switch( info ) {
                case TARGET_URI_LIST:
                        text = g_strconcat( "file:", path, "\r\n", NULL );
                        break;
                }
		
                gtk_selection_data_set( selectionData, selectionData->target,
                                        8, text, strlen( text ) );
                g_free( text );
        } else {
                gtk_selection_data_set( selectionData, selectionData->target,
                                        8, NULL, 0 );
	}
}

void tree_drag_begin( GtkWidget *widget, GdkDragContext *context,
		      gpointer data )
{
	GtkCTreeNode *node;
        gpointer key;
        
        key = gtk_object_get_data( GTK_OBJECT( widget ), "drag_start" );
	
	g_return_if_fail( key != NULL );
       
	if( GTK_IS_CTREE( widget ) ) {
		node = gtk_ctree_node_nth( GTK_CTREE( widget ), 
					   GPOINTER_TO_INT( key ));
		g_return_if_fail( node != NULL );
		gtk_ctree_select( GTK_CTREE( widget ), node );
	} else {
		gtk_clist_select_row( GTK_CLIST( widget ), 
				      GPOINTER_TO_INT( key ),
				      0 );
}
}

gboolean tree_drag_motion( GtkWidget *widget, 
			   GdkDragContext *context, 
			   gint x, gint y, guint time,
			   gpointer data )
{
	GdkDragAction action;
	
        GtkWidget *source;

        source = gtk_drag_get_source_widget( context );

        if( ( source ) &&
            ( context->suggested_action != GDK_ACTION_ASK ) )
                action = GDK_ACTION_MOVE; 
        else
                action = context->suggested_action;

        gdk_drag_status( context, action, time );

        return TRUE;
}

void tree_drop_data( GtkWidget *widget, GdkDragContext *context,
		     gint x, gint y, GtkSelectionData *selectionData,
		     guint info, guint time )
{

	Site *site;
	GtkWidget *popup;
	gint item;
	gboolean move;
	gchar *text;
	gint row;
	gint col;
	GtkCTreeNode *node = NULL;
	gchar *path;
    	gchar *file;
	gint len;
	gboolean res;

	gboolean is_dir;
	gboolean is_dir2;

	const gchar *site_path;
	Page *page;

	site = current_site;

	g_return_if_fail( site != NULL );

	site_path = screem_site_get_pathname( site );

	/* if its a middle drag we need to ask the user what to do */
	if( context->action == GDK_ACTION_ASK ) {
		popup = gnome_popup_menu_new( tree_dnd_menu );
		item = gnome_popup_menu_do_popup_modal( popup, 0, 0, 0, 0 );
		switch( item ) {
		case 0:
			context->action = GDK_ACTION_MOVE;
			break;
		case 1:
			context->action = GDK_ACTION_COPY;
			break;
		default:
			return;
			break;
		}
	}

	/* set flag if we are performing a move */
	move = ( context->action == GDK_ACTION_MOVE );

	/* id the drop type */
	switch( info ) {
	case TARGET_URI_LIST:
		text = selectionData->data + strlen( "file:" );
		text[ strlen( text ) - 2 ] = 0;
		is_dir = g_file_test( text, G_FILE_TEST_ISDIR );

		gtk_clist_get_selection_info( GTK_CLIST( widget ), x, y,
					      &row, &col );
		/* id the node we dropped onto */
		if( row != -1 && GTK_IS_CTREE( widget ) ) {
			node = gtk_ctree_node_nth( GTK_CTREE( widget ),
						   ( guint )row );
			g_return_if_fail( node != NULL );
			path = 	(gchar*)
				gtk_ctree_node_get_row_data( GTK_CTREE(widget),
							     node );
		} else {
			g_return_if_fail( row != -1 );
			path = gtk_clist_get_row_data( GTK_CLIST(widget),row );
		}

		g_return_if_fail( path != NULL );

		/* if node isn't a directory node then get its parent
		   ( which will be a directory ) */
		is_dir2 = g_file_test( path, G_FILE_TEST_ISDIR );
		if( (! is_dir2) && GTK_IS_CTREE( widget ) ) {
			node = GTK_CTREE_ROW( node )->parent;
			/* node should never be NULL */
			g_return_if_fail( node != NULL );
			/* get the path for the parent */
			path = (gchar*)
				gtk_ctree_node_get_row_data( GTK_CTREE(widget),
							     node );
		} else if( ! is_dir2 ) {
			/* browse clist, rather than site ctree */
			path = (gchar*)gtk_object_get_data( GTK_OBJECT(widget),
							    "current_dir" );
		}

		g_return_if_fail( path != NULL );

		/* check to see if text and path are the same */
		if( ! strcmp( text, path ) )
			return;

		/* 
		 * path == the directory to drop onto,
		 * now get the file / directory name to drop
		 * (not the full path for it though)
		 */
		file = text + strlen( text ) - ( 1 + is_dir );
		while( *file != G_DIR_SEPARATOR )
			file --;
			
		path = g_strconcat( path, ++ file, NULL );

		/* check to see if text and path are the same now */
		if( ! strcmp( text, path ) ) {
			g_free( path );
			return;
		}

		if( is_dir )
			res = copy_dir( text, path, FALSE );
		else
			res = copy_file( text, path );

		if( ! res ) {
			/* we failed */
			g_free( path );
			return;
		}

		screem_site_rename_pages_with_path( site, text, path );

		/* update any links etc if we performed a move */
		if( move ) {
			screem_editor_buffer_text();
			screem_site_file_change( site, text, path );
			if( is_dir )
				delete_dir( text );
			else
				delete_file( text );

			if( ( page = screem_site_get_current_page( site ) ) ) {
				/* insert the possibly changed page */
				screem_editor_clear();
				screem_editor_insert( 0, page->data );
			}
		}
		g_free( path );

		/* copy/move performed, update the tree */
		refresh_file_tree();

		break;
	}

}

static void tree_open_file()
{
	GtkWidget *file_tree;
	gpointer key;
        GtkCTreeNode *node;

	file_tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

        key = gtk_object_get_data( GTK_OBJECT( file_tree ), "drag_start" );

        if( ! key )
                return;

        node = gtk_ctree_node_nth( GTK_CTREE( file_tree ), 
				   GPOINTER_TO_INT( key ) );
        if( ! node )
                return;

	tree_item_clicked( GTK_CTREE( file_tree ), node, 0 );
}

static void tree_delete_file()
{
	GtkWidget *file_tree;
	GtkWidget *box;
        const gchar *file = _( "Delete selected file" );
        const gchar *dir =  _( "Delete selected directory" );
        const gchar *message;

        gpointer key;
        GtkCTreeNode *node;
        gchar *path;
	gint button;
	gboolean status;
	Site *site;

	site = current_site;

	g_return_if_fail( site != NULL );

	file_tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

        key = gtk_object_get_data( GTK_OBJECT( file_tree ),
                                   "drag_start" );
        if( ! key )
                return;
        node = gtk_ctree_node_nth(GTK_CTREE( file_tree ), 
                                  GPOINTER_TO_INT( key ) );
        if( ! node )
                return;

        path = (gchar*)gtk_ctree_node_get_row_data( GTK_CTREE( file_tree ),
                                                    node );

        if( g_file_test( path,G_FILE_TEST_ISDIR ) )
                message = dir;
        else
                message = file;

        box = gnome_message_box_new( message,
                                     GNOME_MESSAGE_BOX_QUESTION,
                                     GNOME_STOCK_BUTTON_YES,
                                     GNOME_STOCK_BUTTON_NO, NULL );
        gnome_dialog_set_parent( GNOME_DIALOG( box ), GTK_WINDOW( app ) );
        gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );

	button = gnome_dialog_run_and_close( GNOME_DIALOG( box ) );

	if( button == 1 )
		return;

	/* delete confirmed */
	if( ! g_file_test( path, G_FILE_TEST_ISDIR ) )
                status = delete_file( path );
        else
                status = delete_dir( path );

	if( ! status ) {
		/* failed to delete */
		return;
	}

	/* perform cvs delete? */
	if( screem_site_get_cvs_root( site ) ) {
		box = gnome_message_box_new( _( "perform CVS delete?" ),
					     GNOME_MESSAGE_BOX_QUESTION,
					     GNOME_STOCK_BUTTON_YES,
					     GNOME_STOCK_BUTTON_NO, NULL );
		gnome_dialog_set_parent( GNOME_DIALOG( box ), 
					 GTK_WINDOW( app ) );
		gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );
		if( gnome_dialog_run_and_close( GNOME_DIALOG( box ) ) == 0 ) {
			/* yes */
			 cvs_delete_with_filename( path );
		}
	}

	screem_site_purge_pages_with_path( current_site, path );

	refresh_file_tree();
}

static void tree_rename_file()
{
	GtkWidget *file_tree;
	GtkWidget *dialog;
        GtkWidget *table;
        GtkWidget *label;
        GtkWidget *entry;

	GtkWidget *box;

        const gchar *file = _( "Old file" );
        const gchar *file2 = _( "New file" );
        const gchar *dir =  _( "Old directory" );
        const gchar *dir2 = _( "New directory" );
        const gchar *fileTitle = _( "Rename file" );
        const gchar *dirTitle = _( "Rename directory" );
        const gchar *title;
        const gchar *message;
        const gchar *message2;

	const gchar *site_path;
 
        gpointer key;
        GtkCTreeNode *node;
        gchar *path;
	gint button;
	Site *site;
	gchar *new_path;
	gchar *temp;
	gboolean status;
	Page *page;
	Page *cpage;

	gboolean is_dir;
	gint tab;
	GtkWidget *pagebook;
	GtkWidget *note;
	gchar *pname;

	site = current_site;

	g_return_if_fail( site != NULL );

	site_path = screem_site_get_pathname( site );

	file_tree = gtk_object_get_data( GTK_OBJECT( app ), "file_tree" );

        key = gtk_object_get_data( GTK_OBJECT( file_tree ),
                                   "drag_start" );
        if( ! key )
                return;
        node = gtk_ctree_node_nth( GTK_CTREE( file_tree ), 
				   GPOINTER_TO_INT( key ) );
        if( ! node )
                return;

        path = (gchar*)gtk_ctree_node_get_row_data( GTK_CTREE( file_tree ),
                                                    node );

        if( g_file_test( path, G_FILE_TEST_ISDIR ) ) {
                message = dir;
                message2 = dir2;
                title = dirTitle;
        } else {
                message = file;
                message2 = file2;
                title = fileTitle;
        }

        table = gtk_table_new( 2, 2, FALSE );

        label = gtk_label_new( message );
        gtk_table_attach( GTK_TABLE( table ), label, 0, 1, 0, 1,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );
        entry = gtk_entry_new();
        gtk_entry_set_editable( GTK_ENTRY( entry ), FALSE );
        /* we chop off the site_path though */
        gtk_entry_set_text( GTK_ENTRY( entry ), path + strlen( site_path ) );
        gtk_table_attach( GTK_TABLE( table ), entry, 1, 2, 0, 1,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

        label = gtk_label_new( message2 );
        gtk_table_attach( GTK_TABLE( table ), label, 0, 1, 1, 2,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );
        entry = gnome_file_entry_new( NULL, NULL );
        gtk_table_attach( GTK_TABLE( table ), entry, 1, 2, 1, 2,
                          GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

        dialog = gnome_dialog_new( title,
                                   GNOME_STOCK_BUTTON_OK,
                                   GNOME_STOCK_BUTTON_CANCEL,
                                   NULL );
        gtk_object_set_data( GTK_OBJECT( dialog ), "entry", entry );
        gtk_box_pack_start( GTK_BOX( GNOME_DIALOG( dialog )->vbox ),
                            table, TRUE, TRUE, GNOME_PAD );
        gnome_dialog_set_default( GNOME_DIALOG( dialog ), 0 );

	gtk_widget_show_all( table );
	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	/* rename ok'ed */
	entry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( entry ) );
	new_path = g_strdup_printf( "%s%s", site_path,
				    gtk_entry_get_text( GTK_ENTRY( entry ) ) );

	gtk_widget_destroy( dialog );

	if( button == 1 ) {
		g_free( new_path );
		return;
	}

	is_dir = g_file_test( path, G_FILE_TEST_ISDIR );

	/* if it is a dir then we must have a G_DIR_SEPARATOR on the end */
	if( is_dir && new_path[ strlen( new_path ) - 1 ] != G_DIR_SEPARATOR ) {
		temp = new_path;
		new_path = g_strconcat( temp, G_DIR_SEPARATOR_S, NULL );
		g_free( temp );
	}

	if( ! is_dir )
		status = copy_file( path, new_path );
	else
		status = copy_dir( path, new_path, FALSE );

	if( ! status ) {
		/* rename failed */
		g_free( new_path );
		return;
	}

	screem_site_rename_pages_with_path( site, path, new_path );

	/* update any links */
	cpage = screem_site_get_current_page( site );
	if( cpage )
		screem_editor_buffer_text();
       
	screem_site_file_change( site, path, new_path );
	
	if( ! is_dir )
		delete_file( path );
	else
		delete_dir( path );

	if( cpage ) {
		/* we had a page open in the editor, show
		   the possible changes to it */
		screem_editor_clear();
		screem_editor_insert( 0, cpage->data );
	}

	/* perform cvs delete + cvs add ? */
	if( screem_site_get_cvs_root( site ) ) {
		box = gnome_message_box_new( _( "Rename in CVS?" ),
					     GNOME_MESSAGE_BOX_QUESTION,
					     GNOME_STOCK_BUTTON_YES,
					     GNOME_STOCK_BUTTON_NO, NULL );
		gnome_dialog_set_parent( GNOME_DIALOG( box ), 
					 GTK_WINDOW( app ) );
		gnome_dialog_set_default( GNOME_DIALOG( box ), 0 );
		if( gnome_dialog_run_and_close( GNOME_DIALOG( box ) ) == 0 ) {
			/* yes */
			 cvs_delete_with_filename( path );
			 cvs_add_with_filename( new_path );
		}
	}
	g_free( new_path );

	/* add the new node */
	refresh_file_tree();
}

void screem_browse_list_show_dir( gchar *dir )
{
	GtkWidget *list;
	DIR *d;
	struct dirent *entry = NULL;
	gchar *name;
	Icons *icon;
	const gchar *mime_type;
	gchar *item[ 2 ] = { NULL, NULL };
	gint row;
	gint ld;
	gint data_row;
	gboolean is_dir;

	gchar *path;

	gchar cwd[ 16384 ];
	gchar cwd2[ 16384 ];

	g_return_if_fail( dir != NULL );

	list = gtk_object_get_data( GTK_OBJECT( app ), "browse" );

	load_folder_icons();

	getcwd( cwd, 16384 );

	d = opendir( dir );
	if( d ) {
		chdir( dir );
		getcwd( cwd2, 16384 );
		dir = cwd2;
		gtk_clist_freeze( GTK_CLIST( list ) );
		gtk_clist_clear( GTK_CLIST( list ) );

		for( ld = 0, row = 0; ( entry = readdir( d ) ); row ++ ) {
			name = entry->d_name;
			if( ! strcmp( ".", name ) ) {
				name = dir;
			} else if( strcmp( "..", name ) && name[ 0 ] == '.' ) {
				row --;
				continue;
			}
			item[ 1 ] = g_strdup( name );
			is_dir = g_file_test( name, G_FILE_TEST_ISDIR );
			if( ! is_dir ) {
				mime_type = gnome_mime_type( name );
				icon = icon_from_mime_type( mime_type );
				gtk_clist_append( GTK_CLIST( list ), item );
				gtk_clist_set_pixmap( GTK_CLIST( list ),
						      row, 0, icon->open,
						      icon->open_mask );
				data_row = row;
			} else {
				gtk_clist_insert( GTK_CLIST( list ), ld,item );
				gtk_clist_set_pixmap( GTK_CLIST( list ),
						      ld, 0, folder_close,
						      folder_close_mask );
				data_row = ld;
				ld ++;
			}
			if( strcmp( name, dir ) )
				path = g_strconcat(dir, G_DIR_SEPARATOR_S,
						   name, NULL);
			else
				path = g_strdup( dir );
			gtk_clist_set_row_data( GTK_CLIST( list ), data_row,
						path );
		}
		gtk_clist_columns_autosize( GTK_CLIST( list ) );
		gtk_clist_thaw( GTK_CLIST( list ) );
		closedir( d );
		gtk_object_set_data( GTK_OBJECT( list ), "current_dir", dir );
	}

	chdir( cwd );
}

void browse_list_clicked( GtkWidget *widget, GdkEventButton *event,
			  gpointer data )
{
	GtkCTreeNode *node = NULL;
        GtkWidget *menu;
        gint row = -1;
        gint col;

	gchar *path;

	static guint32 time = 0;

	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );

	if( row != - 1 ) {
		gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", GINT_TO_POINTER( row ) );
        } else {
                gtk_object_set_data( GTK_OBJECT( widget ),
                                     "drag_start", NULL);
	}

	/* what button was clicked? */
	if( event->button == 1 && row ) {
		if( time == event->time ) {
			path = gtk_clist_get_row_data( GTK_CLIST( widget ),
						       row );
			if( g_file_test( path, G_FILE_TEST_ISDIR ) )
				screem_browse_list_show_dir( path );
			else {

			}
		}
                time = event->time;
        } 
}


void resource_list_display()
{
	const gchar *resource_root = RESOURCE_PATH;
	xmlDocPtr doc;
	GtkWidget *tree;

	doc = build_directory_tree( NULL, resource_root, NULL );

	tree = gtk_object_get_data( GTK_OBJECT( app ), "resource_tree" );
	gtk_clist_clear( GTK_CLIST( tree ) );
	build_file_tree( tree, NULL, doc );
}
