/* this file is a part of Ami software, (C) Hwang chi-deok 1999 */

#include "config.h"

#include <gdk/gdkkeysyms.h>
#include <sys/stat.h>
#include <unistd.h>

#include "amitype.h"
#include "ic.h"
#include "edit.h"
#include "chartable.h"
#include "ami.h"

typedef struct {
    off_t filesize;
    int fd;
    char *buf;
    int ref;
} HanjaDic;

HanjaDic *hanjadic = NULL;

static void hanja_dialog_destroyed(GtkWidget *widget);
static void hanja_dict_load(void);
static void hanja_dialog_selected(GtkWidget *widget, char *selected);
static void hanja_dict_close(void);
static char * hanja_find_word(char *search);


guchar **
hanja_find_candidates(guchar *hanbuf, int current_pos, int *num)
{
    guchar *search;
    guchar *buf[1000];
    guchar **candidates;
    int i;
    *num = 0;
    if (current_pos <= 0) return NULL;
    search = g_new(guchar, current_pos + 2);
    strncpy(search, hanbuf, current_pos);
    search[current_pos] = ' ';
    search[current_pos+1] = '\0';
    for (i=0;i<current_pos;i += 2) {
	char *s = hanja_find_word(search + i);
	if (s) {
	    char *last = strchr(s, '\n');
	    char *s2;
	    int len;
	    if (last == NULL) len = strlen(s);
	    else len = last - s;
	    s2 = g_new0(char, len+1);
	    strncpy(s2, s, len);
	    s = strtok(s2, " ");
	    while (s) {
		s = strtok(NULL, " ");
		if (s) buf[(*num)++] = g_strdup(s);
	    }
	}
    }
    g_free(search);
    if (*num == 0) return NULL;
    candidates = g_new (guchar *, (*num)+1);
    for (i=0;i<(*num);i++) candidates[i] = buf[i];
    candidates[(*num)] = NULL;
    return candidates;
}

static int
key_press_cb(GtkWidget *w, GdkEventKey *ev)
{
    if (ev->keyval == GDK_Escape) {
	gtk_widget_destroy(w);
	return TRUE;
    }
    return FALSE;
}

static int
hanja_get_total_len(guchar **words, int max_col)
{
    int total_len = 0;
    int col = 0;
    while(*words) {
	int len;
	len = strlen(*words)/2;
	if (col + len > max_col) {
	    total_len += max_col - col;
	    col = len;
	} else {
	    col += len;
	}
	total_len += len;
	words++;
	if (col == max_col) col = 0;
    }
    return total_len;
}

static void
hanja_subst_mode_toggle_cb(GtkWidget *w, gpointer data)
{
    if (GTK_TOGGLE_BUTTON(w)->active) {
	CharTable *table;
	ami_hanja_subst_mode = GPOINTER_TO_INT(data);
	table = gtk_object_get_data((GtkObject *)gtk_widget_get_toplevel(w), "ctable");
	char_table_selected(table);
    }
}

GtkWidget *
hanja_dialog(HangulState *han)
{
    GtkWidget *win, *vbox, *ctable, *frame;
    GtkWidget *hbox, *b;
    guchar **candidates;
    guchar **table_buf;
    guchar **s;
    gchar **hanja_formats;
    int col, max_col, total_len, len, max_len;
    int i, k, num;
    GSList *group;

    if (!hanjadic) hanja_dict_load();
    if (!hanjadic) return NULL;
    candidates = hanja_find_candidates(han->buf, han->pos, &num);
    if (candidates == NULL) return NULL;

    max_len = strlen(candidates[0])/2;

    if (num < 20) max_col = MAX(max_len, 5);
    else max_col = MAX(max_len, 10);

    total_len = hanja_get_total_len(candidates, max_col);

    if (total_len < max_col) max_col = total_len;
    else if (total_len < max_len*max_len) {
	max_col = max_len;
	total_len = hanja_get_total_len(candidates, max_len);
    }

    table_buf = g_new(guchar *, total_len);
    col = 0;
    k = 0;
    s = candidates;
    while(*s) {
	len = strlen(*s)/2;
	if (col + len > max_col) {
	    while(col++<max_col) table_buf[k++] = NULL;
	    col = len;
	} else {
	    col += len;
	}
	table_buf[k++] = g_strdup(*s);
	if (len > 1) {
	    for(i=1;i<len;i++) table_buf[k++] = NULL;
	}
	s++;
	if (col == max_col) col = 0;
    }

    win = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_position(GTK_WINDOW(win), GTK_WIN_POS_CENTER);
    gtk_window_set_title(GTK_WINDOW(win), "ƹ:  Է");
    gtk_signal_connect((GtkObject *)win, "key_press_event", (GtkSignalFunc)key_press_cb, NULL);
    gtk_container_set_border_width(GTK_CONTAINER(win), 10);
    gtk_signal_connect(GTK_OBJECT(win), "destroy", 
    	(GtkSignalFunc)hanja_dialog_destroyed, NULL);

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_container_add(GTK_CONTAINER(win), vbox);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type((GtkFrame *)frame, GTK_SHADOW_ETCHED_OUT);
    gtk_container_add(GTK_CONTAINER(vbox), frame);

    ctable = char_table_new((char **)table_buf, max_col, total_len);
    gtk_object_set_data(GTK_OBJECT(win), "ctable", ctable);
    gtk_container_add((GtkContainer *)frame, ctable);
    gtk_signal_connect(GTK_OBJECT(ctable), "selected", 
    	(GtkSignalFunc)hanja_dialog_selected, NULL);

    for(i=0;i<num;i++) g_free(candidates[i]);
    g_free(candidates);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(vbox), hbox);

    hanja_formats = g_strsplit(ami_hanja_subst_format, ":", -1);
    group = NULL;
    for(i=0;hanja_formats[i] != NULL; i++) {
	b = gtk_radio_button_new_with_label(group, hanja_formats[i]);
	group = gtk_radio_button_group(GTK_RADIO_BUTTON(b));
	gtk_container_add(GTK_CONTAINER(hbox), b);
	if (ami_hanja_subst_mode == (i+1))
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b), TRUE);
	gtk_signal_connect(GTK_OBJECT(b), "toggled", 
		(GtkSignalFunc)hanja_subst_mode_toggle_cb, 
		GINT_TO_POINTER(i+1));
    }
    g_strfreev(hanja_formats);

    gtk_widget_show_all(win);
    gtk_widget_grab_focus(ctable);
    return win;
}

static void
hanja_dialog_selected(GtkWidget *widget, char *selected)
{
    GtkWidget *top;
    IC *ic;

    top = gtk_widget_get_toplevel(widget);
    ic = (IC *)gtk_object_get_data((GtkObject *)top, "ic");
    if (ic == NULL || !ic_is_valid(ic) || ic->han == NULL) {
	g_warning("%s: Invalid IC", __FUNCTION__);
    } else {
	if (ic->han == NULL) {
	    g_warning("%s: Hangul Buffer attached to special chars dialog is null",__FUNCTION__);
	} else {
	    if (ic->han == gtk_object_get_data((GtkObject *)top, "han")) 
		editing_hanja_replace(ic, selected);
	    else g_warning("%s: han is different from original.",__FUNCTION__);
	}
    }
    gtk_widget_destroy(top);
}

static void
hanja_dialog_destroyed(GtkWidget *widget)
{
    IC *ic;
    HangulState *han;
    ic = (IC *)gtk_object_get_data((GtkObject *)widget, "ic");
    if (ic == NULL) return;
    han = (HangulState *)gtk_object_get_data((GtkObject *)widget, "han");
    if (ic->han != han || han == NULL) return;
    han->aux_input = NULL;
    hanja_dict_close();
}


static void
hanja_dict_load(void)
{
    struct stat statbuf;
    off_t filesize;
    int fd;
    char *buf;

    if (hanjadic) {
	hanjadic->ref++;
	return;
    }

    fd = open("./hanja.dic", O_RDONLY);
    if (fd < 0) {
	fd = open(hanja_dic_filename, O_RDONLY);
    }
    if (fd < 0) {
	g_warning("fail to open hanja dictionary %s", hanja_dic_filename);
	g_print("hanja.dic Ͻõ ƴϸ hanja-hwp2ami ̿ؼ\n"
		"hwp ڻ ƹ̿ ȯϿ Ͻʽÿ\n");
	return ;
    }
    fstat(fd, &statbuf);
    filesize = statbuf.st_size;
    buf = g_new(guchar, filesize + 1);
    read(fd, buf, filesize);
    buf[filesize] = '\0';
    hanjadic = g_new(HanjaDic, 1);
    hanjadic->filesize = filesize;
    hanjadic->buf = buf;
    hanjadic->ref = 1;
    close(fd);
}

static void
hanja_dict_close(void)
{
    g_return_if_fail (hanjadic != NULL);
    hanjadic->ref--;
    if (hanjadic->ref != 0) return;
    g_free(hanjadic->buf);
    g_free(hanjadic);
    hanjadic = NULL;
}


/*  Լ  gtkutilִ gtkdic  ͵Դϴ.*/


static char *
go_next_line(char *s, char *last)
{
   while (s < last && *s != '\n')
      s++;
   if (s == last)
      return NULL;
   return s + 1;
}

/* sͽϴ  ó ̵Ѵ.
 * start   ó̹Ƿ lower bound ȴ.
 */
static char *
go_start_of_line(char *s, char *start)
{
   if (s == start)
      return start;
   while (s > start && *s != '\n')
      s--;
   if (*s == '\n')
      return s + 1;
   return start;
}

static char *
hanja_find_word(char *search)
{
    char *e, *b, *s;
    int len, comp;
    if ((unsigned char)search[0] < 0xb0) return NULL;
    if ((unsigned char)search[0] > 0xc8) return NULL;

    len = strlen(search);
    e = hanjadic->buf + hanjadic->filesize;
    b = hanjadic->buf;
    if (strncmp(b, search, len) == 0) return b;

   /* bi-section ġϴ  ִ  ãƳ. */
   while (1)
   {

      s = b + (e - b) / 2;
      if ((s = go_start_of_line(s, b)) == NULL)
         return NULL;
      if (s == b)
      {
	 s = go_next_line(s, e);
	 if (s == NULL || s == e)
            return NULL;
      }
      comp = strncmp(s, search, len);
      if (comp == 0)
	 break;
      else if (comp > 0)
	 e = s;
      else
	 b = s;
   }
   return s;
}