
/*
  
   1998 Roberto Alameda <roberto@myokay.net>
  You may modify and distribute this file under the terms of the GNU
  General Public License, version 2, or any later version, at your
  convenience. See the file COPYING for details. 
  

*/ 


#include <config.h>

#if HAVE_UNISTD_H
# include <unistd.h>
# include <sys/types.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <locale.h>
#include <errno.h>
#include <sys/stat.h>

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

extern "C" {
#include <gif_lib.h>
}

#include "gfont.h"
#include "error.xpm"
#include "font.xpm"
#include "openhand.xpm"

#include "ps.h"


extern GtkWidget *mainwindow, *fontclistw;
extern GtkStyle* stylebold;

extern GdkColor graypalette[];
extern GSList *fontlist;

static GtkWidget *errwindow;  // The window where error messages come
static GtkWidget *tree;       // The root tree in errwindow

static const char* abouttext = 
"Font Viewer for PostScript Type 1 and TrueType fonts\n"
"Version " VERSION "\n"
" Roberto Alameda"
;


// ************** Function prototypes (internally used in this module)
static bool delete_yes_no(GtkWidget *widget, GdkEvent *event, gpointer data);
static void set_variable(GtkWidget *button, gpointer data);

static Callback do_print, selectprintfile;
static void okselprintfile(gpointer diag);
static bool make_gif(GdkImage* image, char* filename);




// ********************* Begin of program functions

/* Note: Dealing with trees, we cannot call gtk_window_show_all as we
usually do, because that would also show the asts of the tree even in
collapsed state.  I think it should be fixed in GTK+, but nevertheless
we work around it by calling gtk_widget_show for each element.  */

// Adds an error entry for the given font
void add_error(FontData* fd, const char *msg, ...)
{
  va_list args;
  int size = 64;
  char* message = NULL;
  GtkTreeItem *font = NULL;

  // Construct a string (message) with the arguments  
  va_start(args, msg);
  while(1) {  
    if (message == NULL) message = (char*)malloc(size);
    else message = (char*)realloc(message, size);
    if (message == NULL) {
      fprintf(stderr, "Unable to reserve %d bytes of memory; exiting ...", 
	      size);
      exit(1);
    }
    int nchars = g_vsnprintf(message, size, msg, args);
    if (nchars < size) break;  // if enough space
    // if not enough space, enlarge size and try again
    size = nchars + 1; /* as in glibc >= 2.1 */
  }
  va_end(args);

  // Look for a tree item with the same font name
  for (GList *list=GTK_TREE(tree)->children; list; list=list->next) {
    gchar *name;
    GtkTreeItem *item = GTK_TREE_ITEM(list->data);
    GtkLabel *label = GTK_LABEL(GTK_BIN(item)->child);
    gtk_label_get(label, &name);
    if (strcmp(name, fd->fontFile) == 0) {
      font = item;
      break;
    }
  }
  // If no such font name, create a subtree with that name
  if (font == NULL) {
    font = GTK_TREE_ITEM(gtk_tree_item_new_with_label(fd->fontFile));
    gtk_tree_append(GTK_TREE(tree), GTK_WIDGET(font));
    gtk_widget_show(GTK_WIDGET(font));
    GtkWidget *subtree = gtk_tree_new();
    gtk_tree_item_set_subtree(font, subtree);
  }

  // Create a tree item in the subtree
  GtkWidget *messageitem = gtk_tree_item_new_with_label(message);
  gtk_tree_append(GTK_TREE(GTK_TREE_ITEM_SUBTREE(font)), messageitem);
  gtk_widget_show(messageitem);

  free(message);
  gtk_widget_show(errwindow);
  gdk_window_raise(errwindow->window);
}




void show_message_window(GtkWidget*, gpointer)
{
  gtk_widget_show(errwindow);
}



// Initialize the error messages window
void make_message_window(void)
{  
  errwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL); 
  gtk_widget_set_name(errwindow, "message window"); 
  gtk_window_set_title(GTK_WINDOW(errwindow), "Font Viewer messages");
  gtk_signal_connect_object(GTK_OBJECT(errwindow), "delete_event", 
			    GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			    GTK_OBJECT(errwindow));
  gtk_container_border_width(GTK_CONTAINER(errwindow), 5);

  GtkWidget *scrolled_win = gtk_scrolled_window_new(NULL, NULL);  
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win), 
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);  
  gtk_widget_set_usize(scrolled_win, 400, 300);  
  gtk_container_add(GTK_CONTAINER(errwindow), scrolled_win);
  gtk_widget_show(scrolled_win);

  tree = gtk_tree_new();
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_win), 
					tree);
  gtk_tree_set_selection_mode(GTK_TREE(tree), GTK_SELECTION_SINGLE);
  gtk_tree_set_view_lines(GTK_TREE(tree), FALSE);
  gtk_tree_set_view_mode(GTK_TREE(tree), GTK_TREE_VIEW_ITEM);
  gtk_widget_show(tree);
}



// Called in ask_yes_no if window closed; equivalent to saying No
static bool delete_yes_no(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  int *variable = (int*)data;
  *variable = 0;  // As if the user clicked 'No'
  return TRUE;    // Do not destroy
}


// Assign the value of the button to the variable; used in ask_yes_no
static void set_variable(GtkWidget *button, gpointer data)
{
  int *variable = (int*)data;
  int value = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "value"));
  *variable = value;
}



// Asks and waits till user responds; stays on top of parent
// Return 1 if yes, 0 if no
bool ask_yes_no(GtkWidget *parent, const char* msg, ...)
{
  int rc = -1;
  va_list args;
  int size = 64;
  char* message = NULL;

  // Construct a string (message) with the arguments
  va_start(args, msg);
  while(1) {  
    if (message == NULL) message = (char*)malloc(size);
    else message = (char*)realloc(message, size);
    if (message == NULL) {
      fprintf(stderr, "Unable to reserve %d bytes of memory; exiting ...", 
	      size);
      exit(1);
    }
    int nchars = g_vsnprintf(message, size, msg, args);
    if (nchars < size) break;  // if enough space
    // if not enough space, enlarge size and try again
    size = nchars + 1; /* as in glibc >= 2.1 */
  }
  va_end(args);

  GtkWidget* qw = gtk_dialog_new();
  gtk_widget_set_name(qw, "question window");
  gtk_window_set_title(GTK_WINDOW(qw), "Font Viewer question");
  // Delete event (closing the window) is equivalent to 'No'
  gtk_signal_connect(GTK_OBJECT(qw), "delete_event", 
		     GTK_SIGNAL_FUNC(delete_yes_no), (gpointer)&rc);
  gtk_widget_realize(qw);

  // Buttons in action_area
  GtkWidget* okb = gtk_button_new(); 
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(qw)->action_area), okb, FALSE, FALSE, 0);
  gtk_object_set_data(GTK_OBJECT(okb), "value", GINT_TO_POINTER(1));  // Return 1 if pressed
  gtk_signal_connect(GTK_OBJECT(okb), "clicked", 
		     GTK_SIGNAL_FUNC(set_variable), (gpointer)&rc);
  GtkWidget* label = gtk_label_new("Yes");
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(okb), label);

  GtkWidget* cancelb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(qw)->action_area), cancelb, 
		     FALSE, FALSE, 0);
  gtk_object_set_data(GTK_OBJECT(cancelb), "value", GINT_TO_POINTER(0));  // Return 0 if pressed
  gtk_signal_connect(GTK_OBJECT(cancelb), "clicked", 
		     GTK_SIGNAL_FUNC(set_variable), (gpointer)&rc);
  label = gtk_label_new("No");
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(cancelb), label);

  // Message area (vbox)
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxm), 20);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(qw)->vbox), hboxm, TRUE, FALSE, 0);

  GtkWidget* qicon = makeicon(qw, HandOpen_xpm);
  gtk_box_pack_start(GTK_BOX(hboxm), qicon, FALSE, FALSE, 0);
  label = gtk_label_new(message);
  //gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hboxm), label, FALSE, FALSE, 10);

  gtk_window_set_transient_for(GTK_WINDOW(qw), GTK_WINDOW(parent));
  gtk_window_position(GTK_WINDOW(qw), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(qw), TRUE);
  gtk_widget_show_all(qw);

  gdk_beep();
  free(message);

  // Loop until user presses Yes or No
  while (rc == -1) gtk_main_iteration();
  // Buf, user pressed something at last; return its value
  gtk_widget_destroy(qw);
  return rc;
}




// Create an error message window with a OK button
void errormsg(const char* msg, ...)
{
  va_list args;
  int size = 64;
  char* message = NULL;

  // Construct a string (message) with the arguments  
  va_start(args, msg);
  while(1) {  
    if (message == NULL) message = (char*)malloc(size);
    else message = (char*)realloc(message, size);
    if (message == NULL) {
      fprintf(stderr, "Unable to reserve %d bytes of memory; exiting ...", 
	      size);
      exit(1);
    }
    int nchars = vsnprintf(message, size, msg, args);
    if (nchars < size) break;  // if enough space
    // if not enough space, enlarge size and try again
    size = nchars + 1; /* as in glibc >= 2.1 */
  }
  va_end(args);

  // Now give 'message' as error message
  GtkWidget* errw = gtk_dialog_new();
  gtk_widget_set_name(errw, "error window");
  gtk_window_set_title(GTK_WINDOW(errw), "Font Viewer error");
  gtk_widget_realize(errw);

  // Buttons in action_area
  GtkWidget* okb = gtk_button_new();  
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(errw)->action_area), okb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(okb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(errw));
  GtkWidget* label = gtk_label_new("OK");
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(okb), label);

  // Message area (vbox)
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxm), 20);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(errw)->vbox), hboxm, TRUE, FALSE, 0);

  GtkWidget* erricon = makeicon(errw, errorxpm);
  gtk_box_pack_start(GTK_BOX(hboxm), erricon, FALSE, FALSE, 0);
  label = gtk_label_new(message);
  //gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  //gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(hboxm), label, FALSE, FALSE, 10);

  gtk_window_position(GTK_WINDOW(errw), GTK_WIN_POS_CENTER);
  gtk_window_set_transient_for(GTK_WINDOW(errw), GTK_WINDOW(mainwindow));
  gdk_window_set_group(errw->window, mainwindow->window);
  gtk_window_set_modal(GTK_WINDOW(errw), TRUE);
  gtk_widget_show_all(errw);

  GTK_WIDGET_SET_FLAGS(okb, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(okb);
  gdk_beep();
  free(message);
}



// Make a GtkPixmap out of the given xpm data; uses style of toplevel
// for transparent color of xpm
GtkWidget* makeicon(GtkWidget* toplevel, char** xpm)
{
  GdkBitmap* mask;

  GtkStyle* style = gtk_widget_get_style(toplevel);
  GdkPixmap* pixmap = gdk_pixmap_create_from_xpm_d(toplevel->window, &mask,
                                      &style->bg[GTK_STATE_NORMAL], xpm);
  GtkWidget* pixmapwid = gtk_pixmap_new(pixmap, mask);
  return pixmapwid;
}



// The name says it all
void makeabout(GtkWidget*, gpointer)
{
  GtkWidget* aboutw = gtk_dialog_new();
  gtk_widget_set_name(aboutw, "about window");
  gtk_window_set_title(GTK_WINDOW(aboutw), "About Font Viewer");
  gtk_widget_realize(aboutw);
  
  GtkWidget* okb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(aboutw)->action_area), okb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(okb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(aboutw));
  GtkWidget* hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(okb), hbox);
  GtkWidget* label = gtk_label_new("OK");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 25);

  // Message area
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxm), 20);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(aboutw)->vbox), hboxm, TRUE, FALSE, 0);

  GtkWidget* icon = makeicon(aboutw, font_xpm);
  gtk_box_pack_start(GTK_BOX(hboxm), icon, FALSE, FALSE, 0);
  label = gtk_label_new(abouttext);
  gtk_box_pack_start(GTK_BOX(hboxm), label, FALSE, FALSE, 10);

  gtk_window_set_position(GTK_WINDOW(aboutw), GTK_WIN_POS_MOUSE);
  gtk_window_set_modal(GTK_WINDOW(aboutw), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(aboutw), GTK_WINDOW(mainwindow));
  GTK_WIDGET_SET_FLAGS(okb, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(okb);
  gtk_widget_show_all(aboutw);
}



// **** Printer dialog

static GtkWidget *prtw;
static GtkWidget *printerentry, *fileentry, *browsebutton;
static GtkWidget *printerbutton, *filebutton;
static GtkWidget *fontbutton, *charsetbutton, *textbutton;
static GtkWidget *sizeentry, *spacingentry;



// Actually print the font sample according to the settings in the options
// window
static void do_print(GtkWidget*, gpointer ptr)
{
  int rc;
  static GString *command = NULL;
  const char *psfile;
  int printtoprinter = GTK_TOGGLE_BUTTON(printerbutton)->active;
  FontData *fd = (FontData*)ptr;

  if (command==NULL) command = g_string_new("");
  const char *printername = gtk_entry_get_text(GTK_ENTRY(printerentry));
  if (*printername == '\0') printername = "lp";
  const char *filename = gtk_entry_get_text(GTK_ENTRY(fileentry));
  double fontsize = atof(gtk_entry_get_text(GTK_ENTRY(sizeentry)));
  double spacing = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(spacingentry));

  if (GTK_TOGGLE_BUTTON(textbutton)->active && 
      (fontsize<1.0 || fontsize>50.0)) {
    errormsg("Invalid size");
    return;
  }
  if (printtoprinter) {     // if printing to printer
    psfile = tmpnam(NULL);  // generate temporary file to print it later
    if (psfile == NULL) {
      errormsg("Cannot generate temporary file: %s", g_strerror(errno));
      gtk_widget_hide(prtw);
      return;
    }
  }
  else {  // if printing to file
    struct stat buf;
    if (stat(filename, &buf)==0) {
      if (S_ISDIR(buf.st_mode)) {
	errormsg("Cannot print to a directory");
	return;
      }
      int rc = ask_yes_no(prtw, 
			  "File %s already exists\nDo you want to overwrite it?", 
			  filename);
      if (rc == 0) return;  // if no
    }
    psfile = filename;  // if yes
  }

  FILE *ofp = fopen(psfile, "w");
  if (ofp == NULL) {
    errormsg("Cannot open file %s: %s", psfile, g_strerror(errno));
    return;
  }
  fputs("%!PS\n\n", ofp);

  switch (fd->fontType) {
  case T1FONT:
    rc = t1_downloadfont(ofp, fd);  // Write the font to the file (in ASCII)
    if (rc == FALSE) {  // Error message already shown
      fclose(ofp);
      unlink(psfile);
      gtk_widget_hide(prtw);
      return;
    }
    // PostScript strings and commands defined in ps.h
    if (GTK_TOGGLE_BUTTON(fontbutton)->active) { // Size sample
      fprintf(ofp, size_samples, fd->fontName);  
    }
    else if (GTK_TOGGLE_BUTTON(textbutton)->active) {
      fputs("\n\n\n/textstring (\\", ofp);
      fputs(sample_text, ofp);
      fputs(") def", ofp);
      fprintf(ofp, textsample_code, fd->fontName, fontsize, spacing);
    }
    else if (GTK_TOGGLE_BUTTON(charsetbutton)->active) {
      fputs(fontchars_code, ofp);
      fprintf(ofp, "(%s) DoFont\n", fd->fontName);
    }
    break;

  case TTFONT:
    fputs(check_type42, ofp);  // check_type42 is defined in ps.h
    rc = tt_downloadfont(ofp, fd);  // Download TT font as Type 42 font
    if (rc == FALSE) {
      fclose(ofp);
      unlink(psfile);
      gtk_widget_hide(prtw);
      return;
    }
    if (GTK_TOGGLE_BUTTON(fontbutton)->active) // Size sample
      fprintf(ofp, size_samples, getfontname(fd, TT_NAME_ID_PS_NAME));  
    else if (GTK_TOGGLE_BUTTON(textbutton)->active) {
      fputs("\n\n\n/textstring (\\", ofp);
      fputs(sample_text, ofp);
      fputs(") def", ofp);
      fprintf(ofp, textsample_code, getfontname(fd, TT_NAME_ID_PS_NAME), 
	      fontsize, spacing);
    }
    else if (GTK_TOGGLE_BUTTON(charsetbutton)->active) {
      fputs(fontchars_code, ofp);
      fprintf(ofp, "(%s) DoFont\n", getfontname(fd, TT_NAME_ID_PS_NAME));
    }
    break;
  }

  rc = fclose(ofp);
  if (rc != 0) {  // if error writing to file
    errormsg("Error writing to file %s: %s", psfile, g_strerror(errno));
    return;
  }

  if (!printtoprinter) {  // If printing to file, we are done
    gtk_widget_hide(prtw);
    return;
  }

  // Print to printer here
  char buf[256];
  g_string_sprintf(command, "%s%s %s 2>&1", PRINTER_SPOOLER, printername, psfile);
  FILE *stream = popen(command->str, "r");
  if (stream == NULL) {
    errormsg("Cannot start printer spooler: %s", g_strerror(errno));
    unlink(psfile);
    return;
  }
  char *err = fgets(buf, sizeof(buf), stream); // Not NULL if lpr generates output
  rc = pclose(stream);
  unlink(psfile);
  if (rc == -1) {
    errormsg("System error in printer spooler command: %s", g_strerror(errno));
    return;
  }
  if (!WIFEXITED(rc)) {  // Abnormal termination of spooler
    errormsg("Printer spooler terminated abnormally");
    return;
  }
  if (WEXITSTATUS(rc) != 0) {
    errormsg("Error in printer spooler\n%s", err?err:"");
    return;
  }
  gtk_widget_hide(prtw);  // Hide only if successful
}



// The user selected a file to print to; put it in the corresponding
// entry field and destroy the file select window
static void okselprintfile(gpointer diag)
{
  GtkWidget *filesel = GTK_WIDGET(diag);
  const char *newfile = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  gtk_entry_set_text(GTK_ENTRY(fileentry), newfile);
  gtk_widget_destroy(filesel);
}


// Called when user wants to select the file to print to
static void selectprintfile(GtkWidget*, gpointer)
{
  set_window_busy(mainwindow, TRUE);
  set_window_busy(prtw, TRUE);

  GtkWidget* filesel = gtk_file_selection_new("Font Viewer file selection");
  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    GTK_OBJECT(filesel));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(okselprintfile), 
			    GTK_OBJECT(filesel));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), 
				  gtk_entry_get_text(GTK_ENTRY(fileentry)));
  gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
  gtk_widget_show(filesel);
  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(mainwindow));
  set_window_busy(mainwindow, FALSE);
  set_window_busy(prtw, FALSE);
}




/* 
   Callback called when user wants to print the selected font in the font list
   It shows a print options dialog. The dialog is not destroyed when finished,
   just hidden. This is done to maintain the settings from the user 
   for the next font to print.
*/
void printfont(GtkWidget*, gpointer)
{
  static GtkWidget *okb;
  int rc;

  GList *selection = GTK_CLIST(fontclistw)->selection;
  if (selection == NULL) {
    errormsg("No row selected");
    return;
  }
  int row = GPOINTER_TO_INT(selection->data);
  FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), row);
  
  // If print dialog already exist
  if (prtw != NULL) {
    // Give actual parameters to the callback do_print
    gtk_signal_handlers_destroy(GTK_OBJECT(okb));
    gtk_signal_connect(GTK_OBJECT(okb), "clicked", 
		       GTK_SIGNAL_FUNC(do_print), (gpointer)fd);
    gtk_widget_show(prtw);
    return;
  }

  // Otherwise make print dialog
  prtw = gtk_dialog_new();
  gtk_widget_set_name(prtw, "print window");
  gtk_window_set_title(GTK_WINDOW(prtw), "Font Viewer print dialog");
  gtk_signal_connect_object(GTK_OBJECT(prtw), "delete_event", 
			    GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			    GTK_OBJECT(prtw));
  
  // Button area (action_area)
  okb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(prtw)->action_area), okb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(okb), "clicked", 
		     GTK_SIGNAL_FUNC(do_print), (gpointer)fd);
  GtkWidget* label = gtk_label_new("OK");
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(okb), label);
  
  GtkWidget* cancelb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(prtw)->action_area), cancelb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(cancelb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_hide), 
			    GTK_OBJECT(prtw));
  label = gtk_label_new("Cancel");
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(cancelb), label);
  
  // **** Dialog area (vbox)
  
  // 'Print to' area
  GtkWidget *printframe = gtk_frame_new("Print to:");
  gtk_container_border_width(GTK_CONTAINER(printframe), 5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(prtw)->vbox), printframe, 
		     TRUE, FALSE, 0);
  
  GtkWidget *table = gtk_table_new(2, 4, FALSE); // A table with 2 rows, 4 columns
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(printframe), table);
  
  printerbutton = gtk_radio_button_new_with_label(NULL, "Printer:");
  gtk_table_attach(GTK_TABLE(table), printerbutton, 0, 1, 0, 1, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  filebutton = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(printerbutton)), "File:");
  gtk_table_attach(GTK_TABLE(table), filebutton, 0, 1, 1, 2, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(printerbutton), TRUE);  // Set default pressed
  
  printerentry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(printerentry), "lp");
  gtk_table_attach(GTK_TABLE(table), printerentry, 1, 2, 0, 1, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  
  fileentry = gtk_entry_new();
  GString *file = g_string_new("");
  g_string_sprintf(file, "%s/sample.ps", getenv("HOME"));
  gtk_entry_set_text(GTK_ENTRY(fileentry), file->str);
  g_string_free(file, TRUE);
  gtk_table_attach(GTK_TABLE(table), fileentry, 1, 3, 1, 2, 
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), GTK_EXPAND, 0, 0);
  
  browsebutton = gtk_button_new();
  label = gtk_label_new("Browse...");
  gtk_misc_set_padding(GTK_MISC(label), 3, 3);
  gtk_container_add(GTK_CONTAINER(browsebutton), label);
  gtk_signal_connect(GTK_OBJECT(browsebutton), "clicked", 
		     GTK_SIGNAL_FUNC(selectprintfile), NULL);
  gtk_table_attach(GTK_TABLE(table), browsebutton, 3, 4, 1, 2, 
		   GTK_FILL, GTK_EXPAND, 10, 10);
  
  
  // 'Select sample' area
  GtkWidget *sampleframe = gtk_frame_new("Select sample:");
  gtk_container_border_width(GTK_CONTAINER(sampleframe), 5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(prtw)->vbox), sampleframe, TRUE, FALSE, 0);
  
  table = gtk_table_new(3, 3, FALSE); // A table with 3 rows, 3 columns
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(sampleframe), table);
  
  fontbutton = gtk_radio_button_new_with_label(NULL, "Size samples");
  gtk_table_attach(GTK_TABLE(table), fontbutton, 0, 1, 0, 1, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  charsetbutton = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(fontbutton)), "Font map");
  gtk_table_attach(GTK_TABLE(table), charsetbutton, 0, 1, 1, 2, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  textbutton = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(fontbutton)), "Text sample");
  gtk_table_attach(GTK_TABLE(table), textbutton, 0, 1, 2, 3, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(fontbutton), TRUE);  // Set default pressed
  
  GtkWidget* hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(hbox), 0);
  gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, 2, 3, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  sizeentry = gtk_entry_new();
  gtk_widget_set_usize(sizeentry, 40, -1);
  gtk_entry_set_text(GTK_ENTRY(sizeentry), "12");
  gtk_box_pack_start(GTK_BOX(hbox), sizeentry, FALSE, FALSE, 0);
  label = gtk_label_new("points");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(hbox), 0);
  gtk_table_attach(GTK_TABLE(table), hbox, 2, 3, 2, 3, 
		   GTK_FILL, GTK_EXPAND, 10, 10);
  GtkAdjustment *adj = (GtkAdjustment*)gtk_adjustment_new(1.0, 1.0, 2.0, 
							  0.1, 0.2, 0.0);
  spacingentry = gtk_spin_button_new(adj, 0, 1);
  gtk_box_pack_start(GTK_BOX(hbox), spacingentry, FALSE, FALSE, 0);
  label = gtk_label_new("spacing");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  
  // End
  gtk_window_position(GTK_WINDOW(prtw), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(prtw), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(prtw), GTK_WINDOW(mainwindow));
  gtk_widget_show_all(prtw);
  gdk_window_set_group(prtw->window, mainwindow->window);
}



// *** Save names


static void do_savenames(GtkWidget *button, GtkWidget *)
{
  struct stat buf;
  int rc;

  GtkWidget *filesel = gtk_widget_get_toplevel(button);
  char *file = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  if (strchr(file, '*')) return;

  if (stat(file, &buf)==0) {
    if (S_ISDIR(buf.st_mode)) {
      errormsg("Selected file is a directory");
      return;
    }
    rc = ask_yes_no(filesel, 
		    "File %s already exists\nDo you want to overwrite it?", 
		    g_basename(file));
    if (rc == 0) return; // if no
  }
  FILE *fp = fopen(file, "w");
  if (fp == NULL) {
    errormsg("Cannot open file %s: %s", file, g_strerror(errno));
    return;
  }
  for (GSList* p=fontlist; p; p=p->next) {
    FontData* fd = (FontData *)p->data;
    fprintf(fp, "%s\t\t%s\n", g_basename(fd->fontFile), fd->fontName);
  }
  rc = fclose(fp);
  if (rc != 0) {
    errormsg("Error writing file %s: %s", file, g_strerror(errno));
    return;
  }
  gtk_widget_hide(filesel);
}


// Save a list of all font files in the directory to a text file
void save_names(GtkWidget*, gpointer)
{
  static GtkWidget *filesel = NULL;
  
  if (filesel == NULL) {
    filesel = gtk_file_selection_new("Save font names");
    gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
    gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
    //gtk_file_selection_complete(GTK_FILE_SELECTION(filesel), "*.txt");
    GtkWidget *cancelb = GTK_FILE_SELECTION(filesel)->cancel_button;
    GtkWidget *okb = GTK_FILE_SELECTION(filesel)->ok_button;
    
    gtk_signal_connect_object(GTK_OBJECT(cancelb), "clicked", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
    gtk_signal_connect(GTK_OBJECT(okb), "clicked",
		       GTK_SIGNAL_FUNC(do_savenames), NULL);
    gtk_signal_connect_object(GTK_OBJECT(filesel), "delete_event", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
  }

  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(mainwindow));
  gtk_widget_show(filesel);
}



// **** Show properties

// Show the properties of the selected font
void show_properties(GtkWidget*, gpointer)
{
  GList *selection = GTK_CLIST(fontclistw)->selection;
  if (selection == NULL) {
    errormsg("No row selected");
    return;
  }
  int row = GPOINTER_TO_INT(selection->data);
  FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), row);

  GtkWidget* window = gtk_dialog_new();
  gtk_widget_set_name(window, "properties window");
  gtk_window_set_title(GTK_WINDOW(window), "Font Properties");
  gtk_signal_connect(GTK_OBJECT(window), "delete_event", 
		     GTK_SIGNAL_FUNC(gtk_false), NULL);
  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_NONE);
  gtk_container_border_width(GTK_CONTAINER(frame), 10);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), frame, TRUE, FALSE, 0);

  GtkWidget* okb = gtk_button_new();  // Button in action_area
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), okb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(okb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    GTK_OBJECT(window));
  GTK_WIDGET_SET_FLAGS(okb, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(okb);
  GtkWidget* label = gtk_label_new("OK");
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(okb), label);

  switch (fd->fontType) {
  case T1FONT:
    t1_showproperties(fd, frame);
    break;
  case TTFONT: 
    tt_showproperties(fd, frame);
    break;
  }
  gtk_widget_show_all(window);
}



/* See function below; keeps track of resize events of window to cover */
static gint busy_configure_event(GtkWidget *widget, 
				 GdkEventConfigure *event, gpointer data)
{
  GdkWindow *busy = (GdkWindow*)data;
  gdk_window_resize(busy, event->width, event->height);
  return FALSE;
}



/*
  Avoid any interaction with the window: the window will not respond to
  any events, and it will have a wait cursor over it if state TRUE
  The function cen be called nested: if called 2 times with TRUE, 
  it has to be called 2 times with FALSE
  It works by superposing an input window of the same size of the window
  to cover (resizes tracked by the configure events) which discards
  any mouse or key events.
 */
void set_window_busy(GtkWidget *window, bool state)
{
  GdkWindowAttr attr; 
  GdkWindow *busy;
  int busy_count;

  g_return_if_fail(window->window != NULL);

  busy = (GdkWindow*)gtk_object_get_data(GTK_OBJECT(window), "busy");
  if (busy == NULL) {
    attr.window_type = GDK_WINDOW_CHILD;
    //attr.wclass = GDK_INPUT_OUTPUT;  // For testing
    attr.wclass = GDK_INPUT_ONLY;
    attr.cursor = gdk_cursor_new(GDK_WATCH);
    attr.width = 0;
    attr.height = 0;
    attr.event_mask = GDK_KEY_PRESS_MASK;
    busy = gdk_window_new(window->window, &attr, GDK_WA_CURSOR);
    gtk_object_set_data(GTK_OBJECT(window), "busy", (gpointer)busy);
    gtk_object_set_data(GTK_OBJECT(window), "busycount", GINT_TO_POINTER(0));
    gtk_signal_connect(GTK_OBJECT(window), "configure_event", 
		       (GtkSignalFunc)busy_configure_event, (gpointer)busy);
    gtk_signal_handler_block_by_func(GTK_OBJECT(window), 
				     (GtkSignalFunc)busy_configure_event, 
				     (gpointer)busy);
    gdk_cursor_destroy(attr.cursor);
  }

  busy_count = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window), 
						   "busycount"));
  if (state) busy_count++;
  else busy_count--;

  if (state && busy_count==1) { // Transition 0->1
    GdkWindowPrivate *priv = (GdkWindowPrivate*)window->window; 
    gdk_window_resize(busy, priv->width, priv->height);
    gdk_window_show(busy);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(window), 
				       (GtkSignalFunc)busy_configure_event, 
				       (gpointer)busy);
  }
  if (busy_count==0) {  // Transition 1->0
    gdk_window_hide(busy);
    gtk_signal_handler_block_by_func(GTK_OBJECT(window), 
				     (GtkSignalFunc)busy_configure_event, 
				     (gpointer)busy);
  }
  if (busy_count < 0) busy_count = 0;
  gtk_object_set_data(GTK_OBJECT(window), "busycount", 
		      GINT_TO_POINTER(busy_count));
  while (gtk_events_pending()) gtk_main_iteration();
}




/*    
      Handling of the 'Save as GIF' option for images
      The window containing the image must have 2 data associated:
      the delete handler (delete_handler) and the GdkImage (image)

*/

// Makes the popup menu for the given window (containing an image, character or string)
GtkWidget* make_imagepopupmenu(GtkWidget *window)
{
  GtkWidget *menu = gtk_menu_new();
  GtkWidget *item = gtk_menu_item_new_with_label("Save as GIF");
  gtk_menu_append(GTK_MENU(menu), item);
  gtk_signal_connect_object(GTK_OBJECT(item), "activate",
                GTK_SIGNAL_FUNC(save_asgif), GTK_OBJECT(window));
  gtk_widget_show(item);
  return menu;
}



static void cancel_saveas(GtkWidget *button, GtkWidget *imagewindow)
{
  GtkWidget *filesel = gtk_widget_get_toplevel(button);
  gtk_widget_hide(filesel);
  GTK_WINDOW(filesel)->transient_parent = NULL;
  int hid = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(imagewindow), 
						"delete_handler"));
  gtk_signal_disconnect(GTK_OBJECT(imagewindow), hid);
  hid = gtk_signal_connect_object(GTK_OBJECT(imagewindow), "delete_event",
				  GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_object_set_data(GTK_OBJECT(imagewindow), "delete_handler", 
		      GINT_TO_POINTER(hid));
}



static void do_savegif(GtkWidget *button, GtkWidget *imagewindow)
{
  struct stat buf;

  GtkWidget *filesel = gtk_widget_get_toplevel(button);
  char *file = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  if (strchr(file, '*')) return;

  int len = strlen(file);
  if (len<4 || strcmp(file+len-4, ".gif")!=0) {
    int rc = ask_yes_no(filesel, "File extension is not gif\nProceed anyway?");
    if (rc == 0) return;  // if no
  }

  if (stat(file, &buf)==0) {
    if (S_ISDIR(buf.st_mode)) {
      errormsg("Selected file is a directory");
      return;
    }
    int rc = ask_yes_no(filesel, 
			"File %s already exists\nDo you want to overwrite it?", 
			g_basename(file));
    if (rc == 0) return; // if no
  }
  GdkImage *image = (GdkImage*)gtk_object_get_data(GTK_OBJECT(imagewindow), 
						   "image");
  bool ret = make_gif(image, file);
  if (ret == FALSE) return;  // sth went wrong
  cancel_saveas(button, imagewindow);
}




// The user selected to save the image in the given window as a GIF file
// Called from popup menu
void save_asgif(GtkWidget *imagewindow)
{
  static GtkWidget *filesel = NULL;

  if (filesel == NULL) {
    filesel = gtk_file_selection_new("Save GIF file");
    gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
    gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
    gtk_file_selection_complete(GTK_FILE_SELECTION(filesel), "*.gif");
    // Ignore delete events
    gtk_signal_connect_object(GTK_OBJECT(filesel), "delete_event", 
			      GTK_SIGNAL_FUNC(gtk_true), NULL);
  }
  GtkWidget *cancelb = GTK_FILE_SELECTION(filesel)->cancel_button;
  GtkWidget *okb = GTK_FILE_SELECTION(filesel)->ok_button;
  gtk_signal_handlers_destroy(GTK_OBJECT(cancelb));
  gtk_signal_handlers_destroy(GTK_OBJECT(okb));

  gtk_signal_connect(GTK_OBJECT(cancelb), "clicked", 
		     GTK_SIGNAL_FUNC(cancel_saveas), GTK_OBJECT(imagewindow));
  gtk_signal_connect(GTK_OBJECT(okb), "clicked",
		     GTK_SIGNAL_FUNC(do_savegif), GTK_OBJECT(imagewindow));

  int hid = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(imagewindow), 
						"delete_handler"));
  gtk_signal_disconnect(GTK_OBJECT(imagewindow), hid);
  hid = gtk_signal_connect_object(GTK_OBJECT(imagewindow), "delete_event",
				  GTK_SIGNAL_FUNC(gtk_true), NULL);
  gtk_object_set_data(GTK_OBJECT(imagewindow), "delete_handler", 
		      GINT_TO_POINTER(hid));

  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(imagewindow));
  gtk_widget_show(filesel);

  /* Do not return until the file save dialog window is closed,
   because after returning the image can be destroyed */
  while (GTK_WIDGET_VISIBLE(filesel)) gtk_main_iteration(); 
}




// Called when user clicked on an image window (character or string window)
// data is the menu to post, widget the window clicked on
gint image_clicked(GtkWidget* widget, GdkEventButton *event, gpointer data)
{
  GtkWidget *menu;

  switch (event->type) {
  case GDK_BUTTON_PRESS:
    if (event->button != 3) return FALSE;
    menu = (GtkWidget*)data;
    gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		   event->button, event->time);
    return TRUE;   // Handled
    break;

  default: break;
  }
  return FALSE;
}




// ************** Handling of GIF files


// Make an error message
static const char* getGIFError(int error)
{
  static GString *msg = NULL;
  if (msg == NULL) msg = g_string_new(NULL);

  switch(error) {
  case D_GIF_ERR_OPEN_FAILED:
  case E_GIF_ERR_OPEN_FAILED:
    g_string_sprintf(msg, "Failed to open given file (%s)", 
		     g_strerror(errno));
    break;
  case E_GIF_ERR_WRITE_FAILED:
    g_string_sprintf(msg, "Failed to Write to given file (%s)", 
		     g_strerror(errno));
    break;
  case E_GIF_ERR_HAS_SCRN_DSCR:
    g_string_assign(msg, "Screen Descriptor already been set");
    break;
  case E_GIF_ERR_HAS_IMAG_DSCR:
    g_string_assign(msg, "Image Descriptor is still active");
    break;
  case E_GIF_ERR_NO_COLOR_MAP:
    g_string_assign(msg, "Neither Global Nor Local color map");
    break;
  case E_GIF_ERR_DATA_TOO_BIG:
    g_string_assign(msg, "#Pixels bigger than Width * Height");
    break;
  case E_GIF_ERR_NOT_ENOUGH_MEM:
    g_string_assign(msg, "Fail to allocate required memory");
    break;
  case E_GIF_ERR_DISK_IS_FULL:
    g_string_sprintf(msg, "Write failed (%s)", g_strerror(errno));
    break;
  case E_GIF_ERR_CLOSE_FAILED:
    g_string_sprintf(msg, "Failed to close given file (%s)", 
		     g_strerror(errno));
    break;
  case E_GIF_ERR_NOT_WRITEABLE:
    g_string_assign(msg, "Given file was not opened for write");
    break;

  case D_GIF_ERR_READ_FAILED:
    g_string_sprintf(msg, "Failed to Read from given file (%s)", 
		     g_strerror(errno));
    break;
  case D_GIF_ERR_NOT_GIF_FILE:
    g_string_assign(msg, "Given file is not a GIF file");
    break;
  case D_GIF_ERR_NO_SCRN_DSCR:
    g_string_assign(msg, "No Screen Descriptor detected");
    break;
  case D_GIF_ERR_NO_IMAG_DSCR:
    g_string_assign(msg, "No Image Descriptor detected");
    break;
  case D_GIF_ERR_NO_COLOR_MAP:
    g_string_assign(msg, "Neither Global Nor Local color map");
    break;
  case D_GIF_ERR_WRONG_RECORD:
    g_string_assign(msg, "Wrong record type detected");
    break;
  case D_GIF_ERR_DATA_TOO_BIG:
    g_string_assign(msg, "#Pixels bigger than Width * Height");
    break;
  case D_GIF_ERR_NOT_ENOUGH_MEM:
    g_string_assign(msg, "Fail to allocate required memory");
    break;
  case D_GIF_ERR_CLOSE_FAILED:
    g_string_sprintf(msg, "Failed to close given file (%s)", g_strerror(errno));
    break;
  case D_GIF_ERR_NOT_READABLE:
    g_string_assign(msg, "Given file was not opened for read");
    break;
  case D_GIF_ERR_IMAGE_DEFECT:
    g_string_assign(msg, "Image is defective, decoding aborted");
    break;
  case D_GIF_ERR_EOF_TOO_SOON:
    g_string_assign(msg, "Image EOF detected, before image was complete");
    break;
  default:
    g_string_assign(msg, "Unknown GIF error");
    break;
  }
  return msg->str;
}




// Make a GIF (89a) file out of the given image
static bool make_gif(GdkImage* image, char* filename)
{
  GifFileType *giffile;
  ColorMapObject *gifColorMap;
  GifByteType *GlblGifBuffer, *GlblGifBufferPtr;
  int i, rc, error;

  GlblGifBufferPtr = GlblGifBuffer = g_new(GifByteType, 
					   image->width * image->height);
  switch (image->depth) {
  case 1:   // Bitmap image
    gifColorMap = MakeMapObject(2, NULL);  // 2 colors (black, white)
    gifColorMap->Colors[0].Red = gifColorMap->Colors[0].Green = 
      gifColorMap->Colors[0].Blue = 0xFF;  // White
    gifColorMap->Colors[1].Red = gifColorMap->Colors[1].Green = 
      gifColorMap->Colors[1].Blue = 0x0;   // Black
    break;
  default:  // Pixmap (antialiased image)
    gifColorMap = MakeMapObject(8, NULL);  // 8 colors, 3 unused
    for (i=0; i<5; i++) { // 5 colors in graypalette[]
      gifColorMap->Colors[i].Red   = graypalette[i].red / 256;
      gifColorMap->Colors[i].Green = graypalette[i].green / 256;
      gifColorMap->Colors[i].Blue  = graypalette[i].blue / 256;
    }
    break;
  }
  // EGifSetGifVersion("89a"); // Triggers giflib bug (%%$##!!)
  giffile = EGifOpenFileName(filename, FALSE);
  if (giffile == NULL) {
    error = GifLastError();
    errormsg(getGIFError(error));
    return FALSE;
  }
  rc = EGifPutScreenDesc(giffile, image->width, image->height, 
			    8, 0, gifColorMap);
  if (rc==GIF_ERROR) {
    errormsg(getGIFError(GifLastError()));
    g_free(GlblGifBuffer);
    EGifCloseFile(giffile);
    return FALSE;
  }

  char buf[] = {0x1, 0, 0, 0};  // Set transparent color to index 0 (white)
  EGifPutExtension(giffile, GRAPHICS_EXT_FUNC_CODE, 4, buf);
  rc = EGifPutImageDesc(giffile, 0, 0, image->width, image->height, 
			     FALSE, NULL);
  if (rc==GIF_ERROR) {
    errormsg(getGIFError(GifLastError()));
    g_free(GlblGifBuffer);
    EGifCloseFile(giffile);
    return FALSE;
  }

  // Scan image pixel by pixel (slow?)
  for (int y=0; y<image->height; y++) 
    for (int x=0; x<image->width; x++) {
      guint32 pixel = gdk_image_get_pixel(image, x, y);
      *GlblGifBufferPtr = 0;
      // Find pixel value in graypalette and put it into gif buffer
      switch (image->depth) {
      case 1:
	*GlblGifBufferPtr = pixel;  // 0 or 1
      default:
	for (int col=0;col<5;col++) 
	  if (pixel == graypalette[col].pixel) {
	    *GlblGifBufferPtr = col;
	    break;
	  } 
	break;
      }
      GlblGifBufferPtr++;
    }

  // GlblGifBuffer contains now the gif colormap values bytewise
  rc = EGifPutLine(giffile, GlblGifBuffer, image->width * image->height);
  if (rc==GIF_ERROR) {
    errormsg(getGIFError(GifLastError()));
    g_free(GlblGifBuffer);
    EGifCloseFile(giffile);
    return FALSE;
  }
 
  g_free(GlblGifBuffer);
  EGifCloseFile(giffile);
  return TRUE;
}



