/* GK.rellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"

#define	RX_LED	0
#define	TX_LED	1

struct _NetIface
	{
	struct _NetIface
					*next;
	GtkWidget		*vbox;		/* Handle for hidding */
	gint			height;		/* Height of chart + panel */

	gchar			*name;
	gint			idx;		/* Major iface type: ppp, plip, eth, other */
	Chart			chart;

	Decal			rxled,
					txled;

	gint			state,
					old_state;

	gint			enabled;		/* An enabled iface is visible if the	*/
	gint			is_visible;		/* iface is up.  A disabled iface is	*/
									/* not visible regardless of up/down.	*/
	gint			config_temp;
	gint			net_extra;

	gint			rxtx_units;
	unsigned long	rx,
					tx,
					oldrx,
					oldtx;

	GtkWidget		*spin_button,	/* For user config settings. */
					*enable_button;
	};

typedef struct _NetIface	NetIface;


typedef struct
	{
	gchar	*name;
	gint	resolution;
	gshort	enable;
/*	gshort	inactivity_hide; */
	}
	NetConfig;

#define		N_NET_CFG	20

NetConfig		net_config[N_NET_CFG];
gint			n_net_cfg;


static NetIface		*net_iface;
static NetIface		*ppp0;			/* Special treatment: button & time */

Panel				timer_panel;		/* For the timer and button	*/
static Style		timer_style;

static GtkWidget	*net_vbox;		/* Where all net monitors live */
static GtkWidget	*dynamic_net_vbox;
static GtkWidget	*timer_button_pixmapwid;

static gchar		*lock_file;

static gint			rx_bytes_index,
					tx_bytes_index,
					rx_packets_index,
					tx_packets_index;

static gint			timer_button_state;	/* Button state if not used for ppp0 */
static time_t		net_timer0;


#define NET_DOWN	0
#define NET_WAIT	1
#define NET_UP		2

#define	LED_RX_OFF		0
#define	LED_RX_ON		1
#define	LED_TX_OFF		2
#define	LED_TX_ON		3


  /* Drawing an led means to draw its bitmap onto the panel stencil and
  |  flag that the led is modified.
  */
void
draw_led(NetIface *iface, int rxtx, int led_index)
	{
	Panel		*p;
	Decal		*led;
	gint		y_src;

	p = &iface->chart.panel;
	if (rxtx == RX_LED)
		led = &iface->rxled;
	else
		led = &iface->txled;

	/* Save some bandwidth and draw the led only if it is changed.
	*/
	if (led->value != led_index)
		{
		led->modified = TRUE;

		/* Write the new led mask into the stencil.
		*/
		y_src = led_index * led->h;
		led->y_src = y_src;
		if (led->mask)			/* Can be NULL if no transparency */
			gdk_draw_pixmap(p->stencil, p->stencil_gc, led->mask,
					0, y_src, led->x, led->y, led->w, led->h);
		else	/* Fill it with white */
			{
			gdk_gc_set_foreground(p->stencil_gc, &GK.white_color);
			gdk_draw_rectangle(p->stencil, p->stencil_gc,
					TRUE, led->x, led->y, led->w, led->h);
			}
		}
	led->value = led_index;
	}


static void
draw_timer(Panel *p, gint seconds)
	{
	static gint	last_time = -1;
	gint		x;
	gint		minutes, hours;
	Label		*lbl;
	gchar		buf[32];

	lbl = &p->label;
	if (p->pixmap == NULL || p->background == NULL)
		return;

	minutes = seconds / 60;
	if (last_time == minutes)
		return;
	last_time = minutes;

	x = timer_style.border_panel.left;

	hours = minutes / 60;
	minutes = minutes % 60;

	sprintf(buf, "%2d:%02d", hours, minutes);

	gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->background,
			0, 0,  0, 0,  p->w, p->h);
	gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->background,
			0, 0,  0, 0,  p->w, p->h);
	if (lbl->textstyle.effect)
		{
		gdk_gc_set_foreground(GK.text_GC, &GK.background_color);
		gdk_draw_string(p->pixmap, lbl->textstyle.font, GK.text_GC,
			x + 1, lbl->y_baseline + 1, buf);
		gdk_draw_string(p->drawing_area->window, lbl->textstyle.font,
						GK.text_GC, x + 1, lbl->y_baseline + 1, buf);
		}
	gdk_gc_set_foreground(GK.text_GC, &lbl->textstyle.color);
	gdk_draw_string(p->pixmap, lbl->textstyle.font, GK.text_GC,
			x, lbl->y_baseline, buf);
	gdk_draw_string(p->drawing_area->window, lbl->textstyle.font, GK.text_GC,
						x, lbl->y_baseline, buf);
	}


static time_t
get_modem_connect_time(void)
	{
	struct stat	st;
	time_t		t;
	gchar		path[256];

	snprintf(path, sizeof(path), "/var/lock/%s", lock_file);
	if (stat(path, &st) == 0)
		t = st.st_mtime;
	else
		{
		/* If lock file is ttySx, then user can make a link like so:
		|  ln -s ~/.gkrellm/LCK..modem /var/lock/LCK..ttySx
		*/
		snprintf(path, sizeof(path), "%s/%s/%s",
					homedir(), GKRELLM_DIR, lock_file);
		if (stat(path, &st) == 0)
			t = st.st_mtime;
		else
			t = 0;
		}
	return t;
	}

void
set_timer_button_pixmap(state)
	{
	GdkPixmap	*pixmap;
	
	if (state == OFF)
		pixmap = GK.ppp_button_off_pixmap;
	else
		pixmap = GK.ppp_button_on_pixmap;
	if (timer_button_pixmapwid)
		gtk_pixmap_set(GTK_PIXMAP(timer_button_pixmapwid), pixmap, NULL);
	}


void
cb_timer_button(GtkWidget *widget, GtkWidget *pixmapwid)
	{
	if (UC.net_timer_is_ppp)
		{
		if (ppp0->state == NET_DOWN)
			{
			system(UC.ppp_on_command);
			set_timer_button_pixmap(ON);
			ppp0->state = NET_WAIT;
			}
		else if (ppp0->state == NET_UP)
			system(UC.ppp_off_command);
		else if (ppp0->state == NET_WAIT)
			{
			system(UC.ppp_off_command);
			set_timer_button_pixmap(OFF);
			ppp0->state = NET_DOWN;
			draw_led(ppp0, RX_LED, LED_RX_OFF);
			draw_led(ppp0, TX_LED, LED_TX_OFF);
			}
		}
	else
		{
		if (timer_button_state == OFF)
			{
			system(UC.ppp_on_command);
			set_timer_button_pixmap(ON);
			timer_button_state = ON;
			time(&net_timer0);
			}
		else
			{
			system(UC.ppp_off_command);
			set_timer_button_pixmap(OFF);
			timer_button_state = OFF;
			}
		}
	}

static const char	*delim	= " :|\t\n";

void
get_io_indices()
	{
	FILE	*f;
	gchar	*s;
	gchar	buf[184];
	gint	i;

	if ((f = fopen("/proc/net/dev", "r")))
		{
		fgets(buf, sizeof(buf), f);		/* Waste the first line.	*/
		fgets(buf, sizeof(buf), f);		/* Look for "units" in this line */
		s = strtok(buf, delim);
		for (i = 0; s; ++i)
			{
			if (strcmp(s, "bytes") == 0)
				{
				if (rx_bytes_index == 0)
					rx_bytes_index = i;
				else
					tx_bytes_index = i;
				}
			if (strcmp(s, "packets") == 0)
				{
				if (rx_packets_index == 0)
					rx_packets_index = i;
				else
					tx_packets_index = i;
				}
			s = strtok(NULL, delim);
			}
		fclose(f);
		}
	if (GK.debug)
		printf("rx_bytes=%d tx_bytes=%d rx_packets=%d tx_packets=%d\n",
			rx_bytes_index, tx_bytes_index, rx_packets_index, tx_packets_index);
	}


void
set_widget_visibility(NetIface *iface)
	{
	/* Can't go toggling and adjusting gkrellm height if the net interface
	|  is not enabled.
	*/
	if (! iface->enabled)
		return;
	switch (iface->state)
		{
		case NET_DOWN:
			if (iface->is_visible == TRUE)
				{
				if (GK.debug)
					printf("Setting visibility of %s to %d\n",
							iface->name, iface->state);
				gtk_widget_hide(iface->chart.hbox);
				GK.monitor_height -= iface->chart.h;

				if (iface != ppp0 || ! UC.net_timer_is_ppp)
					{
					gtk_widget_hide(iface->chart.panel.hbox);
					GK.monitor_height -= iface->chart.panel.h;
					}
				pack_side_frames();		/* Change in monitor area height */
				}
			iface->is_visible = FALSE;
			break;
		case NET_WAIT:
			break;
		case NET_UP:
			if (iface->is_visible == FALSE)
				{
				if (GK.debug)
					printf("Setting visibility of %s to %d\n",
								iface->name, iface->state);
				if (iface != ppp0 || ! UC.net_timer_is_ppp)
					{
					gtk_widget_show(iface->chart.panel.hbox);
					GK.monitor_height += iface->chart.panel.h;
					}
				gtk_widget_show(iface->chart.hbox);
				GK.monitor_height += iface->chart.h;
				pack_side_frames();
				}
			iface->is_visible = TRUE;
			break;
		}
	}


static NetConfig *
lookup_net_config(gchar *name)
	{
	gint	i;

	for (i = 0; i < n_net_cfg; ++i)
		if (strcmp(name, net_config[i].name) == 0)
			return &net_config[i];
	return NULL;
	}

gint
store_net_config(gchar *name, int resolution, int enable)
	{
	NetConfig	*nc;
	gint		i;

	for (i = 0; i < n_net_cfg; ++i)
		{
		nc = &net_config[i];
		if (strcmp(name, nc->name) == 0)
			{
			if (GK.debug)
				printf("Net_config change %s (%d %d) -> (%d %d)\n",
						name, nc->resolution, nc->enable, resolution, enable);
			if (nc->resolution != resolution || nc->enable != enable)
				{
				nc->resolution = resolution;
				nc->enable = enable;
				return TRUE;
				}
			return FALSE;
			}
		}
	net_config[i].name = g_strdup(name);
	net_config[i].resolution = resolution;
	net_config[i].enable = enable;
	if (GK.debug)
		printf("New net_config[%d]: %s=%d %d\n", i, name, resolution, enable);
	if (n_net_cfg < sizeof(net_config) / sizeof(NetConfig))
		++n_net_cfg;
	return TRUE;
	}

void
write_net_config(FILE *f)
	{
	gint	i;

	for (i = 0; i < n_net_cfg; ++i)
		fprintf(f, "net_resolution %s %d %d\n", net_config[i].name,
				net_config[i].resolution, net_config[i].enable);
	}

void
load_net_config(gchar *arg)
	{
	gchar	name[32];
	gint	resolution	= 0,
			enable		= TRUE;

	name[0] = '0';
	sscanf(arg, "%s %d %d", name, &resolution, &enable);
	if (name[0] != '0')
		store_net_config(name, resolution, enable);
	}

void create_interface_widgets(GtkWidget *, NetIface *);

void
register_new_interface(char *s)
	{
	GtkWidget	*vbox;
	NetIface	*ifi, *ifnew;
	gint		id;

	if (GK.trace)
		printf("register_new_interface()\n");
	if (   strncmp(s, "lo", 2) == 0
		|| strncmp(s, "dummy", 5) == 0
	   )
		return;
	vbox = dynamic_net_vbox;
	if (strncmp(s, "ppp", 3) == 0)
		id = PPP;
	else if (strncmp(s, "plip", 4) == 0)
		id = PLIP;
	else if (strncmp(s, "eth", 3) == 0)
		id = ETH;		/* don't use this */
	else
		id = NET_OTHER;

	ifnew = (NetIface *) g_new0(NetIface, 1);

	if (id == PPP && strncmp(s, "ppp0", 4) == 0)
		{
		ppp0 = ifnew;
		vbox = net_vbox;		/* ppp0 interface will be last */
		}

	ifnew->idx = id;

	ifnew->chart.h = UC.chart_height[MON_NET];

	ifnew->next = NULL;
	if (net_iface == NULL)
		net_iface = ifnew;
	else
		for (ifi = net_iface; ifi; ifi = ifi->next)
			if (ifi->next == NULL)
				{
				ifi->next = ifnew;
				break;
				}
	ifnew->name = g_strdup(s);
	ifnew->state = NET_UP;

	create_interface_widgets(vbox, ifnew);
	pack_side_frames();

	ifnew->is_visible = TRUE;

	alloc_chart_data(&ifnew->chart);
	clear_chart(&ifnew->chart);

	draw_led(ifnew, RX_LED, LED_RX_OFF);
	draw_led(ifnew, TX_LED, LED_TX_OFF);

	ifnew->net_extra = UC.enable_extra_info;

	if (GK.debug)
		printf("New net interface: %s label(%d)\n", ifnew->name, ifnew->idx);
	if (GK.trace)
		printf("  <-\n");
	}

void
sync_interfaces_state()
	{
	FILE		*f;
	NetIface	*iface;
	gchar		*s;
	gchar		buf[160];

	for (iface = net_iface; iface; iface = iface->next)
		{
		iface->old_state = iface->state;
		if (!(iface == ppp0 && iface->state == NET_WAIT))
			iface->state = NET_DOWN;
		}
	if ((f = fopen("/proc/net/route", "r")))
		{
		fgets(buf, sizeof(buf), f);		/* Waste the first line */
		while (fgets(buf, sizeof(buf), f))
			{
			s = strtok(buf, " \t\n");
			if (s == NULL)
				continue;
			for (iface = net_iface; iface; iface = iface->next)
				if (strcmp(iface->name, s) == 0)
					{
					iface->state = NET_UP;
					break;
					}
			if (iface == NULL)
				register_new_interface(s);
			}
		fclose(f);
		}
	for (iface = net_iface; iface; iface = iface->next)
		{
		set_widget_visibility(iface);
		if (iface == ppp0 && iface->state != iface->old_state)
			{
			if (iface->state == NET_UP)
				{
				time(&net_timer0);	/* New ppp session just started */
				if (UC.net_timer_is_ppp)
					set_timer_button_pixmap(ON);
				}
			else if (iface->state == NET_DOWN)
				{
				if (UC.net_timer_is_ppp)
					set_timer_button_pixmap(OFF);
				draw_led(ppp0, RX_LED, LED_RX_OFF);
				draw_led(ppp0, TX_LED, LED_TX_OFF);
				}
			}
		}

	}


  /* I read both the bytes (kernel 2.2.x) and packets (all kernels).  Some
  |  net drivers for 2.2.x do not update the bytes counters.
  */
static void
parse_net_dev_io(char *buf)
	{
	NetIface	*iface;
	gchar		*s, *s1;
	gint		i;
	unsigned long	rx_bytes, tx_bytes, rx_packets, tx_packets;
	static gchar	*io_delim	= " \t\n";

	/* Virtual net interfaces have a colon in the name, and a colon seps
	|  the name from data, and there might be no space between data and name!
	|  Eg. this is possible -> eth2:0:11249029    0 ...
	|  So, replace the colon the separates data from the name with a space.
	*/
	s = strchr(buf, (int) ':');
	if (s)
		{
		s1 = strchr(s + 1, (int) ':');
		if (s1)
			*s1 = ' ';
		else
			*s = ' ';
		}

	s = strtok(buf, io_delim);		/* Get name of interface */
	if (s == NULL)
		return;
	for (iface = net_iface; iface; iface = iface->next)
		{
		if (strcmp(iface->name, s) == 0)
			{
			rx_bytes = 0;
			tx_bytes = 0;
			rx_packets = 0;
			tx_packets = 0;
			for (i = 1; s; ++i)
				{
				s = strtok(NULL, io_delim);
				if (s == NULL)
					break;
				if (i == rx_bytes_index)
					rx_bytes = strtoul(s, NULL, 10);
				else if (i == tx_bytes_index)
					tx_bytes = strtoul(s, NULL, 10);
				else if (i == rx_packets_index)
					rx_packets = strtoul(s, NULL, 10);
				else if (i == tx_packets_index)
					tx_packets = strtoul(s, NULL, 10);
				if (i > tx_bytes_index && i > tx_packets_index)
					break;
				}
			if (rx_bytes == 0 && tx_bytes == 0)
				{
				iface->rx = rx_packets;
				iface->tx = tx_packets;
				iface->rxtx_units = NET_UNITS_PACKETS;
				}
			else
				{
				iface->rx = rx_bytes;
				iface->tx = tx_bytes;
				iface->rxtx_units = NET_UNITS_BYTES;
				}
			break;
			}
		}
	}

void
draw_net_extra(NetIface *iface, unsigned long l)
	{
	Chart	*cp;
	gchar	buf[32];
	gint	n;

	cp = &iface->chart;
	n = (gint) l;
	if (n < 1000)
		sprintf(buf, "%d", n);
	else if (n < 19950)
		{
		n += 50;
		sprintf(buf, "%d.%dk", n / 1000, (n % 1000) /100);
		}
	else
		{
		n += 500;
		sprintf(buf, "%dk", n / 1000);
		}
	draw_chart_label(cp, GK.label_font, 4, label_font_ascent + 4, buf);
	}


  /* Read /proc/net/dev at full speed to run the LEDs, but only update the
  |  charts once per second.
  */
void
update_net()
	{
	FILE		*f;
	NetIface	*iface;
	gchar		buf[160];
	unsigned long	l;

	for (iface = net_iface; iface; iface = iface->next)
		iface->rx = iface->tx = 0;

	f = fopen("/proc/net/dev", "r");
	if (f)
		{
		fgets(buf, sizeof(buf), f);		/* Waste first 2 lines */
		fgets(buf, sizeof(buf), f);
		while (fgets(buf, sizeof(buf), f))
			parse_net_dev_io(buf);
		fclose(f);
		}
	for (iface = net_iface; iface; iface = iface->next)
		{
		if (iface->state != NET_DOWN)
			{
			if (iface->rx > iface->oldrx)
				draw_led(iface, RX_LED, LED_RX_ON);
			else
				draw_led(iface, RX_LED, LED_RX_OFF);
			if (iface->tx > iface->oldtx)
				draw_led(iface, TX_LED, LED_TX_ON);
			else
				draw_led(iface, TX_LED, LED_TX_OFF);
			}
		l = iface->rx - iface->chart.prevIn;
		l += iface->tx - iface->chart.prevOut;

		iface->oldrx = iface->rx;
		iface->oldtx = iface->tx;
		if (GK.second_tick)
			{
			store_chart_data(&iface->chart, iface->tx, iface->rx, 0);
			if (iface->state == NET_UP)		/* And visible?? */
				{
				draw_chart(&iface->chart);
				if (iface == ppp0 && UC.net_timer_is_ppp)
					draw_timer(&timer_panel, (int) (time(0) - net_timer0));
				if (iface->net_extra)
					draw_net_extra(iface, l);
				}
			}
		/* I don't care if this unsigned long add overflows because I
		|  care only about differences from one update to the next.
		*/
		update_krell(&iface->chart.panel, iface->chart.panel.krell,
					iface->tx + iface->rx);
		draw_layers(&iface->chart.panel);
		}
	if (GK.second_tick)
		{
		if (! UC.net_timer_is_ppp && timer_button_state == ON)
			draw_timer(&timer_panel, (int) (time(0) - net_timer0));
		sync_interfaces_state();
		}
	}

/* On exposure event, redraw the screen from the backing pixmap
*/
static gint
chart_expose_event(GtkWidget *widget, GdkEventExpose *event)
	{
	NetIface	*iface;

	for (iface = net_iface; iface; iface = iface->next)
		if (iface->chart.drawing_area == widget)
			{
			gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
				iface->chart.pixmap,
				event->area.x, event->area.y,
				event->area.x, event->area.y,
				event->area.width, event->area.height);
			break;
			}

	return FALSE;
	}

static gint
panel_expose_event(GtkWidget *widget, GdkEventExpose *event)
	{
	NetIface	*iface;

	if (timer_panel.drawing_area == widget)
		{
		gdk_draw_pixmap(widget->window,
			  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			  timer_panel.pixmap,
			  event->area.x, event->area.y,
			  event->area.x, event->area.y,
			  event->area.width, event->area.height);
		return FALSE;
		}

	for (iface = net_iface; iface; iface = iface->next)
		{
		if (iface->chart.panel.drawing_area == widget)
			{
			gdk_draw_pixmap(widget->window,
				  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
				  iface->chart.panel.pixmap,
				  event->area.x, event->area.y,
				  event->area.x, event->area.y,
				  event->area.width, event->area.height);
			break;
			}
		}
	return FALSE;
	}

static gint
cb_net_extra(GtkWidget *widget, GdkEventButton *event)
	{
	NetIface	*iface;

	if (event->button == 1)
		for (iface = net_iface; iface; iface = iface->next)
			if (iface->chart.drawing_area == widget)
				iface->net_extra = 1 - iface->net_extra;
	return TRUE;
	}

static int	scale_defaults[N_NET_IFACES] =
	/* PPP  PLIP   ETH    NET_OTHER 	see gkrellm.h */
	{ 2000, 5000, 20000, 10000 };

int
map_x(gint x, gint width)
	{
	gint	xnew;

	xnew = x;
	if (GK.allow_scaling && GK.chart_width_ref != UC.chart_width)
		xnew = xnew * UC.chart_width / GK.chart_width_ref;
	if (x < 0)
		xnew += UC.chart_width - width;
	return xnew;
	}

static void
setup_net_scaling(Chart *cp, gint resolution)
	{
	gint	grids;

	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	cp->scale_min = resolution;
	cp->panel.krell->full_scale = resolution * grids / UC.update_HZ;
	cp->scale_max = 0;		/* Force chart rescale */
	}

void
create_interface_widgets(GtkWidget *vbox, NetIface *iface)
	{
	GdkImlibBorder	*border;
	Style			*style;
	Chart			*cp;
	NetConfig		*nc;
	gint			idx, n;
	gint			h_leds;

	if (GK.trace)
		printf("create_interface_widgets()\n");
	/* Create the main chart for plotting data.
	*/
	idx = iface->idx;
	cp = &iface->chart;

	if ((nc = lookup_net_config(iface->name)) == NULL)
		{
		/* Some nic drivers for kernel 2.2.x do not update byte counters.
		|  For these, I will fall back to packets, but the default
		|  resolution will be way off.  Too bad.  At least I can adjust
		|  the default if I _know_ I will read packets (kernel 2.0.x).
		*/
		n = scale_defaults[idx];
		if (rx_bytes_index == 0)
			n /= 200;
		store_net_config(iface->name, n, TRUE);
		nc = lookup_net_config(iface->name);
		}
	/* A ppp0 interface with timer button linked cannot be disabled.
	*/
	if (iface == ppp0)
		nc->enable = TRUE;

	if (GK.debug)
		printf("   %s idx=%d res=%d\n", iface->name, idx, nc->resolution);

	cp->name = iface->name;
	style = GK.chart_style[NET_IMAGE];
	create_krell(iface->name, GK.krell_panel_image[NET_IMAGE],
						&cp->panel.krell, style);
	setup_net_scaling(cp, nc->resolution);

	iface->vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), iface->vbox);

	create_chart(iface->vbox, cp, NET_IMAGE);
	iface->height = cp->h;
	if ((iface->enabled = nc->enable) == TRUE)
		{
		GK.monitor_height += cp->h;
		gtk_widget_show(iface->vbox);
		}
	gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
			(GtkSignalFunc) chart_expose_event, NULL);
	gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_net_extra, NULL);


	/* Create the panel area.  Will hold the name label of the interface, the
	|  rx and tx leds.  For the ppp0 interface, will hold the timer and button.
	*/
	border = &GK.chart_style[NET_IMAGE]->border_panel;

	h_leds = GK.rx_led_y > GK.tx_led_y ? GK.rx_led_y : GK.tx_led_y;
	h_leds += GK.led_height + style->border_panel.bottom;

	default_textstyle(&cp->panel.label.textstyle, TEXTSTYLE_LABEL);
	if (strlen(cp->name) > 5)
		cp->panel.label.textstyle.font = GK.alt_font;
	configure_panel(&cp->panel, cp->name, style);

	if (cp->panel.label.h_panel < h_leds)
		cp->panel.label.h_panel = h_leds;
	create_panel_area(iface->vbox, &cp->panel, GK.bg_panel_image[NET_IMAGE]);
	if (iface->enabled)
		GK.monitor_height += cp->panel.h;
	iface->height += cp->panel.h;

	gtk_signal_connect(GTK_OBJECT (cp->panel.drawing_area), "expose_event",
			(GtkSignalFunc) panel_expose_event, NULL);

	cp->panel.decal = &iface->rxled;
	iface->rxled.next = &iface->txled;
	iface->txled.next = NULL;

	iface->rxled.x = map_x(GK.rx_led_x, GK.led_width);
	iface->rxled.y = GK.rx_led_y;
	iface->rxled.value = -1;		/* Force initial draw	*/
	iface->rxled.w = GK.led_width;
	iface->rxled.h = GK.led_height;
	iface->rxled.pixmap = GK.led_pixmap;
	iface->rxled.mask = GK.led_mask;
	iface->rxled.type = DECAL_TRANSPARENCY;

	iface->txled.x = map_x(GK.tx_led_x, GK.led_width);
	iface->txled.y = GK.tx_led_y;
	iface->txled.value = -1;
	iface->txled.w = GK.led_width;
	iface->txled.h = GK.led_height;
	iface->txled.pixmap = GK.led_pixmap;
	iface->txled.mask = GK.led_mask;
	iface->txled.type = DECAL_TRANSPARENCY;

	if (GK.debug)
		printf("%s rx_led_x=%d tx_led_x=%d mask=0x%x\n", iface->name,
				iface->rxled.x, iface->txled.x, (gint) GK.led_mask);

	if (GK.trace)
		printf("  <-\n");
	}


  /* Originally, there was only a ppp timer.  Now the timer is more
  |  general so users can control ippp0, etc (or even something unrelated
  |  to net) but some ppp_ naming relics remain.
  */
static void
create_net_timer(GtkWidget *vbox)
	{
	GtkWidget		*box1,
					*button;
	GdkPixmap		*pixmap;
	Panel			*p;
	GdkImlibImage	*im;
	gint			x_dst, h_tim, h_but;
	gint			w, h, y_button;

	h = 2;		/* Just don't allow tall buttons.	*/
	h_but = 8;	/* Image + 3 pixel top and bottom gtk button borders. */
	w = GK.ppp_button_image->rgb_width;
	if (GK.allow_scaling && GK.chart_width_ref != UC.chart_width)
		w = w * UC.chart_width / GK.chart_width_ref;
	gdk_imlib_render(GK.ppp_button_image, w, 2 * h);
	pixmap = gdk_imlib_move_image(GK.ppp_button_image);
	GK.ppp_button_on_pixmap = gdk_pixmap_new(top_window->window, w, h, -1);
	GK.ppp_button_off_pixmap = gdk_pixmap_new(top_window->window, w, h, -1);
	gdk_draw_pixmap(GK.ppp_button_on_pixmap, GK.draw1_GC, pixmap,
					0, h,  0, 0, w, h);
	gdk_draw_pixmap(GK.ppp_button_off_pixmap, GK.draw1_GC, pixmap,
					0, 0,  0, 0, w, h);

	p = &timer_panel;
	if (GK.bg_timer_image == NULL && GK.ppp_border_mode == 2)
		im = GK.bg_bordered_image;
	else
		im = GK.bg_spacer_image;
								
	timer_style.label_position = LABEL_CENTER;		/* XXX */
	if (GK.bg_timer_image == NULL && GK.ppp_border_mode == 1)
		timer_style.border_panel = GK.bordered_bg_border;
	else
		timer_style.border_panel = GK.timer_border;

	default_textstyle(&p->label.textstyle, TEXTSTYLE_TIME);
	if (UC.chart_width < 54)	/* XXX */
		p->label.textstyle.font = GK.alt_font;
	configure_panel(p, "00:00", &timer_style);

	h_tim = p->label.h_panel;
	y_button = (h_tim - h_but) / 2;
	h = (y_button < 0) ? h_but : h_tim;
	if (y_button < 0)
		y_button = 0;

	p->label.h_panel = h;

	p->label.string = NULL;
	create_panel_area(vbox, p, im);
	GK.monitor_height += p->h;

	gtk_signal_connect(GTK_OBJECT (p->drawing_area),
				"expose_event", (GtkSignalFunc) panel_expose_event, NULL);

	/* And there may be an optional border to put around the timer.
	*/
	w = p->label.width + timer_style.border_panel.left
							  + timer_style.border_panel.right + 1;
	im = GK.bg_timer_image ? GK.bg_timer_image : GK.bg_bordered_image;
	if (GK.bg_timer_image || GK.ppp_border_mode == 1)
		{
		gdk_imlib_paste_image(im, p->pixmap,    0, 0, w, h_tim);
		gdk_imlib_paste_image(im, p->background,0, 0, w, h_tim);
		}

	button = gtk_button_new();	

	box1 = gtk_hbox_new (/*homogeneous*/ FALSE, /*spacing*/ 0);
	gtk_container_set_border_width (GTK_CONTAINER (box1), 0);
	timer_button_pixmapwid = gtk_pixmap_new(GK.ppp_button_off_pixmap,
									GK.ppp_button_mask);

	gtk_box_pack_start (GTK_BOX (box1), timer_button_pixmapwid,
			/* expand */ FALSE, /* fill */ FALSE, /* padding */ 0);
	gtk_widget_show(timer_button_pixmapwid);
	gtk_container_add (GTK_CONTAINER (button), box1);
	gtk_widget_show(box1);

	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (cb_timer_button), timer_button_pixmapwid);

	x_dst = map_x(-2,
			((GdkWindowPrivate *) GK.ppp_button_off_pixmap)->width + 6);

	gtk_fixed_put (GTK_FIXED (p->fixed), button, x_dst, y_button);
	gtk_widget_show(button);

	if (! UC.enable_net_timer)
		{
		GK.monitor_height -= timer_panel.h;
		gtk_widget_hide(timer_panel.hbox);
		}
	}

static void
register_ppp0()
	{
	time_t	t;

	register_new_interface("ppp0");	/* Says ppp is up */
	if (ppp0->state == NET_UP)
		{
		t = get_modem_connect_time();
		if (t > 0)
			net_timer0 = t;
		if (UC.net_timer_is_ppp)
			set_timer_button_pixmap(ON);
		}
	}

  /* Net monitors are dynamically created as they are found in
  |  /proc/dev/route, except for ppp0 which needs a control to
  |  start and stop it.
  */
void
create_net(GtkWidget *vbox)
	{

	/* Make a couple of vboxes here so I can control the net layout.
	|  I want ppp interfaces to go last.
	*/
	if (GK.trace)
		printf("create_net()\n");
	dynamic_net_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), dynamic_net_vbox);
	gtk_widget_show(dynamic_net_vbox);

	net_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), net_vbox);
	gtk_widget_show(net_vbox);

	lock_file = "LCK..modem";

	get_io_indices();

	create_net_timer(vbox);

	time(&net_timer0);				/* Sanity initialization */
	if (UC.net_timer_is_ppp)
		register_ppp0();
	draw_timer(&timer_panel, (int) (time(0) - net_timer0));

	/* Create all net interfaces currently up. Also verifies ppp0 state
	*/
	sync_interfaces_state();

	if (GK.trace)
		printf("  <-\n");
	}



/* -------------- User config interface --------------------- */

#define STEP	1

GtkWidget	*pon_combo,
			*poff_combo;

GtkWidget	*net_timer_is_ppp_button,
			*enable_net_timer_button;

static gint	net_res_map []	=
	{
	5,
	10, 20, 50,
	100, 200, 500,
	1000, 2000, 5000,
	10000, 20000, 50000,
	100000, 200000, 500000,
	1000000
	};

gint
map_1_2_5(gint value, gint *table, gint size)
	{
	gint	i;

	for (i = 0; i < size; ++i)
		{
/*printf("  mapping[%d] value=%d table=%d\n", i, value, table[i]); */
		if (value == table[i])
			return table[i];
		else if (value == table[i] - STEP)
			return table[i - 1];
		else if (value == table[i] + STEP)
			return table[i + 1];
		}
	return value;
	}

static void
cb_net_resolution(GtkWidget *widget, GtkSpinButton *spin)
	{
	NetIface	*iface;
	Chart		*cp;
	gint		res;

	for (iface = net_iface; iface; iface = iface->next)
		{
		if (spin == GTK_SPIN_BUTTON(iface->spin_button))
			{
			res = gtk_spin_button_get_value_as_int(spin);
			if (res != iface->config_temp)	/* Avoid recursion */
				{
				res = map_1_2_5(res, net_res_map,
						sizeof(net_res_map)/sizeof(int));
				iface->config_temp = res;
				gtk_spin_button_set_value(spin, (gfloat) res);
				cp = &iface->chart;
				setup_net_scaling(cp, res);
				draw_chart(cp);
				}
			break;
			}
		}
	}


static void
cb_config_timer_button()
	{
	gint	ent;

	ent = GTK_TOGGLE_BUTTON(enable_net_timer_button)->active;
	if (ent == FALSE)
		gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(net_timer_is_ppp_button), FALSE);
	}

static gchar	*net_info_text =
"Routed net interfaces listed in the routing table /proc/net/route\n"
"are charted.  Data is plotted with units of bytes/sec if possible,\n"
"however, units of packets/sec are used for kernel 2.0 and some 2.2\n"
"nic drivers which do not report bytes.\n\n"

"If the timer button is linked to ppp0, then the timer will show an\n"
"elapsed on line time based on the time stamp of the modem lock file\n"
"/dev/modem.  If your pppd setup does not use /dev/modem, then you\n"
"can make the timer work correctly by making the symbolic link:\n"
"    ln  -s  /var/lock/LCK..ttySx   ~/.gkrellm/LCK..modem\n"
"where ttySx is the tty device your modem uses.\n\n"

"The timer button Start Command must run in the background and\n"
"this is automatically the default for most ppp logon scripts. One\n"
"exception, if you use wvdial it needs to be explicitly backgrounded:\n"
"    wvdail &\n"
"and the ppp logoff Stop Command in this case could be:\n"
"    skill -c wvdial\n"
"Otherwise do not append the \"&\" on more common ppp Start\n"
"Command entries such as pon (Debian) or ifup.\n"
"If the timer button is not linked to ppp0, then it can be used for\n"
"isdn or any other purpose.\n\n"

"See General->Info for help on setting Net chart resolutions.\n"
;

void
create_net_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1;
	GtkWidget		*hbox;
	GtkWidget		*label;
	GtkWidget		*separator;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GList			*glist;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;
	NetIface		*iface;
	gchar			buf[160];

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	vbox = create_tab(tabs, "Timer Button");

	enable_net_timer_button = gtk_check_button_new_with_label(
				"Enable Net timer button");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_net_timer_button),
				UC.enable_net_timer);
	gtk_box_pack_start(GTK_BOX(vbox), enable_net_timer_button, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(enable_net_timer_button),"clicked",
			(GtkSignalFunc) cb_config_timer_button, NULL);

	net_timer_is_ppp_button = gtk_check_button_new_with_label(
			"Link ppp0 to the timer button and always show the ppp0 panel.");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(net_timer_is_ppp_button),
				UC.net_timer_is_ppp);
	gtk_box_pack_start(GTK_BOX(vbox), net_timer_is_ppp_button, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(net_timer_is_ppp_button),"clicked",
			(GtkSignalFunc) cb_config_timer_button, NULL);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);

	hbox = gtk_hbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	vbox1 = gtk_vbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 3);
	gtk_container_add(GTK_CONTAINER(hbox), vbox1);
	label = gtk_label_new("Start Command");
	gtk_box_pack_start (GTK_BOX (vbox1), label, FALSE, TRUE, 0);
	pon_combo = gtk_combo_new();
	gtk_box_pack_start (GTK_BOX (vbox1), pon_combo, FALSE, TRUE, 0);
	glist = NULL;
	glist = g_list_append(glist, "pon");
	glist = g_list_append(glist, "pppon");
	gtk_combo_set_popdown_strings( GTK_COMBO(pon_combo), glist);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(pon_combo)->entry),
						UC.ppp_on_command);

	vbox1 = gtk_vbox_new (FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 3);
	gtk_container_add(GTK_CONTAINER(hbox), vbox1);
	label = gtk_label_new("Stop Command");
	gtk_box_pack_start (GTK_BOX (vbox1), label, FALSE, TRUE, 0);
	poff_combo = gtk_combo_new();
	gtk_box_pack_start (GTK_BOX (vbox1), poff_combo, FALSE, TRUE, 0);
	glist = NULL;
	glist = g_list_append(glist, "poff");
	glist = g_list_append(glist, "pppoff");
	gtk_combo_set_popdown_strings( GTK_COMBO(poff_combo), glist);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(poff_combo)->entry),
						UC.ppp_off_command);

	for (iface = net_iface; iface; iface = iface->next)
		{
		vbox = create_tab(tabs, iface->name);

		hbox = gtk_hbox_new (FALSE, 3);
		gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
		gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

		adj = (GtkAdjustment *) gtk_adjustment_new(
			iface->chart.scale_min, 5.0 /*lower*/, 1000000.0 /*upper*/,
			(gfloat) STEP /*step*/, STEP /*page*/, 0.0);
		iface->spin_button = gtk_spin_button_new (adj, 0.95 /*climb rate*/, 0);
		gtk_widget_set_usize (iface->spin_button, 70, 0);

		spin = GTK_SPIN_BUTTON(iface->spin_button);
		gtk_spin_button_set_numeric(spin, TRUE);
		gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
				GTK_SIGNAL_FUNC (cb_net_resolution), (gpointer) spin);
		gtk_box_pack_start(GTK_BOX(hbox), iface->spin_button, FALSE, TRUE, 5);
		sprintf(buf, "Chart resolution in %s/sec per grid",
							(iface->rxtx_units == NET_UNITS_BYTES) ?
							"bytes" : "packets");
		label = gtk_label_new (buf);
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 5);

		/* ppp0 interface is always enabled because of timer button interactions
		*/
		if (iface != ppp0)
			{
			NetConfig	*nc;
			gchar		buf[32];

			separator = gtk_hseparator_new();
			gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);

			nc = lookup_net_config(iface->name);
			sprintf(buf, "Enable %s", iface->name);
			iface->enable_button = gtk_check_button_new_with_label(buf);
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(iface->enable_button),
					nc->enable);
			gtk_box_pack_start(GTK_BOX(vbox), iface->enable_button,
					TRUE, TRUE, 0);
			}
		iface->config_temp = iface->chart.scale_min;
		}

/* --Info tab */
	vbox = create_tab(tabs, "Info");
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, net_info_text, -1);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);
	}

  /* Apply net config tab settings to the net UC and, for the net apply
  |  we also apply to the net_config[].
  */
void
apply_net_config()
	{
	NetIface		*iface;
	Chart			*cp;
	gint			enable, ntp;
	gchar			*s0, *s1;

	enable = GTK_TOGGLE_BUTTON(enable_net_timer_button)->active;
	enable_visibility(enable, &UC.enable_net_timer, timer_panel.hbox,
					timer_panel.h);
	ntp = enable ? GTK_TOGGLE_BUTTON(net_timer_is_ppp_button)->active : FALSE;

	if (ntp && ppp0 == NULL)
		register_ppp0();
	if (UC.net_timer_is_ppp != ntp)
		{
		if (ppp0 == NULL)
			register_ppp0();
		if (ppp0->is_visible == FALSE && ntp == TRUE)
			{
			gtk_widget_show(ppp0->chart.panel.hbox);
			GK.monitor_height += ppp0->chart.panel.h;
			}
		if (ppp0->is_visible == FALSE && ntp == FALSE)
			{
			gtk_widget_hide(ppp0->chart.panel.hbox);
			GK.monitor_height -= ppp0->chart.panel.h;
			}
		pack_side_frames();
		}
	UC.net_timer_is_ppp = ntp;

	for (iface = net_iface; iface; iface = iface->next)
		{
		/* The chart resolution already has been updated in the callback. 
		|  Here I only need to update the net_config entries so they
		|  will be saved.
		*/
		cp = &iface->chart;

		if (iface->enable_button)		/* ppp0 will not have this button */
			enable = GTK_TOGGLE_BUTTON(iface->enable_button)->active;
		else
			enable = TRUE;
		store_net_config(iface->name, cp->scale_min, enable);

		if (iface->is_visible)
			enable_visibility(enable, &iface->enabled, iface->vbox,
								iface->height);
		else
			iface->enabled = enable;
		setup_net_scaling(cp, cp->scale_min);	/* In case grid change */
		}
	s0 = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(pon_combo)->entry));
	s1 = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(poff_combo)->entry));
	if (strcmp(s0, UC.ppp_on_command) || strcmp(s1, UC.ppp_off_command))
		{
		g_free(UC.ppp_on_command);
		g_free(UC.ppp_off_command);
		UC.ppp_on_command = g_strdup(s0);
		UC.ppp_off_command = g_strdup(s1);
		}
	}

