/***************************************************************************
                          qbrew.cpp  -  a brewing recipe calculator
                             -------------------
    begin                : September 20th, 1999
    copyright            : (C) 1999 by David Johnson
    email                : arandir@meer.net

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#include "qbrew.h"

#include <qwindowsstyle.h>
#include <qplatinumstyle.h>
#include <qmotifstyle.h>
#include <qcdestyle.h>

#include <qpaintdevicemetrics.h>

#include "setupdialog.h"

#include "filesave.xpm"
#include "fileopen.xpm"
#include "filenew.xpm"
#include "fileprint.xpm"
#include "qbrew.xpm"

// our instance of styleMap
QMap<QString, int> styleMap;

QBrew::QBrew()
{
	setCaption(IDS_TITLE + VERSION);
	QPixmap qbicon(qbrewIcon);
	setIcon(qbicon);

	doc = NULL;
	view = NULL;
	calc = NULL;
	settings = NULL;
	qmlStyle = NULL;
	filename = IDS_DEFAULT_FILE;
	
	///////////////////////////////////////////////////////////////////
	// call inits to invoke all other construction parts
	initStyles();
	initSettings();
	initMenuBar();
	initToolBar();
	initStatusBar();

	initDoc();
	initCalc();
	initView();
	// clear the document (redundant)
	doc->newDoc();
	
	// setup connections
	connect(doc,SIGNAL(documentModified()),this,SLOT(slotDocumentModified()));
	connect(calc,SIGNAL(calcDone()),doc,SIGNAL(styleChanged())); // piggybacked signals
}

QBrew::~QBrew()
{
	if (settings != NULL)
		delete settings;
	if (qmlStyle != NULL)
		delete qmlStyle;
	if (doc != NULL)
		delete doc;
	if (view != NULL)
		delete view;
	if (calc != NULL)
		delete calc;
}

///////////////////////////////////////////////////////////////////////////////
// initialization

void QBrew::initStyles()
{
	// set up style sheet and themable styles
	// portions of this method courtesy of Troll Tech AS.
	
	// setup style sheet for HTML
	// Ignore any bodytext in <head>...</head>:
	qmlStyle = new QStyleSheetItem( QStyleSheet::defaultSheet(), "head" );
	qmlStyle->setDisplayMode(QStyleSheetItem::DisplayNone);

	// Not in default style sheet, just fake it:
	qmlStyle = new QStyleSheetItem( QStyleSheet::defaultSheet(), "dl" );
	qmlStyle->setDisplayMode(QStyleSheetItem::DisplayBlock);
	qmlStyle = new QStyleSheetItem( QStyleSheet::defaultSheet(), "dt" );
	qmlStyle->setDisplayMode(QStyleSheetItem::DisplayBlock);
	qmlStyle->setContexts("dl");

	// Many HTML files omit the </p> or </li>, so we add this for efficiency:
	QStyleSheet::defaultSheet()->item("p")->setSelfNesting( FALSE );
	QStyleSheet::defaultSheet()->item("li")->setSelfNesting( FALSE );
	
	// setup themable styles
	styleMap.insert(STYLES[STYLE_MOTIF], STYLE_MOTIF);
	styleMap.insert(STYLES[STYLE_PLATINUM], STYLE_PLATINUM);
	styleMap.insert(STYLES[STYLE_WINDOWS], STYLE_WINDOWS);
	styleMap.insert(STYLES[STYLE_CDE], STYLE_CDE);
	styleMap.insert(STYLES[STYLE_DEFAULT], STYLE_DEFAULT);
}

void QBrew::initSettings()
{
	settings = new Settings();

	// load in default in case settings file doesn't have them all set
	settings->insertSetting(IDS_SETTINGS_STATUSBAR, IDS_SETTINGS_STATUSBAR_DEFAULT);
	settings->insertSetting(IDS_SETTINGS_TOOLBAR, IDS_SETTINGS_TOOLBAR_DEFAULT);
	settings->insertSetting(IDS_SETTINGS_QBREWDIR, IDS_SETTINGS_QBREWDIR_DEFAULT);
	settings->insertSetting(IDS_SETTINGS_STYLE, IDS_SETTINGS_STYLE_DEFAULT);
	settings->insertSetting(IDS_SETTINGS_BATCH, IDS_SETTINGS_BATCH_DEFAULT);
	settings->insertSetting(IDS_SETTINGS_EFFICIENCY, IDS_SETTINGS_EFFICIENCY_DEFAULT);
	
	// try to load setting from rc file
	if (!settings->loadSettings(QDir::homeDirPath() + "/" + IDS_SETTINGS_FILE, IDS_TITLE)) {
		// unsuccessful, recover
		// NOTE: I don't like this duplication. Any better way to recover?
		settings->insertSetting(IDS_SETTINGS_STATUSBAR, IDS_SETTINGS_STATUSBAR_DEFAULT);
		settings->insertSetting(IDS_SETTINGS_TOOLBAR, IDS_SETTINGS_TOOLBAR_DEFAULT);
		settings->insertSetting(IDS_SETTINGS_QBREWDIR, IDS_SETTINGS_QBREWDIR_DEFAULT);
		settings->insertSetting(IDS_SETTINGS_STYLE, IDS_SETTINGS_STYLE_DEFAULT);
		settings->insertSetting(IDS_SETTINGS_BATCH, IDS_SETTINGS_BATCH_DEFAULT);
		settings->insertSetting(IDS_SETTINGS_EFFICIENCY, IDS_SETTINGS_EFFICIENCY_DEFAULT);
	}
	// do any stuff that needs to be done
	if ( styleMap.contains(settings->retrieveSetting(IDS_SETTINGS_STYLE)) ) {
		switch ( styleMap[settings->retrieveSetting(IDS_SETTINGS_STYLE)] ) {
			case STYLE_MOTIF:
				QApplication::setStyle(new QMotifStyle);
				break;
			case STYLE_PLATINUM:
				QApplication::setStyle(new QPlatinumStyle);
				break;
			case STYLE_WINDOWS:
				QApplication::setStyle(new QWindowsStyle);
				break;
			case STYLE_CDE:
				QApplication::setStyle(new QCDEStyle);
				break;
			default:
				QApplication::setStyle(new QMotifStyle);
		}
	} else {
		QApplication::setStyle(new QMotifStyle);
	}				
}

void QBrew::initMenuBar()
{
	///////////////////////////////////////////////////////////////////
	// MENUBAR

	///////////////////////////////////////////////////////////////////
	// menuBar entry file_menu
	
	// TODO: don't hardcode the accelerator keys
	
	file_menu = new QPopupMenu();
	file_menu->insertItem(tr("&New"),this,SLOT(slotFileNew()),CTRL+Key_N, ID_FILE_NEW );
	file_menu->insertItem(tr("&Open..."),this,SLOT(slotFileOpen()),CTRL+Key_O, ID_FILE_OPEN );
	// TODO: Open Recent ->
	file_menu->insertSeparator();
	file_menu->insertItem(tr("&Save"),this,SLOT(slotFileSave()),CTRL+Key_S, ID_FILE_SAVE );
	file_menu->insertItem(tr("Save &as..."),this,SLOT(slotFileSaveAs()),0, ID_FILE_SAVE_AS);
    file_menu->insertSeparator();
	file_menu->insertItem(tr("&Print..."),this,SLOT(slotFilePrint()),CTRL+Key_P, ID_FILE_PRINT );
	file_menu->insertSeparator();
	file_menu->insertItem(tr("&Quit"),this,SLOT(slotFileQuit()),CTRL+Key_Q, ID_FILE_QUIT );

	///////////////////////////////////////////////////////////////////
	// menuBar entry edit_menu
	edit_menu = new QPopupMenu();
	edit_menu->insertItem(tr("Cu&t"),this,SLOT(slotEditCut()),CTRL+Key_X, ID_EDIT_CUT );
	edit_menu->insertItem(tr("&Copy"),this,SLOT(slotEditCopy()),CTRL+Key_C, ID_EDIT_COPY );
	edit_menu->insertItem(tr("&Paste"),this,SLOT(slotEditPaste()),CTRL+Key_V, ID_EDIT_PASTE );
 
	///////////////////////////////////////////////////////////////////
	// menuBar entry options_menu
	options_menu = new QPopupMenu();
	options_menu->setCheckable(true);
	options_menu->insertItem(tr("Tool&bar"),this,SLOT(slotOptionsToolBar()),0, ID_OPTIONS_TOOLBAR);
	options_menu->insertItem(tr("&Statusbar"),this,SLOT(slotOptionsStatusBar()),0, ID_OPTIONS_STATUSBAR );
	options_menu->insertSeparator();
	// TODO: Configure Key Bindings...
	options_menu->insertItem(tr("&Configure qbrew..."),this,SLOT(slotOptionsSetup()),0, ID_OPTIONS_SETUP );
	options_menu->insertSeparator();
	options_menu->insertItem(tr("Sa&ve Options"),this,SLOT(slotOptionsSaveSettings()),0, ID_OPTIONS_SAVESETTINGS );

	options_menu->setItemChecked(ID_OPTIONS_TOOLBAR, true);
	options_menu->setItemChecked(ID_OPTIONS_STATUSBAR,true);

	// check if settings for toolbar is set
	if (settings->retrieveSetting(IDS_SETTINGS_TOOLBAR) != IDS_NULL) {
		// setting is in configuration file
		options_menu->setItemChecked(ID_OPTIONS_TOOLBAR,
			((QString)settings->retrieveSetting(IDS_SETTINGS_TOOLBAR) == "TRUE"));
	} else {
		// setting not in configuration file, so set it
		options_menu->setItemChecked(ID_OPTIONS_TOOLBAR, true);
	}
	// now check settings for statusbar
	if (settings->retrieveSetting(IDS_SETTINGS_STATUSBAR) != IDS_NULL) {
		// setting is in configuration file
		options_menu->setItemChecked(ID_OPTIONS_STATUSBAR,
			((QString)settings->retrieveSetting(IDS_SETTINGS_STATUSBAR) == "TRUE"));
	} else {
		// setting not in configuration file, so set it
		options_menu->setItemChecked(ID_OPTIONS_STATUSBAR, true);
	}
	
	///////////////////////////////////////////////////////////////////
	// menuBar entry help_menu
	help_menu = new QPopupMenu();
	help_menu->insertItem(tr("&Contents..."),this,SLOT(slotHelpContents()),Key_F1,ID_HELP_CONTENTS);
	help_menu->insertSeparator();
	help_menu->insertItem(tr("About ") + IDS_TITLE + "...",this,SLOT(slotHelpAbout()),0,ID_HELP_ABOUT);
	help_menu->insertItem(tr("About Qt..."),this,SLOT(slotHelpAboutQt()),0,ID_HELP_ABOUT_QT);

	///////////////////////////////////////////////////////////////////
	// MENUBAR CONFIGURATION

	menuBar()->insertItem(tr("&File"), file_menu);
	menuBar()->insertItem(tr("&Edit"), edit_menu);
	menuBar()->insertItem(tr("&Options"), options_menu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(tr("&Help"), help_menu);
	
	// enable/disable appropriate items in menu
	file_menu->setItemEnabled( ID_FILE_SAVE, false );

	///////////////////////////////////////////////////////////////////
	// CONNECT THE SUBMENU SLOTS WITH SIGNALS

	connect(file_menu,SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	connect(edit_menu,SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	connect(options_menu,SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
	connect(help_menu,SIGNAL(highlighted(int)), SLOT(statusCallback(int)));
}

void QBrew::initToolBar()
{
	QPixmap openIcon,saveIcon,newIcon, printIcon;

	// set the toolbars to be right justified
	setRightJustification(true);
	
	main_toolbar = new QToolBar( this, "file operations" );

	newIcon = QPixmap( filenew );
	QToolButton * fileNew = new QToolButton( newIcon, tr("New File"), 0,this, SLOT(slotFileNew()),
					main_toolbar);
	openIcon = QPixmap( fileopen );
	QToolButton * fileOpen = new QToolButton( openIcon, tr("Open File"), 0,this, SLOT(slotFileOpen()),
					main_toolbar);
	saveIcon = QPixmap( filesave );
	QToolButton * fileSave = new QToolButton( saveIcon, tr("Save File"), 0,this, SLOT(slotFileSave()),
					main_toolbar);
	printIcon = QPixmap( fileprint );
	QToolButton * filePrint = new QToolButton( printIcon, tr("Print File"), 0,this, SLOT(slotFilePrint()),
					main_toolbar);

	// TODO: | print | cut copy paste | help
	
	main_toolbar->addSeparator();
	(void)QWhatsThis::whatsThisButton( main_toolbar );
	QWhatsThis::add(fileNew, tr("Click this button to create a new file.\n\n"
			"You can also select the New command from the File menu."));
	QWhatsThis::add(fileOpen, tr("Click this button to open a new file.\n\n"
			"You can also select the Open command from the File menu."));
	QWhatsThis::add(fileSave, tr("Click this button to save the file you are "
			"editing.  You will be prompted for a file name.\n\n"
			"You can also select the Save command from the File menu."));
	QWhatsThis::add(filePrint, tr("Click this button to print the current "
			"recipe.\n\n"
			"You can also select the Print command from the File menu."));

	// set a filler widget so that right justified toolbars look okay	
	QLabel *mtfiller =  new QLabel( main_toolbar, "mtfiller");
	main_toolbar->setStretchableWidget(mtfiller);
	
	// now show or hide toolbar depending on initial setting
	if (settings->retrieveSetting(IDS_SETTINGS_TOOLBAR) == "TRUE")
		main_toolbar->show();
	else
		main_toolbar->hide();
}

void QBrew::initStatusBar()
{
	statusBar()->message( IDS_DEFAULT_MESSAGE, 2000 );
	
	// now show or hide statusbar depending on initial setting
	if (settings->retrieveSetting(IDS_SETTINGS_STATUSBAR) == "TRUE")
		statusBar()->show();
	else
		statusBar()->hide();	
}

void QBrew::initDoc()
{
	// does doc already exist?
	if (doc != NULL)
		delete doc;
	doc = new QBrewDoc(this);
	filename = IDS_DEFAULT_FILE;
}

void QBrew::initView()
{
	view = new QBrewView(this, "qbrewview", doc, calc);
	setCentralWidget( view );
}

void QBrew::initCalc()
{
	if (calc != NULL)
		delete(calc);
	calc = new QBrewCalc(settings, doc);
}

///////////////////////////////////////////////////////////////////////////////
// Utility functions

int QBrew::querySave()
{
	return QMessageBox::information(this, IDS_TITLE + " - Save?", tr("Do you wish to save your work first?"),
		tr("Yes"), tr("No"), tr("Cancel"), 0, 2);

}

unsigned QBrew::getBatchSetting() { return settings->retrieveSetting(IDS_SETTINGS_BATCH).toInt(); }

QString QBrew::getRecipeText()
{
	QString buffer, newline;
	buffer = newline = "\n";
	
	// title stuff
	buffer += tr("Recipe: ") + doc->getTitle() + newline;
	buffer += tr("Brewer: ") + doc->getBrewer() + newline;
	buffer += tr("Style: ") + doc->getStyle() + newline;
	buffer += tr("Batch: ") + QString::number(double(doc->getSize()) / 100.0, 'f', 2);
	if (doc->getMash()) buffer += tr(", Mashed");
	buffer += newline + newline;
	
	// style stuff
	buffer += tr("Recipe Gravity: ") + QString::number(calc->getOG() + 1000, 'f', 0) + newline;
	buffer += tr("Recipe Bitterness: ") + QString::number(calc->getIBU(), 'f', 0) + newline;
	buffer += tr("Recipe Color: ") + QString::number(calc->getSRM(), 'f', 0) + CHAR_LATIN_DEGREE + newline;
	buffer += tr("Estimated FG: ") + QString::number(calc->getFGEstimate() + 1000, 'f', 0) + newline;
	buffer += tr("Alcohol by Volume: ") + QString::number(calc->getABV(), 'f', 1) + "%\n";
	buffer += tr("Alcohol by Weight: ") + QString::number(calc->getABW(), 'f', 1) + "%\n\n";

	// grains
	GrainList *grainlist = doc->getGrainList();
	GrainList::Iterator itg;
    for (itg=grainlist->begin(); itg != grainlist->end(); ++itg) {
    	buffer += (*itg).getName().leftJustify(30, ' ');
    	buffer += (*itg).getQuantityString() + tr(" lbs, ");
    	buffer += (*itg).getUseString() + newline;
	}
	buffer += newline;
	
	// hops
	HopsList *hopslist = doc->getHopsList();
	HopsList::Iterator ith;
    for (ith=hopslist->begin(); ith != hopslist->end(); ++ith) {
    	buffer += (*ith).getName().leftJustify(30, ' ');
    	buffer += (*ith).getQuantityString() + tr(" oz, ");
    	buffer += (*ith).getForm() + ", ";
    	buffer += (*ith).getTimeString() + tr(" minutes\n");
	}
	buffer += newline;
	
	// misc ingredients
	MiscIngredientList *misclist = doc->getMiscIngredientList();
	MiscIngredientList::Iterator itm;
    for (itm=misclist->begin(); itm != misclist->end(); ++itm) {
    	buffer += (*itm).getName().leftJustify(30, ' ');
    	buffer += (*itm).getQuantityString() + ", ";
    	buffer += (*itm).getNotes() + newline;
	}
	
	buffer += IDS_TITLE + " " + VERSION;
	return buffer;
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

void QBrew::slotFileNew()
{
	if (doc->isModified()) {
		// file needs to be saved, what do we do
		switch (querySave()) {
			case 0: // yes, save the file
				slotFileSave();
				break;
			case 1: // no, don't save the file
				break;
			case 2: // cancel creating new file
				statusBar()->message(tr("Canceled..."), 2000);
				// exit function
				return;
		}
	}
	// create a new file
	statusBar()->message(tr("Creating new recipe..."));
	doc->newDoc();
	file_menu->setItemEnabled( ID_FILE_SAVE, false );
	// no file name yet, so set it as junk
	filename = IDS_DEFAULT_FILE;
	setCaption( IDS_TITLE + (QString)" - " + filename);
	statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
}

void QBrew::slotFileOpen()
{
	if (doc->isModified()) {
		// file needs to be saved, what do we do
		switch (querySave()) {
			case 0: // yes, save the file
				slotFileSave();
				break;
			case 1: // no, don't save the file
				break;
			case 2: // cancel creating new file
				statusBar()->message(tr("Canceled..."), 2000);
				return;
		}
	}
	statusBar()->message(tr("Opening file..."));

	QString fname = QFileDialog::getOpenFileName(0, IDS_FILE_FILTER, this);
	// keep information about the file
	QFileInfo *finfo = new QFileInfo(fname);
	if ( !fname.isEmpty() ){
		// file dialog returned a file name
		if (doc->load( fname )) {
			// load was successful
			file_menu->setItemEnabled( ID_FILE_SAVE, false );
			setCaption( IDS_TITLE + (QString)" - " + finfo->fileName() );
			statusBar()->message(tr("Loaded document: ") + finfo->fileName() , 2000 );
			// save name of file
			filename = fname.copy();
		} else {
			// load was unsuccessful
			QMessageBox::warning(this, IDS_TITLE, IDS_TITLE + tr(" was unable to load the file ") + finfo->fileName());
			statusBar()->message(tr("Error in loading ") + finfo->fileName() , 2000 );
		}
	} else {
		// file dialog didn't return a file name
		statusBar()->message(tr("Opening aborted") , 2000 );
	}
	delete finfo;
}

void QBrew::slotFileSave()
{
	if (doc->isNew())
	{
		// need to get the file to save it to
		slotFileSaveAs();
	} else {
		// file exists so save it
		statusBar()->message(tr("Saving file..."));
		if (doc->save(filename)) {
			// successful in saving file
			file_menu->setItemEnabled( ID_FILE_SAVE, false );
			statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
		} else {
			// error in saving file
			QMessageBox::warning( this, IDS_TITLE, IDS_TITLE + tr(" was unable to save the file " + filename));
			statusBar()->message(tr("Error in saving file"), 2000);
		}
	}
}

void QBrew::slotFileSaveAs()
{
	statusBar()->message(tr("Saving file under new filename..."));
	QString fname = QFileDialog::getSaveFileName(0,IDS_FILE_FILTER,this);
	QFileInfo *finfo = new QFileInfo(fname);
	if ( !fname.isEmpty() ){
		// we got a valid filename
		if (doc->save( fname )) {
			// successfully saved
			file_menu->setItemEnabled( ID_FILE_SAVE, false );
			setCaption(IDS_TITLE + (QString)" - " + finfo->fileName());
			statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
			// save name of file
			filename = fname.copy();
		} else {
			// error in saving
			QMessageBox::warning( this, IDS_TITLE, IDS_TITLE + tr(" was unable to save the file " + finfo->fileName()));
			statusBar()->message(tr("Error in saving file"), 2000);
		}
	} else {
		// no file name chosen
		statusBar()->message( tr("Opening aborted"), 2000 );
	}
	delete finfo;
}

void QBrew::slotFilePrint()
{
	statusBar()->message(tr("Printing..."));
	if (printer.setup(this))
	{
		QPainter painter;
		QString text, line;
		int tpos;
		int ypos = 12;		// y position for each line
		const int margin = 12;	// eighteen point margin
		int pageno = 1;			// keep track of pages
		
		painter.begin( &printer );
		QFontMetrics fm = painter.fontMetrics();
		QFont fnt("courier"); fnt.setPointSize(10);
		painter.setFont( fnt );	// use fixed width font
		QPaintDeviceMetrics pdm( &printer );	// need width/height
		// get doc string...
		text = getRecipeText();
		// for each line of text...
		while (text.length() > 0) {
			// get line of text
			tpos = text.find('\n');
			if (tpos > 0) {
				line = text.left(tpos);
				text.remove(0, tpos+1);
			} else {
				// get last line if text doens't end in newline
				line = text; text = "";
			}
			// is there space for this line on page?
			if ( margin + ypos > pdm.height() - margin ) {
				QString msg( "Printing (page " + QString::number( ++pageno ) + ")...");
				statusBar()->message( msg );
				printer.newPage();		// no more room on this page
				ypos = 12;				// back to top of page
			}
			// print the line
			painter.drawText( margin, margin + ypos, pdm.width(), fm.lineSpacing(),
				ExpandTabs | DontClip, line );
			// update paper position
			ypos += fm.lineSpacing();
		}
		painter.end();		// send job to printer
		statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
	} else {
		// user chose not to print
		statusBar()->message( "Printing cancelled", 2000 );
	}
}

void QBrew::slotFileQuit()
{
	///////////////////////////////////////////////////////////////////
	// exits the Application
	if (doc->isModified()) {
		// file needs to be saved, what do we do
		switch (querySave()) {
			case 0: // yes, save the file
				slotFileSave();
				break;
			case 1: // no, go ahead and exit
				break;
			case 2: // cancel exit
				statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
				return;
		}
	}
	// do we need to do anything before we quit?
	qApp->quit();
}

void QBrew::slotEditCut()
{
	statusBar()->message(tr("Cutting selection..."));
	// temporary placeholder until this is done
	QMessageBox::information(this, IDS_TITLE, tr("Cut is not implemented yet"));	
	statusBar()->message(IDS_DEFAULT_MESSAGE);
}

void QBrew::slotEditCopy()
{
	statusBar()->message(tr("Copying selection to Clipboard..."));
	// temporary placeholder until this is done
	QMessageBox::information(this, IDS_TITLE, tr("Copy is not implemented yet"));	
	statusBar()->message(IDS_DEFAULT_MESSAGE);
}

void QBrew::slotEditPaste()
{
	statusBar()->message(tr("Inserting Clipboard contents..."));
	// temporary placeholder until this is done
	QMessageBox::information(this, IDS_TITLE, tr("Paste is not implemented yet"));	
	statusBar()->message(IDS_DEFAULT_MESSAGE);
}

void QBrew::slotOptionsToolBar()
{
	// turn Toolbar on or off
	if ( main_toolbar->isVisible() ) {
		main_toolbar->hide();
		options_menu->setItemChecked( ID_OPTIONS_TOOLBAR, false);
		settings->insertSetting(IDS_SETTINGS_TOOLBAR, "FALSE");
	} else {
		main_toolbar->show();
		options_menu->setItemChecked( ID_OPTIONS_TOOLBAR, true );
		settings->insertSetting(IDS_SETTINGS_TOOLBAR, "TRUE");
	}
	statusBar()->message(IDS_DEFAULT_MESSAGE);
}

void QBrew::slotOptionsStatusBar()
{
	//turn Statusbar on or off
	if ( statusBar()->isVisible() ) {
		statusBar()->hide();
		options_menu->setItemChecked( ID_OPTIONS_STATUSBAR, false );
		settings->insertSetting(IDS_SETTINGS_STATUSBAR, "FALSE");
	} else {
		statusBar()->show();
		options_menu->setItemChecked( ID_OPTIONS_STATUSBAR, true );
		settings->insertSetting(IDS_SETTINGS_STATUSBAR, "TRUE");
	}
	statusBar()->message(IDS_DEFAULT_MESSAGE);
}

void QBrew::slotOptionsSetup()
{
	// setup dialog
	statusBar()->message(tr("Configuring ") + IDS_TITLE);
	SetupDialog* dialog = new SetupDialog(this, "SetupDialog");
	// load current settings into dialog
	dialog->setCaption(IDS_TITLE + tr(" - Configure"));
	dialog->setQBrewDir(settings->retrieveSetting(IDS_SETTINGS_QBREWDIR));
	dialog->setStyle(settings->retrieveSetting(IDS_SETTINGS_STYLE));
	dialog->setBatch(settings->retrieveSetting(IDS_SETTINGS_BATCH));
	dialog->setEfficiency(settings->retrieveSetting(IDS_SETTINGS_EFFICIENCY));
	
	int ret = dialog->exec();
	if (ret == QDialog::Accepted) {
		settings->insertSetting(IDS_SETTINGS_QBREWDIR, dialog->qbrewDir());
		settings->insertSetting(IDS_SETTINGS_STYLE, dialog->style());
		settings->insertSetting(IDS_SETTINGS_BATCH, dialog->batch());
		settings->insertSetting(IDS_SETTINGS_EFFICIENCY, dialog->efficiency());
	}
	// activate new style
	if ( styleMap.contains(settings->retrieveSetting(IDS_SETTINGS_STYLE)) ) {
		switch ( styleMap[settings->retrieveSetting(IDS_SETTINGS_STYLE)] ) {
			case STYLE_MOTIF:
				QApplication::setStyle(new QMotifStyle);
				break;
			case STYLE_PLATINUM:
				QApplication::setStyle(new QPlatinumStyle);
				break;
			case STYLE_WINDOWS:
				QApplication::setStyle(new QWindowsStyle);
				break;
			case STYLE_CDE:
				QApplication::setStyle(new QCDEStyle);
				break;
			default:
				QApplication::setStyle(new QMotifStyle);
		}
	} else {
		QApplication::setStyle(new QMotifStyle);
	}				

	statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
}

void QBrew::slotOptionsSaveSettings()
{
	statusBar()->message(tr("Saving settings..."));
	settings->saveSettings(QDir::homeDirPath() + "/" + IDS_SETTINGS_FILE, IDS_TITLE);
	statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
}

void QBrew::slotHelpContents()
{
    QString home = QString(settings->retrieveSetting(IDS_SETTINGS_QBREWDIR)) + tr(IDS_HELP_FILE);
    ( new HelpWindow(home, ".", 0, "QBrewHelp") )->show();

	statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
}

void QBrew::slotHelpAboutQt()
{
    QMessageBox::aboutQt( this, IDS_TITLE);
	statusBar()->message(IDS_DEFAULT_MESSAGE, 2000);
}

void QBrew::slotHelpAbout()
{
	QMessageBox::about(this, IDS_TITLE, IDS_APP_ABOUT);
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous slots

void QBrew::slotStatusHelpMsg(const char *text)
{
	///////////////////////////////////////////////////////////////////
	// change status message of whole statusbar temporary (text, msec)
	statusBar()->message(text, 2000);
}

void QBrew::slotDocumentModified()
{
	file_menu->setItemEnabled( ID_FILE_SAVE, true );
}

void QBrew::statusCallback(int id_)
{
	switch (id_) {
    
	case ID_FILE_NEW        :  slotStatusHelpMsg(tr("Creates a new document"));break;
	case ID_FILE_OPEN       :  slotStatusHelpMsg(tr("Opens an existing document"));break;
	case ID_FILE_SAVE       :  slotStatusHelpMsg(tr("Save the document"));break;
	case ID_FILE_SAVE_AS    :  slotStatusHelpMsg(tr("Save the document under a new name"));break;
	case ID_FILE_PRINT      :  slotStatusHelpMsg(tr("Prints the file"));break;
	case ID_FILE_QUIT       :  slotStatusHelpMsg(tr("Quits the program"));break;
	case ID_EDIT_CUT        :  slotStatusHelpMsg(tr("Cuts the selected section and puts it to the clipboard"));break;
	case ID_EDIT_COPY       :  slotStatusHelpMsg(tr("Copys the selected section to the clipboard"));break;
	case ID_EDIT_PASTE      :  slotStatusHelpMsg(tr("Pastes the clipboard contents to actual position"));break;
	case ID_OPTIONS_TOOLBAR    :  slotStatusHelpMsg(tr("Enables / disables the main Toolbar"));break;
	case ID_OPTIONS_STATUSBAR  :  slotStatusHelpMsg(tr("Enables / disables the Statusbar"));break;
	case ID_OPTIONS_SETUP	:  slotStatusHelpMsg(tr("Configures general settings for QBrew"));break;
	case ID_OPTIONS_SAVESETTINGS	:  slotStatusHelpMsg(tr("Save the options"));break;
	case ID_HELP_CONTENTS	:  slotStatusHelpMsg(tr("Displays the main help contents"));break;
	case ID_HELP_ABOUT_QT   :  slotStatusHelpMsg(tr("All about the Qt library"));break;
	case ID_HELP_ABOUT      :  slotStatusHelpMsg(tr("All about QBrew"));break;
	}
}
















































