#include "global.h"
#include "net_shm.h"


int isValidPayload(const Payload *payload)
{
  return (payload != NULL
	  /* && payload->len >= 0 */
	  && payload->len <= MAX_TEXT
	  /* && payload->current >= 0 */
	  && payload->current <= payload->len);
}

Payload* initPayload(Payload *payload)
{
  memset(payload, 0, sizeof(Payload));
  return payload;
}

Payload* rewindPayload(Payload *payload)
{
  assert(isValidPayload(payload));
  payload->current=0;
  return payload;
}

Payload* dumpPayload(FILE *f, Payload *payload)
{
  fprintf(f, "dump_payload: ");
  if (payload == NULL) {
    fprintf(f, "NULL\n");
    return payload;
  }
  else {
    int i;
    fprintf(f, "len=%d, current=%d\n", payload->len, payload->current);
    for (i=0; i<payload->len; i++) {
      int c = (u_int8) payload->content[i];
      fprintf(f, "0x%02x", c);
      if (isprint(c))
        fprintf(f, "(%c)", c);
      else
        fprintf(f, "(-)");
      if (i == payload->current-1)
        fprintf(f, ">");
      else
        fprintf(f, " ");
    }
  }
  fprintf(f, "\n");
  return payload;
}

int writePayload(Payload *payload, u_int8 *format, ...)
{
  va_list ap;

  /* Verification des arguments.  */
  if (! isValidPayload(payload)) {
    warning("writePayload: invalid Payload %s", payload);
    return -1;
  }
  if (format == NULL) {
    warning("writePayload: NULL format");
    return -1;
  }

  va_start(ap, format);
  payload->len = payload->current;   /* mode "rewrite" rather than "append" */

  /* Parse le format */
  while (*format) {
    switch (*format) {
    case 'c': /* char */
      {
	u_int8 i = va_arg(ap, int);  	
	/* A l'appel, le C passe toujours des int */
		
	payload->content[payload->current++] = 'c';
	payload->content[payload->current++] = i;
	break;
      }
    case 'h': /* short */
      {
	int16 i = htons(va_arg(ap, int));  	
	/* A l'appel, le C passe toujours des int */
		
	payload->content[payload->current++] = 'h';
	memcpy(payload->content + payload->current, &i, sizeof(int16));
	payload->current += sizeof(int16);
	break;
      }
    case 'd': /* int32 */
      {
	int32 i = htonl(va_arg(ap, int));

	payload->content[payload->current++] = 'd';
	memcpy(payload->content + payload->current, &i, sizeof(int32));
	payload->current += sizeof(int32);
	break;
      }
    case 'f': /* float */
      {
	float f = va_arg(ap, double);
	/* A l'appel, le C passe toujours des double */
	int32 * p = (int32*) &f;
	int32 j = htonl(*p);
	
	payload->content[payload->current++] = 'f';
	memcpy(payload->content + payload->current, &j, sizeof(int32));
	payload->current += sizeof(int32);
	break;
      }
    case 's': /* string */
      {
	u_int8 *s = va_arg(ap, u_int8*);
	u_int8 len = strlen(s);
	
	payload->content[payload->current++] = 's';
	/* 255 caracteres maxi, meme si MAX_TEXT permettait plus */
	payload->content[payload->current++] = len;
	/* on copie tout sauf le 0, qu'il faudra rajouter */
	memcpy(payload->content + payload->current, s, len);
	payload->current += len;
	break;
      }
    case 'n': /* NetObjectId */
      {
	NetObjectId n = va_arg(ap, NetObjectId);
		
	payload->content[payload->current++] = 'n';

	/* everything is already in network format */
#ifdef WITH_SSRC
	memcpy(payload->content + payload->current, &n.ssrc_id, sizeof(n.ssrc_id));
	payload->current += sizeof(n.ssrc_id);
#endif
	memcpy(payload->content + payload->current, &n.host_id, sizeof(n.host_id));
	payload->current += sizeof(n.host_id);
	memcpy(payload->content + payload->current, &n.port_id, sizeof(n.port_id));
	payload->current += sizeof(n.port_id);
	memcpy(payload->content + payload->current, &n.obj_id, sizeof(n.obj_id));
	payload->current += sizeof(n.obj_id);
	break;
      }
    case 't': /* timeval */
      {
	struct timeval t = va_arg(ap, struct timeval);
	u_int32 sec = htonl(t.tv_sec);
	u_int32 usec = htonl(t.tv_usec);

	payload->content[payload->current++] = 't';
	memcpy(payload->content + payload->current, &sec, sizeof(int32));
	payload->current += sizeof(int32);
	memcpy(payload->content + payload->current, &usec, sizeof(int32));
	payload->current += sizeof(int32);
	break;
      }	

    default: /* unknown type in format */
      warning("writePayload: invalid format '%c' in %s", *format, format);
      va_end(ap);
      return -1;
    }

    format++;
    if (payload->current > payload->len)
      payload->len = payload->current;
	
    /* check the length, if too long -> warning */
    if (payload->len >= MAX_TEXT) {
      warning("writePayload: Payload too long (%d bytes)", payload->len);
      /* just in case : */
      payload->len = payload->current = 0;
      va_end(ap);
      return -1;
    }
  }
  va_end(ap);
  return 0;
}    

int scanPayload(Payload *payload, u_int8 *format, ...)
{
  va_list ap;

  /* Verification des arguments */ 
  if (! isValidPayload(payload)) {
    warning("scanPayload: invalid Payload: %02x%02x%02x%02x", payload[0], payload[1], payload[2], payload[3]); 
    return -1;
  }
  if (format == NULL) {
    warning("scanPayload: NULL format"); 
    return -1;
  }

  va_start(ap, format); 

  while (*format) { 
    /* Format known ? */
    if (strchr("chdfsnt", *format) == NULL) {
      warning("scanPayload invalid format [%c] in %s", *format, format);
      format++;
      continue;
    }

    /* Test matching Payload - format */
    if (payload->content[payload->current] != *format) {
      warning("scanPayload: mismatch '%c' in payload, '%c' in format",
	      payload->content[payload->current], *format);
      payload->current = payload->len = 0; 
      format++;
      continue;
    }
    payload->current++;
	
    switch (*format) { 
    case 'c': /* char */ 
      { 
	u_int8 *p = va_arg(ap, u_int8*);
	
	memcpy(p, payload->content + payload->current, sizeof(u_int8)); 
	payload->current++;
	break; 
      } 
    case 'h': /* int16 */ 
      { 
	int16 i; 
	int16 *p = va_arg(ap, int16*);
	
	memcpy(&i, payload->content + payload->current, sizeof(int16)); 
	*p = ntohs(i);
	payload->current += sizeof(int16);
	break; 
      } 
    case 'd': /* int32 */ 
      { 
	int32 i; 
	int32 *p = va_arg(ap, int32*);
	
	memcpy(&i, payload->content + payload->current, sizeof(int32)); 
	*p = ntohl(i);
	payload->current += sizeof(int32);
	break; 
      } 
    case 'f': /* float */ 
      { 
	float *p = va_arg(ap, float*);
	/* A l'appel, le C passe par contre les float* */
	int32 i;
	
	memcpy(&i, payload->content + payload->current, sizeof(int32)); 
	i = ntohl(i);
	memcpy(p, &i, sizeof(int32));
	payload->current += sizeof(int32);
	break; 
      } 
    case 's': /* string */ 
      { 
	/* Note: il n'y a pas de controle de longueur.
	 * Il est donc sage de prevoir un u_int8[MAX_TEXT]
	 */

	u_int8 *s = va_arg(ap, u_int8*);
	u_int8 len = payload->content[payload->current++];
	
	memcpy(s, payload->content + payload->current, len);
	s[len] = 0; /* le 0 terminal */
	payload->current += len;
	break; 
      }     
    case 'n': /* NetObjectId */
      {
	NetObjectId * n = va_arg(ap, NetObjectId*);

	/* tout reste au format network */
#ifdef WITH_SSRC
	memcpy(&n->ssrc_id, payload->content + payload->current, sizeof(n->ssrc_id));
	payload->current += sizeof(n->ssrc_id);
#endif
	memcpy(&n->host_id, payload->content + payload->current, sizeof(n->host_id));
	payload->current += sizeof(n->host_id);
	memcpy(&n->port_id, payload->content + payload->current, sizeof(n->port_id));
	payload->current += sizeof(n->port_id);
	memcpy(&n->obj_id, payload->content + payload->current, sizeof(n->obj_id));
	payload->current += sizeof(n->obj_id);
	break;
      }
    case 't': /* timeval */
      {
	struct timeval *p = va_arg(ap, struct timeval*);
	u_int32 sec, usec;
	
	memcpy(&sec, payload->content + payload->current, sizeof(int32));
	payload->current += sizeof(int32);
	memcpy(&usec, payload->content + payload->current, sizeof(int32));
	payload->current += sizeof(int32);
	p->tv_sec = ntohl(sec);
	p->tv_usec = ntohl(usec);
	break;
      }	
    default:
      warning("scanPayload: format unimplemented [%c] in %s", *format, format);
      va_end(ap);
      return -1;
    }
    
    /* verify if not too far */
    if (payload->current > payload->len) {
      warning("scanPayload: past end of Payload"); 
      payload->current = payload->len = 0; 
      va_end(ap);
      return -1; 
    }
    format++;
  }
  va_end(ap); 
  return 0; 
}

void appendPayload(Payload *t, const Payload* a)
{
  if (!isValidPayload(t) || !isValidPayload(a)) {
    warning("appendPayload: invalid Payload");
  }
  memcpy(t->content + t->current, a->content, a->len);
  t->current += a->len;
  if (t->current > t->len) t->len = t->current;
} 

