#include "global.h"
#include "net_shm.h"
#include "channel.h"


/* Send a packet
 * Return result of sendto(): -1 if error, else >=0
   * TODO:
   * send queue to group messages into only one packet
 */
Public
int sendPacket(struct sockaddr_in *to, const Payload *ppload)
{
  u_int8 hdr_size, pkt[PKTSIZE], *ppl;
  rtp_hdr_t *rtp_hdr = (rtp_hdr_t *) pkt;
  int r, pkt_len, fd_send_rtp;
  struct sockaddr_in *sin_rtcp;
  Channel *pchan = channels_list;
  u_int32 hostid;
  u_int16 portid;

  assert(isValidPayload(ppload)); assert(to);
  if (ppload->len == 0) {
    trace(DBG_NET, "sendPacket: payload empty");
  }
  if (to == NULL) {
    trace(DBG_FORCE, "sendPacket: socket empty, to=%x", to);
    return -1;
  }

  hdr_size = RTP_HDR_SIZE;
  RtpCreateHeader(pchan->session, rtp_hdr);
  ppl = pkt + hdr_size;
  *ppl = ppload->len + PAYLOAD_HDR_SIZE;
  pkt_len = hdr_size + *ppl;

  /* source address first to allow using simple reflector */
  hostid = htonl(my_hostid);
  portid = htons(my_portid);
  memcpy(ppl + PAYLOAD_HDR_HOSTID, &hostid, sizeof(hostid));
  memcpy(ppl + PAYLOAD_HDR_PORTID, &portid, sizeof(portid));  
  memcpy(ppl + PAYLOAD_HDR_SIZE, ppload->content, ppload->len);

  if ((fd_send_rtp = getfdbysa(pchan, to, SEND_RTP)) <= 0) {
    trace(DBG_RTP, "sendPacket fd error: fd_send_rtp=%d, pchan->sd[RECV_RTP]=%d, pchan->sd[SEND_RTP]=%d, to=%x", fd_send_rtp, pchan->sd[RECV_RTP], pchan->sd[SEND_RTP], to);
    if (pchan->sd[SEND_RTP] != -1)
      fd_send_rtp = pchan->sd[SEND_RTP];	/* Hack !!! */
    else
      return -1;
  }
  r = sendto(fd_send_rtp, pkt, pkt_len, 0, (struct sockaddr*) to, sizeof(*to));
  if (r<0) {
    warning("sendPacket: sendto: %s fd_send_rtp=%d", strerror(errno), fd_send_rtp);
    return r;
  }
  trace(DBG_NET, "S: %d %02x%02x%02x%02x/%02x%02x",
      ppl[0], ppl[1], ppl[2], ppl[3], ppl[4], ppl[5], ppl[6]);

  bytes_sent += pkt_len;
  sum_bytes_sent += pkt_len;
  pkts_sent++;
  sum_pkts_sent++;

  /* update SR */
  pchan->session->sess_sr.psent++;	/* TO CHANGE */
  pchan->session->sess_sr.osent += pkt_len;	/* TO CHANGE */

  if ((pkts_sent % 50) == 0) {	/* send a SR */
    if ((sin_rtcp = getsabysa(pchan, to, SOCK_RTCP)) == NULL) {
      trace(DBG_RTP, "sendSR error: &(pchan->sin_mcast_rtcp)=%x, pchan->sa[0]=%x, pchan->sa[1]=%x", &(pchan->sin_mcast_rtcp), pchan->sa[0], pchan->sa[1]);
      return -1;
    }
    sendRTCPPacket(pchan, sin_rtcp, RTCP_SR);
  }
  
  if ((pkts_sent % 100) == 0) {	/* send a SDES */
    RtpRefreshSDES(pchan->session);	/* TO CHANGE */
    sendRTCPPacket(pchan, sin_rtcp, RTCP_SDES);
  }
  return r;
}

/*
 * Send a RTCP packet
 */
Public
int sendRTCPPacket(Channel *pchannel, struct sockaddr_in *to, u_int8 pt)
{
  u_int8 pkt[PKTSIZE];
  rtcp_common_t rtcp_hdr;
  struct timeval ts;
  int pkt_len = 0, r, fd_send_rtcp;
  u_int32 timestamp;

  gettimeofday(&ts, NULL);
  timestamp = htonl(ts.tv_sec*1000 + ts.tv_sec/1000);

  /* common part of rtcp packet */
  memset(&rtcp_hdr, 0, sizeof(rtcp_common_t));	/* clear P:1, count:5 */
  rtcp_hdr.version = RTP_VERSION;
  rtcp_hdr.pt = pt;

  switch (pt) {

    case RTCP_SR: {
      rtcp_sr_t sr;

      rtcp_hdr.count = 0;
      sr = pchannel->session->sess_sr;
      sr.ssrc = htonl(my_ssrcid);
      rtcp_hdr.length = htons(sizeof (rtcp_sr_t) >>2);
      /* divided by 2 or 4 number of words = n * 32 bits? */
      memcpy(pkt, &rtcp_hdr, sizeof(rtcp_common_t));
      pkt_len += sizeof(rtcp_common_t);
      memcpy(pkt + pkt_len, &sr, sizeof(rtcp_sr_t));
      pkt_len += sizeof(rtcp_sr_t);
      sr.rtp_ts = timestamp;
      trace(DBG_RTP, "send SR: ssrc=%x pkt_len=%d", sr.ssrc, pkt_len);
    }
    break;

    case RTCP_SDES: {
      sdes_item *psd;
      u_int32 ssrc = htonl(my_ssrcid);
      u_int8 lgsd = 0;
      u_int8 pktsd[PKTSIZE];

      memset(pktsd, 0, PKTSIZE);
      rtcp_hdr.count = 1; /* Only one SDES with multiple chunks */
      for (psd = pchannel->session->sess_sdes; psd != NULL; psd = psd->next) {
        pktsd[lgsd++] = psd->type;
        pktsd[lgsd++] = psd->len;
        memcpy(pktsd + lgsd, psd->content, psd->len);
        lgsd += psd->len;
      }
      /* End of chunks is indicated by a zero */
      pktsd[lgsd++] = 0;
      /* Pad to 32 bytes and store zeros there */
      while (lgsd % 4) pktsd[lgsd++] = 0;
      rtcp_hdr.length = htons(1 + (lgsd >> 2));
      memcpy(pkt, &rtcp_hdr, sizeof(rtcp_common_t));
      pkt_len += sizeof(rtcp_common_t);
      memcpy(pkt + pkt_len, &ssrc, sizeof(u_int32));
      pkt_len += sizeof(u_int32);
      memcpy(pkt + pkt_len, pktsd, lgsd);
      pkt_len += lgsd;
      trace(DBG_RTP, "send SDES: ssrc=%x pkt_len=%d", my_ssrcid, pkt_len);
    }
    break;

    case RTCP_BYE: {
      u_int32 ssrc = htonl(my_ssrcid);

      rtcp_hdr.count = 1; /* me only says bye */
      rtcp_hdr.length = htons(1);
      memcpy(pkt, &rtcp_hdr, sizeof(rtcp_common_t));
      pkt_len = sizeof(rtcp_common_t);
      memcpy(pkt + pkt_len, &ssrc, sizeof(u_int32));
      pkt_len += sizeof(u_int32);
      trace(DBG_RTP, "send BYE: ssrc=%x pkt_len=%d", my_ssrcid, pkt_len);
    }
    break;
  }

  if ((fd_send_rtcp = getfdbysa(pchannel, to, SEND_RTCP)) < 0) {
    trace(DBG_FORCE, "sendRTCPPacket fd error: fd_send_rtcp=%d, pchannel->sa[0]=%x, pchannel->sa[1]=%x, to=%x", fd_send_rtcp, pchannel->sa[0], pchannel->sa[1], to);
    return -1;
  }
  trace(DBG_RTP, "fd_send_rtcp=%d pktlen=%d to=%x", fd_send_rtcp, pkt_len, to);

  r = sendto(fd_send_rtcp, pkt, pkt_len, 0, (struct sockaddr*)to, sizeof(*to));
  if (r<0)
    warning("sendRTCPPacket sendto: %s", strerror(errno));
  return r; 
}


/*
 * Receive a packet
 *
 * put datas into the Payload and sender into from
 * -1 if errorr, else >=0
   * TODO:
   * decoding multi-messages packets
 */
Public
int recvPacket(int fd, struct sockaddr_in *from, Payload *ppload)
{
  u_int8 hdr_size, pkt[PKTSIZE], *ppl;
  rtp_hdr_t *rtp_hdr = (rtp_hdr_t *) pkt;
  rtcp_common_t *rtcp_hdr = (rtcp_common_t *) pkt;
  int pkt_len, l = sizeof(struct sockaddr_in);
  source_info *psi;
  source *s;
  Channel *pchan = channels_list;

  assert(from);
  memset(from, 0, sizeof(*from));
  /* recv rather than recvfrom because reflectors */
#ifdef REFLECTOR
  if ((pkt_len = recv(fd, pkt, sizeof(pkt), 0)) < 0) {
#else
  if ((pkt_len = recvfrom(fd, pkt, sizeof(pkt), 0, (struct sockaddr *)from, &l)) <0) {
#endif
    warning("recv: %s on %d", strerror(errno), fd);
    trace(DBG_NET, "recvPacket: %s on %d", strerror(errno), fd);
    return pkt_len;
  }
  bytes_recvd += pkt_len;
  sum_bytes_recvd += pkt_len;
  pkts_recvd++;
  sum_pkts_recvd++;

  if (rtp_hdr->version == RTP_VERSION && rtp_hdr->pt == PAYLOAD_TYPE) {
    /* RTP packet */
    hdr_size = RTP_HDR_SIZE;
    if ((psi = RtpGetSinfo(ntohl(rtp_hdr->ssrc))) != NULL) {
      psi->rr.lost += ntohs(rtp_hdr->seq) - psi->rr.last_seq - 1;
      psi->rr.last_seq = ntohs(rtp_hdr->seq);
      s = &(psi->s);
      RtpUpdateSeq(s, ntohs(rtp_hdr->seq));
      psi->extended_max = s->cycles + s->max_seq;
      psi->expected = psi->extended_max - s->base_seq + 1;
      psi->lost = psi->expected - s->received;	/* number of packets lost */
#ifdef DEBUG
      trace(DBG_RTP, "seq=%x received=%x expected=%x lost=%x extented=%x max_seq=%x", rtp_hdr->seq, s->received, psi->expected, psi->lost, psi->extended_max, s->max_seq);
#endif
    }
  }
  else if (rtcp_hdr->version == RTP_VERSION) { /* RTCP packet */
    switch (rtcp_hdr->pt) {
      case RTCP_SR:
      case RTCP_SDES:
      case RTCP_BYE:
      case RTCP_APP:
        recvRTCPPacket(pchan, from, pkt, pkt_len);
      case RTCP_RR: /* why send a RR ? we only send SR */
      default:
        return 0;
    }
  }
  else { /* old version of VREng <= 1.5.8 */
    hdr_size = 0;
  }

  ppl = pkt + hdr_size;

  trace(DBG_NET, "R: %d %02x%02x%02x%02x/%02x%02x",
      ppl[0], ppl[1], ppl[2], ppl[3], ppl[4], ppl[5], ppl[6]);

  if (pkt_len != *ppl + hdr_size || *ppl < PAYLOAD_HDR_SIZE) {
    warning("recvPacket: wrong size, pkt_len=%d *ppl=%d hdr_size=%d", pkt_len, *ppl, hdr_size);
    trace(DBG_FORCE, "R: %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
      pkt[0], pkt[1], pkt[2], pkt[3], pkt[4], pkt[5], pkt[6],
      pkt[7], pkt[8], pkt[9], pkt[10], pkt[11]);
    trace(DBG_FORCE, "R: %d %02x%02x%02x%02x/%02x%02x",
      ppl[0], ppl[1], ppl[2], ppl[3], ppl[4], ppl[5], ppl[6]);
    trace(DBG_NET, "wrong packet size, pkt_len=%d *ppl=%d", pkt_len, *ppl);
    return -1;
  }

  from->sin_family = AF_INET;
  memcpy(&from->sin_addr.s_addr, ppl + PAYLOAD_HDR_HOSTID, 4);
  memcpy(&from->sin_port, ppl + PAYLOAD_HDR_PORTID, 2);
  initPayload(ppload);
  ppload->len = *ppl - PAYLOAD_HDR_SIZE;
  memcpy(ppload->content, ppl + PAYLOAD_HDR_SIZE, ppload->len);
  return (pkt_len);
}

Public
int recvRTCPPacket(Channel *pchannel, struct sockaddr_in *from, u_int8 *pkt, int pkt_len)
{
  rtcp_common_t rtpc_hdr;
  u_int16 len;
  u_int32 ssrc;

  /* we are supposed to receive compounds, but bye now we get a packet */
  memcpy(&rtpc_hdr, pkt, sizeof(rtcp_common_t));
  pkt += sizeof(rtcp_common_t);
  len = ntohs(rtpc_hdr.length);

  switch (rtpc_hdr.pt) {

    case RTCP_SR: {
      rtcp_sr_t sr;
      source_info *psi;

      trace(DBG_RTP, "Got a SR: len=%d", len<<2);
      memcpy(&sr, pkt, sizeof(rtcp_sr_t));
      if ((psi = RtpGetSinfo(ntohl(sr.ssrc))) != NULL)
        psi->sr = sr;
    }
    break;

    case RTCP_SDES: {
      int i = 0;
      sdes_item *sitem;
      u_int8 *pktmp = pkt;
      source_info *psi;
      u_int32 ssrc = 0;

      trace(DBG_RTP, "Got a SDES: len=%d", len<<2);
      memcpy(&ssrc, pktmp, sizeof(u_int32));
      pktmp += sizeof(u_int32);
      if ((psi = RtpGetSinfo(ntohl(ssrc))) != NULL) {
        sitem = &(psi->sdes);	/* first sitem */ 

        for (i = 0; i < rtpc_hdr.count; i++) {
          /* trace(DBG_RTP, "SDES: %x %x", psi, sitem); */
          sitem->type = pktmp[0];
          sitem->len = pktmp[1];
          if (sitem->content == NULL) {	/* free asumption */
            sitem->content = (char *) malloc(sitem->len + 1);
            memcpy(pktmp+2, sitem->content, sitem->len);
          }
          sitem->content = pktmp+2;
          sitem->content[sitem->len] = 0; /* string is null terminated */
          pktmp += sitem->len + 2;
          trace(DBG_RTP, "SDES: %s", sitem->content);
          if ( i < rtpc_hdr.count) {	/* need next */
            if (sitem->next == NULL) {
              sitem = sitem->next = (sdes_item *) malloc(sizeof(sdes_item));
              sitem->next = NULL;
            }
            else {
              sitem = sitem->next;
            }
          }
        }
      }
    }
    break;

    case RTCP_BYE: {
      rtcp_t bye;
      u_int32 ssrc = 0;

      memcpy(&ssrc, pkt, sizeof(u_int32));
      trace(DBG_RTP, "Got a BYE: ssrc=%x len=%x", ssrc, len);
      RtpDeleteSinfo(pchannel->session, ntohl(ssrc));
    }
    break;

    case RTCP_APP:
      trace(DBG_RTP, "Got a APP");
    default:
      break;
  }
}
