/*******************************************************************/
/* Les fonctions suivantes sont utilisees pour gerer des solides 
 * disposes dans le monde. On n'a pas specifie ici :
 * - la gestion des sources lumineuses et des modeles lumineux
 * - la gestion des hierarchies d'objets
 * - les fonctions d'entree/sortie pour les geometries et les scenes
 * - comment gerer plusieurs mondes ou affichages
 */

#ifndef ZLIB_H
#define ZLIB_H

#include <zmath.h>


/******************************************************************
 * Fonctions de gestion des solides
 ******************************************************************/

/* handle sur un solide */
typedef struct ZVSolid * Solid;

/*
 * Retourne True si les deux solides 's1' et 's2' s'intersectent
 * Non implemente pour l'instant: retourne 'true'
 */
int SolidIntersect(Solid s1, Solid s2);

/*
 * Retourne une bounding box pour le solide 's'. La bounding box est
 * alignee avec le repere du monde
 */
void SolidGetBB(Solid s, V3 *center, V3 *size);

/*
 * Modifie la position du solide 's'. La position est specifiee par la 
 * matrice 4x4 'mpos'. La matrice de position donne la transformation 
 * a effectuer sur les coordonnees locales du solide pour l'amener dans les
 * coordonnees du monde.
 */ 
void SolidSetPosition(Solid s, M4 *mpos);

/* Change la "frame" de l'objet (ie sa gomtrie) */
void SolidSetFrame(Solid s, int frame) ;

/* Pour les murs */
int Solid_intersect_Walls(V3 *bb_center, V3 *bb_size, V3 *norm) ;

/*
 * Lecture de la position du solide 's'
 */
void SolidGetPosition(Solid s, M4 *mpos);

/*
 * Rends le solide 's' visible (flag=True) ou invisible (flag=False). Un
 * solide est visible par defaut lors de sa creation.
 */
void SolidSetVisibilty(Solid s, int flag);

/*
 * Creation d'un nouveau solide. La chaine 'id' donne la geometrie du solide.
 * Le handle du nouveau solide est retourne. Le fichier 'zv/doc.txt' documente
 * le format de la chaine 'geom'.
 */
Solid SolidParser(char *geom);

/*
 * Le solide 's' est efface du monde
 */
void SolidDelete(Solid s);

/*
 * Ajout d'information complementaires au solide 's'
 */
void SolidSetOptBuf(Solid s, char *buf, int size);

/* 
 * Lecture des informations complementaires associees a 's'. 'size' indique
 * la taille maximale du buffer.
 */
int SolidGetOptBuf(Solid s, char *buf, int size);

/******************************************************************
 * Fonctions de gestion de l'affichage
 ******************************************************************/

/*
 * Initialisation de l'affichage 3d
 * quality = 0 : normal 3D quality
 * quality = 1 : high 3D quality
 */
void RenderInit(int quality);

/*
 * Affichage de la scene courante dans le buffer courant
 */
void Render(void);

/*
 * Fin de l'affichage 3d
 */
void RenderClose(void);

/* Matrice donnant la position de la camera dans le monde.  Elle
 * specifie la transformation orthogonale a effectuer des coordonnes
 * mondes vers les coordonees ecran. Dans les coordonnees ecran, l'axe
 * z est normal a l'ecran et dirige vers l'observateur et l'axe x est
 * dirige de la gauche vers la droite. Le repere est direct.  (Pas
 * totalement implemente) */

void ZLib_setCameraPosition(M4 *cpos);

/*
 * Dfinition de la projection.
 * fovy : largeur de champ (en degrs)
 * Near : distance du plan de projection  l'oeil
 * Far  : distance  l'oeil du plan de clipping arrire
 */
void ZLib_setCameraProjection(float Fovy, float Near, float Far);
/*
 * Parametres indiquant la position du buffer d'affichage dans la fenetre
 * d'affichage (qui se trouve a l'adresse 'buf'). 
 * Note: 'xsize' et 'x0' doivent etre pairs.
void ZLib_setFrameBuffer(char *buf, int x0, int y0, int xsize, int ysize);
 */

/*
 * Taille du buffer d'affichage.
 * Note: 'xsize' doit etre un multiple de 16.
 */
void RenderSetWindow(int xsize, int ysize);

/*
 * Retourne le handle du solide s'affichant en (x,y) sur l'ecran. On utilise
 * ici les coordonnees DANS LE ZBUFFER et non dans le fenetre d'affichage.
 */
Solid RenderGetSelection(int x, int y);


extern int count_triangles;
extern int count_triangles_textured;
extern int count_pixels;

#endif  /* ZLIB_H */
