/*
 *  Copyright (C) 1998,9 by Marco G"otze.
 *
 *  This code is part of the wmpinboard source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#include <stdlib.h>
#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "features.h"
#include "misc.h"
#include "wmpinboard.h"
#include "xmisc.h"
#include "notes.h"

/*
 * adds a new note and returns its number (or -1 if notes limit exceeded)
 */
int
add_note()
{
  int cols[C_NUM];
  int rep = 20;
  int i, j, n;

  if (notes_count >= MAX_NOTES)
    return -1;
  n = notes_count++;

  /* find a reasonable color */
  for (i = 0; i < C_NUM; i++) cols[i] = 1;
  for (i = 0; i < n; i++) cols[ndata[i].col] = 0;
  for (i = j = 0; i < C_NUM; i++)
    if (cols[i]) cols[j++] = i;
  ndata[n].col = j ? cols[rand() % j] : rand() % C_NUM;

  /* choose a random place to pin it, some pixels off any other note's origin
     (possible if MAX_NOTES is sufficiently small) */
  do {
    ndata[n].x =  6 + rand() % 36;
    ndata[n].y = 12 + rand() % 34;  /* avoid "TO DO" label */
    j = 1;
    for (i = 0; i < n; i++)
      if (ndata[i].x - ndata[n].x < 5 && ndata[i].x - ndata[n].x > -5 &&
          ndata[i].y - ndata[n].y < 5 && ndata[i].y - ndata[n].y > -5)
      {
        j = 0;
        break;
      }
  } while (!j && --rep);

  memset(ndata[n].text, 32, 59);
  ndata[n].text[59] = '\0';
  ndata[n].cursor = 0;
  memset(ndata[n].sketch, 0, 512);
  memset(ndata[n].creases, 0, 32);

  return n;
}

/*
 * removes a note
 */
void
remove_note(int number)
{
  int i;

  if (number >= notes_count) return;
  /* in case memcpy() can't do overlapping copies... */
  for (i = number; i < notes_count-1; i++)
    memcpy(&ndata[i], &ndata[i+1], sizeof(data_t));
  notes_count--;
}

/*
 * moves a note on top of all others and returns the quote's new ID
 */
int
raise_note(int number)
{
  data_t tmp;
  int i;

  if (number >= notes_count-1 || number < 0) return number;
  memcpy(&tmp, &ndata[number], sizeof(data_t));
  /* in case memcpy() can't do overlapping copies... */
  for (i = number; i < notes_count-1; i++)
    memcpy(&ndata[i], &ndata[i+1], sizeof(data_t));
  memcpy(&ndata[notes_count-1], &tmp, sizeof(data_t));
  return notes_count-1;
}

/*
 * returns a value corresponding to how tilted a note is to be displayed
 */
int
note_tilt(int note)
{
  return ndata[note].x < 14 ? 0 : (ndata[note].x < 36 ? 1 : 2);
}

/*
 * returns true if the note in question is to be considered empty
 */
int
note_empty(int note)
{
  char *ptr = ndata[note].sketch;
  int i;

  if (!string_empty(ndata[note].text, 0)) return 0;
  for (i = 0; i < 511; i++, ptr++)  /* last byte ignored */
    if (*ptr) return 0;
  return 1;
}

/*
 * creates templates for the icons suiting the specified note
 */
void
color_notes(int note)
{
  XGetSubImage(display, app, 64, 0, 16, 48, ~0, ZPixmap, img, 64, 0);
  if (ndata[note].sketch[511] & 1)
    replace_color(img, 64, 0, 16, 48, C_EXTRA, palette[ndata[note].col].fg);
  else
    replace_color(img, 64, 0, 16, 48, C_EXTRA, palette[ndata[note].col].bg);
  replace_color(img, 64, 0, 16, 48, C_INNER, palette[ndata[note].col].bg);
#ifdef CREASES
  render_wear(note);
#endif
}

/*
 * draws a note's miniature representation on the pinboard
 */
void
pin_note(int note)
{
  merge_masked(img, 64, 16*note_tilt(note), ndata[note].x,
    ndata[note].y, 16, 16, C_OUTER);
}

/*
 * renders the pinboard with all of its notes excluding <exclude> if >= 0
 */
void
render_pinboard(int exclude)
{
  int i;

  XPutImage(display, app, normalGC, img, 0, 0, 0, 0, 64, 64);
  for (i = 0; i < notes_count; i++)
    if (i != exclude) {
      color_notes(i);
      pin_note(i);
    }
}

/*
 * if (<x>, <y>) is within a quote's area, returns the quote's ID, otherwise -1
 */
int
selected_note(int x, int y)
{
  int i;

  for (i = notes_count-1; i >= 0; i--)
    if (x >= ndata[i].x && x < ndata[i].x+16 &&
        y >= ndata[i].y && y < ndata[i].y+16 &&
        XGetPixel(img, 128+x-ndata[i].x, note_tilt(i)*16+y-ndata[i].y) !=
          C_OUTER)
    {
      return i;
    }
  return -1;
}

/*
 * prints a letter identified by a note and a position in its text string, and
 * overlays it by a possible sketch if sketch has a true value
 */
void
print_letter(int note, int letter, int sketch)
{
  int a = 2+6*(letter%10), b = 2+10*(letter/10);


  if (sketch) {
    XFillRectangle(display, app, fillGC, a, b, 6, 10);
#ifdef CREASES
    render_edit_wear_area(note, a, b, 6, 10);
#endif
    draw_sketch_area(note, a, b, 6, 10);
  }
  XDrawString(display, app, fontGC, a, b + font->ascent,
    &ndata[note].text[letter], 1);
}

/*
 * prints a note's entire text *without* drawing a possibly existing sketch
 */
void
print_text(int note)
{
  int i;

  for (i = 0; i < 59; i++) print_letter(note, i, 0);
}

/*
 * shifts part of a note's text to the right, inserting <ch> (step >= 1)
 */
void
shift_string(int note, int at, int margin, int step, int ch)
{
  int i;

  if (step < 1) return;

  for (i = margin; i >= at+step; i--) {
    ndata[note].text[i] = ndata[note].text[i-step];
    print_letter(note, i, 1);
  }
  for (i = at; i < at+step; i++) {
    ndata[note].text[i] = ch;
    print_letter(note, i, 1);
  }
}

/*
 * draws a rectangular section of a sketch, according to the data saved along
 * with a note
 */
void
draw_sketch_area(int note, int x, int y, int w, int h)
{
  int i, j;

  for (i = x; i < x+w; i++)
    for (j = y; j < y+h; j++)
      if (ndata[note].sketch[8*j + i/8] & (1<<(i%8)))
        XDrawPoint(display, app, fontGC, i, j);
}

/*
 * returns the number of the button bar's button corresponding to the given
 * coordinates, or -1
 */
int
bbar_button(int x, int y)
{
  int i;

  for (i = 0; i < 8; i++)
    if (x >= 4 + (i%4)*14 && x < 18 + (i%4)*14 && y >= 32 + (i/4)*14 &&
      y < 46 + (i/4)*14)
    {
      return i;
    }
  return -1;
}

/*
 * displays note <note> in edit mode view
 */
void
render_note(int note)
{
  int i;

  XSetForeground(display, fillGC, palette[ndata[note].col].bg);
  XSetForeground(display, fontGC, BlackPixel(display, DefaultScreen(display)));
  XSetBackground(display, fontGC, palette[ndata[note].col].bg);
  set_mask(0);
  XFillRectangle(display, app, fillGC, 0, 0, 64, 64);
  XDrawRectangle(display, app, fontGC, 0, 0, 63, 63);
  XSetForeground(display, fontGC, palette[ndata[note].col].fg);
  for (i = 55; i < 62; i += 2)  /* draw triangle in the bottom right corner */
    XDrawLine(display, app, fontGC, i, 62, 62, i);
}

/*
 * prepares edit mode for note <note> (does NOT update the display)
 */
void
init_edit_mode(int note)
{
#ifdef CREASES
  XGCValues gcv;

  gcv.foreground = palette[ndata[note].col].cr;
  XChangeGC(display, creaseGC, GCForeground, &gcv);
#endif
  render_note(note);
#ifdef CREASES
  render_edit_wear(note);
#endif
  print_text(note);
  draw_sketch(note);
}

/*
 * converts <note>'s contents to what's referred to as "cooked" format (as
 * opposed to raw data); returns a malloc()ed string that has to be freed
 * later on!
 */
char*
cook(int note, int pos, int len)
{
  char *s, *t;
  int i;

  s = smalloc(70);
  for (t = s, i = pos; i < (pos+len > 59 ? 59 : pos+len); i++) {
    if (!string_empty(&ndata[note].text[i], (i >= 50 ? 59 : 10*(i/10+1))-i))
    {
      if (t > s && !(i%10) && t[-1] != ' ' && t[-1] != '-')
        *t++ = ' ';  /* replace virtual newlines by spaces */
      if ((t > s && (ndata[note].text[i] != ' ' || t[-1] != ' ') &&
        !(t[-1] == '-'
        && string_empty(&ndata[note].text[10*(i/10)], i%10+1))) ||
        (t == s && ndata[note].text[i] != ' '))
      {
        *t++ = ndata[note].text[i];
      }
    }
  }
  *t = '\0';
  return s;
}

/*
 * dumps all notes' contents on STDOUT (raw dump unless <cooked> is true)
 */
void
dump_notes(int cooked)
{
  char *s;
  int c = 0, i;

#ifndef LOWESTCOLOR
  for (; c < 4; c++)
#endif
    for (i = 0; i < notes_count; i++)
      if (c_group[ndata[i].col] == c) {
        s = cooked ? cook(i, 0, 59) : ndata[i].text;
        printf("#%02d: %s\n", i, s);
        if (cooked) free(s);
      }
}

/*
 * adds the (linear) number of the character (<x>, <y>) belongs to on a note;
 * disallowing char #59 if <strict>
 */
int
char_at(int x, int y, int strict)
{
  int i;

  if (x < 2) x = 2;
  if (x > 62) x = 62;  /* intentional! */
  if (y > 61) y = 61;
  if (y < 2) y = 2;
  i = (x-2)/6 + ((y-2)/10)*10;
  if (i > 59) i = 59;
  return i;
}

#ifdef CREASES
/*
 * applies random wear & tear to a note (the default probability is multiplied
 * by <factor>)
 */
void
wear_note(int note)
{
  /* prefabricated crease patterns */
  static const int pats = 8;
  static const unsigned char pat[8][5][5] = {
    { { 1,0,0,0,0 },
      { 0,1,0,0,0 },
      { 0,0,1,0,0 },
      { 0,0,0,1,0 },
      { 0,0,0,0,1 } },
    { { 0,0,0,0,1 },
      { 0,0,0,1,0 },
      { 0,0,1,0,0 },
      { 0,1,0,0,0 },
      { 1,0,0,0,0 } },
    { { 0,0,0,1,0 },
      { 0,0,0,1,0 },
      { 0,0,1,0,0 },
      { 1,1,0,0,0 },
      { 0,0,0,0,0 } },
    { { 0,1,0,0,0 },
      { 0,1,0,0,0 },
      { 0,0,1,0,0 },
      { 0,0,0,1,1 },
      { 0,0,0,0,0 } },
    { { 0,0,0,0,0 },
      { 0,0,0,1,1 },
      { 0,0,1,0,0 },
      { 0,1,0,0,0 },
      { 0,1,0,0,0 } },
    { { 0,0,0,0,0 },
      { 1,1,0,0,0 },
      { 0,0,1,0,0 },
      { 0,0,0,1,0 },
      { 0,0,0,1,0 } },
    { { 0,0,0,0,0 },
      { 0,1,0,0,0 },
      { 0,0,1,0,0 },
      { 0,0,0,1,0 },
      { 0,0,0,0,0 } },
    { { 0,0,0,0,0 },
      { 0,0,0,1,0 },
      { 0,0,1,0,0 },
      { 0,1,0,0,0 },
      { 0,0,0,0,0 } }
  };
  int i, j, k = 0;

  /* determine number of crease bits */
  for (i = 0; i < 32; i++)
   for (j = 0; j < 8; j++)
     k += ndata[note].creases[i]>>(j) & 1;
  /* don't cover more than 35% of a note's area with creases */
  if (100*k/256 < 35 && !(rand() % (4+k/64))) {
    int x, y;
    int t = rand() % pats;  /* choose a prefab crease pattern to apply... */
    int a = rand() % 11;    /* ...as well as its location */
    int b = rand() % 11;

    for (y = 0; y < 5; y++)
      for (x = 0; x < 5; x++)
        if (pat[t][y][x])
          ndata[note].creases[2*(b+y) + (a+x<8)] |= 1<<(7-(a+x)%8);
  }
}

/*
 * makes an otherwise pre-rendered note look worn out
 */
void
render_wear(int note)
{
  int x, y, z;

  for (z = 0; z <= 48; z += 16)
    for (y = 0; y < 16; y++)
      for (x = 0; x < 16; x++)
        if (ndata[note].creases[2*y+x/8]>>(7-x%8) & 1 &&
          XGetPixel(img, 64+x, z+y) == palette[ndata[note].col].bg)
        {
          XPutPixel(img, 64+x, z+y, palette[ndata[note].col].cr);
        }
}

/*
 * renders wear & tear in edit mode in a specific window
 */
void
render_edit_wear_area_win(Window win, int note, int a, int b, int w, int h)
{
#define m(x, y) (m[2*(y) + (x)/8]>>(7-(x)%8) & 1)
#define l(x, y) (l[8*(y) + (x)/8]>>(7-(x)%8) & 1)
  static unsigned char l[(64/8)*64];  /* high res crease map */
  static unsigned char m[32];         /* low res crease map */
  int x, y, z;

  /* lazily render a high-res crease pattern */
  if (memcmp(m, ndata[note].creases, sizeof(m))) {
    memcpy(m, ndata[note].creases, sizeof(m));
    memset(l, 0, sizeof(l));
    /* now we have to somehow extrapolate a 64x64 crease pattern from a 16x16
       sketch... */
    for (z = 0; z < 3; z++)  /* three passes */
      for (y = 2; y < 62; y++)
        for (x = 2; x < 62; x++) {
          int sx = x/4, sy = y/4;
          int xx = x%4, yy = y%4;
          int xi = (xx<<1)%3, yi = (yy<<1)%3;
          int xd = xx<2 ? -1 : 1, yd = yy<2 ? -1 : 1;

          if (z < 2 && (  /* during passes #0 and #1 */
            (!z && xi && yi && m(sx, sy) &&
              (m(sx+xd, sy+yd) || m(sx, sy+yd) || m(sx+xd, sy) ) &&
              ((xd<0 && sx<2) || (xd>0 && sx>14) || (yd<0 && sy<2) ||
               (yd>0 && sy>14) || m(sx+2*xd, sy+2*yd))
            ) || (z && (
              (!xi && !yi && l(x-xd, y-yd) && l(x+2*xd, y+2*yd)) ||
              (!xi && yi && l(x-xd, y) && l(x+2*xd, y)) ||
              (xi && !yi && l(x, y-yd) && l(x, y+2*yd))
            ))))
          {
            l[8*y + x/8] |= 1<<(7-x%8);
          }
          /* during pass #2, remove isolated dots */
          if (z == 2 && l(x, y) && !(l(x-1, y-1) || l(x, y-1) ||
            l(x+1, y-1) || l(x-1, y) || l(x+1, y) || l(x-1, y+1) ||
            l(x, y+1) || l(x+1, y+1)))
          {
            l[8*y + x/8] &= 0xff - (1<<(7-x%8));
          }
        }
  }
  for (y = b; y < b+h; y++)
    for (x = a; x < a+w; x++)
      if (x > 1 && y > 1 && x < 62 && y < 62 && x+y <= 116 && l(x, y))
        XDrawPoint(display, win, creaseGC, x, y);
#undef m
#undef l
}

/* the algorithm previously used...
void
render_edit_wear_area_win(Window win, int note, int a, int b, int w, int h)
{
  static char m[16][16];
  static unsigned char prev[32];
  int x, y, i, j;

  if (memcmp(prev, ndata[note].creases, 32)) {
    memcpy(prev, ndata[note].creases, 32);
    for (y = 0; y < 16; y++)
      for (x = 0; x < 16; x++)
        m[x][y] = prev[2*y + x/8]>>(7-x%8) & 1;
  }
  for (y = b; y < b+h; y++)
    for (x = a; x < a+w; x++)
      if (x > 1 && y > 1 && x < 62 && y < 62 && x+y <= 116) {
        int sx = x/4, sy = y/4;
        if (m[sx][sy]) {
          int xx = x%4, yy = y%4;
          int xi = xx&1 != (xx>>1)&1, yi = yy&1 != (yy>>1)&1;
          int xd = xx<2 ? -1 : 1, yd = yy<2 ? -1 : 1;

          if (
              (m[sx+xd][sy+yd] &&
                (
                  (!xi && !yi && m[sx+xd][sy] && m[sx][sy+yd]) ||
                  (!xi && yi && m[sx+xd][sy]) ||
                  (xi && !yi && m[sx][sy+yd])
                )
              ) ||
              (m[sx+xd][sy] + m[sx+xd][sy+yd] + m[sx][sy+yd] == 1)
            )
          {
            XDrawPoint(display, win, creaseGC, x, y);
          }
        }
      }
}
*/
#endif

#ifdef FUNSTUFF
extern unsigned int state_bits;

/*
 * atek sparppoirta ecaitnoo  npsceai lcoacisno,sr terusnt ur efit ah tahppnede
 */
void
check_occasion(int d, int m, int y)
{
  static const unsigned char sketch0[256] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x87, 0x3d, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xe7, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xaf, 0x4d, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xb3, 0x5d, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xbe, 0x0d, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0x1f, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x18, 0x78, 0x9f, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0xfe, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x60, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x30, 0xf0, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xf0, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xe0, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x3e, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
  };
  static const unsigned char sketch1[512] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x07, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0a, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x0a, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x0a, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x1b, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x12, 0x30, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x66, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x48, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x6f, 0x61, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x97, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x90, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xf8, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xc0, 0x0d, 0xc0, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x80, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x60, 0x40, 0x80, 0x02,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x80, 0x03, 0x02,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x08, 0x56, 0x06,
    0x00, 0x00, 0x00, 0x00, 0xe0, 0xf8, 0x1f, 0x06,
    0x00, 0x00, 0x00, 0x00, 0xf8, 0x6f, 0xfd, 0x07,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x01, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x07, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01
  };
  char *s;
  int note;

  if (!notes_count && !(state_bits & 1)) {  /* ewclmo een wsures */
    if ((note = add_note()) >= 0) {
#ifdef LOWCOLOR
      ndata[note].col = 8;
#else
      ndata[note].col = 15;
#endif
      memcpy(&ndata[note].sketch[256], sketch0, 256);
      s = csarbmel(
        "EWCLMO EotmwipbnaodrhTnaskf rortiygni  tuo!t               ");
      strcpy(ndata[note].text, s);
      free(s);
      ndata[note].cursor = 50;
      state_bits |= 1;
    }
  } else if (notes_count) state_bits |= 1;  /* purgdadeu essrw not'g tet ihs */

  if (m == 12 && d >= 25 && d <= 26) {  /* mXsa */
    if (!(state_bits & 2) && (note = add_note()) >= 0) {
#ifdef LOWCOLOR
      ndata[note].col = 8;
#else
      ndata[note].col = 14;
#endif
      memcpy(ndata[note].sketch, sketch1, 512);
      s = csarbmel(
        "          A M REYR    MXSA      T  O    Y UO !             ");
      strcpy(ndata[note].text, s);
      free(s);
      state_bits |= 2;
    }
  } else state_bits &= ~2;

  if (m == 1 && d <= 3) {  /* eN weYra */
    if (!(state_bits & 4) && (note = add_note()) >= 0) {
#ifdef LOWCOLOR
      ndata[note].col = 1;
#else
      ndata[note].col = 11;
#endif
      ndata[note].sketch[511] = 0x01;
      s = csarbmel(
        "nOeca agni aeyrah saapssde.. .A ynaw,y  A H PAYP  EN WEYRA!");
      strcpy(ndata[note].text, s);
      free(s);
      state_bits |= 4;
    }
  } else state_bits &= ~4;
}
#endif

