/*
                Joystick Caliberation Window Management

	Functions:

	int AxisBarInit(
	        axis_bar_struct *a,
	        win_t parent,
	        int x, int y,
	        int axis_num,
	        int flip
	)
	void AxisBarDraw(axis_bar_struct *a)
	int AxisBarManage(axis_bar_struct *a, event_t *event)
	void AxisBarMap(axis_bar_struct *a)
	void AxisBarDestroy(axis_bar_struct *a)

	int JCWInit(jcw_struct *j, int argc, char *argv[])
	void JCWDraw(jcw_struct *j, int amount)
	int JCWManage(jcw_struct *j, event_t *event)
	void JCWMap(jcw_struct *j)
	void JCWUnmap(jcw_struct *j)
	void JCWDestroy(jcw_struct *j)

	---

 */

#include <stdio.h>
#include <malloc.h>
#include <db.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "../include/disk.h"
#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/osw-x.h"
#include "../include/widget.h"

#include "xjc.h"
#include "config.h"


#include "xjscal_icon.h"


/*
 *	Default window sizes:
 */
#define JCW_DEF_WIDTH	640
#define JCW_DEF_HEIGHT	480

#define JCW_MENUBAR_HEIGHT	30
#define JCW_STATUS_HEIGHT	25


/*
 *	Initializes an axis bar widget on window parent.
 */
int AxisBarInit(
	axis_bar_struct *a,
	win_t parent,
	int x, int y,
	int axis_num,
	int flip
)
{
	if(a == NULL)
	    return(-1);
	if(parent == 0)
	    return(-1);


	a->x = x;
	a->y = y;

	a->pos = 0;
	a->cur = 0;
	a->min = 0;
	a->max = 0;

	/* Axis number prompt. */
        if(
            PromptInit(
                &a->axis_num_prompt,
                parent,
                x,
		y,
                85,
                30,
                PROMPT_STYLE_FLUSHED,
                "Axis:",
                64,
                0,
                NULL
            )
        )
            return(-1);
	PromptSetI(&a->axis_num_prompt, axis_num);

	/* Position window. */
	if(
	    OSWCreateWindow(
		&a->pos_win,
		parent,
		x + 85,
		y + 3,
		180,
		24
	    )
	)
	    return(-1);

	a->pos_win_buf = 0;


	/* Caliberate. */
	if(
	    TgBtnInit(
                &a->caliberate_tb,
                parent,
                x + 85 + 180 + 5,
		y + 5,
                False,
                "Caliberate"
	    )
        )
	    return(-1);

        /* Axis number prompt. */
        if(
            PromptInit(  
                &a->null_zone_prompt,
                parent,
                x + 85 + 180 + 5 + a->caliberate_tb.width + 10,
                y,
                130,
                30,
                PROMPT_STYLE_FLUSHED,
                "Null Zone:",
                64,
                0,
                NULL   
            )
        )
            return(-1);
        PromptSetI(&a->null_zone_prompt, 0);

        /* Flip. */ 
        if(
            TgBtnInit(
                &a->flip_tb,
                parent,
                x + 85 + 180 + 5 + a->caliberate_tb.width + 10 + 130,
                y + 5,
                ((flip) ? True : False),
                "Flip"
            )
        )
            return(-1);


	return(0);
}

/*
 *	Redraws an axis bar widget a.
 */
void AxisBarDraw(axis_bar_struct *a)
{
	int x, r, nz;
	double coeff;
	win_t w;
	pixmap_t pixmap;
	char text[80];
	win_attr_t wattr;


	if(a == NULL)
            return;


        /* Recreate buffers as needed. */
        if(a->pos_win_buf == 0)
        {
            OSWGetWindowAttributes(a->pos_win, &wattr);
            if(
                OSWCreatePixmap(&a->pos_win_buf,
                    wattr.width, wattr.height
            ))
                return;
        }

	/* Redraw position window. */
	if(1)
	{
	    w = a->pos_win;
	    pixmap = a->pos_win_buf;

            OSWGetWindowAttributes(w, &wattr);

	    OSWCopyDrawables(
		pixmap,
		jcw.axis_bar_bkg_pm,
		wattr.width, wattr.height
	    );

	    if(widget_global.force_mono)
	    {

	    }
	    else
	    {
		/* In caliberate mode? */
		if(a->caliberate_tb.state)
		{
		    sprintf(text, "[%i, %i]", a->min, a->max);
		    OSWSetFgPix(widget_global.editable_text_pix);
		    OSWDrawString(
			pixmap,
			5,
			17,
			text
		    );
		}
		else
		{
		    r = (int)wattr.width - 10;
                    x = (r / 2) * a->pos;
                    x += ((r / 2) + 5);
 
                    OSWSetFgPix(widget_global.editable_text_pix);
		    OSWDrawLine(
		        pixmap,
		        x, 13,
		        x, 16
		    );
                    OSWDrawLine(
                        pixmap,
                        x - 1, 14,
                        x - 1, 16
                    );
                    OSWDrawLine(
                        pixmap,
                        x + 1, 14,
                        x + 1, 16
                    );
                    OSWDrawLine(
                        pixmap,   
                        x - 2, 15,
                        x - 2, 16
                    );
                    OSWDrawLine(
                        pixmap,
                        x + 2, 15,
                        x + 2, 16
                    );


		    nz = PromptGetI(&a->null_zone_prompt);
		    coeff = (double)r / (double)(a->max - a->min);

		    x = ((r / 2) + 5) - (nz * coeff);
		    OSWSetFgPix(widget_global.scroll_frame_pix);
		    OSWDrawSolidRectangle(
                        pixmap,
                        x, 4,
                        (nz * 2 * coeff) + 1, 3
                    );

		}

                WidgetFrameButtonPixmap(
                    pixmap,
                    True,
                    wattr.width, wattr.height,
                    widget_global.surface_highlight_pix,
                    widget_global.surface_shadow_pix
                );
	    }

            OSWPutBufferToWindow(w, pixmap);
	}


	return;
}

/*
 *	Manages axis bar widget a.
 */
int AxisBarManage(axis_bar_struct *a, event_t *event)
{
	bool_t prev_state;
	int axis_num;
        int events_handled = 0;
                    
                    
        if((event == NULL) ||
           (a == NULL)
        )
            return(events_handled);


	/* Flip. */
	if(events_handled == 0)
	    events_handled += TgBtnManage(&a->flip_tb, event);

	/* Caliberate. */
	if(events_handled == 0)
	{
	    prev_state = a->caliberate_tb.state;
	    events_handled += TgBtnManage(&a->caliberate_tb, event);
	    if(prev_state != a->caliberate_tb.state)
	    {
	        if(a->caliberate_tb.state)
	        {
	            a->min = a->cur;
	            a->max = a->cur;
	        }
	        AxisBarDraw(a);
	    }
	}

	/* Null zone prompt. */
	if(events_handled == 0)
	    events_handled += PromptManage(&a->null_zone_prompt, event);

	/* Axis number prompt. */
        if(events_handled == 0)
	    events_handled += PromptManage(&a->axis_num_prompt, event);


	return(events_handled);
}

/*
 *	Maps axis bar widget a.
 */
void AxisBarMap(axis_bar_struct *a)
{
        if(a == NULL)
            return;


	TgBtnMap(&a->flip_tb);
	PromptMap(&a->null_zone_prompt);
        TgBtnMap(&a->caliberate_tb);
        OSWMapWindow(a->pos_win);
        PromptMap(&a->axis_num_prompt);


	return;
}

/*
 *	Destroys axis bar widget a and all its resources.
 */
void AxisBarDestroy(axis_bar_struct *a)
{
	if(a == NULL)
	    return;

	TgBtnDestroy(&a->flip_tb);
        PromptDestroy(&a->null_zone_prompt);
	TgBtnDestroy(&a->caliberate_tb);
	OSWDestroyPixmap(&a->pos_win_buf);
	OSWDestroyWindow(&a->pos_win);
	PromptDestroy(&a->axis_num_prompt);


	a->x = 0;
	a->y = 0;

	a->pos = 0;
	a->min = 0;
	a->max = 0;


	return;
}


/*
 *	Initializes joystick caliberation window j.
 */
int JCWInit(jcw_struct *j, int argc, char *argv[])
{
        int i, x, y;
	char *strptr;
	int strc;
	char *strv[10];
        unsigned int width, height;
	win_attr_t wattr;
	pixmap_t pixmap;
	char startup_device[PATH_MAX + NAME_MAX];


        if(j == NULL)
            return(-1);


	/* Set default values. */
        strncpy(
            startup_device,
            DEF_DEVICE_NAME,
            PATH_MAX + NAME_MAX
        );
        startup_device[PATH_MAX + NAME_MAX - 1] = '\0';
	if(!ISPATHABSOLUTE(startup_device))
	{
	    strptr = getenv("HOME");
	    if(strptr != NULL)
	    {
		strptr = PrefixPaths(strptr, startup_device);
		strncpy(
                    startup_device,
		    ((strptr == NULL) ? DEF_DEVICE_NAME : strptr),
		    PATH_MAX + NAME_MAX
                );
		startup_device[PATH_MAX + NAME_MAX - 1] = '\0';
	    }
	}

	strncpy(
	    j->calib_file,
	    DEF_CALIB_FILE,
	    PATH_MAX + NAME_MAX
        );
	j->calib_file[PATH_MAX + NAME_MAX - 1] = '\0';
        if(!ISPATHABSOLUTE(j->calib_file))
        {
            strptr = getenv("HOME");
            if(strptr != NULL)
            {
                strptr = PrefixPaths(strptr, j->calib_file);
                strncpy(
                    j->calib_file,
                    ((strptr == NULL) ? DEF_CALIB_FILE : strptr),
                    PATH_MAX + NAME_MAX
                );
                j->calib_file[PATH_MAX + NAME_MAX - 1] = '\0';
            }
        }


	/* Reset values. *
	j->map_state = 0;
	j->is_in_focus = 0;


        /* Get position and sizes. */
        if(osw_gui[0].def_geometry_set)
        {
            j->x = osw_gui[0].def_toplevel_x;
            j->y = osw_gui[0].def_toplevel_y;
            j->width = osw_gui[0].def_toplevel_width;
            j->height = osw_gui[0].def_toplevel_height;
        }
        else
        {
            j->x = 0;
            j->y = 0;
            j->width = JCW_DEF_WIDTH;
            j->height = JCW_DEF_HEIGHT;
        }

	/* Parse arguments. */
        for(i = 1; i < argc; i++)
        {
            if(argv[i] == NULL)
		continue;


	    if(strcasepfx(argv[i], "--d") ||
               strcasepfx(argv[i], "-d")
	    )
	    {
		i++;
		if(i < argc)
		{
		    strncpy(
		        startup_device,
		        argv[i],
			PATH_MAX + NAME_MAX
		    );
		    startup_device[PATH_MAX + NAME_MAX - 1] = '\0';
		}
	    }
            else if(strcasepfx(argv[i], "--f") ||
                    strcasepfx(argv[i], "-f")
            )
            {
                i++;
                if(i < argc)
                {
                    strncpy(
                        j->calib_file,
                        argv[i],
                        PATH_MAX + NAME_MAX
                    );
                    j->calib_file[PATH_MAX + NAME_MAX - 1] = '\0';
                }
            }

	}


        /* Create toplevel. */
        if(
            OSWCreateWindow(
                &j->toplevel,
                osw_gui[0].root_win,
                j->x, j->y,
                j->width, j->height
            )
        )
            return(-1);

	pixmap = WidgetLoadPixmapFromTgaData(xjscal_icon_tga);

        OSWSetWindowWMProperties(
            j->toplevel,
            PROG_NAME,
            PROG_NAME,
            ((pixmap == 0) ? widget_global.std_icon_pm : pixmap),
            !osw_gui[0].def_geometry_set,	/* Let WM set coordinates? */
            j->x, j->y,
            j->width, j->height,
            j->width, j->height,
            WindowFrameStyleFixed,
            NULL, 0
        );
        OSWSetWindowInput(
            j->toplevel,
            OSW_EVENTMASK_TOPLEVEL
        );


        /* Menu bar. */
        if(
            MenuBarInit(
                &j->mb,
                j->toplevel,
                0, 0,
                0, JCW_MENUBAR_HEIGHT,
                JCWMenuBarCB,
                j
            )
        )
            return(-1);

        /* File menu. */
        x = 0;
        y = 0;
        i = 0;
        if(
            MenuBarAddItem(
                &j->mb, -1,
                "File",
                x, y,
                0, JCW_MENUBAR_HEIGHT
            )
        )
            return(-1);

        MenuBarAddItemMenuItem(
            &j->mb,
            i,
            "Open Device...",
            MENU_ITEM_TYPE_ENTRY,
            NULL,
            JCW_MENU_CODE_OPEN_DEVICE,
            -1
        );

        MenuBarAddItemMenuItem(
            &j->mb,
            i,
            NULL,
            MENU_ITEM_TYPE_HR,
            NULL,
            JCW_MENU_CODE_NONE,
            -1
        );

        MenuBarAddItemMenuItem(
            &j->mb,
            i,
            "Save Caliberation",
            MENU_ITEM_TYPE_ENTRY,
            NULL,
            JCW_MENU_CODE_SAVE_CAL,
            -1
        );

        MenuBarAddItemMenuItem(
            &j->mb,
            i,
            "Save Caliberation As...",
            MENU_ITEM_TYPE_ENTRY,
            NULL,
            JCW_MENU_CODE_SAVE_CAL_AS,
            -1
        );

        MenuBarAddItemMenuItem(
            &j->mb,
            i,
            NULL,
            MENU_ITEM_TYPE_HR,
            NULL,
            JCW_MENU_CODE_NONE,
            -1
        );

        MenuBarAddItemMenuItem(
            &j->mb,
            i,
            "Exit",
            MENU_ITEM_TYPE_ENTRY,
            NULL,
            JCW_MENU_CODE_EXIT,
            -1
        );



        /* Device bar. */
        if(
            OSWCreateWindow(
                &j->device_bar,
                j->toplevel,
                0, JCW_MENUBAR_HEIGHT,
                j->width, 34
            )   
        )
            return(-1);   


	/* Device prompt. */
        if(
            PromptInit(
                &j->device_prompt,
                j->device_bar,
                (int)j->width - 420,
                2,
                418,
                30,
                PROMPT_STYLE_FLUSHED,
                "Device:",
                PATH_MAX + NAME_MAX,
                0,
                NULL
            )
        )
            return(-1);


	/* Allocate axises. */
	j->total_axis_bars = 8;
	j->axis_bar = (axis_bar_struct **)calloc(
	    j->total_axis_bars,
	    sizeof(axis_bar_struct *)
	);
	if(j->axis_bar == NULL)
	    j->total_axis_bars = 0;

	x = 10;
	y = 74;
	for(i = 0; i < j->total_axis_bars; i++)
	{
	    j->axis_bar[i] = (axis_bar_struct *)calloc(
		1,
		sizeof(axis_bar_struct)
	    );
	    if(j->axis_bar[i] == NULL)
		continue;

	    AxisBarInit(
		j->axis_bar[i],
		j->toplevel,
		x,
		y,
		i,	/* Axis number. */
		0	/* Flip. */
	    );

	    y += 35;
	}


	/* Render readout backgrounds as needed. */
	if(1)
	{
	    wattr.width = 180;
	    wattr.height = 24;

	    OSWCreatePixmap(
		&j->axis_bar_bkg_pm,
		wattr.width, wattr.height
	    );

	    pixmap = j->axis_bar_bkg_pm;

	    if(widget_global.force_mono)
	    {
		OSWClearPixmap(
                    pixmap,
                    wattr.width, wattr.height,
                    osw_gui[0].black_pix
                );

		OSWSetFgPix(osw_gui[0].white_pix);
	    }
	    else
	    {
	        OSWClearPixmap(
		    pixmap,
                    wattr.width, wattr.height,
		    widget_global.surface_editable_pix
	        );

                WidgetFrameButtonPixmap(
                    pixmap,
                    True,
                    wattr.width, wattr.height,
                    widget_global.surface_highlight_pix,
                    widget_global.surface_shadow_pix
                );

	        OSWSetFgPix(widget_global.scroll_frame_pix);
	    }

	    OSWDrawLine(
		pixmap,
		5, 3,
		(int)wattr.width - 5, 3
	    );
	    OSWDrawLine(
                pixmap,
                5, 4,
                5, 11
            );
            OSWDrawLine(
                pixmap,
                (int)wattr.width / 2, 4,
                (int)wattr.width / 2, 11
            );
            OSWDrawLine(
                pixmap,
                ((int)wattr.width / 2) -
                    (((int)wattr.width - 10) / 4),
		4,
                ((int)wattr.width / 2) -
                    (((int)wattr.width - 10) / 4),
                9
            );
            OSWDrawLine(
                pixmap,
                ((int)wattr.width / 2) +
                    (((int)wattr.width - 10) / 4),
                4,
                ((int)wattr.width / 2) +
                    (((int)wattr.width - 10) / 4),
                9
            );
            OSWDrawLine(
                pixmap,
                (int)wattr.width - 5, 4,
                (int)wattr.width - 5, 11
            );
	}


        /* X axis number prompt. */
        if(
            PromptInit(
                &j->readout_x_axis_prompt,
                j->toplevel,
                10,
                (int)j->height - 90 - JCW_STATUS_HEIGHT - 10,
                110,
                30,
                PROMPT_STYLE_FLUSHED,
                "X Axis:",
                64,
                0,
                NULL
            )
        )
            return(-1);
	PromptSetI(&j->readout_x_axis_prompt, 0);

	/* Y axis number prompt. */
        if(
            PromptInit(
                &j->readout_y_axis_prompt,
                j->toplevel,  
                10,
                (int)j->height - 90 - JCW_STATUS_HEIGHT - 10 + 35,
                110,
                30,
                PROMPT_STYLE_FLUSHED,
                "Y Axis:",
                64,
                0,
                NULL
            )
        )
            return(-1);
        PromptSetI(&j->readout_y_axis_prompt, 1);

	/* Coordinates readout window. */
	if(
	    OSWCreateWindow(
		&j->readout_win,
		j->toplevel,
		130,
		(int)j->height - 90 - JCW_STATUS_HEIGHT - 10,
		90,
		90
	    )
	)
	    return(-1);

	/* Render background for readout. */
	if(1)
	{
            wattr.width = 90;
            wattr.height = 90;

            OSWCreatePixmap(
                &j->readout_bkg_pm,
                wattr.width, wattr.height
            );

            pixmap = j->readout_bkg_pm;

            if(widget_global.force_mono)
            {
                OSWClearPixmap(
                    pixmap,
                    wattr.width, wattr.height,
                    osw_gui[0].black_pix
                );

                OSWSetFgPix(osw_gui[0].white_pix);
            }
            else  
            {
                OSWClearPixmap(
                    pixmap,
                    wattr.width, wattr.height,
                    widget_global.surface_editable_pix
                );  

                WidgetFrameButtonPixmap(
                    pixmap,
                    True,
                    wattr.width, wattr.height,
                    widget_global.surface_highlight_pix,
                    widget_global.surface_shadow_pix
                );

                OSWSetFgPix(widget_global.scroll_frame_pix);
            }

            OSWDrawLine(
                pixmap,
                3,
                ((int)wattr.height / 2),
                (int)wattr.width - 3,
                ((int)wattr.height / 2)
            );
            OSWDrawLine(
                pixmap,
                ((int)wattr.width / 2),
		3,
                ((int)wattr.width / 2),  
		(int)wattr.height - 3
            );

	    /* X axis ticks. */
	    x = 3;
            OSWDrawLine(
                pixmap,
                x,
		((int)wattr.height / 2) - 3,
                x,
		((int)wattr.height / 2) + 3
            );  
            x = (int)wattr.width - 3;
            OSWDrawLine(
                pixmap,
                x,
                ((int)wattr.height / 2) - 3,  
                x,
                ((int)wattr.height / 2) + 3
            );
            x = ((int)wattr.width / 2) -
                (((int)wattr.width - 6) / 4);
            OSWDrawLine(
                pixmap,
                x,  
                ((int)wattr.height / 2) - 3,
                x,
                ((int)wattr.height / 2) + 3
            );
            x = ((int)wattr.width / 2) +
                (((int)wattr.width - 6) / 4);
            OSWDrawLine(
                pixmap,
                x,
                ((int)wattr.height / 2) - 3,
                x,
                ((int)wattr.height / 2) + 3
            );

	    /* Y axis ticks. */
            y = 3;
            OSWDrawLine(
                pixmap,
                ((int)wattr.width / 2) - 3,
		y,
                ((int)wattr.width / 2) + 3,
		y
            );
            y = (int)wattr.height - 3;
            OSWDrawLine(
                pixmap,
                ((int)wattr.width / 2) - 3,
                y,
                ((int)wattr.width / 2) + 3,
		y
            );
            y = ((int)wattr.height / 2) -
                (((int)wattr.height - 6) / 4);
            OSWDrawLine(
                pixmap,
                ((int)wattr.width / 2) - 3,
                y,
                ((int)wattr.width / 2) + 3,
		y
            );
            y = ((int)wattr.height / 2) +
                (((int)wattr.height - 6) / 4);
            OSWDrawLine(
                pixmap,
                ((int)wattr.width / 2) - 3,
                y,
                ((int)wattr.width / 2) + 3,
		y
            );



	}

	/* Status bar. */
        if(
            OSWCreateWindow(
                &j->status_bar_win,
                j->toplevel,
                0,
                (int)j->height - JCW_STATUS_HEIGHT,
                (int)j->width,
                JCW_STATUS_HEIGHT
            )
        )
            return(-1);


        /* Need to reset jsd values explicitly. */
        bzero(&j->jsd, sizeof(js_data_struct));
	j->jsd.name = NULL;
        j->jsd.device_name = NULL;
        j->jsd.caliberation_file = NULL;  
        j->jsd.fd = -1;
              
        /* Open device. */
        JCWDoOpenDevice(j, startup_device, j->calib_file);


	return(0);
}

/*
 *	Redraws joystick caliberation window j.
 */
void JCWDraw(jcw_struct *j, int amount)
{
	int i, x, y, axis_num;
	win_t w;
	pixmap_t pixmap;
	font_t *prev_font;
	win_attr_t wattr;


	if(j == NULL)
	    return;

	/* Map as needed. */
	if(!j->map_state)
	{
	    OSWMapRaised(j->toplevel);
	    OSWMapWindow(j->device_bar);

	    MenuBarMap(&j->mb);
            PromptMap(&j->device_prompt);

	    for(i = 0; i < j->total_axis_bars; i++)
		AxisBarMap(j->axis_bar[i]);


            PromptMap(&j->readout_x_axis_prompt);
            PromptMap(&j->readout_y_axis_prompt);
	    OSWMapWindow(j->readout_win);

	    OSWMapWindow(j->status_bar_win);

	    j->map_state = 1;
            amount = JCW_DRAW_AMOUNT_COMPLETE;
	}


	/* Recreate buffers as needed. */
	if(j->toplevel_buf == 0)
        {
            OSWGetWindowAttributes(j->toplevel, &wattr);
            if(
                OSWCreatePixmap(&j->toplevel_buf,
                    wattr.width, wattr.height
            ))
                return;
        }

        prev_font = OSWQueryCurrentFont();


        /* Redraw toplevel. */
        if(amount == JCW_DRAW_AMOUNT_COMPLETE)
        {
            w = j->toplevel;
            pixmap = j->toplevel_buf;
            OSWGetWindowAttributes(w, &wattr);

            if(widget_global.force_mono)
            {
                OSWClearPixmap(
                    pixmap,
                    wattr.width, wattr.height,
                    osw_gui[0].black_pix
                );
            }
	    else
	    {
		WidgetPutImageTile(
                    pixmap, widget_global.std_bkg_img,
		    wattr.width, wattr.height
		);
	    }

            OSWPutBufferToWindow(w, pixmap);


	    /* Device bar. */
            w = j->device_bar;
            pixmap = j->toplevel_buf;
            OSWGetWindowAttributes(w, &wattr);

            if(widget_global.force_mono)
            {
                OSWClearPixmap(
                    pixmap,
                    wattr.width, wattr.height,
                    osw_gui[0].black_pix
                );
            }
            else
            {
                WidgetPutImageTile(
                    pixmap, widget_global.std_bkg_img,
                    wattr.width, wattr.height
                );

                WidgetFrameButtonPixmap(
                    pixmap,
                    False,
                    wattr.width, wattr.height,
                    widget_global.surface_highlight_pix,
                    widget_global.surface_shadow_pix
                );
            }

            OSWPutBufferToWindow(w, pixmap);
        }

        /* Redraw axis bars. */
        if((amount == JCW_DRAW_AMOUNT_COMPLETE) ||
           (amount == JCW_DRAW_AMOUNT_AXISBARS)
	)
        {
            for(i = 0; i < j->total_axis_bars; i++)
                AxisBarDraw(j->axis_bar[i]);
	}

        /* Redraw coordinates. */
        if((amount == JCW_DRAW_AMOUNT_COMPLETE) ||
           (amount == JCW_DRAW_AMOUNT_COORDINATES)
        )
        {
            w = j->readout_win;
            pixmap = j->toplevel_buf;
            OSWGetWindowAttributes(w, &wattr);

	    OSWCopyDrawables(
		pixmap,
		j->readout_bkg_pm,
		wattr.width, wattr.height
	    );

	    axis_num = PromptGetI(&j->readout_x_axis_prompt);
	    x = ((int)wattr.width / 2) +
		((((int)wattr.width - 6) / 2) *
		JSGetAxisCoeff(&j->jsd, axis_num));
            axis_num = PromptGetI(&j->readout_y_axis_prompt);
            y = ((int)wattr.height / 2) +
                ((((int)wattr.height - 6) / 2) *
                JSGetAxisCoeff(&j->jsd, axis_num) * -1);

	    if(widget_global.force_mono)
		OSWSetFgPix(osw_gui[0].white_pix);
	    else
		OSWSetFgPix(widget_global.editable_text_pix);

	    OSWDrawLine(
		pixmap,
		x - 3, y,
		x + 3, y
	    );
            OSWDrawLine(
                pixmap,
                x, y - 3,
                x, y + 3
            );

	    if(!widget_global.force_mono)
                WidgetFrameButtonPixmap(
                    pixmap,
                    True,
                    wattr.width, wattr.height,
                    widget_global.surface_highlight_pix,
                    widget_global.surface_shadow_pix
                );


            OSWPutBufferToWindow(w, pixmap);
	}

	/* Redraw status bar. */
        if((amount == JCW_DRAW_AMOUNT_COMPLETE) ||
           (amount == JCW_DRAW_AMOUNT_STATUSBAR)
	)
        {
            w = j->status_bar_win;
            pixmap = j->toplevel_buf;
            OSWGetWindowAttributes(w, &wattr);

            if(widget_global.force_mono)
            {
                OSWClearPixmap(
                    pixmap,
                    wattr.width, wattr.height,
                    osw_gui[0].black_pix
                );
            }
            else
            {
                WidgetPutImageTile(
                    pixmap, widget_global.std_bkg_img,
                    wattr.width, wattr.height
                );

                WidgetFrameButtonPixmap(
                    pixmap,
                    False,
                    wattr.width, wattr.height,
                    widget_global.surface_highlight_pix,
                    widget_global.surface_shadow_pix
                );
            }

            OSWPutBufferToWindow(w, pixmap);
        }

	OSWSetFont(prev_font);


	return;
}

/*
 *	Manages joystick caliberation window j.
 */
int JCWManage(jcw_struct *j, event_t *event)
{
	int i, n;
	char *strptr;
	char tmp_path[PATH_MAX + NAME_MAX];
	int events_handled = 0;
	keycode_t keycode;

	axis_bar_struct **axis_bar;
	bool_t prev_flip_state;


        if((event == NULL) ||
           (j == NULL)
	)
            return(events_handled);

	if(!j->map_state &&
           (event->type != MapNotify)
	)
	    return(events_handled);


	switch(event->type)
	{
	  case KeyPress:
	    if(!j->is_in_focus)
		return(events_handled);

	    keycode = event->xkey.keycode;

	    /* Enter. */
	    if((keycode == osw_keycode.enter) ||
               (keycode == osw_keycode.np_enter) 
            )
	    {
                strptr = PromptGetS(&j->device_prompt);
                strncpy(
                    tmp_path,
                    ((strptr == NULL) ? DEF_DEVICE_NAME : strptr),
                    PATH_MAX + NAME_MAX
		);
		tmp_path[PATH_MAX + NAME_MAX - 1] = '\0';

		PromptManage(&j->device_prompt, event);

		PromptSetS(&j->device_prompt, tmp_path);
		if(j->device_prompt.map_state)
		    PromptDraw(&j->device_prompt, PROMPT_DRAW_AMOUNT_TEXTONLY);

		JCWDoOpenDevice(
		    j,
		    tmp_path,
		    j->calib_file
		);

                events_handled++;
		return(events_handled);
	    }
	    break;

	  case KeyRelease:
            if(!j->is_in_focus)
                return(events_handled);

            keycode = event->xkey.keycode;

	    /* Enter. */
            if((keycode == osw_keycode.enter) ||
               (keycode == osw_keycode.np_enter)
            )
            {
                events_handled++;
                return(events_handled);
            }
	    break;

	  case Expose:
	    if(event->xany.window == j->toplevel)
	    {
		JCWDraw(j, JCW_DRAW_AMOUNT_COMPLETE);
	    }
	    break;

          case FocusIn:
            if(event->xany.window == j->toplevel)
            {
                j->is_in_focus = 1;
                events_handled++;
                return(events_handled);
            }
            break;

          case FocusOut:
            if(event->xany.window == j->toplevel)
            {
                j->is_in_focus = 0;

                events_handled++;
                return(events_handled);
            }
            break;

          case MapNotify:
            if(event->xany.window == j->toplevel)
            {
                if(!j->map_state)
                    JCWMap(j);

                events_handled++;
                return(events_handled);
            }
            break;

          case UnmapNotify:
            if(event->xany.window == j->toplevel)
            {
                if(j->map_state)
                    JCWUnmap(j);

                events_handled++;
                return(events_handled);
            }
            break;

          case ClientMessage:
            if(OSWIsEventDestroyWindow(j->toplevel, event))
            {
		runlevel = 1;

                events_handled++;
                return(events_handled);
            }
            break;
	}


	/* Manage widgets. */

	/* Menu bar. */
	if(events_handled == 0)
	    events_handled += MenuBarManage(&j->mb, event);

	/* Device prompt. */
	events_handled += PromptManage(&j->device_prompt, event);

	/* Axis number prompts. */
	events_handled += PromptManage(&j->readout_x_axis_prompt, event);
	events_handled += PromptManage(&j->readout_y_axis_prompt, event);

	/* Axis readout bars. */
	if(events_handled == 0)
	{
	    for(i = 0, axis_bar = j->axis_bar;
                i < j->total_axis_bars;
                i++, axis_bar++
	    )
	    {
		if(*axis_bar == NULL)
		    continue;

		prev_flip_state = (*axis_bar)->flip_tb.state;

		events_handled += AxisBarManage(
		    *axis_bar,
		    event
		);

		if(events_handled > 0)
		{
		    /* Change in flip state? */
		    if(prev_flip_state != (*axis_bar)->flip_tb.state)
		    {
		        n = PromptGetI(&((*axis_bar)->axis_num_prompt));
		        if(JSIsAxisAllocated(&j->jsd, n))
                        {
			    j->jsd.axis[n]->flip = (((*axis_bar)->flip_tb.state) ?
			        1 : 0);
		        }
		    }

		    AxisBarDraw(*axis_bar);
		    JCWDraw(j, JCW_DRAW_AMOUNT_COORDINATES);

		    /* Stop handling events after one is handled. */
		    break;
		}

	    }
	}


	return(events_handled);
}

/*
 *	Maps joystick caliberation window j.
 */
void JCWMap(jcw_struct *j)
{
        if(j == NULL)
            return;

	j->map_state = 0;
	JCWDraw(j, JCW_DRAW_AMOUNT_COMPLETE);

	return;
}

/*
 *	Unmaps joystick caliberation window j.
 */
void JCWUnmap(jcw_struct *j)
{       
        if(j == NULL)
            return;

	OSWUnmapWindow(j->toplevel);
	j->map_state = 0;
	j->is_in_focus = 0;

	OSWDestroyPixmap(&j->toplevel_buf);

	return;
}

/*
 *	Destroys joystick caliberation window j.
 *
 *	Deallocates all its resources, closes the joystick device
 *	opened by it (if any).
 */
void JCWDestroy(jcw_struct *j)
{
	int i;


	if(j == NULL)
	    return;


	OSWDestroyWindow(&j->status_bar_win);

	OSWDestroyPixmap(&j->readout_bkg_pm);
	OSWDestroyWindow(&j->readout_win);

	PromptDestroy(&j->readout_x_axis_prompt);
	PromptDestroy(&j->readout_y_axis_prompt);


	OSWDestroyPixmap(&j->axis_bar_bkg_pm);

	for(i = 0; i < j->total_axis_bars; i++)
	{
	    AxisBarDestroy(j->axis_bar[i]);
	    free(j->axis_bar[i]);
	}
	free(j->axis_bar);
	j->axis_bar = NULL;

	j->total_axis_bars = 0;

        MenuBarDestroy(&j->mb);

        PromptDestroy(&j->device_prompt);
	OSWDestroyWindow(&j->device_bar);

	OSWDestroyPixmap(&j->toplevel_buf);
	OSWDestroyWindow(&j->toplevel);


	JSClose(&j->jsd);


	return;
}
