/***************************************************************** -*- C -*- **
 *
 * main.c
 * xrootconsole
 * Copyright (C) 1998, 1999  Eric Youngblut
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * The author of this software can be reached by e-mail at
 * yngblut@cs.washington.edu.  The latest version of this software can be
 * found at http://www.cs.washington.edu/homes/yngblut/
 */


#include "console.h"
#include "util.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>


#define USAGE \
"Usage: xrootconsole [OPTION]...\n" \
"Redirect the console to a transparent window placed on the X root window.\n" \
"\n" \
"  -geometry GEO   the geometry of the window (default 80x25+0+0)\n" \
"  -fn FONT        the font to use (default fixed)\n" \
"  -fg COLOR       foreground color of the text (default white)\n" \
"  -bd COLOR       border color (default white)\n" \
"  -bw WIDTH       border width (default 0)\n" \
"  --topdown       insert lines at the top and scroll the rest down\n" \
"  --wrap          wrap long lines, instead of cutting them off\n" \
"\n" \
"This program may have to be run as root.\n" \
"\n" \
"Report bugs to <yngblut@cs.washington.edu>.\n"


#define DEFAULT_X		0
#define DEFAULT_Y		0
#define DEFAULT_COLS		80
#define DEFAULT_ROWS		25
#define DEFAULT_FONT		"fixed"
#define DEFAULT_COLOR		"white"
#define DEFAULT_BORDER_WIDTH	0
#define DEFAULT_BORDER_COLOR	"white"


Display* dpy;
char* text;			/* circular buffer to store the characters */
int pos = 0;			/* where the actual first line in text is */

Window win;
XFontStruct* fn = NULL;
GC gc;

int wrap = False;
int topdown = False;

int cols = DEFAULT_COLS;	/* size of the window (in characters) */
int rows = DEFAULT_ROWS;


static void init(char** argv);
static void handle_expose(XExposeEvent* xexpose);
static void put(const char* s);
static void event_loop(int fd);


/*
 * main
 */

int main(int argc, char** argv)
{
    int fd;
    
    init(argv);

    fd = open_console();
    event_loop(fd);

    XCloseDisplay(dpy);
    exit(EXIT_SUCCESS);
}


/*
 * init
 */

void init(char** argv)
{
    XSetWindowAttributes a;
    XGCValues values;
    int geo_mask = 0;
    int x_root = DEFAULT_X, y_root = DEFAULT_Y;
    const char* fg = NULL;
    const char* bd = NULL;
    int width, height;
    int bw = 0;
    
    /* Connect to the X server */
    dpy = XOpenDisplay(NULL);
    if (dpy == NULL)
    {
	fprintf(stderr, "Cannot open display\n");
	exit(EXIT_FAILURE);
    }

    /* Process command-line arguments */
    while (*++argv != NULL)
    {
	if (!strcmp(*argv, "-geometry") || !strcmp(*argv, "-geo"))
	{
	    geo_mask = XParseGeometry(*++argv, &x_root, &y_root,
				      (unsigned*)&cols, (unsigned*)&rows);
	}
	else if (!strcmp(*argv, "-font") || !strcmp(*argv, "-fn"))
	    fn = load_font(*++argv);
	else if (!strcmp(*argv, "-foreground") || !strcmp(*argv, "-fg"))
	    fg = *++argv;
	else if (!strcmp(*argv, "-bordercolor") || !strcmp(*argv, "-bd"))
	    bd = *++argv;
	else if (!strcmp(*argv, "-borderwidth") || !strcmp(*argv, "-bw"))
	    bw = atoi(*++argv);
	else if (!strcmp(*argv, "--wrap"))
	    wrap = True;
	else if (!strcmp(*argv, "--topdown"))
	    topdown = True;
	else
	{
	    fprintf(stderr, "%s", USAGE);
	    exit(EXIT_FAILURE);
	}
    }

    /* Create the 2d array We can display "rows" # of rows, and the extra one
       is for the line being read. */
    text = (char*)malloc(cols * (rows + 1));
    memset(text, ' ', cols * (rows + 1));

    /* Load the default font if none has already been loaded */
    if (fn == NULL)
	fn = load_font(DEFAULT_FONT);

    /* Calculate the position of window on the screen */
    width = cols * fn->max_bounds.width;
    height = rows * (fn->ascent + fn->descent);

    x_root = (geo_mask & XNegative) ?
	(DisplayWidth(dpy, DefaultScreen(dpy)) - width - x_root) : x_root;
    y_root = (geo_mask & YNegative) ?
	(DisplayWidth(dpy, DefaultScreen(dpy)) - height - y_root) : y_root;

    /* Create the window */
    a.background_pixmap = ParentRelative;
    a.border_pixel = load_color((bd != NULL) ? bd : DEFAULT_BORDER_COLOR);
    a.event_mask = ExposureMask;
    a.override_redirect = True;
 
    win = XCreateWindow(dpy, DefaultRootWindow(dpy), x_root, y_root,
			width, height, bw, CopyFromParent, CopyFromParent,
			DefaultVisual(dpy, DefaultScreen(dpy)),
			CWOverrideRedirect | CWBackPixmap | CWEventMask |
			CWBorderPixel, &a);
    XMapWindow(dpy, win);
    XLowerWindow(dpy, win);

    /* Create the GC */
    values.font = fn->fid;
    values.foreground = load_color((fg != NULL) ? fg : DEFAULT_COLOR);
    gc = XCreateGC(dpy, win, GCFont | GCForeground, &values);

#ifdef STARTUP_MESSAGE
    /* Display a message */
    put(STARTUP_MESSAGE);
#endif
}


/*
 * handle_expose
 */

void handle_expose(XExposeEvent* ev)
{
    int col, width;
    int row, height;
    int x_text, y_text;
    
    XClearArea(dpy, win, ev->x, ev->y, ev->width, ev->height, False);

    col = ev->x / fn->max_bounds.width;
    width = (ev->x + ev->width - 1) / fn->max_bounds.width - col + 1;
    assert(width > 0 && col + width <= cols);
	
    row = ev->y / (fn->ascent + fn->descent);
    height = (ev->y + ev->height - 1) / (fn->ascent + fn->descent) - row + 1;
    assert(height > 0 && row + height <= rows);
	
    x_text = col * fn->max_bounds.width;
    y_text = row * (fn->ascent + fn->descent) + fn->ascent;

    /* row is the row index into text */
    row += pos + (topdown ? 1 : rows);
    while (height--)
    {
	XDrawString(dpy, win, gc, x_text, y_text,
		    text + (row % rows) * cols + col, width);

	row++;
	y_text += fn->ascent + fn->descent;
    }
}


/*
 * put
 */

void put(const char* s)
{
    static int x = 0, y = 0;
    
    while (*s != '\0')
    {
	switch (*s)
	{
	case '\n':
	    x = 0;
	    if (topdown)
	    {
		pos = (pos + rows - 1) % rows;
		memset(text + pos * cols, ' ', cols);
	    }
	    else if (y == rows - 1)
	    {
		memset(text + pos * cols, ' ', cols);
		pos = (pos + 1) % rows;
	    }
	    else
		y++;
	    break;

	case '\t':
	    x = (x / 8 + 1) * 8;
	    if (x >= cols && wrap)
		put("\n");
	    break;

	default:
	    if (*s >= ' ' && x < cols)
	    {
		text[x + ((y + pos) % rows) * cols] = *s;
		if (++x == cols && wrap)
		    put("\n");
	    }
	    break;
	}

	s++;
    }
}


/*
 * event_loop
 */

void event_loop(int Cfd)
{
    fd_set rfds;
    fd_set efds;
    int Xfd = ConnectionNumber(dpy);
    int maxfd = (Cfd > Xfd ? Cfd : Xfd) + 1;
    
    FD_ZERO(&rfds);
    FD_ZERO(&efds);
    
    for (;;)
    {
	while (XPending(dpy) > 0)
	{
	    XEvent ev;
	    XMaskEvent(dpy, ExposureMask, &ev);
	    handle_expose((XExposeEvent*)&ev.xexpose);
	}

	FD_SET(Xfd, &rfds);
	FD_SET(Cfd, &rfds);
	FD_SET(Cfd, &efds);
	if (select(maxfd, &rfds, NULL, &efds, NULL) == -1)
	{
	    perror("select");
	    exit(EXIT_FAILURE);
	}

	if (FD_ISSET(Cfd, &efds))
	{
	    close(Cfd);
	    return;
	}

	if (FD_ISSET(Cfd, &rfds))
	{
	    char buf[1024];
	    int n;

	    do
	    {
		n = read(Cfd, buf, sizeof(buf) - 1);
		if (n == -1)
		{
		    if (errno != EAGAIN)
		    {
			close(Cfd);
			return;
		    }
		}
		else if (n > 0)
		{
		    buf[n] = '\0';
		    put(buf);
		}
	    }
	    while (n > 0);

	    XClearArea(dpy, win, 0, 0, 0, 0, True);
	}
    }
}
