/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPStream.cpp

	Contains:	Implementation of RTPStream class. 
					
	$Log: RTPStream.cpp,v $
	Revision 1.4  1999/02/24 19:27:20  ds
	RTPStream now uses new UDPSocketPool interface for getting its UDP sockets
	
	Revision 1.2  1999/02/19 23:08:27  ds
	Created
	

*/

#include <stdlib.h>
#include "OS.h"

#include "RTPStream.h"
#include "RTPModule.h"
#include "RTPServerInterface.h"
#include "RTCPPacket.h"
#include "RTCPAPPPacket.h"

#define RTP_PACKET_CONTENT_LOGGING 0

#if RTP_PACKET_CONTENT_LOGGING
UInt32 gPacketCount = 0;
FILE*	gPacketFile = NULL;
bool gDone = false;
#endif

RTPStream::RTPStream(UInt32 inStreamID, RTPSession* inSession,
						StrPtrLen* inCodecName, UInt32 inCodecType)
: 	fStreamID(inStreamID),
	fSession(inSession),
	fSockets(NULL),
	fRemoteAddr(0),
	fRemoteRTPPort(0),
	fRemoteRTCPPort(0),
	fLastSenderReportTime(0),
	fPacketCount(0),
	fByteCount(0),
	fSsrcStringPtr(fSsrcString, 0),
	fSeqRandomOffset(0),
	fLastSequenceNumber(0),
	fFirstSeqNumber(0),
	fHasFirstSeqNumber(false),
	fFirstRandomOffset(0),
	fTimestampRandomOffset(0),
	fFirstTimeStamp(0),
	fHasFirstTimeStamp(false),
	fStreamURLPtr(fStreamURL, 0),
	fQueueElem(this),
	fSendingRTCP(true),
	fWritingHeader(true),
	fQualityLevel(0),
	fCodecType(inCodecType),
	fNextSeqNumber(0),
	fSeqNumberOffset(0)
{
	Assert(inSession != NULL);
	Assert(inSession->GetStreamQueue() != NULL);
	
	//set the ssrc. This should be a random number, unique wrt all other
	//streams in this session.
	bool isUnique = false;
	while (!isUnique)
	{
		isUnique = true;
		fSsrc = (SInt32)::rand();
		for (OSQueueIter qIter(inSession->GetStreamQueue()); !qIter.IsDone(); qIter.Next())
		{
			Assert(qIter.GetCurrent() != NULL);
			Assert(qIter.GetCurrent()->GetEnclosingObject() != NULL);
			if (((RTPStream*)qIter.GetCurrent()->GetEnclosingObject())->fSsrc == fSsrc)
				isUnique = false;
		}
	}
	
	//format the ssrc as a string
	::sprintf(fSsrcString, "%ld", fSsrc);
	fSsrcStringPtr.Len = ::strlen(fSsrcString);
	Assert(fSsrcStringPtr.Len < kMaxSsrcSizeInBytes);
	
	//copy the Codec name and store it
	if ((inCodecName != NULL) && (inCodecName->Len > 0) && (inCodecName->Ptr != NULL))
	{
		fCodecName.Ptr = new ('cdnm') char[inCodecName->Len + 2];
		::memcpy(fCodecName.Ptr, inCodecName->Ptr, inCodecName->Len);
		fCodecName.Ptr[inCodecName->Len] = '\0';
		fCodecName.Len = inCodecName->Len;
	}
	
	//add the stream to the queue.
	inSession->GetStreamQueue()->EnQueue(&fQueueElem);
}

RTPStream::~RTPStream()
{
	QTSS_ErrorCode err = QTSS_NoErr;
	if (fSockets != NULL)
	{
		Assert(fSockets->GetSocketB()->GetDemuxer() != NULL);
		fSockets->GetSocketB()->GetDemuxer()->
			UnregisterTask(fRemoteAddr, fRemoteRTCPPort, this);
		Assert(err == QTSS_NoErr);
	
		RTPServerInterface::GetSocketPool()->ReleaseUDPSocketPair(fSockets);
	}
	
	//make sure to free up the codec name if we've allocated it
	if (fCodecName.Ptr != NULL)
		delete [] fCodecName.Ptr;
		
	fQueueElem.Remove();//make sure to get off the session's queue!	
}

RTSPProtocol::RTSPStatusCode RTPStream::Setup(RTSPRequestInterface* request)
{
	//Get the URL for this track
	StrPtrLen* fileName = request->GetQTSSParameter(qtssFileNameParam);
	Assert(fileName != NULL);
	Assert(fileName->Len < kMaxStreamURLSizeInBytes);
	::memcpy(fStreamURL, fileName->Ptr, fileName->Len);
	fStreamURLPtr.Len = fileName->Len;
	fStreamURL[fileName->Len] = '\0';//just in case someone wants to use string routines
	
	//Get and store the remote addresses provided by the client
	fRemoteAddr = request->GetSession()->GetSocket()->GetRemoteAddr();
	fRemoteRTPPort = request->GetClientPortA();
	fRemoteRTCPPort = request->GetClientPortB();

	if (fRemoteRTPPort == 0)
		return request->SendErrorResponse(RTSPProtocol::kClientBadRequest,
									RTSPMessages::kNoClientPortInTransport);
		
	if (fRemoteRTCPPort == 0)
		return request->SendErrorResponse(RTSPProtocol::kClientBadRequest,
										RTSPMessages::kNoClientPortInTransport);
	
	//make sure that the client is advertising an even-numbered RTP port,
	//and that the RTCP port is actually one greater than the RTP port
	if ((fRemoteRTPPort & 1) != 0)
		return request->SendErrorResponse(RTSPProtocol::kClientBadRequest,
										RTSPMessages::kRTPPortMustBeEven);
		
	if (fRemoteRTCPPort != (fRemoteRTPPort + 1))
		return request->SendErrorResponse(RTSPProtocol::kClientBadRequest,
										RTSPMessages::kRTCPPortMustBeOneBigger);
	
	//get a UDP socket pair for this stream
	UInt32 clientAddr = request->GetSession()->GetSocket()->GetLocalAddr();
	fSockets = RTPServerInterface::GetSocketPool()->GetUDPSocketPair(clientAddr, 0, fRemoteAddr, 
																		fRemoteRTCPPort);
	if (fSockets == NULL)
		return request->SendErrorResponse(RTSPProtocol::kServerInternal,
										RTSPMessages::kOutOfPorts);
		
	//finally, register with the demuxer to get RTCP packets from the proper address
	Assert(fSockets->GetSocketB()->GetDemuxer() != NULL);
	QTSS_ErrorCode err = fSockets->GetSocketB()->GetDemuxer()->
		RegisterTask(fRemoteAddr, fRemoteRTCPPort, this);
	//errors should only be returned if there is a routing problem, there should be none
	Assert(err == QTSS_NoErr);
	return RTSPProtocol::kSuccessOK;
}

void RTPStream::Play(RTSPRequestInterface* request, bool inSendRTCP, bool inShouldWriteHeader)
{
	Assert(fSession != NULL);
	Assert(request != NULL);
	fSendingRTCP = inSendRTCP;
	fWritingHeader = inShouldWriteHeader;
	
	if (fSession->GetState() == RTPSession::kPlayingState)
		return;
	
	//get all the necessary information from the module
	SInt16 firstSequenceNumber = 0;
	SInt16 seqNumRandomOffset = 0;
	SInt32 rtpTimescale = 0;
	SInt32 firstRTPTimestamp = 0;
	bool hasSeqNumRandomOffset = false;
	
	fHasFirstSeqNumber = false;
	fHasFirstTimeStamp = false;
	
	//Retrieve the relevent content parameters from the module. Modules may or may
	//not implement these, and whether they do affects what the server does with the
	//request. Therefore, we must mark (with a flag variable) whether the values
	//were successfully retreived.
	
	//firstSequenceNumber
	UInt32 paramSize = sizeof(firstSequenceNumber);
	QTSS_ErrorCode err = fSession->GetModule()->
			GetParameter(qtssFirstSequenceNumber, fSession, this, &firstSequenceNumber, &paramSize);
	if ((err == QTSS_NoErr) && (paramSize == sizeof(firstSequenceNumber)))
		fHasFirstSeqNumber = true;
		
	//first rtp timestamp
	paramSize = sizeof(firstRTPTimestamp);
	err = fSession->GetModule()->
			GetParameter(qtssFirstTimestamp, fSession, this, &firstRTPTimestamp, &paramSize);
	if ((err == QTSS_NoErr) && (paramSize == sizeof(firstRTPTimestamp)))
		fHasFirstTimeStamp = true;
	
	//seqNumRandomOffset
	paramSize = sizeof(seqNumRandomOffset);
	err = fSession->GetModule()->
			GetParameter(qtssSeqNumRandomOffset, fSession, this, &seqNumRandomOffset, &paramSize);
	if ((err == QTSS_NoErr) && (paramSize == sizeof(seqNumRandomOffset)))
		hasSeqNumRandomOffset = true;	
	
	//If RTP timescale isn't available to the module, just assume its 0. This messes up
	//the random offsets, but, well, what else are we supposed to do?
	
	paramSize = sizeof(rtpTimescale);
	err = fSession->GetModule()->
			GetParameter(qtssTimescale, fSession, this, &rtpTimescale, &paramSize);
	if ((err != QTSS_NoErr) || (paramSize != sizeof(rtpTimescale)))
		rtpTimescale = 0;

	//ok, we have all the necessary parameters, now lets do the necessary computation on
	//this info.
	
	//compute the random sequence number offset for this play spurt
	if (fSession->IsFirstPlay())
	{
		if (hasSeqNumRandomOffset)
			fSeqRandomOffset = seqNumRandomOffset;
		else
		{
			UInt8 randomChar = (UInt8)::rand();
			fSeqRandomOffset = (SInt16)randomChar;
		}
	}
	else
		//this ensures that the seq#'s will always increase
		fSeqRandomOffset = (fLastSequenceNumber + 1) - firstSequenceNumber;

	//If the server isn't writing the headers, it shouldn't do any random offset
	if (!fWritingHeader)
		fSeqRandomOffset = 0;
		
	fFirstSeqNumber = firstSequenceNumber + fSeqRandomOffset;
	fLastSequenceNumber = fFirstSeqNumber;
	
	//RTP timestamp random offset computation
	if (fSession->IsFirstPlay())
		fFirstRandomOffset = (SInt32)::rand();
	fTimestampRandomOffset = (SInt32)(fFirstRandomOffset + 
							(rtpTimescale * (fSession->GetPlayTime() -
												fSession->GetFirstPlayTime()) / 1000));
	fTimestampRandomOffset -= firstRTPTimestamp;

	//same as above.
	if (!fWritingHeader)
		fTimestampRandomOffset = 0;

	fFirstTimeStamp = firstRTPTimestamp + fTimestampRandomOffset;
}

void RTPStream::AppendTransport(RTSPRequestInterface* request)
{
	//RTP sockets should already be bound to the proper IP address... report that to client
	UDPSocket* theRTPSocket = fSockets->GetSocketA();
	UDPSocket* theRTCPSocket = fSockets->GetSocketB();
	Assert(theRTPSocket->GetLocalAddr() != INADDR_ANY);
	
	//We don't have to append the source IP addr to the transport header, because
	//it is always the same as the RTSP IP addr (the code in Setup() ensures this.
	request->AppendTransportHeader(NULL, theRTPSocket->GetLocalPortStr(), theRTCPSocket->GetLocalPortStr());
}

void	RTPStream::AppendRTPInfo(RTSPProtocol::RTSPHeader inHeader, RTSPRequestInterface* request)
{
	//format strings for the various numbers we need to send back to the client
	char rtpTimeBuf[20];
	StrPtrLen rtpTimeBufPtr;
	if (fHasFirstTimeStamp)
	{
		::sprintf(rtpTimeBuf, "%ld", fFirstTimeStamp);
		rtpTimeBufPtr.Set(rtpTimeBuf, ::strlen(rtpTimeBuf));
		Assert(rtpTimeBufPtr.Len < 20);
	}	
	
	char seqNumberBuf[20];
	StrPtrLen seqNumberBufPtr;
	if (fHasFirstSeqNumber)
	{
		::sprintf(seqNumberBuf, "%d", fFirstSeqNumber);
		seqNumberBufPtr.Set(seqNumberBuf, ::strlen(seqNumberBuf));
		Assert(seqNumberBufPtr.Len < 20);
	}
	
	//only advertise our ssrc to the client if we are in fact sending RTCP
	//sender reports. Otherwise, this ssrc is invalid!
	UInt32 theSsrcStringLen = fSsrcStringPtr.Len;
	if (!fSendingRTCP)
		fSsrcStringPtr.Len = 0;
	
#if RTPSESSION_LOGGING
	if (fSession->GetDebugFile() != NULL)
		fprintf(fSession->GetDebugFile(), "First seq number: %s, first timestamp: %s. ssrc: %s\n",seqNumberBufPtr.Ptr, rtpTimeBufPtr.Ptr, fSsrcStringPtr.Ptr);
#endif

	request->AppendRTPInfoHeader(inHeader, &fStreamURLPtr, &seqNumberBufPtr,
							&fSsrcStringPtr, &rtpTimeBufPtr);
	fSsrcStringPtr.Len = theSsrcStringLen;
}

void	RTPStream::SendRTCP(char* inData, UInt32 inLen)
{
	fSockets->GetSocketB()->SendTo(fRemoteAddr, fRemoteRTCPPort, inData, inLen);
}

void	RTPStream::SendRTP(char* inData, UInt32 inLen)
{
	Assert(inData != NULL);
	Assert(inLen > 12);
	if ((inData == NULL) || (inLen < 12))
		return;
	
	//append random offsets
	UInt16* seqNumberP = (UInt16*)inData;
	UInt32* timeStampP = (UInt32*)inData;

	//Write the packet header if we've been instructed to by the module
	if (fWritingHeader)
	{
		seqNumberP[0] |= 0x8000;//this is the RTP version number (2)
		seqNumberP[1] += fSeqRandomOffset;
		timeStampP[1] += fTimestampRandomOffset;
		timeStampP[2] = fSsrc;//ssrc belongs here.
	}

	//mark the last seq number for the purposes of appending a new random offset
	fLastSequenceNumber = ntohs(seqNumberP[1]);

#if RTPSESSION_LOGGING
	if (fSession->GetDebugFile() != NULL)
		fprintf(fSession->GetDebugFile(), "Sending a packet: seq number %d, timestamp %ld, ssrc %ld\n", seqNumberP[1], timeStampP[1], timeStampP[2]);
#endif

#if RTP_PACKET_CONTENT_LOGGING
	if (!gDone)
	{
		if (gPacketFile == NULL)
			gPacketFile = fopen("TimeSharePackets", "w");
		Assert(gPacketFile != NULL);
		if (fCodecType == kAudioCodecType)
		{
			fprintf(gPacketFile, "Packet # %d\n",gPacketCount);
			fwrite(inData+12, inLen-12, 1, gPacketFile);
			fprintf(gPacketFile, "\n");
			gPacketCount++;
			if (gPacketCount == 1000)
			{
				fclose(gPacketFile);
				gDone = true;
			}
			//printf("Packet length: %d\n",inLen-12);
		}
	}
#endif

	fSockets->GetSocketA()->SendTo(fRemoteAddr, fRemoteRTPPort, inData, inLen);
	fSession->UpdatePacketsSent(1);
	fSession->UpdateBytesSent(inLen);
	RTPServerInterface::IncrementTotalBytes(inLen);
	RTPServerInterface::IncrementTotalPackets();
	
	//stream statistics
	fPacketCount++;
	fByteCount += inLen;
	
	//only send RTCP packets if we're supposed to!
	if (!fSendingRTCP)
		return;
			
	//Send an RTCP sender report if it's time
	SInt64 theTime = OS::Milliseconds();
	if (theTime > (fLastSenderReportTime + (kSenderReportIntervalInSecs * 1000)))
	{
		fLastSenderReportTime = theTime;
		
		UInt32* theReport = (UInt32*)fSession->GetRTCPSR();
		theReport++;
		*theReport = htonl(fSsrc);
		theReport++;
		SInt64* theNTPTimestampP = (SInt64*)theReport;
		*theNTPTimestampP = OS::HostToNetworkSInt64(fSession->GetNTPPlayTime() +
								OS::ConvertMsecToFixed64Sec(theTime - fSession->GetPlayTime()));
		theReport += 2;
		*theReport = htonl(timeStampP[1]);
		theReport++;
		*theReport = htonl(fPacketCount);
		theReport++;
		*theReport = htonl(fByteCount);
		theReport += 2;
		*theReport = htonl(fSsrc);
		
		fSockets->GetSocketB()->SendTo(fRemoteAddr, fRemoteRTCPPort,
								fSession->GetRTCPSR(), fSession->GetRTCPSRLen());
	}
}

void RTPStream::ProcessPacket(StrPtrLen* inPacket)
{
#if RTPSESSION_DEBUGGING
	printf("RTPStream %ld: received RTCP packet\n",(SInt32)this);
#endif

	StrPtrLen currentPtr(*inPacket);
	
	//no matter what happens (whether or not this is a valid packet) reset the timeout
	fSession->RearmTimeout();
	while ( currentPtr.Len > 0 )
	{
		/*
			Due to the variable-type nature of RTCP packets, this is a bit unusual...
			We initially treat the packet as a generic RTCPPacket in order to determine its'
			actual packet type.  Once that is figgered out, we treat it as its' actual packet type
		*/
		RTCPPacket rtcpPacket;
		if (!rtcpPacket.ParsePacket((UInt8*)currentPtr.Ptr, currentPtr.Len))
			return;//abort if we discover a malformed RTCP packet

		//if this is the first packet, verify that it is an RR and the padding bit is 0 (see RTP RFC)
		if (currentPtr.Ptr == inPacket->Ptr)
			if ((rtcpPacket.GetPacketType() != RTCPReceiverPacket::kPacketType) ||
			 	(rtcpPacket.GetHasPadding() == true))
			 	return;
		
		switch (rtcpPacket.GetPacketType())
		{
			case RTCPReceiverPacket::kPacketType:
			{
				RTCPReceiverPacket receiverPacket;
				if (!receiverPacket.ParseReceiverReport((UInt8*)currentPtr.Ptr, currentPtr.Len))
					return;//abort if we discover a malformed receiver report

				//stuff values into the session's dictionary
				UInt32 uint32Val = 0;
				
				uint32Val = receiverPacket.GetCumulativeFractionLostPackets();
				fStreamDictionary.SetStandardValue( StreamDictionary::kFractionLostPackets, &uint32Val, sizeof(uint32Val) );
				
				uint32Val = receiverPacket.GetCumulativeTotalLostPackets();
				fStreamDictionary.SetStandardValue( StreamDictionary::kTotalLostPackets, &uint32Val, sizeof(uint32Val) );
			
				uint32Val = receiverPacket.GetCumulativeJitter();
				fStreamDictionary.SetStandardValue( StreamDictionary::kJitter, &uint32Val, sizeof(uint32Val) );
				
			}
			break;
			
			case RTCPCompressedQTSSPacket::kPacketType:
			{
				RTCPCompressedQTSSPacket compressedQTSSPacket;
				if (!compressedQTSSPacket.ParseCompressedQTSSPacket((UInt8*)currentPtr.Ptr, currentPtr.Len))
					return;//abort if we discover a malformed app packet

				//stuff values into the session's dictionary
				UInt16 uint16Val = 0;
				UInt32 uint32Val = 0;

				uint32Val = compressedQTSSPacket.GetReceiverBitRate();
				fStreamDictionary.SetStandardValue( StreamDictionary::kReceiverBitRate, &uint32Val, sizeof(uint32Val) );
				
				uint16Val = compressedQTSSPacket.GetAverageLateMilliseconds();
				fStreamDictionary.SetStandardValue( StreamDictionary::kAvgLateMilliseconds, &uint16Val, sizeof(uint16Val) );
				
				uint16Val = compressedQTSSPacket.GetPercentPacketsLost();
				fStreamDictionary.SetStandardValue( StreamDictionary::kPercentPacketsLost, &uint16Val, sizeof(uint16Val) );
				
				uint16Val = compressedQTSSPacket.GetAverageBufferDelayMilliseconds();
				fStreamDictionary.SetStandardValue( StreamDictionary::kAverageBufferDelayMilliseconds, &uint16Val, sizeof(uint16Val) );

				uint16Val = (int)compressedQTSSPacket.GetIsGettingBetter();
				fStreamDictionary.SetStandardValue( StreamDictionary::kIsGettingBetter, &uint16Val, sizeof(uint16Val) );

				uint16Val = (int)compressedQTSSPacket.GetIsGettingWorse();
				fStreamDictionary.SetStandardValue( StreamDictionary::kIsGettingWorse, &uint16Val, sizeof(uint16Val) );

				uint32Val = compressedQTSSPacket.GetNumEyes();
				fStreamDictionary.SetStandardValue( StreamDictionary::kNumEyes, &uint32Val, sizeof(uint32Val) );

				uint32Val = compressedQTSSPacket.GetNumEyesActive();
				fStreamDictionary.SetStandardValue( StreamDictionary::kNumEyesActive, &uint32Val, sizeof(uint32Val) );

				uint32Val = compressedQTSSPacket.GetNumEyesPaused();
				fStreamDictionary.SetStandardValue( StreamDictionary::kNumEyesPaused, &uint32Val, sizeof(uint32Val) );

				uint32Val = compressedQTSSPacket.GetTotalPacketReceived();
				fStreamDictionary.SetStandardValue( StreamDictionary::kTotalPacketsReceived, &uint32Val, sizeof(uint32Val) );
				
				uint16Val = compressedQTSSPacket.GetTotalPacketsDropped();
				fStreamDictionary.SetStandardValue( StreamDictionary::kTotalPacketsDropped, &uint16Val, sizeof(uint16Val) );
				
				uint16Val = compressedQTSSPacket.GetTotalPacketsLost();
				fStreamDictionary.SetStandardValue( StreamDictionary::kTotalPacketsLost, &uint16Val, sizeof(uint16Val) );
				
				uint16Val = compressedQTSSPacket.GetClientBufferFill();
				fStreamDictionary.SetStandardValue( StreamDictionary::kClientBufferFill, &uint16Val, sizeof(uint16Val) );
				
				uint16Val = compressedQTSSPacket.GetFrameRate();
				fStreamDictionary.SetStandardValue( StreamDictionary::kFrameRate, &uint16Val, sizeof(uint16Val) );
				
				uint16Val = compressedQTSSPacket.GetExpectedFrameRate();
				fStreamDictionary.SetStandardValue( StreamDictionary::kExpectedFrameRate, &uint16Val, sizeof(uint16Val) );
				
				uint16Val = compressedQTSSPacket.GetAudioDryCount();
				fStreamDictionary.SetStandardValue( StreamDictionary::kAudioDryCount, &uint16Val, sizeof(uint16Val) );
				
			}
			break;
			
			case SourceDescriptionPacket::kPacketType:
			{
#ifdef DEBUG_RTCP_PACKETS
				SourceDescriptionPacket sedsPacket;
				if (!sedsPacket.Parse(currentPtr.Ptr, currentPtr.Len))
					return;//abort if we discover a malformed app packet

				sedsPacket.Dump();
#endif
			}
			break;
			
			//default:
			//	WarnV(false, "Unknown RTCP Packet Type");
			break;
		
		}
		
		
		currentPtr.Ptr += (rtcpPacket.GetPacketLength() * 4 ) + 4;
		currentPtr.Len -= (rtcpPacket.GetPacketLength() * 4 ) + 4;
	}

#ifdef DEBUG_RTCP_PACKETS
	this->DumpDictionary();
#endif

	//make sure to lock down the session before invoking RTCP modules
	OSMutexLocker locker(fSession->GetSessionMutex());
	// Seems a little redundant to make each module parse again...but probably the right thing to do?
	RTPModuleInterface::ExecuteRTCPModules(this, inPacket);
}

void RTPStream::DumpDictionary()
{	
	//stuff values into the session's dictionary
	int intVal = 0;
	UInt16 uint16Val = 0;
	UInt32 uint32Val = 0;
	UInt32 actualLen = 0;
	
	
	printf("\nStreamDictionary::  \n	");

	fStreamDictionary.GetStandardValue( StreamDictionary::kReceiverBitRate, &uint32Val, sizeof(uint32Val), &actualLen );
	printf("kReceiverBitRate=%lu[%lu]", uint32Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kAvgLateMilliseconds, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kAvgLateMilliseconds=%u[%lu]", uint16Val, actualLen);
	
	fStreamDictionary.GetStandardValue( StreamDictionary::kPercentPacketsLost, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kPercentPacketsLost=%u[%lu]", uint16Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kAverageBufferDelayMilliseconds, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kAverageBufferDelayMilliseconds=%u[%lu]", uint16Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kIsGettingBetter, &intVal, sizeof(intVal), &actualLen );
	printf(", kIsGettingBetter=%d[%lu]", intVal, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kIsGettingWorse, &intVal, sizeof(intVal), &actualLen );
	printf(", kIsGettingWorse=%d[%lu]", intVal, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kNumEyes, &uint32Val, sizeof(uint32Val), &actualLen );
	printf(", kNumEyes=%lu[%lu]", uint32Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kNumEyesActive, &uint32Val, sizeof(uint32Val), &actualLen );
	printf(", kNumEyesActive=%lu[%lu]", uint32Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kNumEyesPaused, &uint32Val, sizeof(uint32Val), &actualLen );
	printf(", kNumEyesPaused=%lu[%lu]", uint32Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kTotalPacketsReceived, &uint32Val, sizeof(uint32Val), &actualLen );
	printf(", kTotalPacketsReceived=%lu[%lu]", uint32Val, actualLen);
	
	fStreamDictionary.GetStandardValue( StreamDictionary::kTotalPacketsDropped, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kTotalPacketsDropped=%u[%lu]", uint16Val, actualLen);
	
	fStreamDictionary.GetStandardValue( StreamDictionary::kClientBufferFill, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kClientBufferFill=%u[%lu]", uint16Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kFrameRate, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kFrameRate=%u[%lu]", uint16Val, actualLen);

	fStreamDictionary.GetStandardValue( StreamDictionary::kExpectedFrameRate, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kExpectedFrameRate=%u[%lu]", uint16Val, actualLen);
	
	fStreamDictionary.GetStandardValue( StreamDictionary::kAudioDryCount, &uint16Val, sizeof(uint16Val), &actualLen );
	printf(", kAudioDryCount=%u[%lu]", uint16Val, actualLen);
	
	printf("\n\n");
}





