/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPRequestStream.cpp

	Contains:	Implementation of RTSPRequestStream class. 
					
	$Log: RTSPRequestStream.cpp,v $
	Revision 1.2  1999/02/19 23:08:33  ds
	Created
	

*/


#include "RTSPRequestStream.h"
#include "StringParser.h"
#include "OS.h"


RTSPRequestStream::RTSPRequestStream(TCPSocket* sock)
: 	fSocket(sock), fRetreatBytes(0), 
	fRequest(fRequestBuffer, 0),
	fRequestPtr(NULL)
{}

SInt32 RTSPRequestStream::GetRetreatBytes(char *buffer, SInt32 length)
{
	SInt32 lengthtoread = 0;
	
	//in case data has been put back into the stream
	if (fRetreatBytes > 0) {

		if (length > fRetreatBytes)
			lengthtoread = fRetreatBytes;
		else
			lengthtoread = length;

		Assert(fRequest.Ptr != NULL);
		char *end = fRequest.Ptr + fRequest.Len;
		
		if (end != buffer)
			//use memmove. These might be overlapping buffers
			::memmove(buffer, end, lengthtoread);
			
		fRetreatBytes -= lengthtoread;
		fRequest.Len += lengthtoread;
		Assert(fRequest.Len < kRequestBufferSizeInBytes);
	}
	return lengthtoread;
}

QTSS_ErrorCode
RTSPRequestStream::ReadRequest(FILE* inDebugFile)
{
	while (true)
	{
		UInt32 newOffset = 0;
		
		//If this is the case, we already HAVE a request on this session, and we now are done
		//with the request and want to move onto the next one. The first thing we should do
		//is check whether there is any lingering data in the stream. If there is, the parent
		//session believes that is part of a new request
		if (fRequestPtr != NULL)
		{
			fRequestPtr = NULL;//flag that we no longer have a complete request
			
			//calling this function will move all the data to the beginning
			//of the buffer
#if DEBUG
			SInt32 tempRetreatBytes = fRetreatBytes;
#endif
			newOffset = GetRetreatBytes(fRequest.Ptr, fRetreatBytes);
#if DEBUG
			Assert((SInt32)newOffset == tempRetreatBytes);
			Assert(fRetreatBytes == 0);
#endif
			fRequest.Len = newOffset;
			if (inDebugFile != NULL)
			{
				fRequest.Ptr[fRequest.Len] = '\0';
				fprintf(inDebugFile, "Time: %qd. ReadRequest: Got %ld retreat bytes. Data: %s\n", OS::Milliseconds(), newOffset, fRequest.Ptr);
			}
		}	

		//if we have no new data yet, get some from the socket
		if (newOffset == 0)
		{
			QTSS_ErrorCode sockErr = fSocket->Read(fRequest.Ptr + fRequest.Len, 
												(kRequestBufferSizeInBytes - fRequest.Len) - 1, &newOffset);
			fRequest.Len += newOffset;
		
			//assume the client is dead if we get an error back
			if (sockErr != QTSS_NoErr)
			{
				if (inDebugFile != NULL)
					fprintf(inDebugFile, "Time: %qd. ReadRequest. Got %ld error back from fSocket->Read()\n", OS::Milliseconds(), sockErr);
				Assert(!fSocket->IsConnected());
				return sockErr;
			}	
			if (inDebugFile != NULL)
			{
				fRequest.Ptr[fRequest.Len] = '\0';
				fprintf(inDebugFile, "Time: %qd. ReadRequest: Got %ld bytes from read. Data: %s\n", OS::Milliseconds(), newOffset, fRequest.Ptr);
			}
		}

		//if we've gotten no data back, just return
		if (newOffset == 0)
		{
			if (inDebugFile != NULL)
				fprintf(inDebugFile, "Time: %qd. ReadRequest: Got no bytes. Returning AWSM_NoErr\n", OS::Milliseconds());
			return QTSS_NoErr;
		}
		
		//use a StringParser object to search for a double EOL, which signifies the end of
		//the header.
		bool weAreDone = false;
		StringParser headerParser(&fRequest);
		
		while (headerParser.GetThruEOL(NULL))
		{
			if (headerParser.ExpectEOL())
			{
				//The legal end-of-header sequences are \r\r, \r\n\r\n, & \n\n. NOT \r\n\r!
				//If the packets arrive just a certain way, we could get here with the latter
				//combo, and not wait for a final \n.
				if ((headerParser.GetDataParsedLen() > 2) &&
					(memcmp(headerParser.GetCurrentPosition() - 3, "\r\n\r", 3) == 0))
					continue;
				weAreDone = true;
				break;
			}
		}
		
		//weAreDone means we have gotten a full request
		if (weAreDone)
		{
			//put back any data that is not part of the header
			Retreat(headerParser.GetDataReceivedLen() - headerParser.GetDataParsedLen());
			fRequestPtr = &fRequest;
			if (inDebugFile != NULL)
				fprintf(inDebugFile, "Time: %qd. ReadRequest: Got a complete request. Returning AWSM_RequestArrived\n", OS::Milliseconds());
			return QTSS_RequestArrived;
		}
		
		//check for a full buffer
		if (fRequest.Len == kRequestBufferSizeInBytes - 1)
		{
			fRequestPtr = &fRequest;
			if (inDebugFile != NULL)
				fprintf(inDebugFile, "Time: %qd. ReadRequest: Buffer is full. Returning AWSM_NotEnoughSpace\n", OS::Milliseconds());
			return QTSS_NotEnoughSpace;
		}
	}
}

void RTSPRequestStream::Retreat(SInt32 length)
{ 
	fRequest.Len -= length;
	fRetreatBytes += length;
}
