------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                 I N T E R F A C E S . C . P T H R E A D S                --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.6 $                             --
--                                                                          --
--      Copyright (C) 1991,1992,1993,1994,1995 Florida State University     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is a Solaris version of this package.

--  This package interfaces with Solaris specific thread calls. It is not a
--  complete interface; it only includes what is needed to implement the
--  Ada runtime. The changes made to this Solaris specific version involve
--  the absorbtion of the various attribute structures, and minor interface
--  differences.

with System;
with Interfaces.C.POSIX_RTE;
--  Used for, Signal,
--            Signal_Set

with Interfaces.C.POSIX_error; use Interfaces.C.POSIX_error;
--  Used for, Return_Code
--            Failure

with Interfaces.C.POSIX_Timers;
--  Used for, timespec

with Unchecked_Conversion;

package body Interfaces.C.Sthreads is

   package RTE renames Interfaces.C.POSIX_RTE;

   pragma Linker_Options ("-lthread");
   --  This pragma ensures that the Solaris threads library will be linked
   --  in.

   pragma Linker_Options ("-lposix4");
   --  This pragma is need to for linker to link in the posix4 functions.

   --  These unchecked conversion functions are used to convert a variable
   --  to an access value referencing that variable.  The expression
   --  Address_to_Pointer(X'Address) evaluates to an access value referencing
   --  X; if X is of type T, this expression returns a value of type
   --  access T.  This is necessary to allow structures to be passed to
   --  C functions, since some compiler interfaces to C only allows scalers,
   --  access values, and values of type System.Address as actual parameters.

   --  ??? it would be better to use the routines in System.Storage_Elements
   --  ??? for conversion between pointers and access values. In any case
   --  ??? I don't see the point of these conversions at all, why not pass
   --  ??? Address values directly to the C routines (I = RBKD)

   Failure : POSIX_Error.Return_Code renames POSIX_Error.Failure;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, RTE.sigset_t_ptr);

   type thread_t_ptr is access thread_t;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, thread_t_ptr);

   type mutex_t_ptr is access mutex_t;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, mutex_t_ptr);

   type cond_t_ptr is access cond_t;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, cond_t_ptr);

   type thread_key_t_ptr is access thread_key_t;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, thread_key_t_ptr);

   type Address_Pointer is access System.Address;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, Address_Pointer);

   type timespec_ptr is access POSIX_Timers.timespec;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, timespec_ptr);

   type Int_Ptr is access int;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, Int_Ptr);

   type Priority_Type_Ptr is access Priority_Type;

   function Address_to_Pointer is new
     Unchecked_Conversion (System.Address, Priority_Type_Ptr);

   --------------------
   -- thr_create --
   --------------------

   procedure thr_create
     (
      stack_base    : System.Address;
      stack_size    : size_t;
      start_routine : System.Address;
      arg           : System.Address;
      flags         : Integer;
      thread        : out thread_t;
      result        : out Return_Code)
   is
      function thr_create_base
        (stack_base    : System.Address;
         stack_size    : size_t;
         start_routine : System.Address;
         arg           : System.Address;
         flags         : int;
         thread        : thread_t_ptr)
         return          Return_Code;

      pragma Import (C, thr_create_base, "thr_create");

   begin
      result :=
        thr_create_base (stack_base, stack_size, start_routine, arg,
                         int (flags), Address_to_Pointer (thread'Address));
   end thr_create;

   ------------------------
   -- mutex_init --
   ------------------------

   procedure mutex_init
     (mutex      : out mutex_t;
      mtype      : Integer;
      arg        : System.Address;
      result     : out Return_Code)
   is
      function mutex_init_base
        (mutex : mutex_t_ptr;
         mtype : int;
         arg   : System.Address)
         return  Return_Code;
      pragma Import
        (C, mutex_init_base, "mutex_init");

   begin
      result :=
        mutex_init_base (Address_to_Pointer (mutex'Address),
          int (mtype), arg);
   end mutex_init;

   ---------------------------
   -- mutex_destroy --
   ---------------------------

   procedure mutex_destroy
     (mutex  : in out mutex_t;
      result : out Return_Code)
   is
      function mutex_destroy_base
        (mutex : mutex_t_ptr)
         return  Return_Code;
      pragma Import (C, mutex_destroy_base, "mutex_destroy");

   begin
      result :=
        mutex_destroy_base (Address_to_Pointer (mutex'Address));
   end mutex_destroy;

   ------------------------
   -- mutex_lock --
   ------------------------

   procedure mutex_lock
     (mutex  : in out mutex_t;
      result : out Return_Code)
   is
      function mutex_lock_base
        (mutex : mutex_t_ptr)
         return Return_Code;
      pragma Import (C, mutex_lock_base, "mutex_lock");

   begin
      result := mutex_lock_base (Address_to_Pointer (mutex'Address));
   end mutex_lock;

   --------------------------
   -- mutex_unlock --
   --------------------------

   procedure mutex_unlock
     (mutex  : in out mutex_t;
      result : out Return_Code)
   is
      function mutex_unlock_base
        (mutex : mutex_t_ptr)
         return Return_Code;
      pragma Import (C, mutex_unlock_base, "mutex_unlock");

   begin
      result := mutex_unlock_base (Address_to_Pointer (mutex'Address));
   end mutex_unlock;

   -----------------------
   -- cond_init --
   -----------------------

   procedure cond_init
     (condition  : out cond_t;
      ctype      : Integer;
      arg        : Integer;
      result     : out Return_Code)
   is
      function cond_init_base
        (cond  : cond_t_ptr;
         ctype : int;
         arg   : int)
         return Return_Code;
      pragma Import (C, cond_init_base, "cond_init");

   begin
      result :=
        cond_init_base (Address_to_Pointer (condition'Address),
          int (ctype), int (arg));
   end cond_init;

   -----------------------
   -- cond_wait --
   -----------------------

   procedure cond_wait
     (condition : in out cond_t;
      mutex     : in out mutex_t;
      result    : out Return_Code)
   is
      function cond_wait_base
        (cond  : cond_t_ptr;
         mutex : mutex_t_ptr)
         return  Return_Code;
      pragma Import (C, cond_wait_base, "cond_wait");

   begin
      result :=
        cond_wait_base (Address_to_Pointer (condition'Address),
          Address_to_Pointer (mutex'Address));
   end cond_wait;

   ----------------------------
   -- cond_timedwait --
   ----------------------------

   procedure cond_timedwait
     (condition     : in out cond_t;
      mutex         : in out mutex_t;
      absolute_time : POSIX_Timers.timespec;
      result        : out Return_Code)
   is
      function cond_timedwait_base
        (cond    : cond_t_ptr;
         mutex   : mutex_t_ptr;
         abstime : timespec_ptr)
         return    Return_Code;
      pragma Import (C, cond_timedwait_base, "cond_timedwait");

   begin
      result :=
        cond_timedwait_base (
          Address_to_Pointer (condition'Address),
          Address_to_Pointer (mutex'Address),
          Address_to_Pointer (absolute_time'Address));
   end cond_timedwait;

   -------------------------
   -- cond_signal --
   -------------------------

   procedure cond_signal
     (condition : in out cond_t;
      result    : out Return_Code)
   is
      function cond_signal_base
        (cond : cond_t_ptr)
         return Return_Code;
      pragma Import (C, cond_signal_base, "cond_signal");

   begin
      result :=
        cond_signal_base (Address_to_Pointer (condition'Address));
   end cond_signal;

   ----------------------------
   -- cond_broadcast --
   ----------------------------

   procedure cond_broadcast
     (condition : in out cond_t;
      result    : out Return_Code)
   is
      function cond_broadcast_base
        (cond : cond_t_ptr)
         return Return_Code;
      pragma Import (C, cond_broadcast_base, "cond_broadcast");

   begin
      result :=
        cond_broadcast_base (Address_to_Pointer (condition'Address));
   end cond_broadcast;

   --------------------------
   -- cond_destroy --
   --------------------------

   procedure cond_destroy
     (condition : in out cond_t;
      result    : out Return_Code)
   is
      function cond_destroy_base
        (cond : cond_t_ptr)
         return Return_Code;
      pragma Import (C, cond_destroy_base, "cond_destroy");

   begin
      result :=
        cond_destroy_base (Address_to_Pointer (condition'Address));
   end cond_destroy;

   -------------------------
   -- thr_setspecific --
   -------------------------

   --  Suppress all checks to prevent stack check on entering routine
   --  which routine does this comment belong in???
   --  need pragma Suppress in spec for routine???
   --  Also need documentation of why suppress is needed ???

   procedure thr_setspecific
     (key    : thread_key_t;
      value  : System.Address;
      result : out Return_Code)
   is
      function thr_setspecific_base
        (key   : thread_key_t;
         value : System.Address)
         return  Return_Code;
      pragma Import (C, thr_setspecific_base, "thr_setspecific");

   begin
      result := thr_setspecific_base (key, value);
   end thr_setspecific;

   -------------------------
   -- thr_getspecific --
   -------------------------

   procedure thr_getspecific
     (key    : thread_key_t;
      value  : out System.Address;
      result : out Return_Code)
   is
      function thr_getspecific_base
        (key   : thread_key_t;
         value : Address_Pointer)
         return  Return_Code;
      pragma Import (C, thr_getspecific_base, "thr_getspecific");

   begin
      result :=
        thr_getspecific_base (key, Address_to_Pointer (value'Address));
   end thr_getspecific;

   ------------------------
   -- thr_keycreate --
   ------------------------

   procedure thr_keycreate
     (key        : in out thread_key_t;
      destructor : System.Address;
      result     : out Return_Code)
   is
      function thr_keycreate_base
        (key        : thread_key_t_ptr;
         destructor : System.Address)
         return       Return_Code;
      pragma Import (C, thr_keycreate_base, "thr_keycreate");

   begin
      result :=
        thr_keycreate_base (Address_to_Pointer (key'Address), destructor);
   end thr_keycreate;

   --------------------------
   -- thr_setprio --
   --------------------------

   procedure thr_setprio
     (thread   : in out thread_t;
      priority : Priority_Type;
      result   : out Return_Code)
   is
      function thr_setprio_base
        (thread   : thread_t;
         priority : Priority_Type)
         return     Return_Code;
      pragma Import (C, thr_setprio_base, "thr_setprio");

   begin
      result :=
        thr_setprio_base (thread, priority);
   end thr_setprio;

   --------------------------
   -- thr_getprio --
   --------------------------

   procedure thr_getprio
     (thread   : thread_t;
      priority : out Priority_Type;
      result   : out Return_Code)
   is
      function thr_getprio_base
        (thread   : thread_t;
         Priority : Priority_Type_Ptr)
         return Return_Code;
      pragma Import (C, thr_getprio_base, "thr_getprio");

   begin
      result :=
        thr_getprio_base (thread, Address_to_Pointer (priority'Address));
   end thr_getprio;

   ------------------------
   -- thr_setconcurrency --
   ------------------------

   procedure thr_setconcurrency
     (new_level : Integer;
      result    : out Return_Code) is

      function thr_setconcurrency_base
        (new_level : int) return Return_Code;
      pragma Import (C, thr_setconcurrency_base, "thr_setconcurrency");

   begin
      result := thr_setconcurrency_base (int (new_level));
   end thr_setconcurrency;

   ------------------------
   -- thr_getconcurrency --
   ------------------------

   function thr_getconcurrency return Integer is
      function thr_getconcurrency_base return int;
      pragma Import (C, thr_getconcurrency_base, "thr_getconcurrency");
   begin
      return Integer (thr_getconcurrency_base);
   end thr_getconcurrency;

   ------------------
   -- thr_self --
   ------------------

   function thr_self return thread_t is
      function thr_self_base return thread_t;
      pragma Import (C, thr_self_base, "thr_self");

   begin
      return thr_self_base;
   end thr_self;

   -------------
   -- sigwait --
   -------------

   procedure sigwait
     (set         : RTE.Signal_Set;
      sig         : out RTE.Signal;
      result      : out Return_Code)
   is
      Temp_Result : Return_Code;

      function sigwait_base
        (set : RTE.sigset_t_ptr) return Return_Code;
      pragma Import (C, sigwait_base, "sigwait");

   begin
      Temp_Result := sigwait_base (Address_to_Pointer (set'Address));

      if Temp_Result /= Failure then
         sig := RTE.Signal (Temp_Result);
      else
         sig := 0;
      end if;

      result := Temp_Result;
   end sigwait;

   ------------------
   -- thr_kill --
   ------------------

   procedure thr_kill
     (thread : thread_t;
      sig    : RTE.Signal;
      result : out Return_Code)
   is
      function thr_kill_base
        (thread : thread_t;
         sig    : RTE.Signal)
         return   Return_Code;
      pragma Import (C, thr_kill_base, "thr_kill");

   begin
      result := thr_kill_base (thread, sig);
   end thr_kill;


   --------------------
   -- thr_sigsetmask --
   --------------------

   procedure thr_sigsetmask
     (how    : int;
      set    : access POSIX_RTE.Signal_Set;
      oset   : access POSIX_RTE.Signal_Set;
      Result : out Return_Code)
   is
      function thr_sigsetmask_base
        (how  : int;
         set  : access POSIX_RTE.Signal_Set;
         oset : access POSIX_RTE.Signal_Set)
         return POSIX_Error.Return_Code;
      pragma Import (C, thr_sigsetmask_base, "thr_sigsetmask");

   begin
      Result := thr_sigsetmask_base (how, set, oset);
   end thr_sigsetmask;

end Interfaces.C.Sthreads;
