/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/
/***********************************************************************
 *
 * mwm menu code
 *
 ***********************************************************************/

#include "mwm.h"

#include <ctype.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <Xm/MessageB.h>

extern XEvent   Event;
extern MwmWindow *Tmp_win;
extern int      menuFromFrameOrWindowOrTitlebar;
extern          DoHandlePageing;

extern char   **g_argv;

static Boolean  cancel = False;

static void
cancel_cb(Widget w, XtPointer calldata, XtPointer cbs)
{
    cancel = True;
}

static void
quit_cb(Widget w, XtPointer calldata, XtPointer cbs)
{
    Done(0, NULL);
}

static void
restart_cb(Widget w, XtPointer calldata, XtPointer cbs)
{
    char           *action;

    XtVaGetValues(w, XmNuserData, (XtPointer) & action, NULL);
    Done(1, action);
}

/***********************************************************************
 *
 *  Procedure:
 *	ExecuteFunction - execute a mwm built in function
 *
 *  Inputs:
 *	func	- the function to execute
 *	action	- the menu action to execute 
 *	w	- the window to execute this function on
 *	tmp_win	- the mwm window structure
 *	event	- the event that caused the function
 *	context - the context in which the button was pressed
 *      val1,val2 - the distances to move in a scroll operation 
 *
 ***********************************************************************/
void
ExecuteFunction(int func, char *action, Window in_w, MwmWindow * tmp_win,
		XEvent * eventp, unsigned long context, long val1, long val2,
		int val1_unit, int val2_unit, MenuRoot * menu)
{
    MwmWindow      *t, *temp;
    char           *junk, *junkC;
    unsigned long   junkN;
    int             junkD;
    int             x, y;
    Window          w;
    int             delta_x, delta_y;
    int             warp_x = 0, warp_y = 0;
    Pixel           TextColor, BackColor;
    Pixmap          BackPixmap;

    /* Defer Execution may wish to alter this value */
    w = in_w;

    switch (func) {
    case F_NOP:
    case F_TITLE:
	break;

    case F_BEEP:
	XBell(dpy, Scr.screen);
	break;

    case F_RESIZE:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   MOVE_CURS, ButtonPress))
	    break;

	if (tmp_win == NULL)
	    break;
	if (check_allowed_function2(func, tmp_win) == 0) {
	    XBell(dpy, Scr.screen);
	    break;
	}
	tmp_win->flags &= ~MAXIMIZED;
	resize_window(w, tmp_win, val1, val2, val1_unit, val2_unit);
	break;

    case F_MOVE:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   MOVE_CURS, ButtonPress))
	    break;

	if (tmp_win == NULL)
	    break;

	move_window(eventp, w, tmp_win, context, val1, val2, val1_unit, val2_unit);
	break;

    case F_SCROLL:
	if ((val1 > -100000) && (val1 < 100000))
	    x = Scr.Vx + val1 * val1_unit / 100;
	else
	    x = Scr.Vx + (val1 / 1000) * val1_unit / 100;

	if ((val2 > -100000) && (val2 < 100000))
	    y = Scr.Vy + val2 * val2_unit / 100;
	else
	    y = Scr.Vy + (val2 / 1000) * val2_unit / 100;

	if (((val1 <= -100000) || (val1 >= 100000)) && (x > Scr.VxMax)) {
	    x = 0;
	    y += Scr.d_height;
	    if (y > Scr.VyMax)
		y = 0;
	}
	if (((val1 <= -100000) || (val1 >= 100000)) && (x < 0)) {
	    x = Scr.VxMax;
	    y -= Scr.d_height;
	    if (y < 0)
		y = Scr.VyMax;
	}
	if (((val2 <= -100000) || (val2 >= 100000)) && (y > Scr.VyMax)) {
	    y = 0;
	    x += Scr.d_width;
	    if (x > Scr.VxMax)
		x = 0;
	}
	if (((val2 <= -100000) || (val2 >= 100000)) && (y < 0)) {
	    y = Scr.VyMax;
	    x -= Scr.d_width;
	    if (x < 0)
		x = Scr.VxMax;
	}
	MoveViewport(x, y, True);
	break;
    case F_MOVECURSOR:
	XQueryPointer(dpy, Scr.root_win, &JunkRoot, &JunkChild,
		      &x, &y, &JunkX, &JunkY, &JunkMask);
	delta_x = 0;
	delta_y = 0;
	warp_x = 0;
	warp_y = 0;
	if (x >= Scr.d_width - 2) {
	    delta_x = Scr.EdgeScrollX;
	    warp_x = Scr.EdgeScrollX - 4;
	}
	if (y >= Scr.d_height - 2) {
	    delta_y = Scr.EdgeScrollY;
	    warp_y = Scr.EdgeScrollY - 4;
	}
	if (x < 2) {
	    delta_x = -Scr.EdgeScrollX;
	    warp_x = -Scr.EdgeScrollX + 4;
	}
	if (y < 2) {
	    delta_y = -Scr.EdgeScrollY;
	    warp_y = -Scr.EdgeScrollY + 4;
	}
	if (Scr.Vx + delta_x < 0)
	    delta_x = -Scr.Vx;
	if (Scr.Vy + delta_y < 0)
	    delta_y = -Scr.Vy;
	if (Scr.Vx + delta_x > Scr.VxMax)
	    delta_x = Scr.VxMax - Scr.Vx;
	if (Scr.Vy + delta_y > Scr.VyMax)
	    delta_y = Scr.VyMax - Scr.Vy;
	if ((delta_x != 0) || (delta_y != 0)) {
	    MoveViewport(Scr.Vx + delta_x, Scr.Vy + delta_y, True);
	    XWarpPointer(dpy, Scr.root_win, Scr.root_win, 0, 0, Scr.d_width,
			 Scr.d_height,
			 x - warp_x,
			 y - warp_y);
	}
	XWarpPointer(dpy, Scr.root_win, Scr.root_win, 0, 0, Scr.d_width,
		   Scr.d_height, x + val1 * val1_unit / 100 - warp_x,
		     y + val2 * val2_unit / 100 - warp_y);

	break;
    case F_ICONIFY:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;

	if (tmp_win == NULL)
	    break;
	if (tmp_win->flags & ICONIFIED) {
	    if (val1 <= 0)
		DeIconify(tmp_win);
	}
	else {
	    if (check_allowed_function2(func, tmp_win) == 0) {
		XBell(dpy, Scr.screen);
		break;
	    }
	    if (val1 >= 0) {
		if (check_allowed_function2(func, tmp_win) == 0) {
		    XBell(dpy, Scr.screen);
		    break;
		}
		Iconify(tmp_win, eventp->xbutton.x_root - 5, eventp->xbutton.y_root - 5);
	    }
	}
	break;

    case F_RAISE:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;

	if (tmp_win)
	    RaiseWindow(tmp_win);

#if 0
	The original code asked the configuration if this client was supposed
	to stay on top.
	tmp_win->flags |= ONTOP;
#endif
	KeepOnTop();
	break;

    case F_LOWER:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;

	if (tmp_win == NULL)
	    break;
	LowerWindow(tmp_win);

	tmp_win->flags &= ~ONTOP;
	break;

    case F_DESTROY:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   DESTROY_CURS, ButtonRelease))
	    break;

	if (tmp_win == NULL)
	    break;
	if (check_allowed_function2(func, tmp_win) == 0) {
	    XBell(dpy, Scr.screen);
	    break;
	}
	/* Dont delete the pager - it crashes the program! */
	if ((tmp_win->w == Scr.Pager_w) || (tmp_win == Scr.MwmPager))
	    break;

	if (XGetGeometry(dpy, tmp_win->w, &JunkRoot, &JunkX, &JunkY,
			 &JunkWidth, &JunkHeight, &JunkBW, &JunkDepth) == 0)
	    Destroy(tmp_win);
	else
	    XKillClient(dpy, tmp_win->w);
	XSync(dpy, 0);
	break;

    case F_DELETE:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   DESTROY_CURS, ButtonRelease))
	    break;

	if (tmp_win == NULL)
	    break;
	if (check_allowed_function2(func, tmp_win) == 0) {
	    XBell(dpy, Scr.screen);
	    break;
	}

	/* Dont delete the pager - it crashes the program! */
	if ((tmp_win->w == Scr.Pager_w) || (tmp_win == Scr.MwmPager))
	    break;
	if (tmp_win->flags & WM_DELS_WINDOW) {
	    send_clientmessage(tmp_win->w, _XA_WM_DELETE_WINDOW, CurrentTime);
	    break;
	}
	else
	    XBell(dpy, Scr.screen);
	XSync(dpy, 0);
	break;

    case F_CLOSE:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   DESTROY_CURS, ButtonRelease))
	    break;

	if (tmp_win == NULL)
	    break;
	if (check_allowed_function2(func, tmp_win) == 0) {
	    XBell(dpy, Scr.screen);
	    break;
	}

	/* Dont delete the pager - it crashes the program! */
	if ((tmp_win->w == Scr.Pager_w) || (tmp_win == Scr.MwmPager))
	    break;
	if (tmp_win->flags & WM_DELS_WINDOW) {
	    send_clientmessage(tmp_win->w, _XA_WM_DELETE_WINDOW, CurrentTime);
	    break;
	}
	else if (XGetGeometry(dpy, tmp_win->w, &JunkRoot, &JunkX, &JunkY,
			 &JunkWidth, &JunkHeight, &JunkBW, &JunkDepth) == 0)
	    Destroy(tmp_win);
	else
	    XKillClient(dpy, tmp_win->w);
	XSync(dpy, 0);
	break;

    case F_RESTART:
#if 0
	gotta get rid of the menu grabs
	    if (restart_mb == NULL) {
	    Widget          tmp;
	    XmString        lab;
	    Arg             decor[1];

	    XtSetArg(decor[0], XmNmwmDecorations, MWM_DECOR_BORDER);
	    restart_mb = XmCreateQuestionDialog(toplevel, "restart_question",
						decor, 1);
	    if (!restart_mb)
		Done(0, NULL);
	    tmp = XmMessageBoxGetChild(restart_mb, XmDIALOG_HELP_BUTTON);
	    XtUnmanageChild(tmp);
	    tmp = XmMessageBoxGetChild(restart_mb, XmDIALOG_OK_BUTTON);
	    XtAddCallback(tmp, XmNactivateCallback, restart_cb, NULL);
	    tmp = XmMessageBoxGetChild(restart_mb, XmDIALOG_MESSAGE_LABEL);
	    lab = XmStringCreateSimple("Restart Mwm?");
	    XtVaSetValues(tmp, XmNlabelString, lab, NULL);
	    XmStringFree(lab);
	}
	{
	    Widget          tmp;

	    /* MLM: this probably won't work */
	    tmp = XmMessageBoxGetChild(restart_mb, XmDIALOG_OK_BUTTON);
	    XtVaSetValues(tmp, XmNuserData, action, NULL);
	}
	XtManageChild(restart_mb);
#endif
	Done(1, "mwm");
	break;

    case F_EXEC:
	{
	    char *shell;
	    extern char *getenv();

	    if ((shell = getenv(MWM_SHELL_NAME)) == NULL) {
		if ((shell = getenv(SHELL_NAME)) == NULL) {
		    shell = DEFAULT_SHELL;
		}
	    }
	    XGrabPointer(dpy, Scr.root_win, True,
			    ButtonPressMask | ButtonReleaseMask,
			    GrabModeAsync, GrabModeAsync,
			    Scr.root_win, Scr.MwmCursors[WAIT_CURS], CurrentTime);
	    XSync(dpy, 0);

	    if (!fork()) {
		if (execl(shell, shell, "-c", action, (char *) 0) == -1)
			exit(100);
	    }
	    XUngrabPointer(dpy, CurrentTime);
	    XSync(dpy, 0);
	}
	break;

    case F_REFRESH:
	{
	    XSetWindowAttributes attributes;
	    unsigned long   valuemask;

	    valuemask = (CWBackPixel);
	    attributes.background_pixel = Mwm.components[MWM_BORDER].foreground;
	    attributes.backing_store = NotUseful;
	    w = XCreateWindow(dpy, Scr.root_win, 0, 0,
			      (unsigned int) Scr.d_width,
			      (unsigned int) Scr.d_height,
			      (unsigned int) 0,
			      CopyFromParent, (unsigned int) CopyFromParent,
			      (Visual *) CopyFromParent, valuemask,
			      &attributes);
	    XMapWindow(dpy, w);
	    XDestroyWindow(dpy, w);
	    XFlush(dpy);
	}
	break;

    case F_STICK:
	/* stick/unstick a window */
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;

	if (tmp_win == NULL)
	    break;

	if (Scr.Hilite != tmp_win) {
	    /* Need to make SetBorder change the window back color */
	    temp = Scr.Hilite;
	    SetBorder(tmp_win, True, True, True, None);
	    SetBorder(tmp_win, False, True, True, None);
	    SetBorder(temp, True, True, True, None);
	}
	MoveResizePagerView(tmp_win);

	/* Need to re-draw pager_view in case the window
	 * is unsticking */
	if (Scr.Hilite == tmp_win) {
	    TextColor = Mwm.components[MWM_PAGER].active_foreground;
	    BackPixmap = Mwm.components[MWM_PAGER].active_background_pixmap;
	    BackColor = Mwm.components[MWM_PAGER].active_background;
	}
	else {
	    TextColor = Mwm.components[MWM_PAGER].foreground;
	    BackPixmap = Mwm.components[MWM_PAGER].background_pixmap;
	    BackColor = Mwm.components[MWM_PAGER].background;
	}
	if (Scr.d_depth < 2 && BackPixmap != XmUNSPECIFIED_PIXMAP)
	    XSetWindowBackgroundPixmap(dpy, tmp_win->pager_view, BackPixmap);
	else
	    XSetWindowBackground(dpy, tmp_win->pager_view, BackColor);
	XClearWindow(dpy, tmp_win->pager_view);
	if ((tmp_win->icon_image != NULL) && (Scr.PagerFont.height > 0)) {
	    NewFontAndColor(Scr.PagerFont.font->fid, TextColor, BackColor);
	    XDrawString(dpy, tmp_win->pager_view, Scr.FontGC, 2, Scr.PagerFont.y + 2,
			tmp_win->icon_image, strlen(tmp_win->icon_image));
	}
	break;

    case F_GOTO_PAGE:
	/* back up 1 virtual desktop page */
	x = val1 * Scr.d_width;
	y = val2 * Scr.d_height;
	MoveViewport(x, y, True);
	break;

    case F_TOGGLE_PAGE:
	if (DoHandlePageing)
	    DoHandlePageing = 0;
	else
	    DoHandlePageing = 1;
	checkPanFrames();
	break;

    case F_CIRCULATE_UP:
	t = Circulate(tmp_win, action, UP);
	if (t)
	    FocusOn(t, 0);
	break;

    case F_CIRCULATE_DOWN:
	t = Circulate(tmp_win, action, DOWN);
	if (t)
	    FocusOn(t, 0);
	break;

    case F_WARP:
	t = Circulate(tmp_win, action, DOWN);
	if ((t) && (t->flags & ICONIFIED)) {
	    FocusOn(t, 0);
	    DeIconify(t);
	}
	if (t)
	    FocusOn(t, 0);
	break;

    case F_WAIT:
	{
	    Bool            done = False;

	    if (val1 == 0)
		val1 = 1;
	    while (!done) {
		if (My_XNextEvent(dpy, &Event)) {
		    DispatchEvent();
		    if (Event.type == MapNotify) {
			if ((Tmp_win) && (matchWildcards(action, Tmp_win->name) == True))
			    done = True;
			if ((Tmp_win) && (Tmp_win->class.res_class) &&
			    (matchWildcards(action, Tmp_win->class.res_class) == True))
			    done = True;
			if ((Tmp_win) && (Tmp_win->class.res_name) &&
			    (matchWildcards(action, Tmp_win->class.res_name) == True))
			    done = True;
		    }
		}
	    }
	}
	XSync(dpy, 0);
	break;
    case F_RAISE_IT:
	if (val1 != 0) {
	    FocusOn((MwmWindow *) val1, 0);
	    if (((MwmWindow *) (val1))->flags & ICONIFIED) {
		DeIconify((MwmWindow *) val1);
		FocusOn((MwmWindow *) val1, 0);
	    }
	}

	break;
    case F_FOCUS:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;
	FocusOn(tmp_win, 0);
	break;

    case F_CHANGE_WINDOWS_DESK:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;
	if (tmp_win == NULL)
	    break;

	changeWindowsDesk(tmp_win, val1);
	break;
    case F_DESK:
	changeDesks(val1, val2);
	break;

    case F_RAISELOWER:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;
	if (tmp_win == NULL)
	    break;

	if ((tmp_win == Scr.LastWindowRaised) ||
	    (tmp_win->flags & VISIBLE)) {
	    LowerWindow(tmp_win);
	    tmp_win->flags &= ~ONTOP;
	}
	else {
	    RaiseWindow(tmp_win);
#if 0
	The original code asked the configuration if this client was supposed
	to stay on top.
	    tmp_win->flags |= ONTOP;
#endif
	    KeepOnTop();
	}
	break;

    case F_POPUP:
	ActiveItem = NULL;
	ActiveMenu = NULL;
	menuFromFrameOrWindowOrTitlebar = FALSE;
	do_menu(menu);
	break;

    case F_MAXIMIZE:
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonRelease))
	    break;
	if (tmp_win == NULL)
	    break;

	if (check_allowed_function2(func, tmp_win) == 0) {
	    XBell(dpy, Scr.screen);
	    break;
	}
	Maximize(tmp_win, val1, val2, val1_unit, val2_unit);
	break;

    case F_QUIT:
#if 0
	gotta get rid of the menu grabs
	    if (quit_mb == NULL) {
	    Widget          tmp;
	    XmString        lab;
	    Arg             decor[1];

	    XtSetArg(decor[0], XmNmwmDecorations, MWM_DECOR_BORDER);
	    quit_mb = XmCreateQuestionDialog(toplevel, "quit_question",
					     decor, 1);
	    if (!quit_mb)
		Done(0, NULL);
	    tmp = XmMessageBoxGetChild(quit_mb, XmDIALOG_HELP_BUTTON);
	    if (tmp)
		XtUnmanageChild(tmp);
	    tmp = XmMessageBoxGetChild(quit_mb, XmDIALOG_OK_BUTTON);
	    XtAddCallback(tmp, XmNactivateCallback, quit_cb, NULL);
	    tmp = XmMessageBoxGetChild(quit_mb, XmDIALOG_CANCEL_BUTTON);
	    XtAddCallback(tmp, XmNactivateCallback, cancel_cb, NULL);
	    tmp = XmMessageBoxGetChild(quit_mb, XmDIALOG_MESSAGE_LABEL);
	    lab = XmStringCreateSimple("QUIT Mwm?");
	    XtVaSetValues(tmp, XmNlabelString, lab, NULL);
	    XmStringFree(lab);
	}
	XtManageChild(quit_mb);
	while (!cancel) {
	    if (My_XNextEvent(dpy, &Event))
		DispatchEvent();
	}
#endif
	Done(0, NULL);
	break;

    case F_WINDOWLIST:
	do_windowList(val1, val2);
	break;

    case F_FUNCTION:
	ComplexFunction(w, tmp_win, eventp, context, menu);
	break;

    }

    /* Only wait for an all-buttons-up condition after calls from
     * regular built-ins, not from complex-functions or modules. */
    WaitForButtonsUp();

    return;
}

MwmWindow      *
Circulate(MwmWindow * tmp_win, char *action, Bool Direction)
{
    MwmWindow      *t, *selected;
    Bool            found;
    int             count, pass = 1;
    int             base, best;

    while (pass < 3) {
	if (tmp_win)
	    base = tmp_win->focus_sequence;
	else
	    base = -1;
	if (Direction == DOWN)
	    best = -1;
	else
	    best = 10000;
	selected = tmp_win;

	/* move focus to the next window */
	found = FALSE;
	t = tmp_win;
	count = 0;
	while (count < 3) {
	    if (Direction == DOWN) {
		if ((t == (MwmWindow *) 0) || (t->next == NULL)) {
		    t = Scr.MwmRoot.next;
		    count++;
		}
		else
		    t = t->next;
	    }
	    else {		/* Direction Up */
		if ((t == (MwmWindow *) 0) || (t == &Scr.MwmRoot) ||
		    (t->prev == &Scr.MwmRoot) || (t->prev == (MwmWindow *) NULL)) {
		    for (t = Scr.MwmRoot.next; t->next != (MwmWindow *) NULL; t = t->next);
		    count++;
		}
		else
		    t = t->prev;
	    }
	    found = TRUE;

	    if (t->Desk != Scr.CurrentDesk)
		found = False;

	    if ((t) && (t->wmhints) && (t->wmhints->flags & InputHint) &&
		(t->wmhints->input == False) &&
		!(t->flags & WM_TAKES_FOCUS))
		found = False;

	    if (t->flags & CIRCULATESKIP)
		found = FALSE;
	    /* optional skip over icons */

	    if ((t->flags & ICONIFIED) && (Scr.flags & CirculateSkipIcons))
		found = FALSE;


	    /* Make CirculateUp and CirculateDown take args. by Y.NOMURA */
	    if (action && (strlen(action) > 0) &&
		!(matchWildcards(action, t->name)) &&
		!(matchWildcards(action, t->icon_image)) &&
		t->class.res_name &&
		!(matchWildcards(action, t->class.res_name)))
		found = FALSE;
	    if ((found) && (Direction == DOWN) && (t->focus_sequence > best)) {
		best = t->focus_sequence;
		selected = t;
	    }
	    if ((found) && (Direction != DOWN) && (t->focus_sequence < best)
		&& (t->focus_sequence > base)) {
		best = t->focus_sequence;
		selected = t;
	    }
	}
	if ((selected) && (selected == tmp_win) && (base > 0)) {
	    if (Direction == DOWN) {
		ClearCirculatedFlag();
		tmp_win->focus_sequence = 0;
	    }
	    else {
		MwmWindow      *temp;

		temp = Scr.MwmRoot.next;
		while (temp != NULL) {
		    temp->focus_sequence++;
		    if (temp == tmp_win)
			temp->focus_sequence = 0;
		    temp = temp->next;
		}
	    }
	    pass++;
	}
	else
	    pass = 3;
    }

    return selected;
}


/***********************************************************************
 *
 *  Procedure:
 *	DeferExecution - defer the execution of a function to the
 *	    next button press if the context is C_ROOT
 *
 *  Inputs:
 *      eventp  - pointer to XEvent to patch up
 *      w       - pointer to Window to patch up
 *      tmp_win - pointer to MwmWindow Structure to patch up
 *	context	- the context in which the mouse button was pressed
 *	func	- the function to defer
 *	cursor	- the cursor to display while waiting
 *      finishEvent - ButtonRelease or ButtonPress; tells what kind of event to
 *                    terminate on.
 *
 ***********************************************************************/
int
DeferExecution(XEvent * eventp, Window * w, MwmWindow ** tmp_win,
	       unsigned long *context, int cursor, int FinishEvent)
{
    int             done;
    int             finished = 0;
    Window          dummy;
    Window          original_w;

    original_w = *w;

    if ((*context != C_ROOT) && (*context != C_NO_CONTEXT)) {
	if ((FinishEvent == ButtonPress) || ((FinishEvent == ButtonRelease) &&
					   (eventp->type != ButtonPress))) {
	    return FALSE;
	}
    }
    if (!GrabEm(cursor)) {
	XBell(dpy, Scr.screen);
	return True;
    }

    while (!finished) {
	done = 0;
	/* block until there is an event */
	XMaskEvent(dpy, ButtonPressMask | ButtonReleaseMask |
		   ExposureMask | KeyPressMask | VisibilityChangeMask |
		   ButtonMotionMask | PointerMotionMask		/* | EnterWindowMask | 
								   LeaveWindowMask */ , eventp);
	StashEventTime(eventp);

	if (eventp->type == KeyPress)
	    Keyboard_shortcuts(eventp, FinishEvent);
	if (eventp->type == FinishEvent)
	    finished = 1;
	if (eventp->type == ButtonPress) {
	    XAllowEvents(dpy, ReplayPointer, CurrentTime);
	    done = 1;
	}
	if (eventp->type == ButtonRelease)
	    done = 1;
	if (eventp->type == KeyPress)
	    done = 1;

	if (!done) {
	    DispatchEvent();
	}

    }


    *w = eventp->xany.window;
    if (((*w == Scr.root_win) || (*w == Scr.no_focus_win))
	&& (eventp->xbutton.subwindow != (Window) 0)) {
	*w = eventp->xbutton.subwindow;
	eventp->xany.window = *w;
    }
    if (*w == Scr.root_win) {
	*context = C_ROOT;
	XBell(dpy, Scr.screen);
	UngrabEm();
	return TRUE;
    }
    if (XFindContext(dpy, *w, MwmContext, (XPointer *) tmp_win) == XCNOENT) {
	*tmp_win = NULL;
	XBell(dpy, Scr.screen);
	UngrabEm();
	return (TRUE);
    }

    if (*w == (*tmp_win)->Parent)
	*w = (*tmp_win)->w;

    if (original_w == (*tmp_win)->Parent)
	original_w = (*tmp_win)->w;

    /* this ugly mess attempts to ensure that the release and press
     * are in the same window. */
    if ((*w != original_w) && (original_w != Scr.root_win) &&
	(original_w != None) && (original_w != Scr.no_focus_win))
	if (!((*w == (*tmp_win)->frame) &&
	      (original_w == (*tmp_win)->w))) {
	    *context = C_ROOT;
	    XBell(dpy, Scr.screen);
	    UngrabEm();
	    return TRUE;
	}

    *context = GetContext(*tmp_win, eventp, &dummy);

    UngrabEm();
    return FALSE;
}


/****************************************************************************
 *
 * This is used to tell applications which windows on the screen are
 * top level appication windows, and which windows are the icon windows
 * that go with them.
 *
 ****************************************************************************/
void
SetMapStateProp(MwmWindow * tmp_win, int state)
{
    unsigned long   data[2];	/* "suggested" by ICCCM version 1 */

    data[0] = (unsigned long) state;
    data[1] = (unsigned long) tmp_win->icon_w;
/*  data[2] = (unsigned long) tmp_win->icon_pixmap_w; */

    XChangeProperty(dpy, tmp_win->w, _XA_WM_STATE, _XA_WM_STATE, 32,
		    PropModeReplace, (unsigned char *) data, 2);
    return;
}

/****************************************************************************
 *
 * Keeps the "StaysOnTop" windows on the top of the pile.
 * This is achieved by clearing a flag for OnTop windows here, and waiting
 * for a visibility notify on the windows. Eception: OnTop windows which are
 * obscured by other OnTop windows, which need to be raised here.
 *
 ****************************************************************************/
void
KeepOnTop()
{
    MwmWindow      *t;

    /* flag that on-top windows should be re-raised */
    for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	if ((t->flags & ONTOP) && !(t->flags & VISIBLE)) {
	    RaiseWindow(t);
	    t->flags &= ~RAISED;
	}
	else
	    t->flags |= RAISED;
    }
}


/***************************************************************************
 *
 *  Moves the viewport within thwe virtual desktop
 *
 ***************************************************************************/
void
MoveViewport(int newx, int newy, Bool grab)
{
    MwmWindow      *t;
    int             deltax, deltay;

    if (grab)
	XGrabServer(dpy);


    if (newx > Scr.VxMax)
	newx = Scr.VxMax;
    if (newy > Scr.VyMax)
	newy = Scr.VyMax;
    if (newx < 0)
	newx = 0;
    if (newy < 0)
	newy = 0;

    deltay = Scr.Vy - newy;
    deltax = Scr.Vx - newx;

    Scr.Vx = newx;
    Scr.Vy = newy;

    if ((deltax != 0) || (deltay != 0)) {
	for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	    /* If the window is iconified, and sticky Icons is set,
	     * then the window should essentially be sticky */
	    if (!((t->flags & ICONIFIED) && (Scr.flags & StickyIcons)) &&
		(!(t->flags & STICKY))) {
		if (!(Scr.flags & StickyIcons)) {
		    t->icon_x_loc += deltax;
		    t->icon_xl_loc += deltax;
		    t->icon_y_loc += deltay;
		    if (t->icon_pixmap_w != None)
			XMoveWindow(dpy, t->icon_pixmap_w, t->icon_x_loc,
				    t->icon_y_loc);
		    if (t->icon_w != None)
			XMoveWindow(dpy, t->icon_w, t->icon_x_loc,
				    t->icon_y_loc + t->icon_p_height);
		}
	    }
	    if (!(t->flags & STICKY))
		SetupFrame(t, t->frame_x + deltax, t->frame_y + deltay,
			   t->frame_width, t->frame_height, FALSE);
	}
	for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	    /* If its an icon, and its sticking, autoplace it so
	     * that it doesn't wind up on top a a stationary
	     * icon */
	    if (((t->flags & STICKY) || (Scr.flags & StickyIcons)) &&
		(t->flags & ICONIFIED) && (!(t->flags & ICON_MOVED)) &&
		(!(t->flags & ICON_UNMAPPED)))
		AutoPlace(t);
	}

    }
    /* fix up the viewport indicator */
    MoveResizeViewPortIndicator();
    checkPanFrames();

    /* do this with PanFrames too ??? HEDU */
    while (XCheckTypedEvent(dpy, MotionNotify, &Event))
	StashEventTime(&Event);
    if (grab)
	XUngrabServer(dpy);
}


/**************************************************************************
 *
 * Moves focus to specified window 
 *
 *************************************************************************/
void
FocusOn(MwmWindow * t, int DeIconifyOnly)
{
    int             dx, dy;
    int             cx, cy;
    int             x, y;

    if (t == (MwmWindow *) 0)
	return;

    if (t->Desk != Scr.CurrentDesk) {
	changeDesks(0, t->Desk);
    }

    if (t->flags & ICONIFIED) {
	cx = t->icon_xl_loc + t->icon_w_width / 2;
	cy = t->icon_y_loc + t->icon_p_height + ICON_HEIGHT / 2;
    }
    else {
	cx = t->frame_x + t->frame_width / 2;
	cy = t->frame_y + t->frame_height / 2;
    }

    /* Put center of window on the visible screen */
    if ((!DeIconifyOnly) && (Scr.flags & CenterOnCirculate)) {
	dx = cx - Scr.d_width / 2 + Scr.Vx;
	dy = cy - Scr.d_height / 2 + Scr.Vy;
    }
    else {
	dx = (cx + Scr.Vx) / Scr.d_width * Scr.d_width;
	dy = (cy + Scr.Vy) / Scr.d_height * Scr.d_height;
    }
    MoveViewport(dx, dy, True);

    if (t->flags & ICONIFIED) {
	x = t->icon_xl_loc + t->icon_w_width / 2;
	y = t->icon_y_loc + t->icon_p_height + ICON_HEIGHT / 2;
    }
    else {
	x = t->frame_x;
	y = t->frame_y;
    }
    if (!(Mwm.keyboard_focus_policy == XmEXPLICIT))
	XWarpPointer(dpy, None, Scr.root_win, 0, 0, 0, 0, x + 2, y + 2);
    RaiseWindow(t);
    KeepOnTop();

    /* If the window is still not visible, make it visible! */
    if (((t->frame_x + t->frame_height) < 0) || (t->frame_y + t->frame_width < 0) ||
	(t->frame_x > Scr.d_width) || (t->frame_y > Scr.d_height)) {
	SetupFrame(t, 0, 0, t->frame_width, t->frame_height, False);
	if (!(Mwm.keyboard_focus_policy == XmEXPLICIT))
	    XWarpPointer(dpy, None, Scr.root_win, 0, 0, 0, 0, 2, 2);
    }
    UngrabEm();
    SetFocus(t->w, t);
}



/***********************************************************************
 *
 *  Procedure:
 *	(Un)Maximize a window.
 *
 ***********************************************************************/
void
Maximize(MwmWindow * tmp_win, int val1, int val2,
	 int val1_unit, int val2_unit)
{
    int             new_width, new_height, new_x, new_y;

    if (tmp_win->flags & MAXIMIZED) {
	tmp_win->flags &= ~MAXIMIZED;
	SetupFrame(tmp_win, tmp_win->orig_x, tmp_win->orig_y, tmp_win->orig_wd,
		   tmp_win->orig_ht, TRUE);
	SetBorder(tmp_win, True, True, True, None);
    }
    else {
	new_width = tmp_win->frame_width;
	new_height = tmp_win->frame_height;
	new_x = tmp_win->frame_x;
	new_y = tmp_win->frame_y;
	if (val1 > 0) {
	    new_width = val1 * val1_unit / 100 - 2;
	    new_x = 0;
	}
	if (val2 > 0) {
	    new_height = val2 * val2_unit / 100 - 2;
	    new_y = 0;
	}
	if ((val1 == 0) && (val2 == 0)) {
	    new_x = 0;
	    new_y = 0;
	    new_height = Scr.d_height - 2;
	    new_width = Scr.d_width - 2;
	}
	tmp_win->flags |= MAXIMIZED;
	ConstrainSize(tmp_win, &new_width, &new_height);
	SetupFrame(tmp_win, new_x, new_y, new_width, new_height, TRUE);
	SetBorder(tmp_win, Scr.Hilite == tmp_win,
		  True, True, tmp_win->maximizeb);
    }
    RedrawPager();
}

/*****************************************************************************
 *
 * Grab the pointer and keyboard
 *
 ****************************************************************************/
Bool
GrabEm(int cursor)
{
    int             i = 0, val = 0;
    unsigned int    mask;

    XSync(dpy, 0);
    /* move the keyboard focus prior to grabbing the pointer to
     * eliminate the enterNotify and exitNotify events that go
     * to the windows */
    if (Scr.PreviousFocus == NULL)
	Scr.PreviousFocus = Scr.Focus;
    SetFocus(Scr.no_focus_win, NULL);
    mask = ButtonPressMask | ButtonReleaseMask | ButtonMotionMask | PointerMotionMask
	| EnterWindowMask | LeaveWindowMask;
    while ((i < 1000) && (val = XGrabPointer(dpy, Scr.root_win, True, mask,
				     GrabModeAsync, GrabModeAsync, Scr.root_win,
				     Scr.MwmCursors[cursor], CurrentTime) !=
			  GrabSuccess)) {
	i++;
	/* If you go too fast, other windows may not get a change to release
	 * any grab that they have. */
	sleep_a_little(1000);
    }

    /* If we fall out of the loop without grabbing the pointer, its
       time to give up */
    XSync(dpy, 0);
    if (val != GrabSuccess) {
	return False;
    }
    return True;
}


/*****************************************************************************
 *
 * UnGrab the pointer and keyboard
 *
 ****************************************************************************/
void
UngrabEm()
{
    Window          w;

    XSync(dpy, 0);
    XUngrabPointer(dpy, CurrentTime);

    if (Scr.PreviousFocus != NULL) {
	w = Scr.PreviousFocus->w;

	/* if the window still exists, focus on it */
	if (w) {
	    SetFocus(w, Scr.PreviousFocus);
	}
	Scr.PreviousFocus = NULL;
    }
    XSync(dpy, 0);
}


/*****************************************************************************
 *
 * Waits Scr.ClickTime, or until it is evident that the user is not
 * clicking, but is moving the cursor
 *
 ****************************************************************************/
Bool
IsClick(int x, int y, unsigned EndMask, XEvent * d)
{
    int             xcurrent, ycurrent, total = 0;

    xcurrent = x;
    ycurrent = y;
    while ((total < Scr.ClickTime) &&
	   (x - xcurrent < 5) && (x - xcurrent > -5) &&
	   (y - ycurrent < 5) && (y - ycurrent > -5)) {
	sleep_a_little(10000);
	total += 10;
	if (XCheckMaskEvent(dpy, EndMask, d)) {
	    StashEventTime(d);
	    return True;
	}
	if (XCheckMaskEvent(dpy, ButtonMotionMask | PointerMotionMask, d)) {
	    xcurrent = d->xmotion.x_root;
	    ycurrent = d->xmotion.y_root;
	    StashEventTime(d);
	}
    }
    return False;
}

/*****************************************************************************
 *
 * Builtin which determines if the button press was a click or double click...
 *
 ****************************************************************************/
void
ComplexFunction(Window w, MwmWindow * tmp_win, XEvent * eventp,
		unsigned long context, MenuRoot * mr)
{
    char            type = MOTION;
    char            c;
    XEvent         *ev;
    MenuItem       *mi;
    XEvent          d;
    Bool            Persist = False;
    Bool            HaveDoubleClick = False;
    Bool            NeedsTarget = False;
    int             x, y;

    if (mr == NULL)
	return;

    /* These built-ins require a selected window 
     * The function code is >= 100 and < 1000
     * F_RESIZE
     * F_MOVE
     * F_ICONIFY
     * F_RAISE
     * F_LOWER
     * F_DESTROY
     * F_DELETE
     * F_STICK
     * F_RAISELOWER
     * F_MAXIMIZE
     * F_FOCUS
     *
     * These do not:
     * The function code is < 100
     * F_NOP
     * F_TITLE
     * F_BEEP
     * F_SCROLL
     * F_MOVECURSOR
     * F_RESTART
     * F_EXEC
     * F_REFRESH
     * F_GOTO_PAGE
     * F_TOGGLE_PAGE
     * F_CIRCULATE_UP
     * F_CIRCULATE_DOWN
     * F_WARP
     * F_DESK
     * F_POPUP
     * F_QUIT
     * F_WINDOWLIST
     * F_FUNCTION
     */

    mi = mr->first;
    while (mi != NULL) {
	/* make lower case */
	c = *(mi->item);
	if ((mi->func >= 100) && (mi->func < 1000))
	    NeedsTarget = True;
	if (isupper(c))
	    c = tolower(c);
	if (c == DOUBLE_CLICK)
	    HaveDoubleClick = True;
	else if (c == IMMEDIATE) {
	    if (tmp_win)
		w = tmp_win->frame;
	    else
		w = None;
	    ExecuteFunction(mi->func, mi->action, w,
			    tmp_win, eventp, context, mi->val1, mi->val2,
			    mi->val1_unit, mi->val2_unit,
			    mi->menu);
	}
	else
	    Persist = True;
	mi = mi->next;
    }

    if (!Persist)
	return;

    /* Only defer execution if there is a possibility of needing
     * a window to operate on */
    if (NeedsTarget) {
	if (DeferExecution(eventp, &w, &tmp_win, &context,
			   SELECT_CURS, ButtonPress)) {
	    WaitForButtonsUp();
	    return;
	}
    }

    if (!GrabEm(SELECT_CURS)) {
	XBell(dpy, Scr.screen);
	return;
    }
    XQueryPointer(dpy, Scr.root_win, &JunkRoot, &JunkChild,
		  &x, &y, &JunkX, &JunkY, &JunkMask);

    /* A window has already been selected */
    ev = eventp;

    /* Wait and see if we have a click, or a move */
    /* wait 100 msec, see if the used releases the button */
    if (IsClick(x, y, ButtonReleaseMask, &d)) {
	type = CLICK;
	ev = &d;
    }

    /* If it was a click, wait to see if its a double click */
    if ((HaveDoubleClick) && (type == CLICK) &&
	(IsClick(x, y, ButtonPressMask, &d))) {
	type = ONE_AND_A_HALF_CLICKS;
	ev = &d;
    }
    if ((HaveDoubleClick) && (type == ONE_AND_A_HALF_CLICKS) &&
	(IsClick(x, y, ButtonReleaseMask, &d))) {
	type = DOUBLE_CLICK;
	ev = &d;
    }
    /* some functions operate on button release instead of 
     * presses. These gets really weird for complex functions ... */
    if (eventp->type == ButtonPress)
	eventp->type = ButtonRelease;

    mi = mr->first;
    while (mi != NULL) {
	/* make lower case */
	c = *(mi->item);
	if (isupper(c))
	    c = tolower(c);
	if (c == type) {
	    if (tmp_win)
		w = tmp_win->frame;
	    else
		w = None;
	    ExecuteFunction(mi->func, mi->action, w,
			    tmp_win, eventp, context,
			    mi->val1, mi->val2,
			    mi->val1_unit, mi->val2_unit, mi->menu);
	}
	mi = mi->next;
    }
    WaitForButtonsUp();
    UngrabEm();
}


/* For Ultrix 4.2 */
#include <sys/types.h>
#include <sys/time.h>


/**************************************************************************
 * 
 * Move to a new desktop
 *
 *************************************************************************/
void
changeDesks(int val1, int val2)
{
    int             oldDesk;
    MwmWindow      *t;
    MwmWindow      *FocusWin = 0;
    static MwmWindow *StickyWin = 0;

    oldDesk = Scr.CurrentDesk;

    if (val1 != 0) {
	Scr.CurrentDesk = Scr.CurrentDesk + val1;
    }
    else {
	Scr.CurrentDesk = val2;
	if (Scr.CurrentDesk == oldDesk)
	    return;
    }

    /* Scan the window list, mapping windows on the new Desk,
     * unmapping windows on the old Desk */
    XGrabServer(dpy);
    for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	/* Only change mapping for non-sticky windows */
	if (!((t->flags & ICONIFIED) && (Scr.flags & StickyIcons)) &&
	    (!(t->flags & STICKY)) && (!(t->flags & ICON_UNMAPPED))) {
	    if (t->Desk == oldDesk) {
		if (Scr.Focus == t)
		    t->FocusDesk = oldDesk;
		else
		    t->FocusDesk = -1;
		UnmapIt(t);
	    }
	    else if (t->Desk == Scr.CurrentDesk) {
		MapIt(t);
		if (t->FocusDesk == Scr.CurrentDesk) {
		    FocusWin = t;
		}
	    }
	}
	else {
	    /* Window is sticky */
	    t->Desk = Scr.CurrentDesk;
	    if (Scr.Focus == t) {
		t->FocusDesk = oldDesk;
		StickyWin = t;
	    }
	}
    }
    XUngrabServer(dpy);
    for (t = Scr.MwmRoot.next; t != NULL; t = t->next) {
	/* If its an icon, and its sticking, autoplace it so
	 * that it doesn't wind up on top a a stationary
	 * icon */
	if (((t->flags & STICKY) || (Scr.flags & StickyIcons)) &&
	    (t->flags & ICONIFIED) && (!(t->flags & ICON_MOVED)) &&
	    (!(t->flags & ICON_UNMAPPED)))
	    AutoPlace(t);
    }
    /* Better re-draw the pager now */
    RedrawPager();

    if (Mwm.keyboard_focus_policy == XmEXPLICIT) {
#ifndef NO_REMEMBER_FOCUS
	if (FocusWin)
	    SetFocus(FocusWin->w, FocusWin);
	else if (StickyWin && (StickyWin->flags && STICKY))
	    SetFocus(StickyWin->w, StickyWin);
	else
#endif
	    SetFocus(Scr.no_focus_win, NULL);
    }
}



/**************************************************************************
 * 
 * Move to a new desktop
 *
 *************************************************************************/
void
changeWindowsDesk(MwmWindow * t, int val1)
{
    if (val1 == t->Desk)
	return;

    /* Scan the window list, mapping windows on the new Desk,
     * unmapping windows on the old Desk */
    /* Only change mapping for non-sticky windows */
    if (!((t->flags & ICONIFIED) && (Scr.flags & StickyIcons)) &&
	(!(t->flags & STICKY)) && (!(t->flags & ICON_UNMAPPED))) {
	if (t->Desk == Scr.CurrentDesk) {
	    t->Desk = val1;
	    UnmapIt(t);
	}
	else if (val1 == Scr.CurrentDesk) {
	    t->Desk = val1;
	    /* If its an icon, auto-place it */
	    if (t->flags & ICONIFIED)
		AutoPlace(t);
	    MapIt(t);
	}
	else
	    t->Desk = val1;

    }
    /* Better re-draw the pager now */
    RedrawPager();
}


/**************************************************************************
 * 
 * Unmaps a window on transition to a new desktop
 *
 *************************************************************************/
void
UnmapIt(MwmWindow * t)
{
    XWindowAttributes winattrs;
    unsigned long   eventMask;

    /*
     * Prevent the receipt of an UnmapNotify, since that would
     * cause a transition to the Withdrawn state.
     */
    XGetWindowAttributes(dpy, t->w, &winattrs);
    eventMask = winattrs.your_event_mask;
    XSelectInput(dpy, t->w, eventMask & ~StructureNotifyMask);
    if (t->flags & ICONIFIED) {
	if (t->icon_pixmap_w != None)
	    XUnmapWindow(dpy, t->icon_pixmap_w);
	if (t->icon_w != None)
	    XUnmapWindow(dpy, t->icon_w);
    }
    else if (t->flags & (MAPPED | MAP_PENDING)) {
	XUnmapWindow(dpy, t->frame);
    }
    XSelectInput(dpy, t->w, eventMask);
    MoveResizePagerView(t);
}

/**************************************************************************
 * 
 * Maps a window on transition to a new desktop
 *
 *************************************************************************/
void
MapIt(MwmWindow * t)
{
    if (t->flags & ICONIFIED) {
	if (t->icon_pixmap_w != None)
	    XMapWindow(dpy, t->icon_pixmap_w);
	if (t->icon_w != None)
	    XMapWindow(dpy, t->icon_w);
    }
    else if (t->flags & MAPPED) {
	XMapWindow(dpy, t->frame);
	t->flags |= MAP_PENDING;
	XMapWindow(dpy, t->Parent);
    }
    MoveResizePagerView(t);
}
