/* 
 * tixGrData.c --
 *
 *	This module manipulates the data structure for a Grid widget.
 *
 * Copyright (c) 1996, Expert Interface Technologies
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include <tkInt.h>
#include <tixInt.h>
#include <tixGrid.h>

static TixGridRowCol * InitRowCol(index)
    int index;
{
    TixGridRowCol * rowCol = (TixGridRowCol *)ckalloc(sizeof(TixGridRowCol));

    rowCol->dispIndex      = index;
    rowCol->size.sizeType  = TIX_GR_DEFAULT;
    rowCol->size.sizeValue = 0;
    rowCol->size.charValue = 0;
    rowCol->size.pad0      = 2;
    rowCol->size.pad1      = 2;
    rowCol->size.pixels    = 0;

    Tcl_InitHashTable(&rowCol->list, TCL_ONE_WORD_KEYS);

    return rowCol;
}

/*----------------------------------------------------------------------
 * TixGridDataSetInit --
 *
 *	Create an instance of the TixGridDataSet data structure.
 *
 *----------------------------------------------------------------------
 */
TixGridDataSet*	TixGridDataSetInit()
{
    TixGridDataSet * dataSet =(TixGridDataSet*)ckalloc(sizeof(TixGridDataSet));

    Tcl_InitHashTable(&dataSet->index[0], TCL_ONE_WORD_KEYS);
    Tcl_InitHashTable(&dataSet->index[1], TCL_ONE_WORD_KEYS);

    dataSet->maxIdx[0] = -1;
    dataSet->maxIdx[1] = -1;

    return dataSet;
}

/*----------------------------------------------------------------------
 * TixGridDataSetFree --
 *
 *	Frees an instance of the TixGridDataSet data structure.
 *
 *----------------------------------------------------------------------
 */
void TixGridDataSetFree(dataSet)
    TixGridDataSet* dataSet;
{
    /** ToDo: free all the entries, etc */
    ckfree((char*)dataSet);
}

char * TixGridDataFindEntry(dataSet, x, y)
    TixGridDataSet * dataSet;
    int x;
    int y;
{
    TixGridRowCol *col, *row;
    Tcl_HashEntry *hashPtr;

    /* (1) Find the row and column */
    if (!(hashPtr = Tcl_FindHashEntry(&dataSet->index[0], (char*)x))) {
	return NULL;
    }
    col = (TixGridRowCol *)Tcl_GetHashValue(hashPtr);

    if (!(hashPtr = Tcl_FindHashEntry(&dataSet->index[1], (char*)y))) {
	return NULL;
    }
    row = (TixGridRowCol *)Tcl_GetHashValue(hashPtr);

    /* (2) Find the entry */
    if (row->list.numEntries < col->list.numEntries) {
	if (!(hashPtr = Tcl_FindHashEntry(&row->list, (char*)col))) {
	    return NULL;
	}
    }
    else {
	if (!(hashPtr = Tcl_FindHashEntry(&col->list, (char*)row))) {
	    return NULL;
	}
    }

    return (char *)Tcl_GetHashValue(hashPtr);
}

/* find or create the entry at the specified index */
char * TixGridDataCreateEntry(dataSet, x, y, defaultEntry)
    TixGridDataSet * dataSet;
    int x;
    int y;
    char * defaultEntry;
{
    TixGridRowCol *rowcol[2];
    Tcl_HashEntry *hashPtr;
    int isNew, i, dispIndex[2];

    dispIndex[0] = x;
    dispIndex[1] = y;

    for (i=0; i<2; i++) {
	hashPtr = Tcl_CreateHashEntry(&dataSet->index[i],
	    (char*)dispIndex[i], &isNew);

	if (!isNew) {
	    rowcol[i] = (TixGridRowCol *)Tcl_GetHashValue(hashPtr);
	} else {
	    rowcol[i] = InitRowCol(dispIndex[i]);
	    Tcl_SetHashValue(hashPtr, (char*)rowcol[i]);

	    if (dataSet->maxIdx[i] < dispIndex[i]) {
		dataSet->maxIdx[i] = dispIndex[i];
	    }
	}
    }

    hashPtr = Tcl_CreateHashEntry(&rowcol[0]->list,
	(char*)rowcol[1], &isNew);

    if (!isNew) {
	return (char *) Tcl_GetHashValue(hashPtr);
    }
    else {
	Tcl_SetHashValue(hashPtr, (char*)defaultEntry);

	hashPtr = Tcl_CreateHashEntry(&rowcol[1]->list,
	    (char*)rowcol[0], &isNew);
	Tcl_SetHashValue(hashPtr, (char*)defaultEntry);

	return defaultEntry;
    }
}

/* return value: has the size of the grid changed as a result of sorting */
int
Tix_GridDataUpdateSort(dataSet, axis, start, end, items)
    TixGridDataSet * dataSet;
    int axis;
    int start;
    int end;
    Tix_GrSortItem *items;
{
    TixGridRowCol **ptr;
    Tcl_HashEntry *hashPtr;
    int numItems = end - start + 1;
    int i, k, max;

    if (numItems <= 0) {	/* sanity check */
	return 0;		/* size not changed */
    }

    ptr = (TixGridRowCol **)ckalloc(numItems * sizeof(TixGridRowCol *));

    for (k=0,i=start; i<=end; i++,k++) {
	if (!(hashPtr = Tcl_FindHashEntry(&dataSet->index[axis], (char*)i))) {
	    /* this row/col doesn't exist */
	    ptr[k] = NULL;
	} else {
	    ptr[k] = (TixGridRowCol *)Tcl_GetHashValue(hashPtr);
	    Tcl_DeleteHashEntry(hashPtr);
	}
    }

    for (k=0,i=start; i<=end; i++,k++) {
	int pos, isNew;
	pos = items[k].index - start;

	if (ptr[pos] != NULL) {
	    hashPtr = Tcl_CreateHashEntry(&dataSet->index[axis], (char*)i,
	        &isNew);
	    Tcl_SetHashValue(hashPtr, (char*)ptr[pos]);
	    ptr[pos]->dispIndex = i;
	    max = i;
	}
    }

    ckfree((char*)ptr);

    if (end+1 >= dataSet->maxIdx[axis]) {
	if (dataSet->maxIdx[axis] != max+1) {
	    dataSet->maxIdx[axis] = max+1;
	    return 1;				/* size changed */
	}
    }
    return 0;					/* size not changed */
}


static int RowColMaxSize(wPtr, which, rowCol, defSize)
    WidgetPtr wPtr;
    int which;				/* 0=cols, 1=rows */
    TixGridRowCol *rowCol;
    TixGridSize * defSize;
{
    Tcl_HashSearch hashSearch;
    Tcl_HashEntry *hashPtr;
    TixGrEntry * chPtr;
    int maxSize = 1;

    if (rowCol->list.numEntries == 0) {
	return defSize->pixels;
    }

    for (hashPtr = Tcl_FirstHashEntry(&rowCol->list, &hashSearch);
	 hashPtr;
	 hashPtr = Tcl_NextHashEntry(&hashSearch)) {

	chPtr = (TixGrEntry *)Tcl_GetHashValue(hashPtr);
	if (maxSize < chPtr->iPtr->base.size[which]) {
	    maxSize = chPtr->iPtr->base.size[which];
	}
    }

    return maxSize;
}

/* width of column or height of rows */
int Tix_GridDataGetRowColSize(wPtr, dataSet, which, index, defSize, pad0, pad1)
    WidgetPtr wPtr;
    TixGridDataSet * dataSet;
    int which;				/* 0=cols, 1=rows */
    int index;
    TixGridSize * defSize;
    int *pad0;
    int *pad1;
{
    TixGridRowCol *rowCol;
    Tcl_HashEntry *hashPtr;
    int size;

    if (!(hashPtr = Tcl_FindHashEntry(&dataSet->index[which], (char*)index))) {
	size  = defSize->pixels;
	*pad0 = defSize->pad0;
	*pad1 = defSize->pad1;
    }
    else {
	rowCol = (TixGridRowCol *)Tcl_GetHashValue(hashPtr);
    
	switch (rowCol->size.sizeType) {
	  case TIX_GR_AUTO:
	    size  = RowColMaxSize(wPtr, which, rowCol, defSize);
	    *pad0 = rowCol->size.pad0;
	    *pad1 = rowCol->size.pad1;
	    break;

	  case TIX_GR_DEFINED_PIXEL:
	    size  = rowCol->size.sizeValue;
	    *pad0 = rowCol->size.pad0;
	    *pad1 = rowCol->size.pad1;
	    break;

	  case TIX_GR_DEFINED_CHAR:
	    size  = rowCol->size.charValue * wPtr->fontSize[which];
	    *pad0 = rowCol->size.pad0;
	    *pad1 = rowCol->size.pad1;
	    break;

	  case TIX_GR_DEFAULT:
	  default:			/* some error ?? */
	    if (defSize->sizeType == TIX_GR_AUTO) {
		size = RowColMaxSize(wPtr, which, rowCol, defSize);
	    } else {
		size = defSize->pixels;
	    }
	    *pad0 = defSize->pad0;
	    *pad1 = defSize->pad1;
	}
    }

    return size;
}

int Tix_GridDataGetIndex(interp, wPtr, xStr, yStr, xPtr, yPtr)
    Tcl_Interp * interp;
    WidgetPtr wPtr;
    char * xStr;
    char * yStr;
    int * xPtr;
    int * yPtr;
{
    char * str[2];
    int * ptr[2];
    int i;

    str[0] = xStr;
    str[1] = yStr;
    ptr[0] = xPtr;
    ptr[1] = yPtr;

    for (i=0; i<2; i++) {
	if (str[i] == NULL) {		/* ignore this index */
	    continue;
	}

	if (strcmp(str[i], "max") == 0) {
	    *ptr[i] = wPtr->dataSet->maxIdx[i];
	    if (*ptr[i] < wPtr->hdrSize[i]) {
		*ptr[i] = wPtr->hdrSize[i];
	    }
	}
	else if (strcmp(str[i], "end") == 0) {
	    *ptr[i] = wPtr->dataSet->maxIdx[i] + 1;
	    if (*ptr[i] < wPtr->hdrSize[i]) {
		*ptr[i] = wPtr->hdrSize[i];
	    }
	} else {
	    if (Tcl_GetInt(interp, str[i], ptr[i]) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	if (*ptr[i] < 0) {
	    *ptr[i] = 0;
	}
    }

    return TCL_OK;
}

/* configure width of column or height of rows */
int
Tix_GridDataConfigRowColSize(interp, wPtr, dataSet, which, index, argc, argv,
	argcErrorMsg, changed_ret)
    Tcl_Interp * interp;
    WidgetPtr wPtr;
    TixGridDataSet * dataSet;
    int which;				/* 0=cols, 1=rows */
    int index;
    int argc;
    char ** argv;
    char * argcErrorMsg;
    int *changed_ret;
{
    TixGridRowCol *rowCol;
    Tcl_HashEntry *hashPtr;
    int isNew, code;

    hashPtr = Tcl_CreateHashEntry(&dataSet->index[which],(char*)index, &isNew);

    if (!isNew) {
	rowCol = (TixGridRowCol *)Tcl_GetHashValue(hashPtr);
    } else {
	rowCol = InitRowCol(index);
	Tcl_SetHashValue(hashPtr, (char*)rowCol);

	if (dataSet->maxIdx[which] < index) {
	    dataSet->maxIdx[which] = index;
	}
    }

    code = Tix_GrConfigSize(interp, wPtr, argc, argv, &rowCol->size,
	argcErrorMsg, changed_ret);

    if (changed_ret) {
	*changed_ret |= isNew;
    }

    return code;
}

void
Tix_GridDataGetGridSize(dataSet, width_ret, height_ret)
    TixGridDataSet * dataSet;
    int *width_ret;
    int *height_ret;
{
    int maxSize[2], i;
    Tcl_HashEntry *hashPtr;
    Tcl_HashSearch hashSearch;
    TixGridRowCol * rowCol;

    maxSize[0] = 1;
    maxSize[1] = 1;

    if (dataSet->index[0].numEntries == 0 || dataSet->index[1].numEntries==0) {
	goto done;
    }


    for (i=0; i<2; i++) {
	
	for (hashPtr = Tcl_FirstHashEntry(&dataSet->index[i], &hashSearch);
	     hashPtr;
	     hashPtr = Tcl_NextHashEntry(&hashSearch)) {

	    rowCol = Tcl_GetHashValue(hashPtr);
	    if (maxSize[i] < rowCol->dispIndex+1) {
		maxSize[i] = rowCol->dispIndex+1;
	    }
	}
    }

  done:
    if (width_ret) {
	*width_ret  = maxSize[0];
    }
    if (height_ret) {
	*height_ret = maxSize[1];
    }
}


/*
 * the following four functions return true if done -- no more rows or cells
 * are left to traverse
 */

int
Tix_GrDataFirstRow(dataSet, rowSearchPtr)
    TixGridDataSet* dataSet;
    Tix_GrDataRowSearch * rowSearchPtr;
{
    rowSearchPtr->hashPtr = Tcl_FirstHashEntry(&dataSet->index[0],
	&rowSearchPtr->hashSearch);

    if (rowSearchPtr->hashPtr != NULL) {
	rowSearchPtr->row = (TixGridRowCol *)Tcl_GetHashValue(
		rowSearchPtr->hashPtr);
	return 0;
    } else {
	rowSearchPtr->row = NULL;
	return 1;
    }
}

int
Tix_GrDataNextRow(rowSearchPtr)
    Tix_GrDataRowSearch * rowSearchPtr;
{
    rowSearchPtr->hashPtr = Tcl_NextHashEntry(&rowSearchPtr->hashSearch);

    if (rowSearchPtr->hashPtr != NULL) {
	rowSearchPtr->row = (TixGridRowCol *)Tcl_GetHashValue(
		rowSearchPtr->hashPtr);
	return 0;
    } else {
	rowSearchPtr->row = NULL;
	return 1;
    }
}

int
Tix_GrDataFirstCell(rowSearchPtr, cellSearchPtr)
    Tix_GrDataRowSearch * rowSearchPtr;
    Tix_GrDataCellSearch * cellSearchPtr;
{
    cellSearchPtr->hashPtr = Tcl_FirstHashEntry(&rowSearchPtr->row->list,
	&cellSearchPtr->hashSearch);

    if (cellSearchPtr->hashPtr != NULL) {
	cellSearchPtr->data = (char *)Tcl_GetHashValue(
		cellSearchPtr->hashPtr);
	return 0;
    } else {
	cellSearchPtr->data = NULL;
	return 1;
    }
}


int
Tix_GrDataNextCell(cellSearchPtr)
    Tix_GrDataCellSearch * cellSearchPtr;
{
    cellSearchPtr->hashPtr = Tcl_NextHashEntry(&cellSearchPtr->hashSearch);

    if (cellSearchPtr->hashPtr != NULL) {
	cellSearchPtr->data = (char *)Tcl_GetHashValue(
		cellSearchPtr->hashPtr);
	return 0;
    } else {
	cellSearchPtr->data = NULL;
	return 1;
    }
}

