

/* The sound server for Maelstrom!

   Much of this code has been adapted from 'sfxserver', written by
   Terry Evans, 1994.  Thanks! :)
*/

#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>

#ifdef linux
#include <linux/soundcard.h>
#else
#ifdef sparc	/* Solaris 2.4 */
#include <sys/audioio.h>
#else
#ifdef _INCLUDE_HPUX_SOURCE
#include <sys/audio.h>
#endif /* HPUX */
#endif /* sparc */
#endif /* linux */


#include "mixer.h"

#if defined(_SGI_SOURCE) || defined(_INCLUDE_HPUX_SOURCE) || defined(__mips)
extern void usleep(unsigned long usec);
#endif

extern "C" {
	extern char *getenv(char *);
};
static unsigned char snd2au(int sample);


Mixer:: Mixer(char *device, unsigned short initvolume)
{
	int   i;

	/* Set the sound card device */
	/* AUDIODEV is a Solaris-2.4 recommended environment variable */
	if ( (dsp_device=getenv("AUDIODEV")) == NULL ) {
#ifdef PLAY_DEV_AUDIO
		dsp_device = _PATH_DEV_AUDIO;
#else
		if ( device )
			dsp_device = device;
		else
			dsp_device = _PATH_DEV_DSP;
#endif /* PLAY_DEV_AUDIO */
	}
	dsp_fd = -1;

	/* Start up with initial volume */
	DSPopen(0);	/* Init device, but don't complain */
	(void) SetVolume(initvolume);

	/* Now set up the channels, and load some sounds */
	for ( i=0; i<NUM_CHANNELS; ++i ) {
		channels[i].position = 0;
		channels[i].in_use = 0;
		channels[i].sample = NULL;
	}
	io_handler = NULL;

	/* Get some sweet blessed silence */
	clipped = new unsigned char[frag_size];
	silence = new unsigned char[frag_size];
	/* Tell the silence to be quiet :) */
	memset(silence, 128, frag_size);
}

Mixer:: ~Mixer()
{
	delete[] clipped;
	delete[] silence;
	close(dsp_fd);
}

int
Mixer:: DSPopen(int complain)
{
#ifdef _INCLUDE_HPUX_SOURCE
	struct audio_describe ainfo;
	int audio_ctl;
	int audio_format;
	struct audio_select_thresholds threshold;
#endif
	int  frag_spec = FRAG_SPEC;

	if ( dsp_fd >= 0 ) {	// The device is already open.
		return(0);
	}
	speed = 11025;			/* Sampling speed of Maelstrom sound */

#ifdef PLAY_DEV_AUDIO
	increment = speed / 8;
	frag_size = (0x01<<(FRAG_SPEC&0x0F));
#else
	frag_size = 512;
#endif /* PLAY_DEV_AUDIO */

	/* Open the sound device (don't hang) */
	if ( (dsp_fd=open(dsp_device, (O_WRONLY|O_NONBLOCK), 0)) < 0 ) {
		if ( complain )
			perror("Mixer: Can't open sound card");
		return(-1);
	}
#ifdef REOPEN_AUDIO
	close(dsp_fd);
	/* Open the sound device (we might hang, but probably not) */
	if ( (dsp_fd=open(dsp_device, O_WRONLY, 0)) < 0 ) {
		if ( complain )
			perror("Mixer: Can't open sound card");
		return(-1);
	}
#endif

/* Do some system specific initialization */
#ifdef linux
#ifndef PLAY_DEV_AUDIO		/* VoxWare */
	if ( ioctl(dsp_fd, SNDCTL_DSP_SETFRAGMENT, &frag_spec) < 0 ) {
		if ( complain )
			perror("Mixer: Can't set frag spec");
		DSPclose();
		return(-1);
	}
	if ( ioctl(dsp_fd, SOUND_PCM_WRITE_RATE, &speed) < 0 ) {
		if ( complain )
			perror("Mixer: Can't set sampling rate");
		DSPclose();
		return(-1);
	}
	if ( ioctl(dsp_fd, SNDCTL_DSP_GETBLKSIZE, &frag_size) < 0 ) {
		if ( complain )
			perror("Mixer: Can't get fragment size");
		DSPclose();
		return(-1);
	}
#endif
#else
#ifdef _INCLUDE_HPUX_SOURCE
	if ( (audio_ctl=open("/dev/audioCtl", (O_WRONLY|O_NDELAY), 0)) < 0 ) {
    		perror("Mixer: Can't open /dev/audioCtl");
		DSPclose();
		return(-1);
	}
	if ( ioctl(audio_ctl, AUDIO_DESCRIBE, &ainfo) < 0 ) {
		perror("Mixer: Can't get audio info");
		DSPclose();
		return(-1);
	}
#ifdef PLAY_DEV_AUDIO
	audio_format = AUDIO_FORMAT_ULAW;
	speed = 8000;
#else	/* This audio format doesn't exist on HPUX... oh well. */
	audio_format = AUDIO_FORMAT_LINEAR8BIT;
#endif
	if ( ioctl(audio_ctl, AUDIO_SET_DATA_FORMAT, audio_format) < 0 ) {
		perror("Mixer: Can't set audio format");
		DSPclose();
		return(-1);
	}
  	if ( ioctl(audio_ctl, AUDIO_SET_CHANNELS, 1) < 0 ) {
		perror("Mixer: Can't set audio to one channel");
		DSPclose();
		return(-1);
	}
	if ( ioctl(audio_ctl, AUDIO_SET_SAMPLE_RATE, speed) < 0 ) {
		perror("Mixer: Can't set sample rate");
		DSPclose();
		return(-1);
	}
	if ( ioctl(audio_ctl, AUDIO_GET_SEL_THRESHOLD, &threshold) < 0 ) {
		perror("Mixer: Couldn't get audio output threshold");
		DSPclose();
		return(-1);
	}
	threshold.write_threshold = frag_size;
	if ( ioctl(audio_ctl, AUDIO_GET_SEL_THRESHOLD, &threshold) < 0 ) {
		perror("Mixer: Couldn't set audio output threshold");
		DSPclose();
		return(-1);
	}
	close(audio_ctl);
#endif /* HPUX */
#endif /* linux */

	/* This is necessary so that the sound server stays in sync */
	long flags;
	flags = fcntl(dsp_fd, F_GETFL, 0);
	flags |= O_SYNC;
	(void) fcntl(dsp_fd, F_SETFL, flags);

	return(0);
}

void
Mixer:: DSPclose(void)
{
	if ( dsp_fd >= 0 ) {
		(void) close(dsp_fd);
		dsp_fd = -1;
	}
}

int
Mixer:: SetVolume(unsigned short Volume)
{
	if ( Volume > 0x08 ) {
		fprintf(stderr, "Mixer: Warning: Volume is a range 0-8\n");
		return(-1);
	}
	if ( Volume ) {	// Don't set the volume if we can't open the device.
		if ( DSPopen(1) < 0 )
			return(-1);
		volume = (float)(Volume*32)/255.0;
	} else {
		volume = 0;
		DSPclose();
	}
	return(0);
}

int
Mixer:: SoundID(unsigned short channel)
{
	if ( channel > NUM_CHANNELS-1 )
		return(-1);
	if ( channels[channel].in_use )
		return(channels[channel].sample->ID);
	return(-1);
}

int
Mixer:: Play_Sample(unsigned short channel, Sample *sample)
{
	if ( channel > NUM_CHANNELS-1 )
		return(-1);

	channels[channel].position = 0;
	channels[channel].in_use = 1;
	channels[channel].sample = sample;
	return(0);
}

void
Mixer:: PlaySleep(void)
{
	/* Sleep for the experimentally determined value of
	   25000 microseconds (the duration of a device write.)
	 */
	for ( long timeleft=25000; timeleft > 0; ) {
		struct timeval then, now;
		long           ticks;

		gettimeofday(&then, NULL);
#ifdef ASYNCHRONOUS_IO
		usleep(timeleft);
#else
		usleep((25000/frag_size)*IO_CHECK);
		Handle_IO(0);
#endif
		gettimeofday(&now, NULL);
		ticks = ((now.tv_sec-then.tv_sec)*1000000);
		ticks += (now.tv_usec-then.tv_usec);
		timeleft -= ticks;
	}
}

/* Set up the I/O handler */
void
Mixer:: Handle_IO(int fd, void (*handler)(int))
{
	io_fd = fd;
	io_handler = handler;
}

void
Mixer:: Play(void)
{
	int frag_offset;
	int num_playing;
	unsigned short i;
	char data;

	/* Check to see if there are sounds playing */
	for( num_playing=0, i=0; i<NUM_CHANNELS; ++i ) {
		if ( channels[i].in_use )
			++num_playing;
	}
	/* If there is nothing to play, just write out silence */
	if ( num_playing == 0 ) {
#ifndef ASYNCHRONOUS_IO
		for ( frag_offset=0; frag_offset<frag_size;
						frag_offset += IO_CHECK ) {
#ifdef PLAY_DEV_AUDIO
			usleep((25000/frag_size)*IO_CHECK);
#else
			if ( dsp_fd >= 0 )
				(void) write(dsp_fd, silence, IO_CHECK);
			else
				PlaySleep();
#endif /* PLAY_DEV_AUDIO */
			Handle_IO(0);
		}
#else
#ifdef PLAY_DEV_AUDIO
		usleep(25000);
#else
		if ( dsp_fd >= 0 )
			(void) write(dsp_fd, silence, frag_size);
		else
			PlaySleep();
#endif /* PLAY_DEV_AUDIO */
#endif /* ASYNCHRONOUS_IO */
//fprintf(stderr, ".");
		return;
	}
#ifdef PLAY_DEV_AUDIO
	int sum=0;
#endif

	/* This is for mono output */
	for( frag_offset=0; frag_offset<frag_size; ++frag_offset ) {
		unclipped = 0;
		num_playing = 0;

#ifndef ASYNCHRONOUS_IO
		if ( (frag_offset%IO_CHECK) == 0 ) 
			Handle_IO(0);
#endif /* ASYNCHRONOUS_IO */

		for( i=0; i<NUM_CHANNELS; ++i ) {
			/* See if the channel is in use */
			if ( channels[i].in_use ) {
				/* Normalize the data */
				data = 
		(*(channels[i].sample->data + channels[i].position)) ^ 0x80;
				unclipped += (int)(volume * (float)data);

				/* See if this sample is done being played */
				if( ++channels[i].position 
						>= channels[i].sample->len ) {
					channels[i].in_use = 0;
					if ( channels[i].sample->callback )
						(*channels[i].sample->callback)(i);
				}
			}
		}
		/* Re-normalize the values */
		unclipped += 128;

#ifdef PLAY_DEV_AUDIO
		sum += increment;
		while ( sum > 0 ) {
			sum -= 1000;
			for( i=0; i<NUM_CHANNELS; ++i ) {
				/* See if the channel is in use */
				if ( channels[i].in_use ) {
					++channels[i].position;
				}
			}
		}
		for( i=0; i<NUM_CHANNELS; ++i ) {
			/* See if the channel is in use */
			if ( channels[i].in_use ) {
				--channels[i].position;
			}
		}
		unclipped = snd2au((0x80-unclipped)*16);
#endif
		if(unclipped < 0)
			clipped[frag_offset] = 0;
		else if(unclipped > 255)
			clipped[frag_offset] = 255;
		else
			clipped[frag_offset] = unclipped;
	}
#ifdef DEBUG
for( i=0; i<NUM_CHANNELS; ++i ) {
	if ( channels[i].in_use )
		fprintf(stderr, "Channel %hu: position = %d, len = %d\n", i, channels[i].position, channels[i].sample->len);
}
#endif
	/* Write out the data */
	if ( dsp_fd >= 0 ) {
#ifdef sparc
	drain_it:
		if ( ioctl(dsp_fd, AUDIO_DRAIN, 0) < 0 ) {
			if ( errno == EINTR )
				goto drain_it;
		}
#endif
//fprintf(stderr, "<");
	write_frag:
		if ( write(dsp_fd, clipped, frag_size) != frag_size ) {
			if ( errno == EINTR )  // Interrupted system call...
				// This should happen (SA_RESTART)
				goto write_frag;
			else {
				perror("Mixer: Can't write to audio device");
				DSPclose();
				return;
			}
		}
//fprintf(stderr, ">");
	} else
		PlaySleep();
}

void
Mixer:: Halt(unsigned short channel)
{
	channels[channel].in_use = 0;
}

void
Mixer:: HaltAll(void)
{
	for ( unsigned short i=0; i<NUM_CHANNELS; ++i )
		Halt(i);
}

/* This function (snd2au()) copyrighted: */
/************************************************************************/
/*      Copyright 1989 by Rich Gopstein and Harris Corporation          */
/*                                                                      */
/*      Permission to use, copy, modify, and distribute this software   */
/*      and its documentation for any purpose and without fee is        */
/*      hereby granted, provided that the above copyright notice        */
/*      appears in all copies and that both that copyright notice and   */
/*      this permission notice appear in supporting documentation, and  */
/*      that the name of Rich Gopstein and Harris Corporation not be    */
/*      used in advertising or publicity pertaining to distribution     */
/*      of the software without specific, written prior permission.     */
/*      Rich Gopstein and Harris Corporation make no representations    */
/*      about the suitability of this software for any purpose.  It     */
/*      provided "as is" without express or implied warranty.           */
/************************************************************************/

static unsigned char snd2au(int sample)
{

	int mask;

	if (sample < 0) {
		sample = -sample;
		mask = 0x7f;
	} else {
		mask = 0xff;
	}

	if (sample < 32) {
		sample = 0xF0 | 15 - (sample / 2);
	} else if (sample < 96) {
		sample = 0xE0 | 15 - (sample - 32) / 4;
	} else if (sample < 224) {
		sample = 0xD0 | 15 - (sample - 96) / 8;
	} else if (sample < 480) {
		sample = 0xC0 | 15 - (sample - 224) / 16;
	} else if (sample < 992) {
		sample = 0xB0 | 15 - (sample - 480) / 32;
	} else if (sample < 2016) {
		sample = 0xA0 | 15 - (sample - 992) / 64;
	} else if (sample < 4064) {
		sample = 0x90 | 15 - (sample - 2016) / 128;
	} else if (sample < 8160) {
		sample = 0x80 | 15 - (sample - 4064) /  256;
	} else {
		sample = 0x80;
	}
	return (mask & sample);
}
