#! /bin/sh
# Boot Disk maker. Bruce Perens, July-September 1995
# Eric Delaunay, 1998 [silo boot]
# Enrique Zanardi, 1998
# This is free software under the GNU General Public License.

# Abort on any non-zero return.
set -e

# Verbose shell execution.
set -x
set -v

. common.sh

#-----------------------------------------------
# write SILO boot infos
#
write_silo () {
	cat > $mnt/etc/silo.conf << EOF
partition=1
message=/debian.txt
root=/dev/fd0
timeout=600
image=/linux
label=linux
append="load_ramdisk=1 prompt_ramdisk=1"
read-write
EOF
	m4 -D__ERROR_MESS__=$scripts/messages/$language/install.sh.txt \
	   -D__version__=$KERNEL_VERSION -D__arch__=$arch \
	   <$scripts/install.sh >$mnt/install.sh
	chmod 755 $mnt/install.sh
}


#-----------------------------------------------
# write PowerPC boot infos
#
write_powerpc () {
	cat > $mnt/etc/quik.conf << EOF
partition=1
message=/debian.txt
root=/dev/fd0
timeout=600
image=/linux
label=Linux
append="load_ramdisk=1 prompt_ramdisk=1"
read-write
EOF
	m4 -D__ERROR_MESS__=$scripts/messages/$language/install.sh.txt \
	   -D__version__=$KERNEL_VERSION -D__arch__=$arch \
	   <$scripts/install.sh >$mnt/install.sh
	chmod 755 $mnt/install.sh
}

#-----------------------------------------------
# write SysLinux boot infos
#
write_syslinux () {
	[ -f "$sfont" ] && cp $sfont $mnt
	{ [ -f "$sfont" ] && echo FONT $sfont; cat $scripts/syslinux.cfg; } |\
		m4 -D__size${blocks}__ -D__tecra__=${revext} | \
		$nlfmt >$mnt/syslinux.cfg
	sed -e s/__arch__/$arch/ \
		<$scripts/rdev.sh >$mnt/rdev.sh
	chmod 777 $mnt/rdev.sh
	m4 -D__ERROR_MESS__=$scripts/messages/$language/install.sh.txt \
	   -D__version__=$KERNEL_VERSION -D__arch__=$arch \
	   <$scripts/install.sh >$mnt/install.sh
	chmod 777 $mnt/install.sh
}

#-----------------------------------------------
# write m68k info
#
write_m68kinfo () {
	m4 -D__ERROR_MESS__=$scripts/messages/$language/install.sh.txt \
	   -D__version__=$KERNEL_VERSION -D__arch__=$arch \
	   <$scripts/install.sh >$mnt/install.sh
	chmod 777 $mnt/install.sh
}

#-----------------------------------------------

# write milo boot infos
#
write_milo () {
  	if test -n "$SRM"; then
           umount $mnt
#           e2writeboot $loopdevice ../alpha-kernel-config/aboot.bootlx
           e2writeboot $loopdevice /boot/bootlx
           mount $fsopts -t $fstype $loopdevice $mnt
        else

           case $SUBARCH in
	     ruffian)
	      cp ../Milos/ruffian/* $mnt/;;
             *)
              cp ../Milos/milo_$SUBARCH $mnt/milo
	      cp ../Milos/linload.exe $mnt/
              umount $mnt
              srmbootfat $loopdevice milo
              mount $fsopts -t $fstype $loopdevice $mnt
           esac
        fi
	sed -e s/__version__/$KERNEL_VERSION/ -e s/__arch__/$arch/ \
		<scripts/rescue/install.sh >$mnt/install.sh
	chmod 777 $mnt/install.sh
}



#-----------------------------------------------
# write VMELILO boot infos
#
write_vmelilo () {
	mkdir $mnt/etc
	mkdir $mnt/boot
	cat > $mnt/etc/vmelilo.conf << EOF
[options]
prompt  = true
default = install
message = /debian.txt
cmdline = root=/dev/ram0 ro console=/dev/ttyS0

[boot]
label   = Install
image   = /linux
ramdisk = /root.bin

[boot]
label   = Rescue
image   = /linux
ramdisk = /root.bin
append  = single
EOF
	m4 -D__ERROR_MESS__=$scripts/messages/$language/install.sh.txt \
	   -D__version__=$KERNEL_VERSION -D__arch__=$arch \
	   <$scripts/install.sh >$mnt/install.sh
	chmod 755 $mnt/install.sh
	vmelilo -f -a $system -b /dev/null -w $mnt -d $loopdevice
}

# Print a usage message and exit if the argument count is wrong.
if [ $# -lt 7 ]; then
echo "Usage: "$0" revision-extension kernel-package root-image blocks debian-version system language sfont" 1>&2
	cat 1>&2 << EOF

	revision-extension: the revision extension of the kernel.
	kernel-package: the Debian package containing the kernel.
	root-image: a compressed disk image to load in ramdisk and mount as root.
	blocks: the size of the floppy in 1K blocks, use 720, 1200, or 1440.
	debian-version: version of the Debian release.
	system: name of the target system.
	language: language for the boot messages and help files.
	sfont: name of the font loaded by syslinux, if any
EOF

	exit -1
fi

revext=$1

# Set this to the location of the package archive.
archive=$2

# Set this to the location of the root filesystem image
rootimage=$3

# Set to the number of blocks per floppy.
blocks=$4

# Set this to the Debian version
debianversion=$5

# Set this to the system
system=$6

# Set this to the language for the boot messages and help files.
language=$7

# Set this to the font loaded by syslinux, if any.
sfont=$8

# Set this to the charset for the boot messages and help files.
# FIXME: this should be a runtime parameter.
charset=latin1

# Set this to the block device for the floppy disk.
floppy=/var/tmp/floppy-image.$$

mnt=/var/tmp/mnt.$$

loopdevice=/dev/loop3

# The home of the scripts used by this script.
scripts=`pwd`/scripts/rescue

# Make sure the kernel package is available.
if [ ! -f $archive ]; then
	echo "Can't find $archive" 1>&2
	exit -1
fi

# Try to unmount anything that should not be mounted. Aborted runs of this
# script may have left things mounted.
(umount /mnt; umount $mnt; umount $floppy; true)2>/dev/null >/dev/null

mkdir -p $mnt

# Zero the entire disk, so that when I compress the raw disk image,
# unused blocks will compress well.
dd if=/dev/zero of=$floppy bs=1k count=$blocks

# Get the kernel version from its package name.
if [ "$arch" = powerpc ]; then
   KERNEL_VERSION=`basename $archive | sed -e 's/kernel-image-//' -e 's/-.*//'`
else
   KERNEL_VERSION=`basename $archive | sed -e 's/kernel-image-//' -e 's/_.*//'`
fi
DEBIAN_KERNEL_IMAGE=`basename $archive .deb | sed -e s/_$arch\$//`
case $arch in
 alpha) SUBARCH=`expr $revext : 'alpha_\(.*\)'`;;
esac

# Do the actual work of making the disk bootable.
if [ $blocks -le 1400 ]; then
    rootimage=/var/tmp/root.$$
    echo -n >$rootimage
fi

# Make the filesystem using the loop device, so that the floppy image file
# won't be truncated if the filesystem is shorter than the image file.
(umount $loopdevice; true)
(/sbin/losetup -d $loopdevice; true)
/sbin/losetup $loopdevice $floppy

case "$arch" in
    alpha)
          case $SUBARCH in
          cabriolet|jensen|noritake|sable*|\
            mikasa|book1|miata-s)
                  mke2fs -F $loopdevice $blocks
                  fstype=ext2
                  fsopts=
                  SRM=1
                  nlfmt="cat -"       #  UNIX newline convention
              ;;
              *)
                  losetup -d $loopdevice
                  mkdosfs -r 112 -F 12 $floppy $blocks
                  losetup $loopdevice $floppy
                  fstype=msdos
                  fsopts="-o fat=12"
                  SRM=
                  nlfmt="cat -"       #  UNIX newline convention
              ;;
          esac
    ;;
    arm)
	echo "No support (yet) for ARM!"
	exit 1
    ;;
    i386)
        if [ -f /usr/lib/syslinux/img$blocks""k.gz ]; then
                gunzip</usr/lib/syslinux/img$blocks""k.gz > $loopdevice
        elif [ -f $scripts/img$blocks""k.gz ]; then
                gunzip<$scripts/img$blocks""k.gz > $loopdevice
        else
                echo "Can't find $blocks KB floppy image"
                exit 1
        fi
        syslinux $loopdevice    # Use the "fast" loader.
        fstype=msdos
        if [ -f "$sfont" ]; then   # MSDOS newline convention
            nlfmt=sed\ s/$/$'\r'/
        else
            nlfmt="recode $charset:ibmpc"
        fi;
        fsopts="-o fat=12"
    ;;
    m68k)
          case "$system" in
              amiga|atari)
                  mkdosfs -r 112 -F 12 $floppy $blocks
                  fstype=msdos
                  fsopts="-o fat=12"
                  nlfmt="recode $charset:ibmpc"       # MSDOS newline convention
              ;;
	      mac)
		  mkdosfs -r 112 -F 12 $floppy $blocks	# just for resc1440; 
		  fstype=msdos				# really want HFS for boot floppy
		  fsopts="-o fat=12"
		  nlfmt="recode $charset:applemac"    # Mac newline convention
	      ;;
              mvme|bvme)
                  # uses ext2 formatted floppies & VMELILO boot loader
                  mke2fs $loopdevice $blocks
                  fstype=ext2
                  nlfmt='cat -'       # UNIX newline convention
              ;;
          esac
    ;;
    mips)
	echo "No support (yet) for Mips!"
	exit 1
    ;;
    powerpc)
            case "$system" in
              apus)
                  echo "Don't know what to do for APUS"
                  echo "Use for the moment the m68k-amiga boot-floppies"
                  exit 1
              ;;
              chrp|prep)
                  mkdosfs -r 112 -F 12 $floppy $blocks
                  fstype=msdos
                  fsopts="-o fat=12"
                  nlfmt="recode $charset:ibmpc"        # MSDOS newline convention
              ;;
              mbx)
                  echo "Don't know what to do for MBX"
                  exit 1
              ;;
              pmac)
                  mkdosfs -r 112 -F 12 $floppy $blocks # just for resc1440; 
                  fstype=msdos                         # really want HFS for boot floppy
                  fsopts="-o fat=12"
                  nlfmt="recode $charset:applemac"     # Mac newline convention
              ;;
              esac
    ;;
    sparc)
        # [ED] sparc uses ext2 formatted floppies & SILO boot loader
	# silo1440k is an floppy image with Sun Disklabel & SILO installed
	# silo1440k-sun4u is the same for ultrasparc
	silo_suffix=`expr "$revext" : '.*\(-sun4u\)';/bin/true`
	gunzip < silo${blocks}k$silo_suffix.gz > $loopdevice
	fstype=ext2
	nlfmt='cat -'	# UNIX newline convention
    ;;
esac

# mount the disk image to install additional files
mount $fsopts -t $fstype $loopdevice $mnt

# For atari copy bootloader and linux on disk
if [ "$system" = atari ]; then
	mkdir $mnt/auto
	# hardcoded bootargs; kernel is in \, bootstrap in \auto
	cat <<EOF > $mnt/auto/bootargs
-s -k \linux -r \root.bin root=/dev/ram load_ramdisk=1 video=keep
EOF
	cp m68k-specials/bootstra.ttp $mnt/auto/bootstra.prg
	# both bootstrap and kernel/ramdisk in \
	cat <<EOF > $mnt/bootargs
-s -k linux -r root.bin root=/dev/ram load_ramdisk=1 video=keep
EOF
	cp m68k-specials/bootstra.ttp $mnt/bootstra.ttp
fi

cp linux$revext $mnt/linux

# only i386 can use rdev
if [ "$arch" = i386 ]; then
	/usr/sbin/rdev -R $mnt/linux 1
       	/usr/sbin/rdev -r $mnt/linux 0
       	/usr/sbin/rdev -v $mnt/linux -1
       	/usr/sbin/rdev $mnt/linux /dev/ram0
fi

if [ ! "$arch" = sparc ]; then
	cp $rootimage $mnt/root.bin
fi

if [ $blocks -le 1400 ]; then
    rm -f $rootimage
fi

# Copy system.map to the floppy.
# FIXME The kernel file is HUGE! There's no space left for the sys_map file.
# if [ "$revext" != lowmem ]; then
# 	cp sys_map${revext}.gz $mnt/sys_map.gz
# fi

# Install the greeting and help messages (f6.txt f7.txt f8.txt contain
#   no information at all, so don't copy them to save some space.
case "$arch/$system" in
*/bvme|*/mvme)
	suffix=.vme
	files=debian.txt
	;;
powerpc/*)
	suffix=.powerpc
	files="debian.txt readme.txt"
	;;
sparc/*)
	suffix=.sparc
	files="debian.txt readme.txt"
	;;
*)
	suffix=".$arch"
	files="debian.txt readme.txt f1.txt f10.txt f2.txt f3.txt f4.txt \
	f5.txt f9.txt" 
	;;
esac
for i in $files; do
	file=$scripts/messages/$language/$i$suffix
	if ! [ -f $file ]; then
		file=$scripts/messages/$language/$i
	fi
	cat $file | \
		m4 -P -D__kernel_version__=$KERNEL_VERSION \
				-D__kernel_image__=$DEBIAN_KERNEL_IMAGE \
				-D__size${blocks}__ \
				-D__debianversion__=$debianversion \
				-D__arch__=$arch \
			| $nlfmt > $mnt/$i
done

# install the boot loader
if [ "$arch" = sparc ]; then
	write_silo
elif [ "$arch" = alpha ]; then
	write_milo
elif [ "$arch" = powerpc ]; then
	write_powerpc
elif [ "$system" = bvme -o "$system" = mvme ]; then
	write_vmelilo
elif [ "$system" = atari -o "$system" = amiga -o "$system" = mac ]; then
	# these LILOs not implemented yet, but need to write install.sh
	echo "Not writing syslinux for $system !"
	write_m68kinfo
else
	write_syslinux
fi

echo rescue >$mnt/type.txt

umount $mnt
/sbin/losetup -d $loopdevice
rmdir $mnt

(rm -f -r /var/tmp/extract-tmp-$$ resc${blocks}${revext}.bin;true)
mv $floppy resc${blocks}${revext}.bin

ls -l resc${blocks}${revext}.bin
exit 0
