#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/klog.h>
#include <sys/reboot.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>

#include "dinstall.h"
#include "../busybox/internal.h"
#include INCLINGUA

char *real_kver = KVER;
char suffix_arch[16] = "";
char *kernel_image_filename;
char *drivers_image_filename;

struct Arg {
  const char *name;
  int isflag;
  void *value;
};

int parsefile(const char *filename, struct Arg *options, 
	int hasinterspace) {
  int fd, bytes, lineend=0;
  char *start= prtbuf, *end;
  struct Arg *a;

  if ((fd = open(filename, O_RDONLY, 0)) < 0) {
    fprintf( stderr, MSG_CANNOT_OPEN_FILE, filename, strerror(errno) );
    return 1;
  }
  while ( (bytes = read(fd,prtbuf,PRTBUFSIZE-1)) != 0) {
    prtbuf[bytes]= '\0';

    /* parse line */
    while (!lineend) {
      /* skip space at the beggining of the line */
      while (isspace(*start)) start++;
      if (!(*start)) break;

      /* take the first word on the string as the option name */
      end = start;
      while (*end && !isspace(*end)) end++;
      if (!*end) lineend = 1;
      *end = '\0';

      a =options;
      while ( a->name != NULL ) {
	if (a->isflag) {
	  if (!strcmp(start, a->name)) {
	    /* the option is valid. It is a flag, switch it on */
	    *((int *)a->value)=1;
	  }
	} else {
	  int namelen=strlen(a->name);
	  if (!strncmp(start, a->name, namelen)) {
	    /* the option is valid. It expects a value */
	    if (hasinterspace) {
	      if (lineend) {
		/* If the file syntax is "name value" (hasinterspace==1) 
		 * and lineend == 1, we have found a name with no value. */
		break;
	      }
	      /* skip whitespace after the option name */
	      namelen=0;
	      start = end + 1;
	      while (isspace(*start)) start++;
	      if (!(*start)) {
		/* We have reached the end of the line, that means we have
		 * found a name with no value. */
		break;
	      }
	      end = start;
	      while (*end && !isspace(*end)) end++;
	      *end = '\0';
	      /* Skip the rest of the line */
	      lineend = 1;
	    }  
	    *((char **)a->value)=strdup(start+namelen);
	  }
	}
	a++;
      }
      start = end + 1;
    }
  }
  close(fd);
  return 0;
}
  
/* Get arguments from boot commmand line */
int readcmdline(void){
  int status;
  struct Arg bootoptions[] = {
    { "root=", 0, &bootargs.root },
    { "bootkbd=", 0, &bootargs.kbd },
    { "tecra", 1, &bootargs.istecra },
    { NULL, 0, NULL}
  };
  memset((void *)&bootargs, 0, sizeof(struct BootArgs));
  status=parsefile("/proc/cmdline",bootoptions,0);
  return status;
}

#if #cpu (m68k)
static const char *get_m68k_model( void )
{
  char *model = NULL;
  int status;
  struct Arg archoptions[] = {
    { "Model:", 0, &model },
    { NULL, 0, NULL}
  };
  status=parsefile("/proc/hardware",archoptions,1);
  if(status) {
    fprintf( stderr, MSG_UNKNOW_M68K_TYPE );
    return NULL;
  }
  if (!model || !*model)
     fprintf( stderr, MSG_UNABLE_TO_FIND_M68K_TYPE );
  return model;
}
#endif

#if #cpu (powerpc)
static const char *get_powerpc_model( void )
{
  char *model = NULL;
  int status;
  struct Arg archoptions[] = {
    { "machine:", 0, &model },
    { NULL, 0, NULL}
  };
  status=parsefile("/proc/cpuinfo",archoptions,2);
  if(status) {
    fprintf( stderr, MSG_UNKNOW_POWERPC_TYPE );
    return NULL;
  }
  if (!model || !*model)
     fprintf( stderr, MSG_UNABLE_TO_FIND_POWERPC_TYPE );
  return model;
}
#endif

/* try to eject root floppy */
#if #cpu(sparc)
void
try_to_eject_root_floppy(void)
{
  if (bootargs.root && 0==strncmp(bootargs.root,"/dev/fd",7)) {
      /* extract floppy device name */
      eject_floppy(bootargs.root);
  }
}
#endif

#ifdef SCSI_FLOPPY
/* scan /proc/scsi/scsi for TEAC FC-1 floppy drive
 * and create /dev/sfd[0-7] symlinks if found
 */
void
find_scsi_floppy()
{
  FILE *fp;
  char buf[100];
  char dev[8];
  int  i = -1;
  int  n = 0;
 
  if ((fp = fopen("/proc/scsi/scsi", "r")) != NULL) {
    while (fgets(buf, sizeof(buf), fp)) {
      if ((strncmp(buf, "Host:", 5) == 0) && strstr(buf, "Channel:"))
        i++;
      else {
        if (strstr(buf, "Vendor:")
        &&  strstr(buf, "TEAC"   )
        &&  strstr(buf, "Model:" )
        &&  strstr(buf, "FC-1"   )) {
          sprintf(dev, "sd%c", 'a' + i);
          sprintf(buf, "/dev/sfd%d", n++);
          unlink(buf);
          symlink(dev, buf);
        }
      }
    }
    fclose(fp);
  }
}
#endif

static void
setup_image_names(void)
{
  char suffix_kver[12]="";
#ifdef MULTI_KERNELS
  /* initialize real_kver & real_arch */
  char key[20];
  FILE *f = fopen("/proc/sys/kernel/osrelease", "r");
  fscanf(f, "%s", key);
  fclose(f);
  real_kver = strdup(key);
  if (strcmp(real_kver, KVER)) {
    /* only set if not the default kernel used for installation */
    sprintf(suffix_kver, "-%s", key);
  }
#if #cpu(sparc)
  f = fopen("/proc/cpuinfo", "r");
  while (fscanf(f, "%s : %s", key, prtbuf)) {
    if (! strcmp(key, "type"))
      break;
  }
  fclose(f);
  if (!strncmp(prtbuf,"sun4",4) &&
      prtbuf[4]!='c' && prtbuf[4]!='d' && prtbuf[4]!='m') {
    /* only set if not sun4[cdm] */
    snprintf(suffix_arch,sizeof(suffix_arch), "-%s", prtbuf);
  }
#elif #cpu(i386)
  /* can use MULTI_KERNELS option to select tecra bootdisks */
  if (bootargs.istecra)
    strcpy(suffix_arch,"tecra");
#endif
#endif
  kernel_image_filename = malloc(strlen(KERDISKIMG)+strlen(suffix_kver)+strlen(suffix_arch));
  sprintf(kernel_image_filename, KERDISKIMG, suffix_kver, suffix_arch);
  drivers_image_filename = malloc(strlen(DRVDISKIMG)+strlen(suffix_kver)+strlen(suffix_arch));
  sprintf(drivers_image_filename, DRVDISKIMG, suffix_kver, suffix_arch);
}

#ifdef INCLUDE_DINSTALL
  int dbootstrap_main(void) {
#else
  int main (void) {
#endif
    char *term;

/* Downgrade kernel verbosity */
    klogctl(8,NULL,4);

    /* setup rescue & drivers disks images filenames
     * (should be done before the first call to stderrToTTY) */
    setup_image_names();

    initMonoScreen();

    readcmdline();

#if #cpu (m68k)
    Arch2=get_m68k_model();
    if ((strncmp(Arch2, "BVME",     4) == 0)
    ||  (strncmp(Arch2, "Motorola", 8) == 0)) Arch2 = "VME";
    else
    if ( (strcmp(Arch2,"Atari")) && (strcmp(Arch2,"Amiga")) &&
	 (strcmp(Arch2,"Macintosh"))) {
      problemBox(MSG_UNKNOW_M68K_ARCH,MSG_PROBLEM);
      reboot(RB_AUTOBOOT);
    }

    /* stop kernel complaining to console about scsi floppy with no disk */
    if (strcmp(Arch2, "VME") == 0)
      klogctl(6,NULL,0);
#endif

#ifdef SCSI_FLOPPY
    find_scsi_floppy();
#endif

#if #cpu (powerpc)
    Arch2=get_powerpc_model();
    if ( (strcmp(Arch2,"CHRP")) && (strcmp(Arch2,"PowerMac")) &&
	 (strcmp(Arch2,"PReP"))) {
      problemBox(MSG_UNKNOW_POWERPC_ARCH,MSG_PROBLEM);
      reboot(RB_AUTOBOOT);
    }
#endif

#if #cpu(sparc)
    try_to_eject_root_floppy();
#endif

    Root = NULL;
    noSwap = 0;
    notCreatedBootFloppy = 1;
    notInstalledLILO = 1;
    Archive_Dir=NULL;

    InstallationRootDevice=block_device("/",NULL);
    if (!InstallationRootDevice) {
      /* net installation with no /dev/root device on nfsroot */
      /* leave time to see any error message */
      sleep(8);
      problemBox(MSG_TROUBLE_ROOT_KO,MSG_PROBLEM);
#if #cpu(alpha)
      exit(1);
#else
      reboot(RB_AUTOBOOT);
#endif
    }

    term = getenv("TERM");
    if (!term || (term && !strncmp(term,"linux",5))) {
      color_or_monochrome();
    }
   
    release_notes();

    is_root_a_floppy ();

    main_menu();

    boxFinished();

#if #cpu (m68k)
    /* restart kernel logging to console */
    if (strcmp(Arch2,"VME") == 0)
      klogctl(7,NULL,0);
#endif

    return 0;
}

