#include "config.h"
#if HAVE_ATARI_PARTITION

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <asm/types.h>
#include "byteorder.h"

#include "fdisk.h"

struct partition_info
{
    __u8  flg;			/* bit 0: active; bit 7: bootable */
    char  id[3];		/* "GEM", "BGM", "XGM", or other */
    __u32 st;			/* start of partition */
    __u32 siz;			/* length of partition */
};

struct rootsector
{
    __u8  unused[0x156];		/* room for boot code */
    struct partition_info icdpart[8];	/* info for ICD-partitions 5..12 */
    __u8  unused2[0xc];
    __u32 hd_siz;			/* size of disk in blocks */
    struct partition_info part[4];
    __u32 bsl_st;			/* start of bad sector list */
    __u32 bsl_cnt;			/* length of bad sector list */
    __u16 checksum;			/* checksum for bootable disks */
} __attribute__ ((__packed__));


/* ++guenther: this should be settable by the user ("make config")?.
 * roman: doesn't use much code, better always include it :-)
 */
#define ICD_PARTS

/* check if a partition entry looks valid -- Atari format is assumed if at
 * least one of the primary entries is ok this way */
#define	VALID_PARTITION(pi,hdsiz)					     \
    (((pi)->flg & 1) &&							     \
     isalnum((pi)->id[0]) && isalnum((pi)->id[1]) && isalnum((pi)->id[2]) && \
     __be32_to_cpu((pi)->st) <= (hdsiz) &&				     \
     __be32_to_cpu((pi)->st) + __be32_to_cpu((pi)->siz) <= (hdsiz))

#define MK_PID(pi) \
    (PTYPE_PREFIX_ATARI|((pi)->id[0] << 16) | ((pi)->id[1] << 8) | (pi)->id[2])
    
int parse_atari_partition (char *device, int fd)
{
    int minor = 1;
    int i;
    struct rootsector *rs;
    struct partition_info *pi;
    struct fdisk_disk *fdisk_device;
    unsigned char data[512], xdata[512];
    __u32 extensect;
    __u32 hd_size;
#ifdef ICD_PARTS
    int part_fmt = 0; /* 0:unknown, 1:AHDI, 2:ICD/Supra */
#endif

    if (!sread( fd, 0, data ))
	return -1;

    rs = (struct rootsector *) data;
    if (!(fdisk_device = fdisk_find_disk( device )))
	return 0;
	/* fdisk_device->size is in kByte, but we need it in sectors for
	 * comparisons. */
    hd_size = 2*fdisk_device->size+1;

    if (!VALID_PARTITION( &rs->part[0], hd_size ) &&
	!VALID_PARTITION( &rs->part[1], hd_size ) &&
	!VALID_PARTITION( &rs->part[2], hd_size ) &&
	!VALID_PARTITION( &rs->part[3], hd_size ))
	/* if there's no valid primary partition, assume that no Atari format
	 * partition table (there's no reliable magic or the like :-() */
	return 0;

    pi = &rs->part[0];
    for (; pi < &rs->part[4]; minor++, pi++) {
	if (pi->flg & 1) {
	    /* active partition */
	    if (memcmp (pi->id, "XGM", 3) == 0) {
		/* extension partition */
		struct rootsector *xrs;
		__u32 partsect;

#ifdef ICD_PARTS
		part_fmt = 1;
#endif
		partsect = extensect = __be32_to_cpu(pi->st);
		while (1) {
		    if (!sread( fd, partsect, xdata ))
			return 0;
		    xrs = (struct rootsector *) xdata;

		    /* ++roman: valid bit must be set in one of the first 3
		     * slots */
		    for( i = 0; i < 3; ++i )
			if (xrs->part[0].flg & 1)
			    break;
		    if (i == 3)
			/* No valid subpartition in extended partition */
			break;
		  
		    fdisk_add_partition(device, minor, MK_PID(&xrs->part[i]),
					__be32_to_cpu(xrs->part[i].siz)/2);

		    /* the slot following must be either invalid (end of list)
		     * or another XGM entry */
		    if (!(xrs->part[i+1].flg & 1))
			break;
		    if (memcmp( xrs->part[i+1].id, "XGM", 3 ) != 0)
			/* Link ID in extended partition is not XGM */
			break;
		    
		    partsect = __be32_to_cpu(xrs->part[i+1].st) + extensect;
		    minor++;
		}
	    }
	    else {
		/* we don't care about other id's */
		fdisk_add_partition(device, minor, MK_PID(pi),
				    __be32_to_cpu(pi->siz)/2);
	    }
	}
    }
#ifdef ICD_PARTS
  if ( part_fmt!=1 ) { /* no extended partitions -> test ICD-format */
      pi = &rs->icdpart[0];
      /* sanity check: no ICD format if first partition invalid */
      if (memcmp (pi->id, "GEM", 3) == 0 ||
	  memcmp (pi->id, "BGM", 3) == 0 ||
	  memcmp (pi->id, "LNX", 3) == 0 ||
	  memcmp (pi->id, "SWP", 3) == 0 ||
	  memcmp (pi->id, "RAW", 3) == 0 ) {
	  for (; pi < &rs->icdpart[8]; minor++, pi++) {
	      /* accept only GEM,BGM,RAW,LNX,SWP partitions */
	      if (pi->flg & 1 && 
		  (memcmp (pi->id, "GEM", 3) == 0 ||
		   memcmp (pi->id, "BGM", 3) == 0 ||
		   memcmp (pi->id, "LNX", 3) == 0 ||
		   memcmp (pi->id, "SWP", 3) == 0 ||
		   memcmp (pi->id, "RAW", 3) == 0) ) {
		  part_fmt = 2;
		  fdisk_add_partition (device, minor, MK_PID(pi),
				       __be32_to_cpu(pi->siz)/2);
	      }
	  }
      }
  }
#endif
  return 1;
}


#endif /* HAVE_ATARI_PARTITION */
