/* xrtools - Color conversion routines and other low-level X support

   Copyright (C) 1998 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@gnu.org>
   Date: Oct 1998
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "xrtools.h"
#include <Foundation/NSArray.h>
#include <Foundation/NSDebug.h>
#define Object X11Object        /* To avoid problems with Objective C */
#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include "wraster.h"
#undef Object

#undef SET_GLOBAL_COLORMAP

static Atom	_XA_GNUSTEP_RGB_MAP = 0;
static Atom	_XA_GNUSTEP_GRAY_RAMP = 0;

/* Maintaining and using colormaps */
static int
xrAllocGrayScale(xr_context_t* context, XStandardColormap *cmap,
		 unsigned long *colors, int ncolors)
{
  long i;
    
  if (!colors)
    return -1;
  for (i=0; i < ncolors; i++) 
    {
      XColor color;
      color.red = color.green = color.blue = i * 65535 / (ncolors-1);
      color.flags = DoRed | DoGreen | DoBlue;
      color.pixel = colors[i];
      XStoreColor(context->display, cmap->colormap, &color);
    }

  cmap->red_max = ncolors - 1;
  cmap->red_mult = 1;
  cmap->green_max = 0;
  cmap->green_mult = 0;
  cmap->blue_max = 0;
  cmap->blue_mult = 0;
  cmap->base_pixel = colors[0];
  return 0;
}

int
xrAllocPseudoColor(xr_context_t* context, XStandardColormap *cmap,
		   unsigned long *colors, int ncolors)
{
  long i, cpc;
  if (!colors)
    return -1;

  cpc = context->attr.colors_per_channel;
  if ((context->attr.flags & XRGamma) && context->attr.rgamma > 0
      && context->attr.ggamma > 0 && context->attr.bgamma > 0) 
    {
      int r, g, b;
      double rg, gg, bg;
      double tmp;
      XColor color;

      /* do gamma correction */
      i = 0;
      rg = 1.0/context->attr.rgamma;
      gg = 1.0/context->attr.ggamma;
      bg = 1.0/context->attr.bgamma;
      for (r=0; r<cpc; r++) {
	for (g=0; g<cpc; g++) {
	  for (b=0; b<cpc; b++) {

	    color.red=(r*0xffff) / (cpc-1);
	    color.green=(g*0xffff) / (cpc-1);
	    color.blue=(b*0xffff) / (cpc-1);
	    color.flags = DoRed|DoGreen|DoBlue;

	    tmp = (double)color.red / 65536.0;
	    color.red = (unsigned short)(65536.0*pow(tmp, rg));

	    tmp = (double)color.green / 65536.0;
	    color.green = (unsigned short)(65536.0*pow(tmp, gg));

	    tmp = (double)color.blue / 65536.0;
	    color.blue = (unsigned short)(65536.0*pow(tmp, bg));
	    color.pixel = colors[i];
	    XStoreColor(context->display, cmap->colormap, &color);

	    i++;
	  }
	}
      }

    } 
  else 
    {
      int r, g, b;
      XColor color;
      i = 0;
      for (r=0; r<cpc; r++) {
	for (g=0; g<cpc; g++) {
	  for (b=0; b<cpc; b++) {
	    color.red=(r*0xffff) / (cpc-1);
	    color.green=(g*0xffff) / (cpc-1);
	    color.blue=(b*0xffff) / (cpc-1);
	    color.flags = DoRed|DoGreen|DoBlue;
	    color.pixel = colors[i];
	    XStoreColor(context->display, cmap->colormap, &color);
	    i++;
	  }
	}
      }
    }

  cmap->red_max = cpc-1;
  cmap->green_max = cpc-1;
  cmap->blue_max = cpc-1;
  cmap->red_mult = cpc*cpc;
  cmap->green_mult = cpc;
  cmap->blue_mult = 1;
  cmap->base_pixel = colors[0];
    return 0;
}

/* Find the offset for a given mask */
unsigned long
mask_offset(unsigned long mask)
{
  unsigned long number = 1;
  if (mask == 0)
    return 0;
  while ((number & mask) == 0)
    number = (number << 1L);
  return number;
}

static XStandardColormap *
xrGetStandardColormap(xr_context_t* context, Atom map_type)
{
  int nitems_ret;
  XStandardColormap *cmap, *retmap;
  
  /* Do we already have a colormap ready? */
  retmap = NULL;
  if (XGetRGBColormaps(context->display,
                       RootWindow(context->display,context->screen_number),
                       &cmap,
                       &nitems_ret,
                       map_type))
    {
      int i = 0;
      while (i < nitems_ret)
        if (cmap[i].visualid == context->vinfo.visualid)
          {
            retmap = XAllocStandardColormap();
            memcpy(retmap, &cmap[i], sizeof(XStandardColormap));
            break;
          }
      XFree(cmap);
    }
  return retmap;
}

static BOOL
xrGetDefaultGraymap(xr_context_t* context)
{
  unsigned long *cells;
  XStandardColormap *cmap;
  
  /* Do we already have colormaps ready? */
  if (_XA_GNUSTEP_GRAY_RAMP == 0)
    _XA_GNUSTEP_GRAY_RAMP = XInternAtom(context->display, "RGB_GRAY_RAMP",
					False);
  if ((context->graymap = xrGetStandardColormap(context, _XA_GNUSTEP_GRAY_RAMP)) 
     != NULL)
    {
      context->ngrays = context->graymap->red_max + 1;
      NSDebugLLog(@"DPS", @"Found default graymap (%d colors)\n", 
		  context->ngrays);
      return YES;
    }
  
  /* No colormap found, create one */
  if (context->vinfo.class == DirectColor || context->vinfo.class == TrueColor)
    {
      /* Use the RGB map */
      context->ngrays = 0;
      return YES;
    }

  context->graymap = XAllocStandardColormap();
  cmap = context->graymap;

  /* See how many colors we can allocate */
  if (context->attr.flags & XRNumberOfGrays)
    context->ngrays = context->attr.number_of_grays;
  else
    context->ngrays = 17;
  NSDebugLLog(@"DPS", @"Trying graymap of %d colors\n", context->ngrays);
  if (context->vinfo.class == GrayScale || context->vinfo.class == PseudoColor)
    {
      int i, ngrays;

      ngrays = context->ngrays;
      if ((context->attr.flags & XRColormapStyle) 
          && context->attr.colormap_style == four_colormap)
	ngrays = 4;
      cmap->colormap = DefaultColormap(context->display, 
                                       context->screen_number);
      i = 0;
      while (i == 0 && ngrays >= 4) 
        {
          OBJC_MALLOC(cells, unsigned long, ngrays);
          i = XAllocColorCells(context->display, 
                               cmap->colormap,
                               TRUE,
                               NULL,
                               0,
                               cells,
                               ngrays);
          if (i == 0) 
            {
              OBJC_FREE(cells);
              cells = NULL;
	      if (ngrays > 8)
		ngrays -= 4;
	      else
		ngrays -= 1;
            }
        }

      if (i == 0 && (context->attr.flags & XRPrivateColormap))
        {
          OBJC_FREE(cells);
          cells = NULL;
          NSLog(@"Warning (xraw): Creating private colormap\n");
          cmap->colormap = XCreateColormap(context->display, 
                       RootWindow(context->display, context->screen_number),
                       context->vinfo.visual,
                       AllocAll);
	  ngrays = context->ngrays;
        }
      else if (i == 0)
        {
          NSLog(@"Warning (xraw): No colors. Using black/white\n");
          context->attr.flags |= XRColormapStyle;
          context->attr.colormap_style = no_colormap;
          context->white = WhitePixel(context->display, context->screen_number);
          context->ltgray = WhitePixel(context->display, context->screen_number);
          context->black = BlackPixel(context->display, context->screen_number);
          context->dkgray = BlackPixel(context->display, context->screen_number);
          context->ngrays = 0;
	  XFree(context->graymap);
	  context->graymap = NULL;
          return NO;
        }
      context->ngrays = ngrays;
    }
  else
    context->ngrays = 0; /* Just use the RGB map */
      
  context->graymap = cmap;
  NSDebugLLog(@"DPS", @"Allocated graymap of %d colors\n", context->ngrays);
  xrAllocGrayScale(context, context->graymap, cells, context->ngrays);
  XFree(cells);

  if (context->graymap->colormap != DefaultColormap(context->display, 
                                       context->screen_number))
    XSetWindowColormap(context->display, context->drawable, 
		       context->graymap->colormap);

  /* Set our new colormap so other apps can use it */
#ifdef SET_GLOBAL_COLORMAP
  XSetRGBColormaps(context->display, 
		   RootWindow(context->display,context->screen_number),
		   context->graymap,
		   1,
		   _XA_GNUSTEP_GRAY_RAMP);
#endif
  return YES;
}

static BOOL
xrGetDefaultRGBmap(xr_context_t* context)
{
  int cpc;
  unsigned long *cells;
  XStandardColormap *cmap;
  
  /* Do we already have colormaps ready? */
  _XA_GNUSTEP_RGB_MAP = (context->attr.colormap_style == rgb_colormap) ? 
    XA_RGB_DEFAULT_MAP : XA_RGB_GRAY_MAP;
  if ((context->rgbmap = xrGetStandardColormap(context, _XA_GNUSTEP_RGB_MAP)) 
      != NULL)
    {
      context->ncolors = (context->rgbmap->red_max + 1)
	* (context->rgbmap->green_max + 1)
	* (context->rgbmap->blue_max + 1);
      NSDebugLLog(@"DPS", @"Found default rgbmap\n");
      return YES;
    }

  /* No colormap found, create one */
  if ((context->attr.flags & XRColormapStyle) 
      && context->attr.colormap_style == four_colormap)
    {
      /* Don't do an RGB map. Just a gray one */
      return NO;
    }

  context->rgbmap = XAllocStandardColormap();
  cmap = context->rgbmap;
  cmap->colormap = context->graymap->colormap;
  if (context->vinfo.class == DirectColor || context->vinfo.class == TrueColor)
    {
      cmap->red_mult = mask_offset(context->vinfo.red_mask);
      cmap->green_mult = mask_offset(context->vinfo.green_mask);
      cmap->blue_mult = mask_offset(context->vinfo.blue_mask);
      cmap->red_max = context->vinfo.red_mask / cmap->red_mult;
      cmap->green_max = context->vinfo.green_mask / cmap->green_mult;
      cmap->blue_max = context->vinfo.blue_mask / cmap->blue_mult;
      cmap->base_pixel = 0;
      context->ncolors = context->vinfo.colormap_size;
      return YES;
    }

  /* See how many colors we can allocate */
  if (context->attr.flags & XRColorsPerChannel)
    cpc = context->attr.colors_per_channel;
  else
    cpc = 5;
  NSDebugLLog(@"DPS", @"Trying rgbmap of %d color cube\n", cpc);
  if (context->vinfo.class == GrayScale || context->vinfo.class == PseudoColor)
    {
      int i, ncolors;
      if (context->vinfo.class == PseudoColor)
	ncolors = cpc * cpc * cpc;
      else
	ncolors = context->vinfo.colormap_size / 2;

      i = 0;
      while (i == 0 && ncolors >= 4) 
        {
          OBJC_MALLOC(cells, unsigned long, ncolors);
          i = XAllocColorCells(context->display, 
                               cmap->colormap,
                               TRUE,
                               NULL,
                               0,
                               cells,
                               ncolors);
          if (i == 0) 
            {
              OBJC_FREE(cells);
              cells = NULL;
	      if (context->vinfo.class == PseudoColor)
		{
		  cpc -= 1;
		  ncolors = cpc * cpc * cpc;
		}
	      else
		ncolors -= 4;
            }
        }

      if (i == 0)
	return NO;

      context->ncolors = ncolors;
    }
  else
    context->ncolors = context->vinfo.colormap_size;
      
  context->rgbmap = cmap;
  context->attr.flags |= XRColorsPerChannel;
  context->attr.colors_per_channel = pow(context->ncolors, 0.3334);

  NSDebugLLog(@"DPS", @"Allocated rgbmap of %d colors\n", context->ncolors);
  if (context->attr.colormap_style == gray_colormap
           || context->vinfo.class == StaticGray
           || context->vinfo.class == GrayScale)
    xrAllocGrayScale(context, context->rgbmap, cells, context->ncolors);
  else if (context->vinfo.class == StaticColor
           || context->vinfo.class == PseudoColor)
    xrAllocPseudoColor(context, context->rgbmap, cells, context->ncolors);
  XFree(cells);

  /* Set our new colormap so other apps can use it */
#ifdef SET_GLOBAL_COLORMAP
  XSetRGBColormaps(context->display, 
		   RootWindow(context->display,context->screen_number),
		   context->rgbmap,
		   1,
		   _XA_GNUSTEP_RGB_MAP);
#endif
  return YES;
}


BOOL
xrGetDefaultColormap(xr_context_t* context)
{
  if (context->rgbmap)
    {
      XFree(context->rgbmap);
      context->rgbmap = NULL;
    }
  context->ncolors = 0;
  if (context->graymap)
    {
      XFree(context->graymap);
      context->graymap = NULL;
    }
  context->ngrays = 0;
  context->nextracolors = 0;

  if ((context->attr.flags & XRColormapStyle) 
           && context->attr.colormap_style == no_colormap)
    {
      context->white = WhitePixel(context->display, context->screen_number);
      context->ltgray = WhitePixel(context->display, context->screen_number);
      context->black = BlackPixel(context->display, context->screen_number);
      context->dkgray = BlackPixel(context->display, context->screen_number);
      return YES;
    }

  /* Grab the server because we might want to set a new colormap */
#ifdef SET_GLOBAL_COLORMAP
  NSDebugLLog(@"DPS", @"Grabbing server to set colormaps\n");
  XGrabServer(context->display);
#endif

  if (xrGetDefaultGraymap(context))
    xrGetDefaultRGBmap(context);

  if (context->graymap || context->rgbmap)
    {
      context->white = xrGrayToPixel(context, 1.000);
      context->ltgray = xrGrayToPixel(context, 0.6667);
      context->dkgray = xrGrayToPixel(context, 0.3334);
      context->black = xrGrayToPixel(context, 0.000);
    }

  
#ifdef SET_GLOBAL_COLORMAP
  XSetCloseDownMode(context->display, RetainTemporary);
  XUngrabServer(context->display);
  NSDebugLLog(@"DPS", @"Ungrabbed server\n");
#endif
  return YES;
}

#define MAX_EXTRA_COLORS 256

unsigned long 
xrAllocActualRGB(xr_context_t* context, float red, float green, float blue)
{
  Colormap cmap;
  XColor color;

  if (context->colors == NULL)
    OBJC_MALLOC(context->colors, XColor, MAX_EXTRA_COLORS);

  if (context->nextracolors > MAX_EXTRA_COLORS)
    return -1;

  if (context->graymap)
    cmap = context->graymap->colormap;
  else if (context->rgbmap)
    cmap = context->rgbmap->colormap;
  else
    return -1;

  if ((context->attr.flags & XRGamma) && context->attr.rgamma > 0
      && context->attr.ggamma > 0 && context->attr.bgamma > 0) 
    {
      double rg, gg, bg;
      double tmp;
      XColor color;

      rg = 1.0/context->attr.rgamma;
      gg = 1.0/context->attr.ggamma;
      bg = 1.0/context->attr.bgamma;
      color.flags = DoRed|DoGreen|DoBlue;
      color.red = (unsigned short)(65536.0*pow(tmp, rg));
      color.green = (unsigned short)(65536.0*pow(tmp, gg));
      color.blue = (unsigned short)(65536.0*pow(tmp, bg));
      color.pixel = 0;
    } 
  else 
    {
      color.flags = DoRed|DoGreen|DoBlue;
      color.red   = red * 65536.0;
      color.green = green * 65536.0;
      color.blue = blue * 65536.0;
      color.pixel = 0;
    }

  if (XAllocColor(context->display, cmap, &color))
    {
      context->colors[context->nextracolors] = color;
      return context->nextracolors++;
    }

  return -1;
}

u_long
xrExactToPixel(xr_context_t* context, float red, float green, float blue)
{
  int i;
  for (i = 0; i < context->nextracolors; i++)
    {
      if ((red == context->colors[i].red)
	  && (green == context->colors[i].green)
	  && (blue == context->colors[i].blue)
	  )
	return context->colors[i].pixel;
    }
  return 0;
}

static XColor *
allocateColorList(xr_context_t* context)
{
  XStandardColormap *map;
  int r, g, b, i;
  XColor *colors;
  
  OBJC_MALLOC(colors, XColor, context->ncolors);
  if (!colors) {
    sprintf(RErrorString, "out of memory");
    return NULL;
  }

  i = 0;
  map = context->rgbmap;
  for (r=0; r< context->rgbmap->red_max+1; r++) {
    for (g=0; g< context->rgbmap->green_max+1; g++) {
      for (b=0; b< context->rgbmap->blue_max+1; b++) {
	colors[i].red = r;
	colors[i].green = g;
	colors[i].blue = b;
	colors[i].pixel = (r * map->red_mult) + (g * map->green_mult)
		     + (b * map->blue_mult) + map->base_pixel;
	colors[i].flags = DoRed|DoGreen|DoBlue;
	i++;
      }
    }
  }
  return colors;
}


/* RContexts are used by the wraster routines, but provide essentially the
   same information as xr_context_t* */
void *
xrRContextFromContext(xr_context_t* context)
{
  RContext *rcontext;
  XGCValues gcv;

  OBJC_MALLOC(rcontext, RContext, 1);
  if (!rcontext) {
    NSLog(@"Warning (xraw): No more memory for creating RContext\n");
    return NULL;
  }
  memset(rcontext, 0, sizeof(RContext));

  rcontext->dpy = context->display;
  rcontext->screen_number = context->screen_number;
    
  OBJC_MALLOC(rcontext->attribs, RContextAttributes, 1);
  if (!rcontext->attribs) {
    OBJC_FREE(rcontext);
    NSLog(@"Warning (xraw): No more memory for creating RContext\n");
    return NULL;
  }
  rcontext->attribs->flags = 
    RC_UseSharedMemory|RC_RenderMode|RC_ColorsPerChannel;
  rcontext->attribs->render_mode = RM_DITHER;
  rcontext->attribs->colors_per_channel = context->attr.colors_per_channel;
  if (context->attr.flags & XRGamma)
    {
      rcontext->attribs->rgamma = context->attr.rgamma;
      rcontext->attribs->ggamma = context->attr.ggamma;
      rcontext->attribs->bgamma = context->attr.bgamma;
    }
  if (context->attr.flags & XRVisualId)
    {
      rcontext->attribs->flags |= RC_VisualID;
      rcontext->attribs->visualid = context->attr.visualid;
    }
  rcontext->attribs->use_shared_memory = YES;

  rcontext->cmap = context->rgbmap->colormap;
  rcontext->visual = context->vinfo.visual;
  rcontext->depth = context->vinfo.depth;
  rcontext->vclass = context->vinfo.class;
  if (rcontext->visual != DefaultVisual(rcontext->dpy, rcontext->screen_number)) {
    XSetWindowAttributes attr;
    attr.colormap = rcontext->cmap;
    attr.override_redirect = True;
    attr.border_pixel = 0;
    rcontext->drawable =
      XCreateWindow(rcontext->dpy, 
		    RootWindow(rcontext->dpy, rcontext->screen_number),
		    1, 1, 1, 1, 0, rcontext->depth,
		    CopyFromParent, rcontext->visual,
		    CWBorderPixel|CWColormap|CWOverrideRedirect, &attr);
  }

  rcontext->black = context->black;
  rcontext->white = context->white;
    
  gcv.function = GXcopy;
  gcv.graphics_exposures = False;
  rcontext->copy_gc = XCreateGC(rcontext->dpy, rcontext->drawable, GCFunction
				|GCGraphicsExposures, &gcv);

  /* Make a color list */
  if (rcontext->vclass == PseudoColor || rcontext->vclass == StaticColor) {
    rcontext->ncolors = context->ncolors;
    rcontext->colors = allocateColorList(context);
    if (!rcontext->colors) {
      OBJC_FREE(rcontext);
      return NULL;
    }
  } else if (rcontext->vclass == GrayScale || rcontext->vclass == StaticGray) {
    rcontext->ncolors = context->ncolors;
    rcontext->colors = allocateColorList(context);
    if (!rcontext->colors) {
      OBJC_FREE(rcontext);
      return NULL;
    }
  } else if (rcontext->vclass == TrueColor) {
    /* calc offsets to create a TrueColor pixel */
    rcontext->red_offset = mask_offset(rcontext->visual->red_mask);
    rcontext->green_offset = mask_offset(rcontext->visual->green_mask);
    rcontext->blue_offset = mask_offset(rcontext->visual->blue_mask);
  }
  /* disable dithering on 24 bits visuals */
  if (rcontext->depth >= 24)
    rcontext->attribs->render_mode = RM_MATCH;
    
  /* check avaiability of MIT-SHM */
#ifdef XSHM
  if (!(rcontext->attribs->flags & RC_UseSharedMemory)) {
    rcontext->attribs->flags |= RC_UseSharedMemory;
    rcontext->attribs->use_shared_memory = True;
  }
  
  if (rcontext->attribs->use_shared_memory) {
    if (!XShmQueryExtension(rcontext->dpy)) {
      rcontext->attribs->use_shared_memory = False;
    }
  } 
#endif

  return rcontext;
}

/* Internal conversion of colors to pixels values */
u_long   
xrGrayToPixel(xr_context_t* context, float gray)
{
  XStandardColormap *map;
  u_long color;

  if (context->nextracolors 
      && (color == xrExactToPixel(context, gray, gray, gray)))
    return color;
  
  if (context->ngrays)
    map = context->graymap;
  else
    map = context->rgbmap;
  if (map == NULL) 
    {
      if (gray < 0.3334/2)
	color = context->black;
      else if (gray < 0.5)
	color = context->dkgray;
      else if (gray < 0.666+0.3334/2)
	color = context->ltgray;
      else
	color = context->white;
    }
  else
    color = ((u_long)(0.5 + (gray * map->red_max)) * map->red_mult)
      + ((u_long)(0.5 + (gray*map->green_max)) * map->green_mult)
      + ((u_long)(0.5 + (gray*map->blue_max)) * map->blue_mult)
      + map->base_pixel;
  return color;
}

/* FIXME: Need to handle gamma correction */
u_long   
xrRGBToPixel(xr_context_t* context, float red, float green, float blue)
{
  XStandardColormap *map;
  u_long color;
  
  if (context->nextracolors 
      && (color == xrExactToPixel(context, red, green, blue)))
    return color;
  
  if (context->ncolors)
    map = context->rgbmap;
  else
    map = context->graymap;
  if (map == NULL) 
    {
      int gray = ((0.3*red) + (0.59*green) + (0.11*blue));
      if (gray < 0.3334/2)
	color = context->black;
      else if (gray < 0.5)
	color = context->dkgray;
      else if (gray < 0.666+0.3334/2)
	color = context->ltgray;
      else
	color = context->white;
    }
  else
    color = ((u_long)(0.5 + (red * map->red_max)) * map->red_mult)
      + ((u_long)(0.5 + (green * map->green_max)) * map->green_mult)
      + ((u_long)(0.5 + (blue * map->blue_max)) * map->blue_mult)
      + map->base_pixel;
  return color;
}

u_long   
xrHSBToPixel(xr_context_t* context, float h, float s, float v)
{
  int i;
  float f, p, q, t;
  float red, green, blue;

  h = fmod(h, 360);

  if (s == 0) 
    return xrRGBToPixel(context, v, v, v);
  i = fmod(h, 60);
  f = fmod(h, 60);
  p = v * (1.0 - s);
  q = v * (1.0 - s * f / 60);
  t = v * (1.0 - s * (60 - f) / 60);
  
  switch (i) 
    {
    case 0:
      red = v;
      green = t;
      blue = p;
      break;
    case 1:
      red = q;
      green = v;
      blue = p;
      break;
    case 2:
      red = p;
      green = v;
      blue = t;
      break;
    case 3:
      red = p;
      green = q;
      blue = v;
      break;
    case 4:
      red = t;
      green = p;
      blue = v;
      break;
    case 5:
      red = v;
      green = p;
      blue = q;
      break;
    }
    return xrRGBToPixel(context, red, green, blue);
}

/* Not implemented. FIXME */
u_long   
xrCMYKToPixel(xr_context_t* context, float c, float m, float y, float k) 
{
    return 0;
}

u_long   
xrColorToPixel(xr_context_t* context, xr_device_color_t  color)
{
  u_long pix;
  switch(color.space)
    {
    case gray_colorspace:
      pix = xrGrayToPixel(context, color.field[0]);
      break;
    case rgb_colorspace:
      pix = xrRGBToPixel(context, color.field[0], 
			 color.field[1], color.field[2]);
      break;
    case hsb_colorspace:
      pix = xrHSBToPixel(context, color.field[0], 
			 color.field[1], color.field[2]);
      break;
    case cmyk_colorspace: 
      pix = xrCMYKToPixel(context, color.field[0], color.field[1],
			  color.field[2], color.field[3]);
      break;
    default:
      break;
    }
    return pix;
}

xr_device_color_t 
xrConvertToGray(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = gray_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      new = color;
      break;
    case hsb_colorspace:
    case cmyk_colorspace: 
      color = xrConvertToRGB(color);
      /* NO BREAK */
    case rgb_colorspace:
      new.field[0] = 
	((0.3*color.field[0]) + (0.59*color.field[1]) + (0.11*color.field[2]));
      break;
    default:
      break;
    }
  return new;
}

xr_device_color_t 
xrConvertToRGB(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = rgb_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      new.field[0] = color.field[0];
      new.field[1] = color.field[0];
      new.field[2] = color.field[0];
      break;
    case rgb_colorspace:
      new = color;
      break;
    case hsb_colorspace: 
    case cmyk_colorspace: 
      break;
    default:
      break;
    }
  return new;
}

xr_device_color_t 
xrConvertToHSB(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = hsb_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      break;
    case rgb_colorspace:
      break;
    case hsb_colorspace: 
      new = color;
      break;
    case cmyk_colorspace: 
      break;
    default:
      break;
    }
  return new;
}

xr_device_color_t 
xrConvertToCMYK(xr_device_color_t color)
{
  xr_device_color_t new;

  new.space = gray_colorspace;
  switch(color.space)
    {
    case gray_colorspace:
      break;
    case rgb_colorspace:
      break;
    case hsb_colorspace:
      break;
    case cmyk_colorspace: 
      new = color;
      break;
    default:
      break;
    }
  return new;
}

/* Internal conversion of image data to pixel values */
/* FIXME: Ignores alpha */
u_char *
xrMapGrays(xr_context_t* context, u_char **src, int colorspace, int bps, int spp, 
	int ncolors, int planar, u_char *dest)
{
  int i;
  XStandardColormap *map;
  u_int  max;
  u_long color;
  u_char *gray;
  
  if (context->ngrays)
    map = context->graymap;
  else
    map = context->rgbmap;
  gray = src[0];
  if (map == NULL)
    return 0;
  max = (1 << bps);
  for (i=0; i < ncolors; i++) 
    {
      /* Just skip the Alpha if there is one */
      if (planar)
	color = (gray[i*bps/8] << ((i * bps) % 8)) % max;
      else
	color = (gray[i*spp*bps/8] << ((i * bps) % 8)) % max;
      color = ((color * map->red_max / max) * map->red_mult
	       + (color * map->green_max / max) * map->green_mult
	       + (color * map->blue_max / max) * map->blue_mult
	       + map->base_pixel) & 0xFFFFFFFF;
      /* FIXME: Need to pack bits for displays with depth < 8 */
      dest[i] = color;
    }
  return dest;
}

/* FIXME: Only handles RGB images (no CMYK) */
u_char   *
xrMapColors(xr_context_t* context, u_char **src, int colorspace, int bps, int spp, 
		int ncolors, int planar, u_char *dest)

{
    int i;
    XStandardColormap *map;
    u_int  max;
    u_long color;
    u_char *red, *green, *blue;
    u_char r, g, b;

    if (spp <= 2)
	return xrMapGrays(context, src, colorspace, bps, 
		spp, ncolors, planar, dest);

    map = context->rgbmap;
    if (map == NULL)
	return 0;
    max = (1 << bps);
    red = src[0];
    if (planar) {
	green = src[1];
	blue = src[2];
    }
    for (i=0; i < ncolors; i++) {
	u_int shift;
	shift = i * bps / 8;
	if (planar) {
	    r = (red[shift]   << ((i * bps) % 8)) % max;
	    g = (green[shift] << ((i * bps) % 8)) % max;
	    b = (blue[shift]  << ((i * bps) % 8)) % max;
	    /* Translate values for gray-scale display */
	    if ( map->green_max == 0 && map->blue_max == 0)
		r = (u_long)((0.3*r) + (0.59*g) + (0.11*b));
	    color = ((r * map->red_max / max) * map->red_mult
		+ (g * map->green_max / max)*map->green_mult
		+ (b * map->blue_max / max)*map->blue_mult
		+ map->base_pixel) & 0xFFFFFFFF;
	} else {
	    r = (red[i*spp*bps/8]     << ((i * bps) % 8)) % max;
	    g = (red[(i*spp+1)*bps/8] << (((i+1) * bps) % 8)) % max;
	    b = (red[(i*spp+2)*bps/8] << (((i+2) * bps) % 8)) % max;
	    /* Translate values for gray-scale display */
	    if ( map->green_max == 0 && map->blue_max == 0)
		r = (u_long)((0.3*r) + (0.59*g) + (0.11*b));
	    color = ((r * map->red_max / max) * map->red_mult
		+ (g * map->green_max / max) * map->green_mult
		+ (b * map->blue_max / max) * map->blue_mult
		+ map->base_pixel) & 0xFFFFFFFF;
	}
	/* FIXME: Need to pack bits for displays with depth < 8 */
	dest[i] = color;
    }

    return dest;
}

