//  UFontFreeType.cpp version 1.1
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998  Gaspar Sinai
// 
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
#include 	"UFont.h"
#include 	"UFontFreeType.h"
#include 	"UCache.h"
#include 	"UCommon.h"
#include 	<string.h>
#include 	<strings.h>
#include 	<X11/Xlib.h>
#include 	<X11/Xutil.h>

static          UCache<UFontFreeTypeCache*>* cache=0;
#define 	INDIRECT_INVALID 65535

UFreeTypeGlyph::UFreeTypeGlyph (TT_Face* face, TT_Instance instance,
	int	indexIn, int ascentIn, int descentIn, int flags)
{
	status = ERROR;
	xPixmap = 0;
#ifdef FREETYPE
	if (face==0)
	{
		status = ERROR;
	}
	else if (TT_New_Glyph (*face, &glyph))
	{
		status=ERROR;
	}
	else if (TT_Load_Glyph (instance, glyph, indexIn, flags))
	{
		status=ERROR;
	}
	else if (TT_Get_Glyph_Metrics (glyph, &metrics))
	{
		status=ERROR;
	}
	else 
	{
		uAscent = ascentIn;
		uDescent = descentIn;
		status = FT_OK;
	}
#endif
}

UFreeTypeGlyph::~UFreeTypeGlyph()
{
	if (status & FT_BIT_OK)
	{
		//hmm. it seems it needs it...
		//delete ttBitmap.bitmap;
	}
	if (status & FT_PIXMAP_OK) 
	{
		XFreePixmap (xDisplay, xPixmap);
	}
#ifdef FREETYPE
	if (status != ERROR) TT_Done_Glyph (glyph);
#endif
}

//
// Load a FreeType font instance
//
UFontFreeTypeCache::UFontFreeTypeCache (UFreeTypeFace* freeTypeFaceIn,
		int pixelIn, int resolutionIn) 
{
	resolution = resolutionIn;
	uPointSize = 72 * pixelIn/resolutionIn;
	freeTypeFace = freeTypeFaceIn;
	face = freeTypeFace->getFace();
	inited=0;
	if (face ==0)
	{
		inited=-1;
	}
	glyphs = 0;
#if FREETYPE
	if (inited==-1)
	{
	}
	else if (TT_New_Instance (*face, &instance))
	{
		inited=-1;
		cerr << "error: can not get font instance of " 
			<< freeTypeFace->getName () << ".\n";
	}
	else if (TT_Set_Instance_Resolutions (instance, resolution, resolution))
	{
		inited=-1;
		cerr << "error: can not get font resolution " << resolution
			<< " of " << freeTypeFace->getName () << ".\n";
	} else if (TT_Set_Instance_PointSize (instance, uPointSize))
	{
		inited=-1;
		cerr << "error: can not get font pointsize " << uPointSize
			<< " of " << freeTypeFace->getName () << ".\n";
	}
	else
	{
		inited=1;
	}
	if (inited!=-1)
	{
		TT_Get_Instance_Metrics (instance, &metrics);
	}
#else
	inited=-1;
#endif
	memset (indirect, 0, 256 * sizeof (UCS2*));
	glyphs = 0;
	glyphArraySize = 0;
	glyphSize = 0;
}

UFontFreeTypeCache::~UFontFreeTypeCache()
{
	int 	i;
	for (i=0; i<256; i++)
	{
		if (indirect[i]!=0)
		{
			delete indirect[i];
		}
	}
	for (i=0; i<glyphSize; i++)
	{
			delete glyphs[i];
	}
#ifdef FREETYPE
	if (inited!=-1) TT_Done_Instance (instance);
#endif
}

int
UFontFreeTypeCache::getDirection()
{
#ifdef FREETYPE
	TT_Face_Properties	*prop;
	prop = freeTypeFace->getProperty();
	if (prop==0) return 1;
	if (prop->header->Font_Direction!=0) return 0;
#endif
	return 1;
}

int
UFontFreeTypeCache::getWidth (const XChar2b& char2B)
{
	UCS2		ucs2;
	UFreeTypeGlyph*	uGlyph;

	ucs2 = (char2B.byte1 << 8) | char2B.byte2;
	if (ucs2 < ' ')
	{
		return  0;
	}
	if (inited==-1) return 1;
#if FREETYPE
	if (indirect[char2B.byte1] == 0 
		|| indirect[char2B.byte1][char2B.byte2]
			==INDIRECT_INVALID) return 0;
	uGlyph = glyphs[indirect[char2B.byte1][char2B.byte2]];
	if (uGlyph==0 || uGlyph->status==UFreeTypeGlyph::ERROR)
	{
		// return metrics.x_ppem;
		return 0;
	}
	return uGlyph->metrics.advance/64 +1;
#else
	return 10;
#endif
}

int
UFontFreeTypeCache::getAscent ()
{
#ifdef FREETYPE
	TT_Face_Properties	*prop;

	prop = freeTypeFace->getProperty();
	if (prop==0) return 1;

	return freeTypeFace->getProperty()->horizontal->Ascender 
		* metrics.y_ppem /
		freeTypeFace->getProperty()->header->Units_Per_EM + 1;
	// return metrics.y_ppem;
#else
	return 1;
#endif
}

int
UFontFreeTypeCache::getDescent ()
{
#ifdef FREETYPE
	TT_Face_Properties	*prop;
	prop = freeTypeFace->getProperty();
	if (prop==0) return 1;

	return (- prop->horizontal->Descender 
//		+ prop->horizontal->Line_Gap
		) * metrics.y_ppem /
		prop->header->Units_Per_EM + 1;
#else
	return 1;
#endif
}

int
UFontFreeTypeCache::getWidth()
{
#ifdef FREETYPE
	TT_Face_Properties	*prop;
	prop = freeTypeFace->getProperty();
	if (prop==0) return 1;

	return prop->horizontal->advance_Width_Max 
		* metrics.y_ppem /
		prop->header->Units_Per_EM + 1;
#else
	return 1;
#endif
}


//
// You have to free up the pixmap return advance
//
int
UFreeTypeGlyph::getXPixmap (Display *displayIn, Pixmap* pixmapOut)
{
	int 		xScreen;
	Visual*   	xVisual;
	Window		xRootWindow;
	XImage*		xImage;
	char*		xBitmap;
	GC		xGC;
	XGCValues	xGCValues;

	if (status == ERROR) return 0;
	if ((status & FT_BIT_OK) ==0)
	{
#ifdef FREETYPE
		ttBitmap.rows = uAscent + uDescent;
		ttBitmap.width = metrics.advance/64 +1;
		ttBitmap.cols =  (ttBitmap.width + 7) / 8;
		ttBitmap.flow  = TT_Flow_Down;
		ttBitmap.size = ttBitmap.rows * ttBitmap.cols;
		xBitmap = new char[ttBitmap.size];
		CHECKNULL (xBitmap);
		memset (xBitmap, 0, ttBitmap.size);
		ttBitmap.bitmap = xBitmap;
		
		
		if (TT_Get_Glyph_Bitmap (glyph, &ttBitmap, 0, 
			(uDescent)  * 64))
		{
			status=ERROR;
			return 0;
		}
		status =  (UStatus) (status | FT_BIT_OK);
#else
		status=ERROR;
		return 0;
#endif
	}
	if (status & FT_PIXMAP_OK)
	{
		*pixmapOut = xPixmap;
#ifdef FREETYPE
		return ttBitmap.width;
#else
		return 0;
#endif
	}
	xDisplay = displayIn;
#ifdef FREETYPE
	xScreen = DefaultScreen  (displayIn);
	xVisual = DefaultVisual (displayIn, xScreen);
	xRootWindow = XRootWindow (displayIn, xScreen);
	xPixmap = XCreatePixmap (displayIn, xRootWindow, 
		ttBitmap.width, ttBitmap.rows, 1);
	xGCValues.foreground = 1;
	xGCValues.background = 0;
	xGC = XCreateGC (displayIn, xPixmap, GCForeground | GCBackground, 
		&xGCValues);

	xImage = XCreateImage (displayIn, xVisual,
			1, XYBitmap, 0, 
			(char*) ttBitmap.bitmap,
			 ttBitmap.width, ttBitmap.rows, 8, 0);
	xImage->byte_order = MSBFirst;
	xImage->bitmap_bit_order = MSBFirst;
	XPutImage (displayIn, xPixmap, xGC, 
		xImage, 0,  0, 0, 0,
		ttBitmap.width, ttBitmap.rows);
	XDestroyImage (xImage);
	XFreeGC (displayIn, xGC);
	status = (UStatus) (status | FT_PIXMAP_OK);
	*pixmapOut = xPixmap;
	return ttBitmap.width;
#else
	return 0;
#endif
}

void
UFontFreeTypeCache::drawImageString (Display* dsp, Window windowIn, GC gcIn, 
    int xIn, int yIn, int ascentIn, const XChar2b* char2B, int countIn,
    UPixmapCache *pmc)
{

	drawStringBase (dsp, windowIn, gcIn, xIn, yIn, ascentIn, char2B,
		countIn, pmc);
}

void
UFontFreeTypeCache::drawString (Display* dsp, Window windowIn, GC gcIn, 
    int xIn, int yIn, int ascentIn, const XChar2b* char2B, int countIn,
		UPixmapCache *pmc)
{
	drawStringBase (dsp, windowIn, gcIn, xIn, yIn, ascentIn, char2B,
		countIn, pmc);
}

void
UFontFreeTypeCache::drawStringBase (Display* dsp, Window windowIn, GC gcIn, 
    int xIn, int yIn, int ascentIn, const XChar2b* char2B, int countIn,
		UPixmapCache *pmc)
{
	int		i;
	Pixmap		uPixmap;
	Pixmap		xPixmap;
	UCS2		ucs2;
	int     	uX;
	int		uAdvance;
	UFreeTypeGlyph*	uGlyph;

	
	uAdvance = xIn;
	for (i=0; i<countIn; i++)
	{
		ucs2 = (char2B[i].byte1 << 8) | char2B[i].byte2;
		if (ucs2 < ' ')
		{
#ifdef FREETYPE
			uAdvance += metrics.x_ppem;
#else
			uAdvance += 10;
#endif
			continue;
		}
		if (indirect[char2B[i].byte1] == 0 
			|| indirect[char2B[i].byte1][char2B[i].byte2]
				==INDIRECT_INVALID) continue;
		uGlyph = glyphs[indirect[char2B[i].byte1][char2B[i].byte2]];
		if (uGlyph==0 || uGlyph->status==UFreeTypeGlyph::ERROR)
		{
#ifdef FREETYPE
			uAdvance += metrics.x_ppem;
#else
			uAdvance += 10;
#endif
			continue;
		}
		uX = uGlyph->getXPixmap (dsp, &uPixmap);
		if (uX==0)
		{
#ifdef FREETYPE
			uX = - uGlyph->metrics.advance/64 +1;
#else
			uX = -10;
#endif
		}
		if (uX < 0)
		{
			uAdvance -= uX; continue;
		}

		if (pmc==0)
		{
			// Draw here
			XCopyPlane (dsp, uPixmap, windowIn, gcIn, 
				0, 0, uX, uGlyph->uAscent + uGlyph->uDescent, 
				uAdvance, yIn, 1);
		}
		else
		{
			xPixmap = pmc->get (uPixmap, 
				indirect[char2B[i].byte1][char2B[i].byte2],
				gcIn, uX, uGlyph->uAscent + uGlyph->uDescent);
			XCopyArea (dsp, xPixmap, windowIn, gcIn,
				0, 0, uX, uGlyph->uAscent + uGlyph->uDescent, 
				uAdvance, yIn);
		}
		uAdvance += uX;
	}
}

UFontFreeType::UFontFreeType (UFontFreeType& freeTypeIn) :  UFont (freeTypeIn.freeType)
{
	fontChanged = 1;
	display = freeTypeIn.display;
	screen = freeTypeIn.screen;
	resolution = (int) ((double) DisplayWidth (display, screen) * 25.42
		/ (double) DisplayWidthMM (display, screen));
	cacheName = 0;
	pixel = freeTypeIn.pixel;
	slant = freeTypeIn.slant;
	spacing = freeTypeIn.spacing;
	weight = freeTypeIn.weight;
	cacheName = 0;
	font = 0;
}

int
UFontFreeType::isA (UFontType ftype)
{
	if (ftype==UFREE) return 1;
	return (UFont::isA (ftype));
}

UFontFreeType::UFontFreeType (UFreeType* freeTypeIn,
		Display *displayIn, int screenIn) 
	: UFont (freeTypeIn)
{
	fontChanged = 1;
	// We suppose both resolution is the same...
	display = displayIn;
	screen = screenIn;
	resolution = (int) ((double) DisplayWidth (displayIn, screenIn) * 25.42
		/ (double) DisplayWidthMM (displayIn, screenIn));
	cacheName = 0;
	pixel = 0;
	font = 0;
}

UFontFreeType::UFontFreeType (UFreeType* freeTypeIn, const int resolutionIn)
	: UFont (freeTypeIn)
{
	fontChanged = 1;
	resolution = resolutionIn;
	cacheName = 0;
	pixel = 0;
	font = 0;
}

UFontFreeType::~UFontFreeType()
{
	if (font!=0) cache->unuseItem (cacheName);
	if (cacheName) delete cacheName;
}

const UFontMapStruct*
UFontFreeType::getFontMapStruct (const UCS2 input)
{
	return 0;
}
void
UFontFreeType::setPixel (const int pixelIn)
{
	if (pixel != pixelIn) clear();
	pixel = pixelIn;
}

void
UFontFreeType::setWeight (const UWeight weightIn)
{
	if (freeType->freeTypeWeight(weight) 
		!= freeType->freeTypeWeight(weightIn)) clear ();
	weight = weightIn;
}
void
UFontFreeType::setSlant (const USlant slantIn)
{
	if (freeType->freeTypeSlant (slant) 
		!= freeType->freeTypeSlant (slantIn)) clear();
	slant = slantIn;
}
void
UFontFreeType::setWeight (const char* weightIn)
{
	clear ();
	UFont::setWeight (weightIn);
}
void
UFontFreeType::setSlant (const char* slantIn)
{
	clear();
	UFont::setSlant (slantIn);
}

void
UFontFreeType::clear ()
{
	if (fontChanged == 1) return;
	if (font!=0)
	{
		if (cacheName!=0)
		{
			cache->unuseItem (cacheName);
			delete cacheName;
		}
		cacheName=0;
		font=0;
	}
	fontChanged = 1;
}

UFontCache*
UFontFreeType::getFont (const UCS2* text, const int from, int* to)
{
	int 			i;
	UFreeTypeFace*		freeTypeFace;

	if (cache == 0)
	{
		cache = new UCache<UFontFreeTypeCache*> (0);
		CHECKNULL (cache);
	}
	if (fontChanged)
	{
		if (freeType==0) return 0;
		fontAscent = fontDescent = fontWidth = fontHeight = 0;
		minFontWidth = 0;

		if (cacheName!=0)
		{
			cache->unuseItem (cacheName);
			delete cacheName;
		}
		freeTypeFace = freeType->getFreeTypeFace (
			freeType->freeTypeWeight(weight), 
			freeType->freeTypeSlant (slant));
		cacheName = getCacheName (freeType->getName(), 
			resolution, pixel, 
			(int) freeType->freeTypeWeight(weight),
			(int) freeType->freeTypeWeight(slant));
		font = cache->getItem (cacheName);
		if (font != 0 ) 
		{
			cache->useItem (cacheName);
		}
		else
		{
			font = new UFontFreeTypeCache (freeTypeFace, pixel, 
					resolution);
			CHECKNULL (font);
			(void) cache->addItem (cacheName, font);
			if (font==0 || font->inited==-1)
			{
				delete cacheName;
				fontChanged = 0;
				cacheName = 0;
				freeType = 0;
				font=0;
				return 0;
			}
		}
		if (font->inited==-1)
		{
			delete cacheName;
			fontChanged = 0;
			cacheName = 0;
			freeType = 0;
			font=0;
			return 0;
		}
		fontWidth = font->getWidth();	
		minFontWidth = fontWidth;
		fontAscent = font->getAscent();
		fontDescent = font->getDescent();

		fontHeight = fontAscent + fontDescent;
		fontChanged = 0;
	}
	if (font==0) return 0;
	// the guy just wanted to set the font bounding box.
	if (text == 0 || text[0]==0) { return 0; }

	// load the missing glyphs and return the cache.
	for (i=0; text[i] != 0; i++)
	{
		if (text[i] < ' ') continue;
		if(font->loadGlyph (text[i]))
		{
			if (to!=0) *to=i;
			return ((i>0) ? font : 0);
		}
	}
	if (to!=0) *to=i;
	return font;
}

//
// Return non-zero if it can not load the glyph
//
int
UFontFreeTypeCache::loadGlyph (const UCS2 ucs2)
{
	UFreeTypeGlyph**	newGlyphs;
	int			newSize;
	int			glyphIndex;
	int			i;

	if (ucs2==0) return 0;
	if (indirect[ucs2>>8]!=0 
		&& indirect[ucs2>>8][ucs2&0xff] != INDIRECT_INVALID)
	{
		if (glyphs[indirect[ucs2>>8][ucs2&0xff]]->status!=UFreeTypeGlyph::ERROR)return 0;
		return -1;
	}
	// Have to load the new glyph.
	if (glyphSize>=glyphArraySize)
	{
		newSize = (glyphArraySize==0) ? 1 : glyphArraySize<<1;
		newGlyphs = new UFreeTypeGlyph*[newSize];
		CHECKNULL (newGlyphs);
		if (glyphs!=0)
		{
			memcpy (newGlyphs, glyphs, glyphSize * sizeof (UFreeTypeGlyph*));
			delete glyphs;
		}
		glyphs = newGlyphs;
		glyphArraySize = newSize;
	}
#ifdef FREETYPE
	glyphIndex = freeTypeFace->getCharIndex (ucs2);
#else
	glyphIndex = -1;
#endif
	if (glyphIndex<0 
#ifdef FREETYPE
		|| freeTypeFace->getProperty() == 0 
		|| freeTypeFace->getProperty()->num_Glyphs <= glyphIndex
#endif
	)	 
	{
		return -1;
	}
	glyphs[glyphSize] = new UFreeTypeGlyph (face, instance, 
		(int) glyphIndex, getAscent(), getDescent());
	CHECKNULL (glyphs[glyphSize]);
	if (indirect[ucs2>>8] == 0)
	{
		indirect[ucs2>>8] = new UCS2[256];
		CHECKNULL (indirect[ucs2>>8]);
		for (i=0; i<256; i++) indirect[ucs2>>8][i] = INDIRECT_INVALID;
	}
	indirect[ucs2>>8][ucs2&0xff] = glyphSize;
	if (glyphs[glyphSize]->status!=UFreeTypeGlyph::ERROR)
	{
		glyphSize++;
		return 0;
	}
	glyphSize++;
	return -1;
}

char *
UFontFreeType::getCacheName (const char *nameIn, const int pixelIn, 
	const int resolutionIn, const int weightIn, const int slantIn)
{
	char*	cname;
	/* The fontname is this */
	cname = new char[strlen (nameIn) + 256 ];
	CHECKNULL (cname);
	sprintf (cname, "%s %d %d %d %d", nameIn, pixelIn, 
		resolutionIn, weightIn, slantIn); 
	return cname;
}

void
UDeinitFontFreeType ()
{
	if (cache != 0)
	{
		delete cache;
		cache = 0;
	}
}

void
UTuneFontFreeType (int cacheSize, int batchSize)
{
	if (cache)
	{
		cache->tune (cacheSize, batchSize);
	}
}
