/*
 * qstat 2.1a
 * by Steve Jankowski
 * steve@activesw.com
 * http://www.activesw.com/people/steve/qstat.html
 *
 * Thanks to Per Hammer for the OS/2 patches (per@mindbend.demon.co.uk)
 * Thanks to John Ross Hunt for the OpenVMS Alpha patches (bigboote@ais.net)
 * Thanks to Scott MacFiggen for the quicksort code (smf@activesw.com)
 *
 * Inspired by QuakePing by Len Norton
 *
 * Copyright 1996,1997,1998 by Steve Jankowski
 *
 *   Permission granted to use this software for any purpose you desire
 *   provided that existing copyright notices are retained verbatim in all
 *   copies and derived works.
 */

#define VERSION "2.1a"
char *qstat_version= VERSION;

/* OS/2 defines */
#ifdef __OS2__
#define BSD_SELECT
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <string.h>
#include <ctype.h>

#ifdef unix
#define _ISUNIX
#endif
#ifdef __unix
#define _ISUNIX
#endif
#ifdef VMS
#define _ISUNIX
#endif
#ifdef _AIX
#define _ISUNIX
#endif


#ifdef _ISUNIX
#include <unistd.h>
#include <sys/socket.h>
#ifndef VMS
#include <sys/param.h>
#endif
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>

#ifndef F_SETFL
#include <fcntl.h>
#endif

#ifdef __hpux
extern int h_errno;
#endif

#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#ifndef INADDR_NONE
#define INADDR_NONE ~0
#endif
#define sockerr()	errno
#endif /* _ISUNIX */

#ifdef _WIN32
/* #define FD_SETSIZE 100 */
#include <winsock.h>
#include <sys/timeb.h>
#define close(a) closesocket(a)
int gettimeofday(struct timeval *now, void *blah)
{
    struct timeb timeb;
    ftime( &timeb);
    now->tv_sec= timeb.time;
    now->tv_usec= (unsigned int)timeb.millitm * 1000;
}
#define sockerr()	WSAGetLastError()
#define strcasecmp      stricmp
#define strncasecmp     strnicmp
#ifndef EADDRINUSE
#define EADDRINUSE	WSAEADDRINUSE
#endif
#endif /* _WIN32 */

#ifdef __OS2__
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <utils.h>
 
#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#define close(a)        soclose(a)
#endif /* __OS2__ */

#include "qstat.h"

extern float HuffFreq[];
void BuildTree(float *freq);
void HuffEncode(unsigned char *in,unsigned char *out,int inlen,int *outlen);
void HuffDecode(unsigned char *in,unsigned char *out,int inlen,int *outlen);

int sendenc(int s, const void *msg, int len, unsigned int flags);
int recvdec(int s, void *buf, int len, unsigned int flags);

#ifndef FD_SETSIZE
#define FD_SETSIZE 64
#endif

/* Values set by command-line arguments
 */

int hostname_lookup= 0;		/* set if -H was specified */
int new_style= 1;		/* unset if -old was specified */
int n_retries= DEFAULT_RETRIES;
int retry_interval= DEFAULT_RETRY_INTERVAL;
int save_retry_interval;
int get_player_info= 0;
int get_server_rules= 0;
int up_servers_only= 0;
int no_full_servers= 0;
int no_empty_servers= 0;
int no_header_display= 0;
int raw_display= 0;
char *raw_delimiter= "\t";
int player_address= 0;
int hex_player_names= 0;
int max_simultaneous= MAXFD_DEFAULT;
int raw_arg= 0;
int progress= 0;
int num_servers_total= 0;
int num_players_total= 0;
int num_servers_returned= 0;
int num_servers_timed_out= 0;
int num_servers_down= 0;
int default_server_type= Q_SERVER;
int default_server_port= DEFAULT_PORT;
int sort= 0;
int first_sort_key= SORT_PING;
int second_sort_key= SORT_GAME;
int show_errors= 0;

#define DEFAULT_COLOR_NAMES_RAW		0
#define DEFAULT_COLOR_NAMES_DISPLAY	1
int color_names= -1;

#define SECONDS 0
#define CLOCK_TIME 1
#define STOPWATCH_TIME 2
#define DEFAULT_TIME_FMT_RAW		SECONDS
#define DEFAULT_TIME_FMT_DISPLAY	CLOCK_TIME
int time_format= -1;

struct qserver *servers= NULL;
struct qserver **connmap= NULL;
int connected= 0;
time_t run_timeout= 0;
time_t start_time;
int waiting_for_masters;

char *DOWN= "DOWN";
char *SYSERROR= "SYSERROR";
char *TIMEOUT= "TIMEOUT";
char *MASTER= "MASTER";
char *SERVERERROR= "ERROR";
char *HOSTNOTFOUND= "HOSTNOTFOUND";

int display_prefix= 0;
char *current_filename;
int current_fileline;


/* MODIFY HERE
 * Change these functions to display however you want
 */
void
display_server(
    struct qserver *server
)
{
    char name[100], *prefix;

    if ( raw_display)  {
	raw_display_server( server);
	return;
    }
    if ( have_server_template())  {
	template_display_server( server);
	return;
    }

    if ( server->type == Q_SERVER)
	prefix= "QS  ";
    else if ( server->type == H2_SERVER)
	prefix= "H2S ";
    else if ( server->type == HW_SERVER)
	prefix= "HWS ";
    else if ( server->type == QW_SERVER)
	prefix= "QW  ";
    else if ( server->type == QW_MASTER)
	prefix= "QWM ";
    else if ( server->type == Q2_SERVER)
	prefix= "Q2  ";
    else if ( server->type == Q2_MASTER)
	prefix= "Q2M ";
    else if ( server->type == UN_SERVER)
	prefix= "UNS ";

    if ( server->server_name == DOWN)  {
	if ( ! up_servers_only)
	    printf( "%s%-16s %10s\n", (display_prefix)?prefix:"",
		(hostname_lookup) ? server->host_name : server->arg, DOWN);
	return;
    }
    if ( server->server_name == TIMEOUT)  {
	if ( server->flags & FLAG_BROADCAST && server->n_servers)
	    printf( "%s%-16s %d servers\n", (display_prefix)?prefix:"",
		server->arg, server->n_servers);
	else if ( ! up_servers_only)
	    printf( "%s%-16s no response\n", (display_prefix)?prefix:"",
		(hostname_lookup) ? server->host_name : server->arg);
	return;
    }

    if ( server->type & MASTER_SERVER)  {
	display_qwmaster(server);
	return;
    }

    if ( no_full_servers && server->num_players >= server->max_players)
	return;

    if ( no_empty_servers && server->num_players == 0)
	return;

    if ( server->error != NULL)  {
	printf( "%s%-22s ERROR <%s>\n",
	    (display_prefix)?prefix:"",
	    (hostname_lookup) ? server->host_name : server->arg,
	    server->error);
	return;
    }

    if ( new_style)  {
	char *game= get_qw_game( server);
	printf( "%s%-22s %2d/%2d %8s %6d / %1d  %*s %s\n",
	    (display_prefix)?prefix:"",
	    (hostname_lookup) ? server->host_name : server->arg,
	    server->num_players, server->max_players,
	    (server->map_name) ? server->map_name : "?",
	    server->ping_total/server->n_requests,
	    server->n_retries,
	    (server->type == QW_SERVER || server->type == Q2_SERVER) ? 9 : 0, game,
	    (server->server_name) ? server->server_name : "?");
	if ( get_server_rules)
	    display_server_rules( server);
	if ( get_player_info)
	    display_player_info( server);
    }
    else  {
	sprintf( name, "\"%s\"", server->server_name);
	printf( "%-16s %10s map %s at %22s %d/%d players %d ms\n", 
	    (hostname_lookup) ? server->host_name : server->arg,
	    name, server->map_name,
	    server->address, server->num_players, server->max_players,
	    server->ping_total/server->n_requests);
    }
}

void
display_qwmaster( struct qserver *server)
{
    char *prefix;
    if ( server->type == QW_MASTER)
	prefix= "QWM";
    else
	prefix= "Q2M";

    if ( server->error != NULL)
	printf( "%s %-17s ERROR <%s>\n", prefix,
		(hostname_lookup) ? server->host_name : server->arg,
		server->error);
    else
	printf( "%s %-17s %d servers %6d / %1d\n", prefix,
	    (hostname_lookup) ? server->host_name : server->arg,
	    server->n_servers,
	    server->ping_total/server->n_requests,
	    server->n_retries);
}

void
display_header()
{
    if ( ! no_header_display)
	printf( "%-16s %8s %8s %15s    %s\n", "ADDRESS", "PLAYERS", "MAP",
		"RESPONSE TIME", "NAME");
}

void
display_server_rules( struct qserver *server)
{
    struct rule *rule;
    int printed= 0;
    rule= server->rules;
    for ( ; rule != NULL; rule= rule->next)  {
	if ( (server->type != Q_SERVER && server->type != H2_SERVER) ||
		! is_default_rule( rule))  {
	    printf( "%c%s=%s", (printed)?',':'\t', rule->name, rule->value);
	    printed++;
	}
    }
    if ( printed)
	puts("");
}

void
display_player_info( struct qserver *server)
{
    if ( server->type == Q_SERVER || server->type == H2_SERVER)
	display_q_player_info(server);
    else if ( server->type == QW_SERVER || server->type == HW_SERVER)
	display_qw_player_info(server);
    else if ( server->type == Q2_SERVER)
	display_q2_player_info(server);
    else if ( server->type == UN_SERVER)
	display_unreal_player_info(server);
}

void
display_q_player_info( struct qserver *server)
{
    char fmt[128];
    struct player *player;

    strcpy( fmt, "\t#%-2d %3d frags %9s ");

    if ( color_names)
	strcat( fmt, "%9s:%-9s ");
    else
	strcat( fmt, "%2d:%-2d ");
    if ( player_address)
	strcat( fmt, "%22s ");
    else
	strcat( fmt, "%s");
    strcat( fmt, "%s\n");

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		player->number,
		player->frags,
		play_time(player->connect_time,1),
		quake_color(player->shirt_color),
		quake_color(player->pants_color),
		(player_address)?player->address:"",
		escape( (unsigned char *) player->name));
    }
}

void
display_qw_player_info( struct qserver *server)
{
    char fmt[128];
    struct player *player;

    strcpy( fmt, "\t#%-6d %3d frags %6s@%-5s %8s");

    if ( color_names)
	strcat( fmt, "%9s:%-9s ");
    else
	strcat( fmt, "%2d:%-2d ");
    strcat( fmt, "%s\n");

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		player->number,
		player->frags,
		play_time(player->connect_time,0),
		ping_time(player->ping),
		player->skin ? player->skin : "",
		quake_color(player->shirt_color),
		quake_color(player->pants_color),
		escape( (unsigned char *) player->name));
    }
}

void
display_q2_player_info( struct qserver *server)
{
    struct player *player;

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( "\t%3d frags %8s  %s\n", 
		player->frags,
		ping_time(player->ping),
		escape( (unsigned char *) player->name));
    }
}

void
display_unreal_player_info( struct qserver *server)
{
    struct player *player;

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( "\t%3d frags team#%-2d %10s  %s\n", 
		player->frags,
		player->team,
		player->skin,
		escape( (unsigned char *) player->name));
    }
}

char *
get_qw_game( struct qserver *server)
{
    struct rule *rule;
    rule= server->rules;
    if ( server->type == Q2_SERVER)  {
	for ( ; rule != NULL; rule= rule->next)
	    if ( strcmp( rule->name, "gamedir") == 0)
		return rule->value;
    }
    else  {
	for ( ; rule != NULL; rule= rule->next)
	    if ( strcmp( rule->name, "*gamedir") == 0 ||
			strcmp( rule->name, "game") == 0)
		return rule->value;
    }
    return "";
}

/* Raw output for web master types
 */

#define RD raw_delimiter

void
raw_display_server( struct qserver *server)
{
    char *prefix;
    if ( server->type == Q_SERVER)
	prefix= "QS";
    else if ( server->type == H2_SERVER)
	prefix= "H2S";
    else if ( server->type == HW_SERVER)
	prefix= "HWS";
    else if ( server->type == QW_SERVER)
	prefix= "QW";
    else if ( server->type == QW_MASTER)
	prefix= "QWM";
    else if ( server->type == Q2_SERVER)
	prefix= "Q2";
    else if ( server->type == Q2_MASTER)
	prefix= "Q2M";
    else if ( server->type == UN_SERVER)
	prefix= "UNS";

    if ( server->server_name == DOWN)  {
	if ( ! up_servers_only)
	    printf( "%s" "%.*s%.*s" "%s%s" "%s%s\n\n",
		prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, (hostname_lookup)?server->host_name:server->arg,
		RD, DOWN);
	return;
    }
    if ( server->server_name == TIMEOUT)  {
	if ( server->flags & FLAG_BROADCAST && server->n_servers)
	    printf( "%s" "%.*s%.*s" "%s%s" "%s%d\n", prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, server->arg,
		RD, server->n_servers);
	else if ( ! up_servers_only)
	    printf( "%s" "%.*s%.*s" "%s%s" "%s%s\n\n",
		prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, (hostname_lookup)?server->host_name:server->arg,
		RD, TIMEOUT);
	return;
    }

    if ( server->error != NULL)  {
        printf( "%s" "%.*s%.*s" "%s%s" "%s%s" "%s%s",
		prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, (hostname_lookup) ? server->host_name : server->arg,
		RD, "ERROR",
		RD, server->error);
    }
    else if ( server->type == Q_SERVER || server->type == H2_SERVER)  {
        printf( "%s" "%.*s%.*s" "%s%s" "%s%s" "%s%s" "%s%d" "%s%s" "%s%d" "%s%d" "%s%d" "%s%d",
		prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, (hostname_lookup) ? server->host_name : server->arg,
		RD, escape((unsigned char *) server->server_name),
		RD, server->address,
		RD, server->protocol_version,
		RD, server->map_name,
		RD, server->max_players,
		RD, server->num_players,
		RD, server->ping_total/server->n_requests,
		RD, server->n_retries
        );
    }
    else if ( server->type == QW_SERVER || server->type == Q2_SERVER ||
		server->type == HW_SERVER)  {
        printf( "%s" "%.*s%.*s" "%s%s" "%s%s" "%s%s" "%s%d" "%s%d" "%s%d" "%s%d",
		prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, (hostname_lookup) ? server->host_name : server->arg,
		RD, escape((unsigned char *) server->server_name),
		RD, (server->map_name) ? server->map_name : "?",
		RD, server->max_players,
		RD, server->num_players,
		RD, server->ping_total/server->n_requests,
		RD, server->n_retries
        );
    }
    else if ( server->type == UN_SERVER)  {
	printf( "%s" "%.*s%.*s" "%s%s" "%s%s" "%s%s" "%s%d",
		prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, (hostname_lookup) ? server->host_name : server->arg,
		RD, escape((unsigned char *) server->server_name),
		RD, (server->map_name) ? server->map_name : "?",
		RD, server->num_players
	);
    }
    else if ( server->type & MASTER_SERVER)  {
        printf( "%s" "%.*s%.*s" "%s%s" "%s%d",
		prefix,
		raw_arg, RD, raw_arg, server->arg,
		RD, (hostname_lookup) ? server->host_name : server->arg,
		RD, server->n_servers
	);
    }
    puts("");

    if ( (server->type & MASTER_SERVER) || server->error != NULL)  {
	puts("");
	return;
    }

    if ( get_server_rules)
	raw_display_server_rules( server);
    if ( get_player_info)
	raw_display_player_info( server);
    puts("");
}

void
raw_display_server_rules( struct qserver *server)
{
    struct rule *rule;
    int printed= 0;
    rule= server->rules;
    for ( ; rule != NULL; rule= rule->next)  {
	printf( "%s%s=%s", (printed)?RD:"", rule->name, rule->value);
	printed++;
    }
    if ( server->missing_rules)
	printf( "%s?", (printed)?RD:"");
    puts("");
}

void
raw_display_player_info( struct qserver *server)
{
    if ( server->type == Q_SERVER || server->type == H2_SERVER)
	raw_display_q_player_info( server);
    else if ( server->type == QW_SERVER || server->type == HW_SERVER)
	raw_display_qw_player_info( server);
    else if ( server->type == Q2_SERVER)
	raw_display_q2_player_info( server);
    else if ( server->type == UN_SERVER)
	raw_display_unreal_player_info( server);
}

void
raw_display_q_player_info( struct qserver *server)
{
    char fmt[128];
    struct player *player;

    strcpy( fmt, "%d" "%s%s" "%s%s" "%s%d" "%s%s");
    if ( color_names)
	strcat( fmt, "%s%s" "%s%s");
    else
	strcat( fmt, "%s%d" "%s%d");

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		player->number,
		RD, escape( (unsigned char *) player->name),
		RD, player->address,
		RD, player->frags,
		RD, play_time(player->connect_time,1),
		RD, quake_color(player->shirt_color),
		RD, quake_color(player->pants_color)
	);
	puts("");
    }
}

void
raw_display_qw_player_info( struct qserver *server)
{
    char fmt[128];
    struct player *player;

    strcpy( fmt, "%d" "%s%s" "%s%d" "%s%s");
    if ( color_names)
	strcat( fmt, "%s%s" "%s%s");
    else
	strcat( fmt, "%s%d" "%s%d");
    strcat( fmt, "%s%d" "%s%s");

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		player->number,
		RD, escape( (unsigned char *) player->name),
		RD, player->frags,
		RD, play_time(player->connect_time,1),
		RD, quake_color(player->shirt_color),
		RD, quake_color(player->pants_color),
		RD, player->ping,
		RD, player->skin ? player->skin : ""
	);
	puts("");
    }
}

void
raw_display_q2_player_info( struct qserver *server)
{
    char fmt[128];
    struct player *player;

    strcpy( fmt, "%s" "%s%d" "%s%d");

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		escape( (unsigned char *) player->name),
		RD, player->frags,
		RD, player->ping
	);
	puts("");
    }
}

void
raw_display_unreal_player_info( struct qserver *server)
{
    static char fmt[128]= "%s" "%s%d" "%s%d" "%s%s" "%s%s";
    struct player *player;

    player= server->players;
    for ( ; player != NULL; player= player->next)  {
	printf( fmt,
		escape( (unsigned char *) player->name),
		RD, player->frags,
		RD, player->team,
		RD, player->skin,
		RD, player->mesh
	);
	puts("");
    }
}

void
display_progress()
{
    fprintf( stderr, "\r%d/%d (%d timed out, %d down)",
	num_servers_returned+num_servers_timed_out+num_servers_down,
	num_servers_total,
	num_servers_timed_out,
	num_servers_down);
}

/* ----- END MODIFICATION ----- Don't need to change anything below here. */


void set_non_blocking( int fd);

/* These packets are fixed size, so don't bother rebuilding them each time.
 * This only works because the packets are an even number in size.
 */
struct {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    char name[6];
    unsigned char version;
} qserverinfo =
	{ Q_FLAG1, Q_FLAG2, sizeof(qserverinfo), Q_CCREQ_SERVER_INFO,
	"QUAKE", Q_NET_PROTOCOL_VERSION };

struct {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    char name[8];
    unsigned char version;
} h2serverinfo =
	{ Q_FLAG1, Q_FLAG2, sizeof(h2serverinfo), Q_CCREQ_SERVER_INFO,
	"HEXENII", HEXEN2_NET_PROTOCOL_VERSION };

struct {
    unsigned char flag1;
    unsigned char flag2;
    unsigned short length;
    unsigned char op_code;
    unsigned char player_number;
} qplayerinfo =
	{ Q_FLAG1, Q_FLAG2, sizeof(qplayerinfo), Q_CCREQ_PLAYER_INFO, 0 };

struct {
    char prefix[4];
    char command[7];
} qwserverstatus = { QW_REQUEST_PREFIX, 's', 't', 'a', 't', 'u', 's', '\n' };

struct {
    char prefix[5];
    char command[7];
} hwserverstatus = { '\377', '\377', '\377', '\377', '\377',
	's', 't', 'a', 't', 'u', 's', '\n' };

char unrealserverstatus[8] = { '\\', 's','t','a','t','u','s', '\\' };
char unrealmasterlist[22] = "\\list\\\\gamename\\unreal";

/* Misc flags
 */

char * NO_SERVER_RULES= NULL;
int NO_PLAYER_INFO= 0xffff;
struct timeval packet_recv_time;
int server_types= 0;
#define FORCE 1

/* Print an error message and the program usage notes
 */

void
usage( char *msg, char **argv, char *a1)
{
    if ( msg)
	fprintf( stderr, msg, a1);

    printf( "Usage: %s [options ...] [-retry retries] [-interval interval]\n",
	argv[0]);
    printf( "\t[-raw delimiter] [-qw host[:port]] [-qws host[:port]]\n");
    printf( "\t[-h2s host[:port]] [-q2m host[:port]] [-default server-type] [-f file] [host[:port]] ...\n");
    printf( "Where host is an IP address or host name\n");
    printf( "Port defaults to %d if not specified\n", DEFAULT_PORT);
    printf( "-H\t\tresolve host names\n");
    printf( "-Hcache\t\thost name cache file\n");
    printf( "-R\t\tserver rules\n");
    printf( "-P\t\tplayer info\n");
    printf( "-f\t\tread hosts from file\n");
    printf( "-qw\t\tget server list from QuakeWorld master server\n");
    printf( "-q2m\t\tget server list from Quake II master server\n");
    printf( "-qws\t\tdisplay stats for QuakeWorld server\n");
    printf( "-hws\t\tdisplay stats for HexenWorld server\n");
    printf( "-h2s\t\tdisplay stats for Hexen II server\n");
    printf( "-hexen2\t\tquery using Hexen II protocol\n");
    printf( "-uns\t\tdisplay stats for Unreal server\n");
    printf( "-q2s\t\tdisplay stats for Quake II server\n");
    printf( "-default\tset default server type: QS, QW, QWM, H2S, HWS, Q2, UNS\n");
    printf( "-sort\t\tsort servers: p=by-ping, g=by-game \n");
    printf( "-u\t\tonly display servers that are up\n");
    printf( "-nf\t\tdo not display full servers\n");
    printf( "-ne\t\tdo not display empty servers\n");
    printf( "-cn\t\tdisplay color names instead of numbers\n");
    printf( "-ncn\t\tdisplay color numbers instead of names\n");
    printf( "-tc\t\tdisplay time in clock format (DhDDmDDs)\n");
    printf( "-tsw\t\tdisplay time in stop-watch format (DD:DD:DD)\n");
    printf( "-ts\t\tdisplay time in seconds\n");
    printf( "-pa\t\tdisplay player address\n");
    printf( "-hpn\t\tdisplay player names in hex\n");
    printf( "-old\t\told style display\n");
    printf( "-progress\tdisplay progress meter (text only)\n");
    printf( "-retry\t\tnumber of retries, default is %d\n", DEFAULT_RETRIES);
    printf( "-interval\tinterval between retries, default is %.2lf seconds\n",
	DEFAULT_RETRY_INTERVAL / 1000.0);
    printf( "-timeout\ttotal time in seconds before giving up\n");
    printf( "-maxsim\t\tset maximum simultaneous queries\n");
    printf( "-raw\t\toutput in raw format using delimiter\n");
    printf( "-Th,-Ts,-Tp,-Tt\toutput templates: header, server, player, and trailer\n");
    printf( "qstat version %s\n", VERSION);
    exit(0);
}

struct server_arg  {
    int type;
    char *arg;
    char *outfilename;
};

void
add_server_arg( char *arg, int type, char *outfilename,
	struct server_arg **args, int *n, int *max)
{
    if ( *n == *max)  {
	if ( *max == 0)  {
	    *max= 4;
	    *args= malloc(sizeof(struct server_arg) * (*max));
	}
	else  {
	    (*max)*= 2;
	    *args= (struct server_arg*) realloc( *args,
		sizeof(struct server_arg) * (*max));
	}
    }
    (*args)[*n].type= type;
    (*args)[*n].arg= arg;
    (*args)[*n].outfilename= outfilename;
    (*n)++;
}

main( int argc, char *argv[])
{
    int pktlen, rc, maxfd, fd;
    long pkt_data[PACKET_LEN/sizeof(long)];
    char *pkt= (char*)&pkt_data[0];
    fd_set read_fds;
    struct timeval timeout;
    int arg, timeout_milli, n_files, i;
    struct qserver *server;
    char **files, *outfilename;
    struct server_arg *server_args= NULL;
    int n_server_args= 0, max_server_args= 0;
 
#ifdef _WIN32
    WORD version= MAKEWORD(1,1);
    WSADATA wsa_data;
    if ( WSAStartup(version,&wsa_data) != 0)  {
	fprintf( stderr, "Could not open winsock\n");
	exit(1);
    }
#endif

    if ( argc == 1)
	usage(NULL,argv,NULL);

    files= (char **) malloc( sizeof(char*) * (argc/2));
    n_files= 0;

    BuildTree( HuffFreq);

    for ( arg= 1; arg < argc; arg++)  {
	if ( argv[arg][0] != '-')
	    break;
	outfilename= NULL;
	if ( strcmp( argv[arg], "-f") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -f\n", argv,NULL);
	    files[n_files++]= argv[arg];
	}
	else if ( strcmp( argv[arg], "-retry") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -retry\n", argv,NULL);
	    n_retries= atoi( argv[arg]);
	    if ( n_retries <= 0)  {
		fprintf( stderr, "retries must be greater than zero\n");
		exit(1);
	    }
	}
	else if ( strcmp( argv[arg], "-interval") == 0)  {
	    double value= 0.0;
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -interval\n", argv,NULL);
	    sscanf( argv[arg], "%lf", &value);
	    if ( value < 0.1)  {
		fprintf( stderr, "retry interval must be greater than 0.1\n");
		exit(1);
	    }
	    retry_interval= (int)(value * 1000);
	}
	else if ( strcmp( argv[arg], "-H") == 0)
	    hostname_lookup= 1;
	else if ( strcmp( argv[arg], "-u") == 0)
	    up_servers_only= 1;
	else if ( strcmp( argv[arg], "-nf") == 0)
	    no_full_servers= 1;
	else if ( strcmp( argv[arg], "-ne") == 0)
	    no_empty_servers= 1;
	else if ( strcmp( argv[arg], "-nh") == 0)
	    no_header_display= 1;
	else if ( strcmp( argv[arg], "-old") == 0)
	    new_style= 0;
	else if ( strcmp( argv[arg], "-P") == 0)
	    get_player_info= 1;
	else if ( strcmp( argv[arg], "-R") == 0)
	    get_server_rules= 1;
	else if ( strcmp( argv[arg], "-raw") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -raw\n", argv,NULL);
	    raw_delimiter= argv[arg];
	    raw_display= 1;
	}
	else if ( strcmp( argv[arg], "-ncn") == 0)  {
	    color_names= 0;
 	}
	else if ( strcmp( argv[arg], "-cn") == 0)  {
	    color_names= 1;
 	}
	else if ( strcmp( argv[arg], "-tc") == 0)  {
	    time_format= CLOCK_TIME;
	}
	else if ( strcmp( argv[arg], "-tsw") == 0)  {
	    time_format= STOPWATCH_TIME;
	}
	else if ( strcmp( argv[arg], "-ts") == 0)  {
	    time_format= SECONDS;
	}
	else if ( strcmp( argv[arg], "-pa") == 0)  {
	    player_address= 1;
	}
	else if ( strcmp( argv[arg], "-hpn") == 0)  {
	    hex_player_names= 1;
	}
	else if ( strncmp( argv[arg], "-qw", 3) == 0 &&
		(argv[arg][3] == '\0' || strcmp(&argv[arg][3], ",outfile") == 0)) {
	    arg++;
	    if ( argv[arg-1][3] == ',')  {
		outfilename= strchr( argv[arg], ',');
		if ( outfilename == NULL)
		    usage( "missing file name for -qw,outfile\n", argv,NULL);
		*outfilename++= '\0';
	    }
	    if ( arg >= argc)
		usage( "missing argument for -qw\n", argv,NULL);
	    add_server_arg( argv[arg], QW_MASTER, outfilename,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strncmp( argv[arg], "-q2m", 4) == 0 &&
		(argv[arg][4] == '\0' || strcmp(&argv[arg][4], ",outfile") == 0)) {
	    arg++;
	    if ( argv[arg-1][4] == ',')  {
		outfilename= strchr( argv[arg], ',');
		if ( outfilename == NULL)
		    usage( "missing file name for -q2m,outfile\n", argv,NULL);
		*outfilename++= '\0';
	    }
	    if ( arg >= argc)
		usage( "missing argument for -q2m\n", argv,NULL);
	    add_server_arg( argv[arg], Q2_MASTER, outfilename,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strcmp( argv[arg], "-qws") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -qws\n", argv,NULL);
	    add_server_arg( argv[arg], QW_SERVER, NULL,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strcmp( argv[arg], "-q2s") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -qws\n", argv,NULL);
	    add_server_arg( argv[arg], Q2_SERVER, NULL,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strncmp( argv[arg], "-maxsimultaneous", 7) == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -maxsimultaneous\n", argv,NULL);
	    max_simultaneous= atoi(argv[arg]);
	    if ( max_simultaneous <= 0)
		usage( "value for -maxsimultaneous must be > 0\n", argv,NULL);
	    if ( max_simultaneous > FD_SETSIZE)
		max_simultaneous= FD_SETSIZE;
 	}
	else if ( strcmp( argv[arg], "-h2s") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -h2s\n", argv,NULL);
	    add_server_arg( argv[arg], H2_SERVER, NULL,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strcmp( argv[arg], "-hws") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -hws\n", argv,NULL);
	    add_server_arg( argv[arg], HW_SERVER, NULL,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strcmp( argv[arg], "-raw-arg") == 0)  {
	    raw_arg= 1000;
 	}
	else if ( strcmp( argv[arg], "-timeout") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -timeout\n", argv,NULL);
	    run_timeout= atoi( argv[arg]);
	    if ( run_timeout <= 0)
		usage( "value for -timeout must be > 0\n", argv,NULL);
	}
	else if ( strcmp( argv[arg], "-progress") == 0)  {
	    progress= 1;
 	}
	else if ( strcmp( argv[arg], "-Hcache") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -Hcache\n", argv,NULL);
	    if ( hcache_open( argv[arg], 0) == -1)
		return 1;
	}
	else if ( strcmp( argv[arg], "-default") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -default\n", argv,NULL);
	    if ( set_default_server_type( argv[arg]) == -1)
		usage( NULL, argv,NULL);
	}
	else if ( strncmp( argv[arg], "-Tserver", 3) == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for %s\n", argv, argv[arg]);
	    if ( read_qserver_template( argv[arg]) == -1)
		return 1;
	}
	else if ( strncmp( argv[arg], "-Theader", 3) == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for %s\n", argv, argv[arg]);
	    if ( read_header_template( argv[arg]) == -1)
		return 1;
	}
	else if ( strncmp( argv[arg], "-Ttrailer", 3) == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for %s\n", argv, argv[arg]);
	    if ( read_trailer_template( argv[arg]) == -1)
		return 1;
	}
	else if ( strncmp( argv[arg], "-Tplayer", 3) == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for %s\n", argv, argv[arg]);
	    if ( read_player_template( argv[arg]) == -1)
		return 1;
	}
	else if ( strcmp( argv[arg], "-sort") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -sort\n", argv, NULL);
	    if ( argv[arg][0] == 'p')
		first_sort_key= SORT_PING;
	    else if ( argv[arg][0] == 'g')
		first_sort_key= SORT_GAME;
	    else
		usage( "Unknown sort key \"%s\", use p or g\n", argv,
			argv[arg]);
	    sort++;
	}
	else if ( strcmp( argv[arg], "-uns") == 0)  {
	    arg++;
	    if ( arg >= argc)
		usage( "missing argument for -uns\n", argv,NULL);
	    add_server_arg( argv[arg], UN_SERVER, NULL,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strncmp( argv[arg], "-unm", 4) == 0 &&
		(argv[arg][4] == '\0' || strcmp(&argv[arg][4], ",outfile") == 0)) {
	    arg++;
	    if ( argv[arg-1][4] == ',')  {
		outfilename= strchr( argv[arg], ',');
		if ( outfilename == NULL)
		    usage( "missing file name for -unm,outfile\n", argv,NULL);
		*outfilename++= '\0';
	    }
	    if ( arg >= argc)
		usage( "missing argument for -unm\n", argv,NULL);
	    add_server_arg( argv[arg], UN_MASTER, outfilename,
		&server_args, &n_server_args, &max_server_args);
	}
	else if ( strcmp( argv[arg], "-errors") == 0)  {
	    show_errors++;
	}
	else  {
	    fprintf( stderr, "unknown option \"%s\"\n", argv[arg]);
	    usage(NULL,argv,NULL);
	}
    }

    start_time= time(0);

    for ( i= 0; i < n_files; i++)
	add_file( files[i]);

    for ( ; arg < argc; arg++)
	add_qserver( argv[arg], default_server_type, NULL);

    for ( i= 0; i < n_server_args; i++)
	add_qserver( server_args[i].arg, server_args[i].type,
		server_args[i].outfilename);

    free( server_args);

    if ( servers == NULL)
	exit(1);

    connmap= (struct qserver**) calloc( 1, sizeof(struct qserver*) *
	( max_simultaneous + 10));

    if ( color_names == -1)
	color_names= ( raw_display) ? DEFAULT_COLOR_NAMES_RAW :
		DEFAULT_COLOR_NAMES_DISPLAY;

    if ( time_format == -1)
	time_format= ( raw_display) ? DEFAULT_TIME_FMT_RAW :
		DEFAULT_TIME_FMT_DISPLAY;

    if ( (server_types & MASTER_SERVER) || count_bits(server_types) > 1)
	display_prefix= 1;

    if ( new_style && ! raw_display && ! have_server_template())
	display_header();
    else if ( have_header_template())
	template_display_header();

    qserverinfo.length= htons( qserverinfo.length);
    h2serverinfo.length= htons( h2serverinfo.length);
    qplayerinfo.length= htons( qplayerinfo.length);

    save_retry_interval= retry_interval;
    if ( server_types & MASTER_SERVER)
	retry_interval*= 2;
    if ( server_types & Q2_MASTER)
	retry_interval*= 2;

    bind_sockets();

    while (connected)  {
	FD_ZERO( &read_fds);
	maxfd= set_fds( &read_fds);

	if ( progress)
	    display_progress();
	get_next_timeout( &timeout);
	rc= select( maxfd+1, &read_fds, NULL, NULL, &timeout);

	if (rc == 0)  {
	    if ( run_timeout && time(0)-start_time >= run_timeout)
		break;
	    send_packets();
	    bind_sockets();
	    continue;
	}
	if (rc == SOCKET_ERROR)  {
	    perror("select");
	    hcache_update_file();
	    return -1;
	}

	gettimeofday( &packet_recv_time, NULL);
#ifdef unix
	fd= 3;
	for ( ; rc; rc--)  {
	    struct sockaddr_in addr;
	    int addrlen= sizeof(addr);
	    while ( fd <= maxfd && ! FD_ISSET( fd, &read_fds))
		fd++;
	    if ( fd > maxfd)
		break;
	    server= connmap[fd++];
	    if ( server->flags & FLAG_BROADCAST)
		pktlen= recvfrom( server->fd, pkt, sizeof(pkt_data), 0,
			(struct sockaddr*)&addr, &addrlen);
	    else
	        pktlen= recv( server->fd, pkt, sizeof(pkt_data), 0);
	    if (pktlen == SOCKET_ERROR)  {
		if ( connection_refused())  {
		    server->server_name= DOWN;
		    num_servers_down++;
		    cleanup_qserver( server, 1);
		    if ( ! connected)
			bind_sockets();
		}
		continue;
	    }
	    if ( server->flags & FLAG_BROADCAST)  {
		struct qserver *broadcast= server;
		/* create new server and init */
		server= add_qserver_byaddr( ntohl(addr.sin_addr.s_addr),
			ntohs(addr.sin_port), server->type);
		server->packet_time1= broadcast->packet_time1;
		server->packet_time2= broadcast->packet_time2;
		server->ping_total= broadcast->ping_total;
		server->n_requests= broadcast->n_requests;
		server->n_packets= broadcast->n_packets;
		broadcast->n_servers++;
	    }

	    deal_with_packet( server, pkt, pktlen);
	}
#endif /* unix */

#ifdef _WIN32
	for ( server= servers; server != NULL; server= server->next)  {
	    struct sockaddr_in addr;
	    struct qserver *broadcast;
	    int addrlen= sizeof(addr);

	    if ( server->fd == -1)
	        continue;
	    if ( ! FD_ISSET( server->fd, &read_fds))
		continue;
	    if ( server->flags & FLAG_BROADCAST)
		pktlen= recvfrom( server->fd, pkt, sizeof(pkt_data), 0,
			(struct sockaddr*)&addr, &addrlen);
	    else
	        pktlen= recv( server->fd, pkt, sizeof(pkt_data), 0);
	    if ( pktlen == SOCKET_ERROR)  {
		if ( connection_refused())  {
		    server->server_name= DOWN;
		    num_servers_down++;
		    cleanup_qserver( server, 1);
		    if ( ! connected)
			bind_sockets();
		}
		continue;
	    }

	    broadcast= NULL;
	    if ( server->flags & FLAG_BROADCAST)  {
		broadcast= server;
		/* create new server and init */
		server= add_qserver_byaddr( ntohl(addr.sin_addr.s_addr),
			ntohs(addr.sin_port), server->type);
		server->packet_time1= broadcast->packet_time1;
		server->packet_time2= broadcast->packet_time2;
		server->ping_total= broadcast->ping_total;
		server->n_requests= broadcast->n_requests;
		server->n_packets= broadcast->n_packets;
		broadcast->n_servers++;
	    }

	    deal_with_packet( server, pkt, pktlen);
	    if ( broadcast)
		server= broadcast;
	}
#endif /* _WIN32 */

	if ( run_timeout && time(0)-start_time >= run_timeout)  {
	    hcache_update_file();
	    break;
	}
	if ( ! connected)
	    bind_sockets();
    }

    hcache_update_file();

    if ( sort)  {
	struct qserver **array, *server;
	array= (struct qserver **) malloc( sizeof(struct qserver *) *
		num_servers_total);
	server= servers;
	for ( i= 0; server != NULL; i++)  {
	    array[i]= server;
	    server= server->next;
	}
	qstatsort( array, num_servers_total);
	if ( progress)
	    fprintf( stderr, "\n");
	for ( i= 0; i < num_servers_total; i++)
	    display_server( array[i]);
	free( array);
    }
    else  {
	struct qserver *server;
	server= servers;
	for ( ; server; server= server->next)
	    if ( server->server_name == HOSTNOTFOUND)
		display_server( server);
    }

    if ( have_trailer_template())
	template_display_trailer();

    return 0;
}


void
add_file( char *filename)
{
    FILE *file;
    char name[200];
    int type;

    if ( strcmp( filename, "-") == 0)  {
	file= stdin;
	current_filename= NULL;
    }
    else  {
	file= fopen( filename, "r");
	current_filename= filename;
    }
    current_fileline= 1;

    if ( file == NULL)  {
	perror( filename);
	return;
    }
    for ( ; fscanf( file, "%s", name) == 1; current_fileline++)  {
	if ( strcmp( name, "QS") == 0)
	    type= Q_SERVER;
	else if ( strcmp( name, "H2S") == 0)
	    type= H2_SERVER;
	else if ( strcmp( name, "HWS") == 0)
	    type= HW_SERVER;
	else if ( strcmp( name, "QW") == 0)
	    type= QW_SERVER;
	else if ( strcmp( name, "Q2") == 0)
	    type= Q2_SERVER;
	else if ( strcmp( name, "QWM") == 0)
	    type= QW_MASTER;
	else if ( strcmp( name, "Q2M") == 0)
	    type= Q2_MASTER;
	else if ( strcmp( name, "UNS") == 0)
	    type= UN_SERVER;
	else if ( strcmp( name, "UNM") == 0)
	    type= UN_MASTER;
	else  {
	    add_qserver( name, default_server_type, NULL);
	    continue;
	}
	if ( fscanf( file, "%s", name) == 1)
	    add_qserver( name, type, NULL);
    }

    if ( file != stdin)
	fclose(file);

    current_fileline= 0;
}

void
print_file_location()
{
    if ( current_fileline != 0)
	fprintf( stderr, "%s:%d: ", current_filename?current_filename:"<stdin>",
		current_fileline);
}

int
add_qserver( char *arg, int type, char *outfilename)
{
    struct sockaddr_in addr;
    struct hostent *ent= NULL;
    struct hostent temp_ent;
    struct hostent *name_ent= NULL;
    struct qserver *server;
    int i, port, flags= 0;
    char **a, *colon= NULL, *arg_copy, *s, *hostname= NULL;
    unsigned long ipaddr;
 
    if ( run_timeout && time(0)-start_time >= run_timeout)
	exit(0);

    if ( type == Q_SERVER)
	port= DEFAULT_PORT;
    else if ( type == H2_SERVER)
	port= HEXEN2_DEFAULT_PORT;
    else if ( type == HW_SERVER)
	port= HW_DEFAULT_PORT;
    else if ( type == QW_SERVER)
	port= QW_DEFAULT_PORT;
    else if ( type == QW_MASTER)
	port= QW_MASTER_DEFAULT_PORT;
    else if ( type == Q2_SERVER)
	port= QUAKE2_DEFAULT_PORT;
    else if ( type == Q2_MASTER)
	port= Q2_MASTER_DEFAULT_PORT;
    else if ( type == UN_SERVER)
	port= UNREAL_DEFAULT_PORT;
    else if ( type == UN_MASTER)
	port= UNREAL_MASTER_DEFAULT_PORT;
    else
	port= default_server_port;

    if ( outfilename && strcmp( outfilename, "-") != 0)  {
	FILE *outfile= fopen( outfilename, "r+");
	if ( outfile == NULL && (errno == EACCES || errno == EISDIR ||
		errno == ENOSPC || errno == ENOTDIR))  {
	    perror( outfilename);
	    return -1;
	}
	if ( outfile)
	    fclose(outfile);
    }

    arg_copy= strdup(arg);

    colon= strchr( arg, ':');
    if ( colon != NULL)  {
	sscanf( colon+1, "%d", &port);
	*colon= '\0';
    }

    if ( *arg == '+')  {
	flags|= FLAG_BROADCAST;
	arg++;
    }

    ipaddr= inet_addr(arg);
    if ( ipaddr == INADDR_NONE)  {
	if ( strcmp( arg, "255.255.255.255") != 0)
	    ipaddr= htonl( hcache_lookup_hostname(arg));
    }
    else if ( hostname_lookup && !(flags&FLAG_BROADCAST))
	hostname= hcache_lookup_ipaddr( ntohl(ipaddr));

    if ( ipaddr == INADDR_NONE && strcmp( arg, "255.255.255.255") != 0)  {
	print_file_location();
	fprintf( stderr, "%s: %s\n", arg, strherror(h_errno));
	server= (struct qserver *) calloc( 1, sizeof( struct qserver));
	init_qserver( server);
	server->arg= arg_copy;
	server->server_name= HOSTNOTFOUND;
	server->error= strdup( strherror(h_errno));
	server->port= port;
	server->type= type;
	server->next= servers;
	servers= server;
	server_types|= type;
        return -1;
    }

    server= servers;
    while ( server != NULL)  {
	if ( server->ipaddr == ipaddr && server->port == port)
	    return 0;
	server= server->next;
    }

    server= (struct qserver *) calloc( 1, sizeof( struct qserver));
    server->arg= arg_copy;
    if ( hostname && colon)  {
	server->host_name= malloc( strlen(hostname) + strlen(colon+1)+1);
	sprintf( server->host_name, "%s:%s", hostname, colon+1);
    }
    else
	server->host_name= strdup((hostname)?hostname:arg);

    server->ipaddr= ipaddr;
    server->port= port;
    server->type= type;
    server->outfilename= outfilename;
    server->flags= flags;
    init_qserver( server);

    if ( server->type & MASTER_SERVER)
	waiting_for_masters++;

    if ( num_servers_total % 10 == 0)
	hcache_update_file();

    server->next= servers;
    servers= server;

    server_types|= type;

    return 0;
}

struct qserver *
add_qserver_byaddr( unsigned long ipaddr, unsigned short port, int type)
{
    char arg[36];
    struct qserver *server;
    struct hostent *name_ent= NULL;
    char *hostname= NULL;

    if ( run_timeout && time(0)-start_time >= run_timeout)
	exit(0);

    ipaddr= htonl(ipaddr);
    server= servers;
    while ( server != NULL)  {
	if ( server->ipaddr == ipaddr && server->port == port)
	    return server;
	server= server->next;
    }

    server= (struct qserver *) calloc( 1, sizeof( struct qserver));
    server->ipaddr= ipaddr;
    ipaddr= ntohl(ipaddr);
    sprintf( arg, "%d.%d.%d.%d:%hu", ipaddr>>24, (ipaddr>>16)&0xff,
	(ipaddr>>8)&0xff, ipaddr&0xff, port);
    server->arg= strdup(arg);

    if ( hostname_lookup)
	hostname= hcache_lookup_ipaddr( ipaddr);
    if ( hostname)  {
	server->host_name= malloc( strlen(hostname) + 6 + 1);
	sprintf( server->host_name, "%s:%hu", hostname, port);
    }
    else
	server->host_name= strdup( arg);

    server->port= port;
    server->type= type;
    init_qserver( server);

    if ( num_servers_total % 10 == 0)
	hcache_update_file();

    server->next= servers;
    servers= server;
    return server;
}

void
add_servers_from_masters()
{
    struct qserver *server;
    unsigned long ipaddr, i;
    unsigned short port;
    char *pkt, *server_key;
    int server_type;
    FILE *outfile;

    for ( server= servers; server != NULL; server= server->next)  {
	if ( !(server->type & MASTER_SERVER) || server->master_pkt == NULL)
	    continue;
	pkt= server->master_pkt;
	switch ( server->type) {
	case QW_MASTER: server_type= QW_SERVER; server_key= "QW"; break;
	case Q2_MASTER: server_type= Q2_SERVER; server_key= "Q2"; break;
	case UN_MASTER: server_type= UN_SERVER; server_key= "UN"; break;
	default: break;
	}
	outfile= NULL;
	if ( server->outfilename)  {
	    if ( strcmp( server->outfilename, "-") == 0)
		outfile= stdout;
	    else
	        outfile= fopen( server->outfilename, "w");
	    if ( outfile == NULL)  {
		perror( server->outfilename);
		continue;
	    }
	}
	for ( i= 0; i < server->master_pkt_len; i+= 6)  {
	    memcpy( &ipaddr, &pkt[i], 4);
	    memcpy( &port, &pkt[i+4], 2);
	    ipaddr= ntohl( ipaddr);
	    port= ntohs( port);
	    if ( outfile)
		fprintf( outfile, "%s %d.%d.%d.%d:%hu\n", server_key,
			(ipaddr>>24)&0xff, (ipaddr>>16)&0xff,
			(ipaddr>>8)&0xff, ipaddr&0xff, port);
	    else
		add_qserver_byaddr( ipaddr, port, server_type);
	}
	free( server->master_pkt);
	server->master_pkt= NULL;
	if ( outfile)
	    fclose( outfile);
    }
    if ( hostname_lookup)
	hcache_update_file();
    bind_sockets();
}

void
init_qserver( struct qserver *server)
{
    server->server_name= NULL;
    server->map_name= NULL;
    server->game= NULL;
    server->num_players= 0;
    server->fd= -1;
    if ( server->flags & FLAG_BROADCAST)  {
	server->retry1= 1;
	server->retry2= 1;
    }
    else  {
	server->retry1= n_retries;
	server->retry2= n_retries;
    }
    server->n_retries= 0;
    server->ping_total= 0;
    server->n_packets= 0;
    server->n_requests= 0;

    server->n_servers= 0;
    server->master_pkt_len= 0;
    server->master_pkt= NULL;
    server->error= NULL;

    server->next_rule= (get_server_rules) ? "" : NO_SERVER_RULES;
    server->next_player_info= (get_player_info) ? 0 : NO_PLAYER_INFO;

    server->n_player_info= 0;
    server->players= NULL;
    server->n_rules= 0;
    server->rules= NULL;
    server->missing_rules= 0;

    num_servers_total++;
}

/* Functions for binding sockets to Quake servers
 */
int
bind_qserver( struct qserver *server)
{
    struct sockaddr_in addr;
    static int one= 1;

    if ( server->type == UN_MASTER)
	server->fd= socket( AF_INET, SOCK_STREAM, 0);
    else
	server->fd= socket( AF_INET, SOCK_DGRAM, 0);

    if ( server->fd == INVALID_SOCKET) {
	if ( sockerr() == EMFILE)  {
	    server->fd= -1;
	    return -2;
	}
        perror( "socket" );
	server->server_name= SYSERROR;
        return -1;
    }

    addr.sin_family = AF_INET;
    if ( server->type == Q2_MASTER)
	addr.sin_port = htons(26500);
    else
	addr.sin_port = htons(0);
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    memset( &(addr.sin_zero), 0, sizeof(addr.sin_zero) );

    if ( bind( server->fd, (struct sockaddr *)&addr,
		sizeof(struct sockaddr)) == SOCKET_ERROR) {
	if ( sockerr() != EADDRINUSE)  {
	    perror( "bind" );
	    server->server_name= SYSERROR;
	}
	close(server->fd);
	server->fd= -1;
        return -1;
    }

    if ( server->flags & FLAG_BROADCAST)
	setsockopt( server->fd, SOL_SOCKET, SO_BROADCAST, (char*)&one,
		sizeof(one));

    if ( server->type != Q2_MASTER && !(server->flags & FLAG_BROADCAST))  {
	addr.sin_family = AF_INET;
	addr.sin_port = htons(server->port);
	addr.sin_addr.s_addr = server->ipaddr;
	memset( &(addr.sin_zero), 0, sizeof(addr.sin_zero) );

	if ( connect( server->fd, (struct sockaddr *)&addr, sizeof(addr)) ==
		SOCKET_ERROR)  {
	    if ( server->type == UN_MASTER)  {
		if ( connection_refused())  {
		/*  server->fd= -2; */
		/* set up for connect retry */
		}
	    }
	    perror( "connect");
	    server->server_name= SYSERROR;
	    close(server->fd);
	    server->fd= -1;
	    return -1;
	}
    }

    if ( server->type == UN_MASTER)
	set_non_blocking( server->fd);

#ifdef unix
    connmap[server->fd]= server;
#endif

    return 0;
}

void
bind_sockets()
{
    struct qserver *server= servers;
    int rc;

    for ( ;	server != NULL && connected < max_simultaneous;
		server= server->next)  {
	if ( server->server_name == NULL && server->fd == -1)  {
	    if ( waiting_for_masters && !(server->type & MASTER_SERVER))
		continue;
	    if ( (rc= bind_qserver( server)) == 0)  {
		send_server_request_packet( server);
		connected++;
	    }
	    else if ( rc == -2)
		break;
	}
    }
}


/* Functions for sending packets
 */
void
send_packets()
{
    struct qserver *server= servers;
    struct timeval now;
    int rc;

    gettimeofday( &now, NULL);

    for ( ; server != NULL; server= server->next)  {
	if ( server->fd == -1)
	    continue;
	if ( server->server_name == NULL || server->type == UN_SERVER)  {
	    if ( server->retry1 != n_retries &&
		    time_delta( &now, &server->packet_time1) <
			(retry_interval*(n_retries-server->retry1+1)))
		continue;
	    if ( ! server->retry1)  {
		cleanup_qserver( server, 0);
		continue;
	    }
	    send_server_request_packet( server);
	    continue;
	}
	if ( server->next_rule != NO_SERVER_RULES)  {
	    if ( server->retry1 != n_retries &&
		    time_delta( &now, &server->packet_time1) <
			(retry_interval*(n_retries-server->retry1+1)))
		continue;
	    if ( ! server->retry1)  {
		server->next_rule= NULL;
		server->missing_rules= 1;
		cleanup_qserver( server, 0);
		continue;
	    }
	    send_rule_request_packet( server);
	}
	if ( server->next_player_info < server->num_players)  {
	    if ( server->retry2 != n_retries &&
		    time_delta( &now, &server->packet_time2) <
			(retry_interval*(n_retries-server->retry2+1)))
		continue;
	    if ( ! server->retry2)  {
		server->next_player_info++;
		if ( server->next_player_info >= server->num_players)  {
		    cleanup_qserver( server, 0);
		    continue;
		}
		server->retry2= n_retries;
	    }
	    send_player_request_packet( server);
	}
    }
}

int
send_broadcast( struct qserver *server, char *pkt, int pktlen)
{
    struct sockaddr_in addr;
    addr.sin_family= AF_INET;
    addr.sin_port= htons(server->port);
    addr.sin_addr.s_addr= server->ipaddr;
    memset( &(addr.sin_zero), 0, sizeof(addr.sin_zero));
    return sendto( server->fd, (const char*) pkt, pktlen, 0,
		(struct sockaddr *) &addr, sizeof(addr));
}

/* Send the first packet to a server
 */
int
send_server_request_packet( struct qserver *server)
{
    if ( server->type == Q_SERVER || server->type == H2_SERVER)
	return send_qserver_request_packet( server);
    else if ( server->type == QW_SERVER || server->type == Q2_SERVER ||
		server->type == HW_SERVER)
	return send_qwserver_request_packet( server);
    else if ( server->type & MASTER_SERVER)
	return send_qwmaster_request_packet( server);
    if ( server->type == UN_SERVER)
	return send_unreal_request_packet( server);
    if ( server->type == UN_MASTER)  {
	return send_unrealmaster_request_packet( server);
    }
    else
	return -1;
}

/* First packet for a normal Quake server
 */
int
send_qserver_request_packet( struct qserver *server)
{
    int rc;
    if ( server->type == H2_SERVER)  {
	if ( server->flags & FLAG_BROADCAST)
	    rc= send_broadcast( server, (char*)&h2serverinfo,
		sizeof(h2serverinfo));
	else
	    rc= send( server->fd, (const char *)&h2serverinfo,
		sizeof(h2serverinfo), 0);
    }
    else  {
	if ( server->flags & FLAG_BROADCAST)
	    rc= send_broadcast( server, (char*)&qserverinfo,
		sizeof(qserverinfo));
	else
	    rc= send( server->fd, (const char *)&qserverinfo,
		sizeof(qserverinfo), 0);
    }

    if ( rc == SOCKET_ERROR)  {
	unsigned long ipaddr= ntohl(server->ipaddr);
	fprintf( stderr,
		"Error on %d.%d.%d.%d, skipping ...\n",
		(ipaddr>>24)&0xff, (ipaddr>>16)&0xff,
		(ipaddr>>8)&0xff, ipaddr&0xff);
	perror( "send");
	cleanup_qserver( server, 1);
	return -1;
    }
    if ( server->retry1 == n_retries || server->flags & FLAG_BROADCAST)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
    return 0;
}

/* First packet for a QuakeWorld server
 */
int
send_qwserver_request_packet( struct qserver *server)
{
    int rc;

    if ( server->type == HW_SERVER)  {
	if ( server->flags & FLAG_BROADCAST)
	    rc= send_broadcast( server, (char*)&hwserverstatus,
		sizeof(hwserverstatus));
	else
	    rc= send( server->fd, (const char *)&hwserverstatus,
		sizeof(hwserverstatus), 0);
    }
    else  {
	if ( server->flags & FLAG_BROADCAST)
	    rc= send_broadcast( server, (char*)&qwserverstatus,
		sizeof(qwserverstatus));
	else
	    rc= send( server->fd, (const char *)&qwserverstatus,
		sizeof(qwserverstatus), 0);
    }

    if ( rc == SOCKET_ERROR)  {
	unsigned long ipaddr= ntohl(server->ipaddr);
	fprintf( stderr,
		"Error on %d.%d.%d.%d, skipping ...\n",
		(ipaddr>>24)&0xff, (ipaddr>>16)&0xff,
		(ipaddr>>8)&0xff, ipaddr&0xff);
	perror( "send");
	cleanup_qserver( server, 1);
	return -1;
    }
    if ( server->retry1 == n_retries || server->flags & FLAG_BROADCAST)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
    return 0;
}

/* First packet for an Unreal server
 */
int
send_unreal_request_packet( struct qserver *server)
{
    int rc;

    rc= send( server->fd, (const char *)&unrealserverstatus,
		sizeof(unrealserverstatus), 0);

    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry1 == n_retries)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
    return 0;
}

/* First packet for an Unreal master
 */
int
send_unrealmaster_request_packet( struct qserver *server)
{
    int rc;

    rc= send( server->fd, (const char *)&unrealmasterlist,
		sizeof(unrealmasterlist), 0);

    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry1 == n_retries)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
    return 0;
}

/* First packet for a QuakeWorld master server
 */
int
send_qwmaster_request_packet( struct qserver *server)
{
    int rc= 0;

    if ( server->type == QW_MASTER)  {
	char pkt[QW_REQUEST_LENGTH];
	memset( pkt, 0, sizeof(pkt));
	sprintf( pkt, "%c%c", QW_GET_SERVERS, QW_NEWLINE);
	rc= send( server->fd, pkt, sizeof(pkt), 0);
    }
    else if ( server->type == Q2_MASTER)  {
	struct sockaddr_in addr;
	addr.sin_family= AF_INET;
	addr.sin_port= htons(server->port);
	addr.sin_addr.s_addr= server->ipaddr;
	memset( &(addr.sin_zero), 0, sizeof(addr.sin_zero));
	rc= sendto( server->fd, "query\n", 7, 0,
		(struct sockaddr *) &addr, sizeof(addr));
    }

    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry1 == n_retries)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
    return 0;
}

int
send_rule_request_packet( struct qserver *server)
{
    struct qpacket request= {Q_FLAG1,Q_FLAG2,0,Q_CCREQ_RULE_INFO,""};
    int rc, len;

    strcpy( (char*)request.data, server->next_rule);
    len= Q_HEADER_SIZE + strlen((char*)request.data) + 1;
    request.length= htons( (short)len);
    rc= send( server->fd, (const char *)&request, len, 0);
    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry1 == n_retries)  {
	gettimeofday( &server->packet_time1, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry1--;
    server->n_packets++;
    return 0;
}

int
send_player_request_packet( struct qserver *server)
{
    int rc;

    qplayerinfo.player_number= server->next_player_info;
    rc= send( server->fd, (const char *)&qplayerinfo, sizeof(qplayerinfo), 0);
    if ( rc == SOCKET_ERROR)
	perror( "send");
    if ( server->retry2 == n_retries)  {
	gettimeofday( &server->packet_time2, NULL);
	server->n_requests++;
    }
    else
	server->n_retries++;
    server->retry2--;
    server->n_packets++;
    return 0;
}

/* Functions for figuring timeouts and when to give up
 */
void
cleanup_qserver( struct qserver *server, int force)
{
    int close_it= force;
    if ( server->server_name == NULL)  {
	server->server_name= TIMEOUT;
	close_it= 1;
	num_servers_timed_out++;
    }
    else if ( server->type == QW_SERVER || (server->type & MASTER_SERVER) ||
		server->type == Q2_SERVER || server->type == HW_SERVER ||
		server->type == UN_SERVER)
	close_it= 1;
    else if ( server->next_rule == NO_SERVER_RULES &&
		server->next_player_info >= server->num_players)
	close_it= 1;

    if ( close_it)  {
	if ( server->fd != -1)  {
	    close( server->fd);
#ifdef unix
	    connmap[server->fd]= NULL;
#endif
	    server->fd= -1;
	    connected--;
	}
	if ( server->server_name != TIMEOUT)  {
	    num_servers_returned++;
	    if ( server->server_name != DOWN)
		num_players_total+= server->num_players;
	}
	if ( server->server_name == TIMEOUT || server->server_name == DOWN)
	    server->ping_total= 999999;
	if ( ! sort)
	    display_server( server);
	if ( server->type & MASTER_SERVER)  {
	    waiting_for_masters--;
	    if ( waiting_for_masters == 0)  {
		retry_interval= save_retry_interval;
		add_servers_from_masters();
	    }
	}
    }
}

void
get_next_timeout( struct timeval *timeout)
{
    struct qserver *server= servers;
    struct timeval now;
    int diff1, diff2, diff, smallest= retry_interval;
    gettimeofday( &now, NULL);
    for ( ; server != NULL; server= server->next)  {
	if ( server->fd == -1)
	    continue;
	diff2= 0xffff;
	diff1= 0xffff;
	if ( server->server_name == NULL)
	    diff1= retry_interval*(n_retries-server->retry1+1) -
		time_delta( &now, &server->packet_time1);
	else  {
	    if ( server->next_rule != NULL)
		diff1= retry_interval*(n_retries-server->retry1+1) -
			time_delta( &now, &server->packet_time1);
	    if ( server->next_player_info < server->num_players)
		diff2= retry_interval*(n_retries-server->retry2+1) -
			time_delta( &now, &server->packet_time2);
	}
	diff= (diff1<diff2)?diff1:diff2;
	if ( diff < smallest)
	    smallest= diff;
    }
    if ( smallest < 10)
	smallest= 10;
    timeout->tv_sec= smallest / 1000;
    timeout->tv_usec= (smallest % 1000) * 1000;
}

int
set_fds( fd_set *fds)
{
    struct qserver *server= servers;
    int maxfd= 1, i;
#ifdef unix
    for ( i= 3; i < max_simultaneous+10; i++)
	if ( connmap[i] != NULL)  {
	    FD_SET( connmap[i]->fd, fds);
	    maxfd= connmap[i]->fd;
	}
#endif /* unix */

#ifdef _WIN32
    while ( server != NULL)  {
	if ( server->fd != -1)  {
	    FD_SET( server->fd, fds);
	    if ( server->fd > maxfd)
		maxfd= server->fd;
	}
	server= server->next;
    }
#endif /* _WIN32 */

    return maxfd;
}


/* Functions for handling response packets
 */
void
deal_with_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    if ( server->type == Q_SERVER || server->type == H2_SERVER)
 	deal_with_q_packet( server, rawpkt, pktlen);
    else if ( server->type == QW_SERVER || server->type == Q2_SERVER ||
		server->type == HW_SERVER)
 	deal_with_qw_packet( server, rawpkt, pktlen);
    else if ( (server->type & MASTER_SERVER) && server->type != UN_MASTER)
 	deal_with_qwmaster_packet( server, rawpkt, pktlen);
    else if ( server->type == UN_SERVER)
	deal_with_unreal_packet( server, rawpkt, pktlen);
    else if ( server->type == UN_MASTER)
	deal_with_unrealmaster_packet( server, rawpkt, pktlen);
}

/* Packet from normal Quake server
 */
void
deal_with_q_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    struct qpacket *pkt= (struct qpacket *)rawpkt;
    int rc;

    if ( ntohs( pkt->length) != pktlen)  {
	fprintf( stderr, "%s Ignoring bogus packet; length %d != %d\n",
		server->arg, ntohs( pkt->length), pktlen);
	cleanup_qserver(server,FORCE);
	return;
    }

    rawpkt[pktlen]= '\0';

    switch ( pkt->op_code)  {
    case Q_CCREP_ACCEPT:
    case Q_CCREP_REJECT:
	return;
    case Q_CCREP_SERVER_INFO:
	server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time1);
	rc= server_info_packet( server, pkt, pktlen-Q_HEADER_SIZE);
	break;
    case Q_CCREP_PLAYER_INFO:
	server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time2);
	rc= player_info_packet( server, pkt, pktlen-Q_HEADER_SIZE);
	break;
    case Q_CCREP_RULE_INFO:
	server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time1);
	rc= rule_info_packet( server, pkt, pktlen-Q_HEADER_SIZE);
	break;
    case Q_CCREQ_CONNECT:
    case Q_CCREQ_SERVER_INFO:
    case Q_CCREQ_PLAYER_INFO:
    case Q_CCREQ_RULE_INFO:
    default:
	return;
    }

    if ( rc == -1)
	fprintf( stderr, "%s error on packet opcode %x\n", server->arg,
		(int)pkt->op_code);

    cleanup_qserver( server, (rc == -1) ? FORCE : 0);
}

/* Packet from QuakeWorld server
 */
void
deal_with_qw_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    char *pkt= &rawpkt[5];
    char *key, *value, *end;
    int len, rc, complete= 0;

    server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time1);

    if ( (rawpkt[0] != '\377' || rawpkt[1] != '\377' ||
	    rawpkt[2] != '\377' || rawpkt[3] != '\377') && show_errors)  {
	unsigned long ipaddr= ntohl(server->ipaddr);
	fprintf( stderr,
		"Odd packet from QW server %d.%d.%d.%d, processing ...\n",
		(ipaddr>>24)&0xff, (ipaddr>>16)&0xff,
		(ipaddr>>8)&0xff, ipaddr&0xff);
	print_packet( rawpkt, pktlen);
    }

    if ( rawpkt[4] == 'n')  {
	server->type= QW_SERVER;
	deal_with_q1qw_packet( server, rawpkt, pktlen);
	return;
    }
    else if ( rawpkt[4] == '\377' && rawpkt[5] == 'n')  {
	server->type= HW_SERVER;
	deal_with_q1qw_packet( server, rawpkt, pktlen);
	return;
    }
    else if ( strncmp( &rawpkt[4], "print\n\\", 7) == 0)  {
	server->type= Q2_SERVER;
	rawpkt[pktlen]= '\0';
	deal_with_q2_packet( server, rawpkt+10, pktlen-10);
	return;
    }
    else if ( strncmp( &rawpkt[4], "print\n", 6) == 0)  {
	/* work-around for occasional bug in Quake II status packets
	*/
	char *c, *p;
	rawpkt[pktlen]= '\0';
	p= &rawpkt[10];
	while ( *p != '\\' && (c= strchr( p, '\n')))
	    p= c+1;
	if ( *p == '\\' && c != NULL)  {
	    server->type= Q2_SERVER;
	    deal_with_q2_packet( server, p, pktlen-(p-rawpkt));
	    return;
	}
    }

    if ( show_errors)  {
	unsigned long ipaddr= ntohl(server->ipaddr);
	fprintf( stderr,
		"Odd packet from QW server %d.%d.%d.%d, skipping ...\n",
		(ipaddr>>24)&0xff, (ipaddr>>16)&0xff,
		(ipaddr>>8)&0xff, ipaddr&0xff);
	print_packet( rawpkt, pktlen);
    }

    cleanup_qserver( server, 0);
}

void
deal_with_q1qw_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    char *key, *value, *end;
    struct rule *rule;
    struct player *player;
    int len, rc, complete= 0;
    int number, frags, connect_time, ping;
    char *pkt= &rawpkt[5];

    if ( server->type == HW_SERVER)
	pkt= &rawpkt[6];

    while ( *pkt && pkt-rawpkt < pktlen)  {
	if ( *pkt == '\\')  {
	    pkt++;
	    end= strchr( pkt, '\\');
	    if ( end == NULL)
		break;
	    *end= '\0';
	    key= pkt;
	    pkt+= strlen(pkt)+1;
	    end= strchr( pkt, '\\');
	    if ( end == NULL)
		end= strchr( pkt, '\n');
	    value= (char*) malloc(end-pkt+1);
	    memcpy( value, pkt, end-pkt);
	    value[end-pkt]= '\0';
	    pkt= end;
	    if ( strcmp( key, "hostname") == 0)
		server->server_name= value;
	    else if  ( strcmp( key, "map") == 0)
		server->map_name= value;
	    else if  ( strcmp( key, "maxclients") == 0)  {
		server->max_players= atoi(value);
		free( value);
	    }
	    else if ( get_server_rules || strncmp( key, "*game", 5) == 0)  {
		rule= (struct rule *) malloc( sizeof( struct rule));
		rule->name= strdup(key);
		rule->value= value;
		rule->next= server->rules;
		server->rules= rule;
		server->n_rules++;
		if ( strcmp( key, "*gamedir") == 0)
		    server->game= value;
	    }
	}
	else if ( *pkt == '\n')  {
	    pkt++;
	    if ( *pkt == '\0')
		break;
	    rc= sscanf( pkt, "%d %d %d %d %n", &number, &frags, &connect_time,
		&ping, &len);
	    if ( rc != 4)  {
		char *nl;	/* assume it's an error packet */
		server->error= malloc( pktlen+1);
	        nl= strchr( pkt, '\n');
		if ( nl != NULL)
		    strncpy( server->error, pkt, nl-pkt);
		else
		    strcpy( server->error, pkt);
		server->server_name= SERVERERROR;
		complete= 1;
		break;
	    }
	    if ( get_player_info)  {
		player= (struct player *) calloc( 1, sizeof( struct player));
		player->number= number;
		player->frags= frags;
		player->connect_time= connect_time * 60;
		player->ping= ping;
	    }
	    else
		player= NULL;

	    pkt+= len;

	    if ( *pkt != '"') break;
	    pkt++;
	    end= strchr( pkt, '"');
	    if ( end == NULL) break;
	    if ( player != NULL)  {
		player->name= (char*) malloc(end-pkt+1);
		memcpy( player->name, pkt, end-pkt);
		player->name[end-pkt]= '\0';
	    }
	    pkt= end+2;

	    if ( *pkt != '"') break;
	    pkt++;
	    end= strchr( pkt, '"');
	    if ( end == NULL) break;
	    if ( player != NULL)  {
		player->skin= (char*) malloc(end-pkt+1);
		memcpy( player->skin, pkt, end-pkt);
		player->skin[end-pkt]= '\0';
	    }
	    pkt= end+2;

	    if ( player != NULL)  {
		sscanf( pkt, "%d %d%n", &player->shirt_color,
			&player->pants_color, &len);
		player->next= server->players;
		server->players= player;
	    }
	    else
		sscanf( pkt, "%*d %*d%n", &len);
	    pkt+= len;

	    server->num_players++;
	}
	else
	    pkt++;
	complete= 1;
    }

    if ( !complete)
	print_packet( rawpkt, pktlen);

    cleanup_qserver( server, 0);
}

void
deal_with_q2_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    char *key, *value, *end;
    struct rule *rule;
    struct player *player;
    int len, rc, complete= 0;
    int frags, ping;
    char *pkt= rawpkt;

/*	print_packet( rawpkt, pktlen);
*/

    while ( *pkt && pkt-rawpkt < pktlen)  {
	if ( *pkt == '\\')  {
	    pkt++;
	    end= strchr( pkt, '\\');
	    if ( end == NULL)
		break;
	    *end= '\0';
	    key= pkt;
	    pkt+= strlen(pkt)+1;
	    end= strchr( pkt, '\\');
	    if ( end == NULL)
		end= strchr( pkt, '\n');
	    value= (char*) malloc(end-pkt+1);
	    memcpy( value, pkt, end-pkt);
	    value[end-pkt]= '\0';
	    pkt= end;
	    if ( strcmp( key, "hostname") == 0)
		server->server_name= value;
	    else if  ( strcmp( key, "mapname") == 0 ||
		    (strcmp( key, "map") == 0 && server->map_name == NULL))  {
		if ( server->map_name != NULL)
		    free( server->map_name);
		server->map_name= value;
	    }
	    else if  ( strcmp( key, "maxclients") == 0)  {
		server->max_players= atoi(value);
		free(value);
	    }
	    else if ( get_server_rules || strncmp( key, "game", 4) == 0)  {
		rule= (struct rule *) malloc( sizeof( struct rule));
		rule->name= strdup(key);
		rule->value= value;
		rule->next= server->rules;
		server->rules= rule;
		server->n_rules++;
		if ( strcmp( key, "game") == 0)
		    server->game= value;
	    }
	}
	else if ( *pkt == '\n')  {
	    pkt++;
	    if ( *pkt == '\0')
		break;
	    rc= sscanf( pkt, "%d %d %n", &frags, &ping, &len);
	    if ( rc != 2)  {
		char *nl;	/* assume it's an error packet */
		free( player);
		server->error= malloc( pktlen+1);
	        nl= strchr( pkt, '\n');
		if ( nl != NULL)
		    strncpy( server->error, pkt, nl-pkt);
		else
		    strcpy( server->error, pkt);
		server->server_name= SERVERERROR;
		complete= 1;
		break;
	    }
	    if ( get_player_info)  {
		player= (struct player *) calloc( 1, sizeof( struct player));
		player->number= -1;
		player->connect_time= -1;
		player->frags= frags;
		player->ping= ping;
	    }
	    else
		player= NULL;

	    pkt+= len;

	    if ( *pkt != '"') break;
	    pkt++;
	    end= strchr( pkt, '"');
	    if ( end == NULL) break;
	    if ( player != NULL)  {
		player->name= (char*) malloc(end-pkt+1);
		memcpy( player->name, pkt, end-pkt);
		player->name[end-pkt]= '\0';
	    }
	    pkt= end+1;

	    if ( player != NULL)  {
		player->skin= NULL;
		player->shirt_color= -1;
		player->pants_color= -1;
		player->next= server->players;
		server->players= player;
	    }
	    server->num_players++;
	}
	else
	    pkt++;
	complete= 1;
    }

    if ( !complete)
	print_packet( rawpkt, pktlen);

    cleanup_qserver( server, 0);
}

/* Packet from QuakeWorld master server
 */
void
deal_with_qwmaster_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    unsigned long ipaddr;
    unsigned short port;
    int i;

    server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time1);

    if ( rawpkt[0] == QW_NACK)  {
	server->error= strdup( &rawpkt[2]);
	server->server_name= SERVERERROR;
	cleanup_qserver( server, 1);
	return;
    }

    if ( *((int*)rawpkt) == -1)  {
	rawpkt+= 4;	/* QW 1.5 */
	pktlen-= 4;
    }

    if ( rawpkt[0] == QW_SERVERS && rawpkt[1] == QW_NEWLINE)  {
	rawpkt+= 2;
	pktlen-= 2;
    }
    else if ( strncmp( rawpkt, "servers", 7) == 0)  {
	rawpkt+= 8;
	pktlen-= 8;
    }
    else if ( show_errors)  {
	unsigned long ipaddr= ntohl(server->ipaddr);
	fprintf( stderr,
		"Odd packet from QW master %d.%d.%d.%d, processing ...\n",
		(ipaddr>>24)&0xff, (ipaddr>>16)&0xff,
		(ipaddr>>8)&0xff, ipaddr&0xff);
	print_packet( rawpkt, pktlen);
    }

    server->master_pkt= (char*)malloc( pktlen+1);
    rawpkt[pktlen]= '\0';
    memcpy( server->master_pkt, rawpkt, pktlen+1);
    server->master_pkt_len= pktlen;
    server->n_servers= server->master_pkt_len / 6;
    server->server_name= MASTER;

    cleanup_qserver( server, 0);
    bind_sockets();
}


int
server_info_packet( struct qserver *server, struct qpacket *pkt, int datalen)
{
    int off= 0;

    /* ignore duplicate packets */
    if ( server->server_name != NULL)
	return 0;

    server->address= strdup((char*)&pkt->data[off]);
    off+= strlen(server->address) + 1;
    if ( off >= datalen)
	return -1;

    server->server_name= strdup((char*)&pkt->data[off]);
    off+= strlen(server->server_name) + 1;
    if ( off >= datalen)
	return -1;

    server->map_name= strdup((char*)&pkt->data[off]);
    off+= strlen(server->map_name) + 1;
    if ( off > datalen)
	return -1;

    server->num_players= pkt->data[off++];
    server->max_players= pkt->data[off++];
    server->protocol_version= pkt->data[off++];

    server->retry1= n_retries;

    if ( get_server_rules)
	send_rule_request_packet( server);
    if ( get_player_info)
	send_player_request_packet( server);

    return 0;
}

int
player_info_packet( struct qserver *server, struct qpacket *pkt, int datalen)
{
    char *name, *address;
    int off, colors, frags, connect_time, player_number;
    struct player *player, *last;

    off= 0;
    player_number= pkt->data[off++];
    name= (char*) &pkt->data[off];
    off+= strlen(name)+1;
    if ( off >= datalen)
	return -1;

    colors= pkt->data[off+3];
    colors= (colors<<8) + pkt->data[off+2];
    colors= (colors<<8) + pkt->data[off+1];
    colors= (colors<<8) + pkt->data[off];
    off+= sizeof(colors);

    frags= pkt->data[off+3];
    frags= (frags<<8) + pkt->data[off+2];
    frags= (frags<<8) + pkt->data[off+1];
    frags= (frags<<8) + pkt->data[off];
    off+= sizeof(frags);

    connect_time= pkt->data[off+3];
    connect_time= (connect_time<<8) + pkt->data[off+2];
    connect_time= (connect_time<<8) + pkt->data[off+1];
    connect_time= (connect_time<<8) + pkt->data[off];
    off+= sizeof(connect_time);

    address= (char*) &pkt->data[off];
    off+= strlen(address)+1;
    if ( off > datalen)
	return -1;

    last= server->players;
    while ( last != NULL && last->next != NULL)  {
	if ( last->number == player_number)
	     return 0;
	last= last->next;
    }
    if ( last != NULL && last->number == player_number)
	return 0;

    player= (struct player *) malloc( sizeof(struct player));
    player->number= player_number;
    player->name= strdup( name);
    player->address= strdup( address);
    player->connect_time= connect_time;
    player->frags= frags;
    player->shirt_color= colors>>4;
    player->pants_color= colors&0xf;
    player->next= NULL;

    if ( last == NULL)
	server->players= player;
    else
	last->next= player;

    server->next_player_info++;
    server->retry2= n_retries;
    if ( server->next_player_info < server->num_players)
	send_player_request_packet( server);

    return 0;
}

int
rule_info_packet( struct qserver *server, struct qpacket *pkt, int datalen)
{
    int off= 0;
    struct rule *rule, *last;
    char *name, *value;

    /* Straggler packet after we've already given up fetching rules */
    if ( server->next_rule == NULL)
	return 0;

    if ( ntohs(pkt->length) == Q_HEADER_SIZE)  {
	server->next_rule= NULL;
	return 0;
    }

    name= (char*)&pkt->data[off];
    off+= strlen( name)+1;
    if ( off >= datalen)
	return -1;

    value= (char*)&pkt->data[off];
    off+= strlen( value)+1;
    if ( off > datalen)
	return -1;

    last= server->rules;
    while ( last != NULL && last->next != NULL)  {
	if ( strcmp( last->name, name) == 0)
	     return 0;
	last= last->next;
    }
    if ( last != NULL && strcmp( last->name, name) == 0)
	return 0;

    rule= (struct rule *) malloc( sizeof( struct rule));
    rule->name= strdup( name);
    rule->value= strdup( value);
    rule->next= NULL;

    if ( last == NULL)
	server->rules= rule;
    else
	last->next= rule;

    server->n_rules++;
    server->next_rule= rule->name;
    server->retry1= n_retries;
    send_rule_request_packet( server);

    return 0;
}

void
add_rule( struct qserver *server, char *key, char *value) 
{
    struct rule *rule;
    for ( rule= server->rules; rule; rule= rule->next)
	if ( strcmp( rule->name, key) == 0)
	    return;

    rule= (struct rule *) malloc( sizeof( struct rule));
    rule->name= strdup(key);
    rule->value= strdup(value);
    rule->next= server->rules;
    server->rules= rule;
    server->n_rules++;
}

struct player *
add_player( struct qserver *server, int player_number)
{
    struct player *player;
    for ( player= server->players; player; player= player->next)
	if ( player->number == player_number)
	    return NULL;

    player= (struct player *) calloc( 1, sizeof( struct player));
    player->number= player_number;
    player->next= server->players;
    server->players= player;
    return player;
}

int
deal_with_unreal_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    char *term= NULL, *s, *key, *value;
    struct player *player= NULL;
    int i, id_major, id_minor, final=0;

    if ( server->server_name == NULL)
	server->ping_total+= time_delta( &packet_recv_time,
		&server->packet_time1);
    else
	gettimeofday( &server->packet_time1, NULL);

    rawpkt[pktlen]= '\0';

    s= rawpkt;
    while ( *s)  {
	while ( *s == '\\') s++;
	if ( !*s) break;
	key= s;
	while ( *s && *s != '\\') s++;
	if ( !*s) break;
	*s++= '\0';
	while ( *s == '\\') s++;
	if ( !*s) break;
	value= s;
	while ( *s && *s != '\\') s++;
	if ( *s)
	    *s++= '\0';

	if ( strcmp( key, "mapname") == 0 && !server->map_name)
	    server->map_name= strdup( value);
	else if ( strcmp( key, "hostname") == 0 && !server->server_name)
	    server->server_name= strdup( value);
	else if ( strcmp( key, "maxplayers") == 0)
	    server->max_players= atoi( value);
	else if ( strcmp( key, "numplayers") == 0)
	    server->num_players= atoi( value);
	else if ( strcmp( key, "queryid") == 0)
	    sscanf( value, "%d.%d", &id_major, &id_minor);
	else if ( strcmp( key, "final") == 0)  {
	    final= 1;
	    break;
	}
	else if ( strncmp( key, "player_", 7) == 0)  {
	    player= add_player( server, atoi(key+7));
	    if ( player) 
		player->name= strdup( value);
	}
	else if ( player && strncmp( key, "frags_", 6) == 0)
	    player->frags= atoi( value);
	else if ( player && strncmp( key, "team_", 5) == 0)
	    player->team= atoi( value);
	else if ( player && strncmp( key, "skin_", 5) == 0)
	    player->skin= strdup( value);
	else if ( player && strncmp( key, "mesh_", 5) == 0)
	    player->mesh= strdup( value);
	else if ( !player)
	    add_rule( server, key, value);
    }

    if ( final)
	cleanup_qserver( server, 1);
    if ( server->num_players < 0 && id_minor >= 3)
	cleanup_qserver( server, 1);

    return 0;
}

int
deal_with_unrealmaster_packet( struct qserver *server, char *rawpkt, int pktlen)
{
    if ( pktlen == 0)  {
	cleanup_qserver( server, 1);
	return 0;
    }
    print_packet( rawpkt, pktlen);
puts( "--");
    return 0;
}

/* Misc utility functions
 */

char *
escape( unsigned char *string)
{
    static char _q[256];

    unsigned char *s= string;
    char *q= _q;

    if ( string == NULL)  {
	q[0]= '?';
	q[1]= '\0';
	return _q;
    }

    if ( hex_player_names)  {
	for ( ; *string; string++, q+= 2)
	    sprintf( q, "%02x", *string);
	*q= '\0';
	return _q;
    }

    for ( ; *string; string++)  {
	if ( isprint(*string))  {
	    *q++= *string;
	    continue;
	}

	if ( *string >= 0xa0)
	    *q++= *string & 0x7f;
	else if ( *string >= 0x92 && *string < 0x9c)
	    *q++= '0' + (*string - 0x92);
	else if ( *string >= 0x12 && *string < 0x1c)
	    *q++= '0' + (*string - 0x12);
	else if ( *string == 0x90 || *string == 0x10)
	    *q++= '[';
	else if ( *string == 0x91 || *string == 0x11)
	    *q++= ']';
	else if ( *string == 0xa || *string == 0xc || *string == 0xd)
	    *q++= ']';
    }
    *q= '\0';
    return _q;
}

int
is_default_rule( struct rule *rule)
{
    if ( strcmp( rule->name, "sv_maxspeed") == 0)
	return strcmp( rule->value, Q_DEFAULT_SV_MAXSPEED) == 0;
    if ( strcmp( rule->name, "sv_friction") == 0)
	return strcmp( rule->value, Q_DEFAULT_SV_FRICTION) == 0;
    if ( strcmp( rule->name, "sv_gravity") == 0)
	return strcmp( rule->value, Q_DEFAULT_SV_GRAVITY) == 0;
    if ( strcmp( rule->name, "noexit") == 0)
	return strcmp( rule->value, Q_DEFAULT_NOEXIT) == 0;
    if ( strcmp( rule->name, "teamplay") == 0)
	return strcmp( rule->value, Q_DEFAULT_TEAMPLAY) == 0;
    if ( strcmp( rule->name, "timelimit") == 0)
	return strcmp( rule->value, Q_DEFAULT_TIMELIMIT) == 0;
    if ( strcmp( rule->name, "fraglimit") == 0)
	return strcmp( rule->value, Q_DEFAULT_FRAGLIMIT) == 0;
    return 0;
}

char *
strherror( int h_err)
{
    static char msg[100];
    switch (h_err)  {
    case HOST_NOT_FOUND:	return "host not found";
    case TRY_AGAIN:		return "try again";
    case NO_RECOVERY:		return "no recovery";
    case NO_ADDRESS:		return "no address";
    default:	sprintf( msg, "%d", h_err); return msg;
    }
}

int
time_delta( struct timeval *later, struct timeval *past)
{
    if ( later->tv_usec < past->tv_usec)  {
	later->tv_sec--;
	later->tv_usec+= 1000000;
    }
    return (later->tv_sec - past->tv_sec) * 1000 +
	(later->tv_usec - past->tv_usec) / 1000;
}

int
connection_refused()
{
#ifdef unix
    return errno == ECONNREFUSED;
#endif

#ifdef _WIN32
    return WSAGetLastError() == WSAECONNABORTED;
#endif
}

void
set_non_blocking( int fd)
{
    int one= 1;
#ifdef unix
#ifdef O_NONBLOCK
    fcntl( fd, F_SETFL, O_NONBLOCK);
#else
    fcntl( fd, F_SETFL, O_NDELAY);
#endif
#endif

#ifdef _WIN32
    ioctlsocket( fd, FIONBIO, (unsigned long*)&one);
#endif
}

void
print_packet( char *buf, int buflen)
{
    int i;
    for ( i= 0; i < buflen; i++)  {
	if ( buf[i] == ' ') fprintf( stderr, " 20 ");
	else if ( isprint( buf[i])) fprintf( stderr, "%c", buf[i]);
	else fprintf( stderr, " %02x ", (unsigned char)buf[i]);
    }
    fprintf(stderr,"\n");
}

char *
quake_color( int color)
{
    static char *colors[] = {
	"White",	/* 0 */
	"Brown",	/* 1 */
	"Lavender",	/* 2 */
	"Khaki",	/* 3 */
	"Red",		/* 4 */
	"Lt Brown",	/* 5 */
	"Peach",	/* 6 */
	"Lt Peach",	/* 7 */
	"Purple",	/* 8 */
	"Dk Purple",	/* 9 */
	"Tan",		/* 10 */
	"Green",	/* 11 */
	"Yellow",	/* 12 */
	"Blue",		/* 13 */
	"Blue",		/* 14 */
	"Blue"		/* 15 */
    };

    if ( color_names)
        return colors[color&0xf];
    else
	return (char*)color;
}

char *
play_time( int seconds, int show_seconds)
{
    static char time_string[24];
    if ( time_format == CLOCK_TIME)  {
	time_string[0]= '\0';
	if ( seconds/3600)
	    sprintf( time_string, "%2dh", seconds/3600);
	else
	    strcat( time_string, "   ");
	if ( (seconds%3600)/60 || seconds/3600)
	    sprintf( time_string+strlen(time_string), "%2dm",
		(seconds%3600)/60);
	else if ( ! show_seconds)
	    sprintf( time_string+strlen(time_string), " 0m");
	else
	    strcat( time_string, "   ");
	if ( show_seconds)
	    sprintf( time_string+strlen(time_string), "%2ds", seconds%60);
    }
    else if ( time_format == STOPWATCH_TIME)  {
	if ( show_seconds)
	    sprintf( time_string, "%02d:%02d:%02d", seconds/3600,
		(seconds%3600)/60, seconds % 60);
	else
	    sprintf( time_string, "%02d:%02d", seconds/3600,
		(seconds%3600)/60);
    }
    else
	sprintf( time_string, "%d", seconds);

    return time_string;
}

char *
ping_time( int ms)
{
    static char time_string[24];
    if ( ms < 1000)
	sprintf( time_string, "%dms", ms);
    else if ( ms < 1000000)
	sprintf( time_string, "%ds", ms/1000);
    else
	sprintf( time_string, "%dm", ms/1000/60);
    return time_string;
}

int
count_bits( int n)
{
    int b= 0;
    for ( ; n; n>>=1)
	if ( n&1)
	    b++;
    return b;
}

int
set_default_server_type( char *type)
{
    char *t= type;
    while ( *t) *t++= toupper( *t);

    if ( strcmp( type, "QS") == 0)  {
	default_server_type= Q_SERVER;
	default_server_port= DEFAULT_PORT;
    }
    else if ( strcmp( type, "Q2") == 0)  {
	default_server_type= Q2_SERVER;
	default_server_port= QUAKE2_DEFAULT_PORT;
    }
    else if ( strcmp( type, "QW") == 0 || strcmp( type, "QWS") == 0)  {
	default_server_type= QW_SERVER;
	default_server_port= QW_DEFAULT_PORT;
    }
    else if ( strcmp( type, "QWM") == 0)  {
	default_server_type= QW_MASTER;
	default_server_port= QW_MASTER_DEFAULT_PORT;
    }
    else if ( strcmp( type, "H2S") == 0)  {
	default_server_type= H2_SERVER;
	default_server_port= HEXEN2_DEFAULT_PORT;
    }
    else if ( strcmp( type, "HWS") == 0)  {
	default_server_type= HW_SERVER;
	default_server_port= HW_DEFAULT_PORT;
    }
    else if ( strcmp( type, "Q2M") == 0)  {
	default_server_type= Q2_MASTER;
	default_server_port= Q2_MASTER_DEFAULT_PORT;
    }
    else if ( strcmp( type, "UNS") == 0)  {
	default_server_type= UN_SERVER;
	default_server_port= UNREAL_DEFAULT_PORT;
    }
    else  {
	fprintf( stderr, "unknown server type \"%s\"\n", type);
	return -1;
    }
    return 0;
}


/*
 * Quick sort
 */
void quicksort( struct qserver **foo, int i, int j);
int  qpartition( struct qserver *foo[],int i,int j);

void
qstatsort(struct qserver **array, int size)
{
    quicksort( array, 0, size-1);
}

void
quicksort( struct qserver **array, int i, int j)
{
    int q= 0;
  
    if ( i < j) {
	q = qpartition(array,i,j);
	quicksort(array,i,q);
	quicksort(array,q+1,j);
    }
}


int
server_compare( struct qserver *one, struct qserver *two)
{
    int rc;

    if ( first_sort_key == SORT_GAME)
	if ( (rc= strcmp_withnull( one->game, two->game)) == 0)
	    return one->ping_total/one->n_requests -
			two->ping_total/two->n_requests;
	else
	    return rc;

    if ( first_sort_key == SORT_PING)
	if ( one->ping_total/one->n_requests == two->ping_total/two->n_requests)
	    return strcmp_withnull( one->game, two->game);
	else
	    return one->ping_total/one->n_requests -
			two->ping_total/two->n_requests;
    return 0;
}

int
qpartition(struct qserver **array, int a, int b)
{
    /* this is our comparison point. when we are done
       splitting this array into 2 parts, we want all the
       elements on the left side to be less then or equal
       to this, all the elements on the right side need to
       be greater then or equal to this
    */
    struct qserver *z;

    /* indicies into the array to sort. Used to calculate a partition
       point
    */
    int i = a-1;
    int j = b+1;

    /* temp pointer used to swap two array elements */
    struct qserver * tmp = NULL;

    z = array[a];

    while (1) {

        /* move the right indice over until the value of that array
           elem is less then or equal to z. Stop if we hit the left
           side of the array (ie, j == a);
        */
	do {
	    j--;
	} while( j > a && server_compare(array[j],z) > 0);

        /* move the left indice over until the value of that
           array elem is greater then or equal to z, or until
           we hit the right side of the array (ie i == j)
        */
	do {
	    i++;
	} while( i <= j && server_compare(array[i],z) < 0);

        /* if i is less then j, we need to switch those two array
           elements, if not then we are done partitioning this array
           section
        */
	if(i < j) {
	    tmp = array[i];
	    array[i] = array[j];
	    array[j] = tmp;
	}
	else
	    return j;
    }
}

int
strcmp_withnull( char *one, char *two)
{
    if ( one == NULL && two == NULL)
	return 0;
    if ( one != NULL && two == NULL)
	return -1;
    if ( one == NULL)
	return 1;
    return strcasecmp( one, two);
}

/* ------------------------------------ */
/* ------------------------------------ */



int
sendenc(int s, const void *msg, int len, unsigned int flags)
{
    char *encstr;
    int ret, enclen;
    encstr = (char *)malloc(len+1);
    HuffEncode((unsigned char *)msg, (unsigned char *)encstr, len, &enclen);
    ret = send(s, encstr, enclen, flags);
    free(encstr);
    return ret;
}       

int
recvdec(int s, void *buf, int len, unsigned int flags)
{
    char *decstr;
    int ret, declen;
    decstr = (char *)malloc(len*20);
    ret = recv(s, decstr, len, flags);
    HuffDecode((unsigned char *)decstr, (unsigned char *)buf, ret, &declen);
    ret = declen;
    free(decstr);
    return ret;
}



int LastCompMessageSize = 0;

typedef struct huffnode_s
{
	struct huffnode_s *zero;
	struct huffnode_s *one;
	unsigned char val;
	float freq;
} huffnode_t;

typedef struct
{
	unsigned int bits;
	int len;
} hufftab_t;
static huffnode_t *HuffTree=0;
static hufftab_t HuffLookup[256];
#if _DEBUG
static int HuffIn=0;
static int HuffOut=0;
#endif

float HuffFreq[256]=
{
 0.14473691,
 0.01147017,
 0.00167522,
 0.03831121,
 0.00356579,
 0.03811315,
 0.00178254,
 0.00199644,
 0.00183511,
 0.00225716,
 0.00211240,
 0.00308829,
 0.00172852,
 0.00186608,
 0.00215921,
 0.00168891,
 0.00168603,
 0.00218586,
 0.00284414,
 0.00161833,
 0.00196043,
 0.00151029,
 0.00173932,
 0.00218370,
 0.00934121,
 0.00220530,
 0.00381211,
 0.00185456,
 0.00194675,
 0.00161977,
 0.00186680,
 0.00182071,
 0.06421956,
 0.00537786,
 0.00514019,
 0.00487155,
 0.00493925,
 0.00503143,
 0.00514019,
 0.00453520,
 0.00454241,
 0.00485642,
 0.00422407,
 0.00593387,
 0.00458130,
 0.00343687,
 0.00342823,
 0.00531592,
 0.00324890,
 0.00333388,
 0.00308613,
 0.00293776,
 0.00258918,
 0.00259278,
 0.00377105,
 0.00267488,
 0.00227516,
 0.00415997,
 0.00248763,
 0.00301555,
 0.00220962,
 0.00206990,
 0.00270369,
 0.00231694,
 0.00273826,
 0.00450928,
 0.00384380,
 0.00504728,
 0.00221251,
 0.00376961,
 0.00232990,
 0.00312574,
 0.00291688,
 0.00280236,
 0.00252436,
 0.00229461,
 0.00294353,
 0.00241201,
 0.00366590,
 0.00199860,
 0.00257838,
 0.00225860,
 0.00260646,
 0.00187256,
 0.00266552,
 0.00242641,
 0.00219450,
 0.00192082,
 0.00182071,
 0.02185930,
 0.00157439,
 0.00164353,
 0.00161401,
 0.00187544,
 0.00186248,
 0.03338637,
 0.00186968,
 0.00172132,
 0.00148509,
 0.00177749,
 0.00144620,
 0.00192442,
 0.00169683,
 0.00209439,
 0.00209439,
 0.00259062,
 0.00194531,
 0.00182359,
 0.00159096,
 0.00145196,
 0.00128199,
 0.00158376,
 0.00171412,
 0.00243433,
 0.00345704,
 0.00156359,
 0.00145700,
 0.00157007,
 0.00232342,
 0.00154198,
 0.00140730,
 0.00288807,
 0.00152830,
 0.00151246,
 0.00250203,
 0.00224420,
 0.00161761,
 0.00714383,
 0.08188576,
 0.00802537,
 0.00119484,
 0.00123805,
 0.05632671,
 0.00305156,
 0.00105584,
 0.00105368,
 0.00099246,
 0.00090459,
 0.00109473,
 0.00115379,
 0.00261223,
 0.00105656,
 0.00124381,
 0.00100326,
 0.00127550,
 0.00089739,
 0.00162481,
 0.00100830,
 0.00097229,
 0.00078864,
 0.00107240,
 0.00084409,
 0.00265760,
 0.00116891,
 0.00073102,
 0.00075695,
 0.00093916,
 0.00106880,
 0.00086786,
 0.00185600,
 0.00608367,
 0.00133600,
 0.00075695,
 0.00122077,
 0.00566955,
 0.00108249,
 0.00259638,
 0.00077063,
 0.00166586,
 0.00090387,
 0.00087074,
 0.00084914,
 0.00130935,
 0.00162409,
 0.00085922,
 0.00093340,
 0.00093844,
 0.00087722,
 0.00108249,
 0.00098598,
 0.00095933,
 0.00427593,
 0.00496661,
 0.00102775,
 0.00159312,
 0.00118404,
 0.00114947,
 0.00104936,
 0.00154342,
 0.00140082,
 0.00115883,
 0.00110769,
 0.00161112,
 0.00169107,
 0.00107816,
 0.00142747,
 0.00279804,
 0.00085922,
 0.00116315,
 0.00119484,
 0.00128559,
 0.00146204,
 0.00130215,
 0.00101551,
 0.00091756,
 0.00161184,
 0.00236375,
 0.00131872,
 0.00214120,
 0.00088875,
 0.00138570,
 0.00211960,
 0.00094060,
 0.00088083,
 0.00094564,
 0.00090243,
 0.00106160,
 0.00088659,
 0.00114514,
	0.00095861,
	0.00108753,
	0.00124165,
	0.00427016,
	0.00159384,
	0.00170547,
	0.00104431,
	0.00091395,
	0.00095789,
	0.00134681,
	0.00095213,
	0.00105944,
	0.00094132,
	0.00141883,
	0.00102127,
	0.00101911,
	0.00082105,
	0.00158448,
	0.00102631,
	0.00087938,
	0.00139290,
	0.00114658,
	0.00095501,
	0.00161329,
	0.00126542,
	0.00113218,
	0.00123661,
	0.00101695,
	0.00112930,
	0.00317976,
	0.00085346,
	0.00101190,
	0.00189849,
	0.00105728,
	0.00186824,
	0.00092908,
	0.00160896,
};


/* ======================================================================== */

void FindTab(huffnode_t *tmp,int len,unsigned int bits)
{
	if(!tmp)
		puts("no huff node");
	if (tmp->zero)
	{
		if(!tmp->one)
			printf("no one in node");
		if(len>=32)
			printf("compression screwd");
		FindTab(tmp->zero,len+1,bits<<1);
		FindTab(tmp->one,len+1,(bits<<1)|1);
		return;
	}
	HuffLookup[tmp->val].len=len;
	HuffLookup[tmp->val].bits=bits;
	return;
}
static unsigned char Masks[8]=
{
	0x1,
	0x2,
	0x4,
	0x8,
	0x10,
	0x20,
	0x40,
	0x80
};

void PutBit(unsigned char *buf,int pos,int bit)
{
	if (bit)
		buf[pos/8]|=Masks[pos%8];
	else
		buf[pos/8]&=~Masks[pos%8];
}

int GetBit(unsigned char *buf,int pos)
{
	if (buf[pos/8]&Masks[pos%8])
		return 1;
	else
		return 0;
}


void BuildTree(float *freq)
{
	float min1,min2;
	int i,j,minat1,minat2;
	huffnode_t *work[256];	
	huffnode_t *tmp;	
	for (i=0;i<256;i++)
	{
		work[i]=malloc(sizeof(huffnode_t));
		work[i]->val=(unsigned char)i;
		work[i]->freq=freq[i];
		work[i]->zero=0;
		work[i]->one=0;
		HuffLookup[i].len=0;
	}
	for (i=0;i<255;i++)
	{
		minat1=-1;
		minat2=-1;
		min1=1E30;
		min2=1E30;
		for (j=0;j<256;j++)
		{
			if (!work[j])
				continue;
			if (work[j]->freq<min1)
			{
				minat2=minat1;
				min2=min1;
				minat1=j;
				min1=work[j]->freq;
			}
			else if (work[j]->freq<min2)
			{
				minat2=j;
				min2=work[j]->freq;
			}
		}
		if (minat1<0)
			printf("minatl: %d",minat1);
		if (minat2<0)
			printf("minat2: %d",minat2);
		tmp=malloc(sizeof(huffnode_t));
		tmp->zero=work[minat2];
		tmp->one=work[minat1];
		tmp->freq=work[minat2]->freq+work[minat1]->freq;
		tmp->val=0xff;
		work[minat1]=tmp;
		work[minat2]=0;
	}		
	HuffTree=tmp;
	FindTab(HuffTree,0,0);
}

void HuffDecode(unsigned char *in,unsigned char *out,int inlen,int *outlen)
{
	int bits,tbits;
	huffnode_t *tmp;	
	if (*in==0xff)
	{
		memcpy(out,in+1,inlen-1);
		*outlen=inlen-1;
		return;
	}
	tbits=(inlen-1)*8-*in;
	bits=0;
	*outlen=0;
	while (bits<tbits)
	{
		tmp=HuffTree;
		do
		{
			if (GetBit(in+1,bits))
				tmp=tmp->one;
			else
				tmp=tmp->zero;
			bits++;
		} while (tmp->zero);
		*out++=tmp->val;
		(*outlen)++;
	}
}

void HuffEncode(unsigned char *in,unsigned char *out,int inlen,int *outlen)
{
	int i,j,bitat;
	unsigned int t;
	bitat=0;
	for (i=0;i<inlen;i++)
	{
		t=HuffLookup[in[i]].bits;
		for (j=0;j<HuffLookup[in[i]].len;j++)
		{
			PutBit(out+1,bitat+HuffLookup[in[i]].len-j-1,t&1);
			t>>=1;
		}
		bitat+=HuffLookup[in[i]].len;
	}
	*outlen=1+(bitat+7)/8;
	*out=8*((*outlen)-1)-bitat;
	if(*outlen >= inlen+1)
	{
		*out=0xff;
		memcpy(out+1,in,inlen);
		*outlen=inlen+1;
	}
}

