/*
    SABRE Fighter Plane Simulator 
    Copyright (c) 1997 Dan Hammer
    Portions Donated By Antti Barck

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*****************************************************
 * pilot.h                                           *
 *****************************************************/
#ifndef __pilot_h
#define __pilot_h

#include "flight.h"
#include "weapons.h"
#include "waypoint.h"
#include "flite.h"

enum pursuit_types { lead, lag, pure };

enum turn_dirs { left_turn, right_turn };

class Control_State
{
public:
  float goal;
  float rate;
  float min;
  int flag;
  int snap_to;
  Control_State()
  {
    goal = 0.0;
    rate = 1.0;
    min = 0.05;
    snap_to = 0;
  }
};

inline void SETCONTROL(Control_State &cs,
		       float goal,
		       float rate,
		       float min = 0.1,
		       int snap_to = 0)
{
  cs.goal = goal;
  cs.rate = rate;
  cs.min = min;
  cs.snap_to = snap_to;
}

class Pilot_Params
{
public:
  int    skill_level;
  int    affiliation; 		// whose side we're on
  float  update_time;     // time interval between updates
  // Some hard-to-name stuff having to do
  // with maneuvering to shoot
  float mparam1,mparam2,mparam3,mparam4,mparam5,mparam6;
  // When to start damping for maneuvering to shoot
  float mdampx,mdampy;
  // More unameable stuff
  float mparam7,mparam8,mparam9,mparam10;
  int c_step1, c_step2, c_step3, c_step4;
  float v_acquire_range; // range for visual target acquisition  (in miles)
  Pilot_Params()
  {
    skill_level = 10;
    affiliation = 0;
    update_time = 0.1;
    mparam1 = 32.0;
    mparam2 = 32.0;
    mparam3 = 8.0;
    mparam4 = 2.0;
    mparam5 = 1.0;
    mparam6 = 6.0;
    mdampy = 2.0;
    mdampx = 4.0;
    mparam7 = mparam8 = mparam9 = mparam10 = 0.0;
    c_step1 = 1;
    c_step2 = 2;
    c_step3 = 3;
    c_step4 = 4;
    v_acquire_range = 5.0;
  }
  friend istream &operator>>(istream &, Pilot_Params &);
};

class Maneuver_State
{
public:
  int       maneuver;
  int  	    stage;
  int       done;
  int       dir;
  float data0;
  float data1;
  float data2;

  Maneuver_State()
  {
    maneuver = 0;
    stage = 0;
    dir = 0;
    data0 = data1 = data2 = 0.0;
    done = 1;
  }
};

class Tracking_Target
{
public:
  // world coords
  R_3DPoint w_position;
  // port coords
  R_3DPoint p_position;
  float sc_x, sc_y;
  float vel_scx, vel_scy;
  float distance;
  // Prev coords
  R_3DPoint prev_w_position;
  R_3DPoint prev_p_position;
  float prev_sc_x, prev_sc_y;
  float prev_distance;
  float x_rate, y_rate, d_rate;
  // Aligning point
  R_3DPoint al_w;
  R_3DPoint al_p;
  float     al_scx,al_scy;
  float     al_ang;
  float     tol_x;
  float     tol_y;

  Tracking_Target()
  {
    vel_scx = vel_scy = 0.0;
    sc_x = sc_y = 0.0;
    distance = 0.0;
    prev_sc_x = prev_sc_y = 0.0;
    prev_distance = 0.0;
    x_rate = y_rate = d_rate = 0.0;
    al_scx = al_scy = al_ang = 0.0;
    tol_x = tol_y = 0.0;
  }

  void copy(const Tracking_Target &tt)
    {
      sc_x = tt.sc_x;
      sc_y = tt.sc_y;
      vel_scx = tt.vel_scx;
      vel_scy = tt.vel_scy;
      distance = tt.distance;
      prev_sc_x = tt.prev_sc_x;
      prev_sc_y = tt.prev_sc_y;
      prev_distance = tt.prev_distance;
      x_rate = tt.x_rate;
      y_rate = tt.y_rate;
      d_rate = tt.d_rate;
      al_w   = tt.al_w;
      al_p   = tt.al_p;
      al_scx = tt.al_scx;
      al_scy = tt.al_scy;
      tol_x  = tt.tol_x;
      tol_y  = tt.tol_y;
    }
  
  Tracking_Target(const Tracking_Target &tt)
    {
      copy(tt);
    }

  Tracking_Target &operator =(const Tracking_Target &tt)
  {
    copy(tt);
    return (*this);
  }

};

class Target_Geometry
{
public:
  R_3DPoint p;      /* position in port coords */
  float angle_off;  /* difference in heading between target and you */
  float aspect;     /*  degrees from tail */
  char  l_r;			/* left or right aspect */
  float range;      /* range */
  float d_range;		/* change in range */
  float b_angle;    /* angle-of-bank necessary to align */
  R_3DPoint bp;     /* bank aligning point */
 
  Target_Geometry()
  {
    angle_off = 0.0;
    aspect = 0.0;
    range = 0.0;
    d_range = 0.0;
    b_angle = 0.0;
  }
};

class Pilot;

class PilotMessage
{
public:
  int flag;
  float time_limit;
  float elapsed_time;
  char *mssg;
  Pilot *frm;
  PilotMessage()
  {
    flag = 0;
    time_limit = 1.0;
    elapsed_time = 0.0;
    mssg = NULL;
    frm = NULL;
  }
  void update();
  int set_message(char *mss, int priority, Pilot *fr);
};

class Flight_Complex;
class Flight;
class C_3DObject_Base;
class Gun_Specs;

class Pilot
{
public:
  static Pilot   *pilots[];
  static int     npilots;
  static int     maxpilots;
  WayPoint 	*waypoint;
  Pilot_Params	*params;
  Flight*flight;
  PilotMessage   message;
  Weapon_Instance *weapons;
  Weapon_Instance *sel_weapon;
  int         sel_wpn;
  int            n_weaps;
  char           *handle;
  Maneuver_State m_state;
  Tracking_Target trk,l_trk,g_trk;
  Tracking_Target wp_trk;
  Tracking_Target *c_trk;
  Target_Geometry tg;
  R_3DPoint gun_point;
  R_3DPoint gun_point_p;
  REAL_TYPE gp_scx,gp_scy;
  float time_to_target;
  float gun_time;
  float tdwell_time;
  float tdwell_max;
  int   tdwell_flg;
  int in_range;
  int engaged;
  int new_damage;
  int flag1;
  int functioning;
  float control_step;
  float takeoff_speed;
  float timer_max;
  float timer_elapsed;
  float t;
  float max_gs;
  int new_target_flag;
  char *dbg;
  Target *target;
  Target *target_obj;
  Flight *target_flight;
  Pilot  *target_pilot;
  DVector target_vel;
  int attkr_cnt;
  char mssbuff[128];
  Control_State pitch_control;
  Control_State speed_control;
  Control_State bank_control;
  Control_State g_control;
  Control_State aoa_control;
  int ground_flag;
  float gh_height;
  Maneuver_State al_state;
  
  static float xpixel_adjust, ypixel_adjust;
  static float bank_30, bank_45;
  static float rcontrol_max;
  static float gcontrol_max;
  static float g_range;

  /* Formation flying stuff */
  Tracking_Target lead_trk;

  Pilot( Flight*flt, Pilot_Params *,
	 Weapon_Instance *, int nw, char *, Target *);

  static initPilot();
  static Pilot *getPilot(char *hndle);

  void checkout();
  void update(int);
  void pause();
  void start();
  void pitch_update();
  void speed_update();
  void bank_update();
  void g_update();
  void aoa_update();
  void shoot();
  void get_target_position();
  void get_waypoint_position();
  void world_2_port(R_3DPoint &w, R_3DPoint *lp, float *sc_x, float *sc_y, Port_3D &port);
  void set_waypoint(WayPoint *wp);
  void engage();
  void pursue();
  void jink();
  void do_pursuit();
  void do_waypoint();
  void do_maneuver();
  void do_cap();
  void do_takeoff();
  void immelman();
  void split_s();
  void barrel_roll();
  void hard_turn(int dir);
  void break_turn(int dir);
  void standard_turn(int dir, float );
  void roll_out();
  void extend();
  void takeoff();
  int ground_check();
  Pilot *threat_check();
  int stall_check();
  int damage_check();
  void get_target_geometry(Target_Geometry &tg, Flight &flt);
  void track_target();
  int acquire_target(int flg = 0);
  void set_target(Pilot *pl);
  inline void set_maneuver(int m, int d = 0)
  {
    m_state.maneuver = m;
    m_state.stage = 0;
    m_state.done = 0;
    m_state.dir = d;
  }
  inline void reset_maneuver()
  {
    m_state.maneuver = 0;
    m_state.done = 1;
  }
  int broadcast(char *mss, int priority, int freq = -1, char * = NULL);
  int brdcst(int priority, int freq, char *mss, ...);
  int align_flight(Tracking_Target &);
  void climb(float );
  void descend(float );
  void coord();
  void selectNextWeapon(int dir);
  void flyFormation();
  void setTrackingInfo(Tracking_Target &, R_3DPoint &);
  void trackToX(Tracking_Target &);
  void trackToY(Tracking_Target &);

  int  has_fuel_tanks;
  int  ft_idx;
  int  dropTanks();
  void selWeapon(int );
};
#endif



