/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>	/* FD_SETSIZE, select, kill */
#include <stdio.h>	/* printf, fprintf, fflush */
#include <string.h>	/* memchr, strchr */
#include <sys/socket.h>	/* struct in_addr, inet_aton, inet_ntoa */
#include <netinet/in.h>	/* struct in_addr, inet_aton, inet_ntoa */
#include <arpa/inet.h>	/* struct in_addr, inet_aton, inet_ntoa */
#include <netdb.h>	/* struct hostent, gethostbyaddr, gethostbyname */
#include <unistd.h>	/* read, write, close, fork, pipe, fcntl, _exit */ 
                        /* select, alarm */
#include <fcntl.h>	/* fcntl */
#include <sys/time.h>	/* FD_SETSIZE, select */
#include <signal.h>	/* kill, signal... */
#include <errno.h>      /* errno */
#include <sys/param.h>	/* MAXHOSTNAMELEN */

#include <gtk/gtk.h>

#include "xqf.h"
#include "utils.h"
#include "dns.h"


#ifndef MAXHOSTNAMELEN
# define MAXHOSTNAMELEN		64
#endif

#define RESOLVE_TIMEOUT		15
#define DNS_BUFFER_SIZE		(MAXHOSTNAMELEN * 2 + 32)


struct worker_data {
  pid_t pid;
  int 	fd;
  char	*buffer;
  char 	*ptr;
};

struct host_hash {
  int num;
  GSList **nodes;
};

static struct host_hash hosts = { 251, NULL };

static char *resolve_arg = NULL;


static int failed (const char *name) {
  fprintf (stderr, "%s() failed: %s\n", name, g_strerror (errno));
  return TRUE;
}


static int host_hash_func (const struct in_addr *ip) {
  unsigned char *ptr;

  if (!ip)
    return 0;

  ptr = (char *) &ip->s_addr;
  return (ptr[0] + (ptr[1] << 2) + (ptr[2] << 4) + (ptr[3] << 6)) % hosts.num;
}


int hosts_total (void) {
  int i;
  int total = 0;

  if (!hosts.nodes)
    return 0;

  for (i = 0; i < hosts.num; i++) {
    total += g_slist_length (hosts.nodes[i]);
  }
  return total;
}


struct host *host_add (const char *address) {
  struct in_addr addr;
  struct host *h;
  GSList *ptr;
  int node;
  int i;

  if (!address || !*address || !inet_aton (address, &addr))
    return NULL;

  node = host_hash_func (&addr);

  if (!hosts.nodes) {
    hosts.nodes = g_malloc (sizeof (GSList *) * hosts.num);
    for (i = 0; i < hosts.num; i++)
      hosts.nodes[i] = NULL;
  }
  else {
    for (ptr = hosts.nodes[node]; ptr; ptr = ptr->next) {
      h = (struct host *) ptr->data;
      if (h->ip.s_addr == addr.s_addr)
	return h;
    }
  }

  h = g_malloc (sizeof (struct host));
  h->address = g_strdup (address);
  h->name = NULL;
  h->ip = addr;
  h->ref_count = 0;

  hosts.nodes[node] = g_slist_prepend (hosts.nodes[node], h);
  return h;
}


void host_unref (struct host *h) {
  int node;

  if (!h || !hosts.nodes)
    return;

  h->ref_count--;

  if (h->ref_count <= 0) {
    node = host_hash_func (&h->ip);
    hosts.nodes[node] = g_slist_remove (hosts.nodes[node], h);

    if (h->address) g_free (h->address);
    if (h->name) g_free (h->name);
    g_free (h);
  }
}


char *dns_lookup_by_addr (char *ip) {
  struct hostent *h;
  struct in_addr addr;

  if (inet_aton (ip, &addr)) {
    h = gethostbyaddr ((char *) &addr.s_addr, sizeof (addr.s_addr), AF_INET);
    if (h)
      return g_strdup (h->h_name);
  }
  return NULL;
}


char *dns_lookup_by_name (char *name) {
  struct hostent *h;
  char *ip;

  h = gethostbyname (name);
  if (h) {
    ip = inet_ntoa ( *((struct in_addr *) h->h_addr_list[0]) );
    if (ip)
      return g_strdup (ip);
  }
  return NULL;
}


GSList *merge_host_to_resolve (GSList *hosts, struct host *h) {

  if (h) {
    if (!h->name && !g_slist_find (hosts, h)) {
      hosts = g_slist_prepend (hosts, h);
      h->ref_count++;
    }
  }

  return hosts;
}


GSList *merge_hosts_to_resolve (GSList *hosts, GSList *servers) {
  struct host *h;

  for (; servers; servers = servers->next) {
    h = ((struct server *) servers->data)->host;
    if (!h->name && !g_slist_find (hosts, h)) {
      hosts = g_slist_prepend (hosts, h);
      h->ref_count++;
    }
  }

  return hosts;
}


static void print_resolved (char *str, char *addr, char *name) {
  printf ("%s%c%s%c%s\n", (str)? str : "",   RESOLVE_DELIM,
	                  (addr)? addr : "", RESOLVE_DELIM, 
                          (name)? name : "");
  fflush (stdout);
}


static void worker_sigalrm_handler (int signum) {
  if (resolve_arg) {
    print_resolved (resolve_arg, NULL, NULL);
#ifdef DEBUG
    fprintf (stderr, "<DNS> timeout: %s\n", resolve_arg);
#endif
  }
  _exit (0);
}


static void worker_sigterm_handler (int signum) {
#ifdef DEBUG
  if (resolve_arg)
    fprintf (stderr, "<DNS> terminated: %s\n", resolve_arg);
#endif
  _exit (0);
}


void resolve_one (char *str) {
  struct hostent *h = NULL;
  struct in_addr ip;

  if (!str || !*str || strchr (str, '\n') || strchr (str, RESOLVE_DELIM))
    return;

  resolve_arg = str;

  if (!inet_aton (str, &ip)) {
    h = gethostbyname (str);
    if (!h) {
      resolve_arg = NULL;
      alarm (0);
      print_resolved (str, NULL, NULL);
      return;
    }
    ip = *((struct in_addr *) h->h_addr_list[0]);
  }

  h = gethostbyaddr ((char *) &ip.s_addr, sizeof (ip.s_addr), AF_INET);
  resolve_arg = NULL;
  alarm (0);
  print_resolved (str, inet_ntoa (ip), (h)? h->h_name : NULL);
}


static void set_nonblock (int fd) {
  int flags;

  flags = fcntl (fd, F_GETFL, 0);
  if (flags < 0 || fcntl (fd, F_SETFL, flags | O_NONBLOCK) < 0) {
    failed ("fcntl");
    return;
  }
}


static int fork_worker (struct worker_data **workers, char *str) {
  pid_t pid;
  int pipefds[2];

  if (pipe (pipefds) < 0) {
    failed ("pipe");
    return -1;
  }

  pid = fork ();
  if (pid < 0) {
    failed ("fork");
    return -1;
  }
  
  if (pid) {	/* parent */
    close (pipefds[1]);
    set_nonblock (pipefds[0]);

    workers[pipefds[0]] = g_malloc (sizeof (struct worker_data));
    workers[pipefds[0]]->pid = pid;
    workers[pipefds[0]]->fd = pipefds[0];
    workers[pipefds[0]]->buffer = g_malloc (DNS_BUFFER_SIZE);
    workers[pipefds[0]]->ptr = workers[pipefds[0]]->buffer;
    return pipefds[0];
  }
  else {	/* child */
    on_sig (SIGHUP,  _exit);
    on_sig (SIGINT,  _exit);
    on_sig (SIGQUIT, _exit);
    on_sig (SIGBUS,  _exit);
    on_sig (SIGSEGV, _exit);
    on_sig (SIGPIPE, _exit);
    on_sig (SIGTERM, worker_sigterm_handler);
    on_sig (SIGALRM, worker_sigalrm_handler);

    close (pipefds[0]);
    dup2 (pipefds[1], 1);
    close (pipefds[1]);

    alarm (RESOLVE_TIMEOUT);
    resolve_one (str);

    _exit (0);
  }
}


static void free_worker_data (struct worker_data *w) {
  if (w) {
    if (w->fd > 0)
      close (w->fd);

    if (w->buffer)
      g_free (w->buffer);

    g_free (w);
  }
}


static int worker_read_handler (struct worker_data *w) {
  char *nl;
  int res;

  if (w->ptr - w->buffer >= DNS_BUFFER_SIZE) {
    fprintf (stderr, "host name is too large\n");
    return FALSE;
  }

  res = read (w->fd, w->ptr, DNS_BUFFER_SIZE - (w->ptr - w->buffer));
  if (res < 0) {
    if (errno == EAGAIN || errno == EWOULDBLOCK)
      return TRUE;
    failed ("read");
    return FALSE;
  }

  if (res == 0) 	/* EOF */
    return FALSE;

  w->ptr += res;

  nl = memchr (w->buffer, '\n', w->ptr - w->buffer);
  if (nl) {
    write (1, w->buffer, nl - w->buffer + 1);
    return FALSE;
  }

  return TRUE;
}


struct worker_data **workers = NULL;
int nworkers = 0;


static void resolve_multiple_cleanup (void) {
  int fd;

  if (workers) {
    if (nworkers) {
      for (fd = 0; fd < FD_SETSIZE; fd++) {
	if (workers[fd]) {
	  kill (workers[fd]->pid, SIGTERM);
	  free_worker_data (workers[fd]);
	  workers[fd] = NULL;
	  nworkers--;
	}
      }
    }
    g_free (workers);
    workers = NULL;
  }
}


static void master_sigterm_handler (int signum) {
  resolve_multiple_cleanup ();
  _exit(0);
}


static void resolve_multiple (GSList *list, int maxworkers) {
  fd_set readfds;
  int fd;
  int n;

  on_sig (SIGTERM, master_sigterm_handler);

  workers = g_malloc0 (FD_SETSIZE * sizeof (struct worker_data *));

  while (list || nworkers) {
    while (list && nworkers < maxworkers) {
      fd = fork_worker (workers, (char *) list->data);
      list = list->next;
      if (fd)
	nworkers++;
    }

    FD_ZERO (&readfds);

    for (fd = 0; fd < FD_SETSIZE; fd++) {
      if (workers[fd])
	FD_SET (fd, &readfds);
    }

    do {
      n = select (FD_SETSIZE, &readfds, NULL, NULL, NULL);
    } while (n == -1 && errno == EINTR);

    if (n < 0) {
      failed ("select");
      resolve_multiple_cleanup ();
      return;
    }

    for (fd = 0; fd < FD_SETSIZE; fd++) {
      if (workers[fd] && FD_ISSET (fd, &readfds)) {
	if (!worker_read_handler (workers[fd])) {
	  free_worker_data (workers[fd]);
	  workers[fd] = NULL;
	  nworkers--;
	}
      }
    }
  }

  resolve_multiple_cleanup ();
}


void resolve (GSList *strings, int maxprocs) {
  if (strings) {
    if (strings->next == NULL)
      resolve_one ((char *) strings->data);
    else
      resolve_multiple (strings, maxprocs);
  }
}

