/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <sys/types.h>	/* getpwuid, getpwnam, readdir, dirent, waitpid */
#include <string.h>	/* strlen, strncpy, strcmp */
#include <ctype.h>	/* isspace */
#include <unistd.h>	/* getuid */
#include <pwd.h>	/* getpwuid, getpwnam */
#include <stdlib.h>	/* getenv */
#include <dirent.h>	/* readdir, dirent */
#include <signal.h>	/* sigaction, sigemptyset */
#include <sys/wait.h>	/* waitpid */

#include <gtk/gtk.h>

#include "utils.h"


#ifndef WAIT_ANY
# define WAIT_ANY	-1
#endif


char *strdup_strip (const char *str) {
  const char *start;
  const char *end;
  char *res;

  if (!str)
    return NULL;

  for (start = str; *start && isspace (*start); start++);

  for (end = str + strlen (str) - 1; end >= start && isspace (*end); end--);

  if (start > end)
    return NULL;

  res = g_malloc (end - start + 1 + 1);
  strncpy (res, start, end - start + 1);
  res[end - start + 1] = '\0';

  return res;
}


char *file_in_dir (const char *dir, const char *file) {
  char *res, *tmp;
  int need_slash = 0;

  if (!dir || dir[0] == '\0')			/* dir "" is current dir */
    return (file)? g_strdup (file) : NULL;

  if (!file)
    return g_strdup (dir);

  if (dir[strlen (dir) - 1] != '/')
    need_slash = 1;

  tmp = res = g_malloc (strlen (dir) + strlen (file) + need_slash + 1);

  strcpy (tmp, dir);
  tmp += strlen (dir);
  if (need_slash)
    *tmp++ = '/';
  strcpy (tmp, file);

  return res;
}


int str_isempty (const char *str) {
  if (!str)
    return TRUE;

  for (; *str; str++) 
    if (!isspace (*str))
      return FALSE;

  return TRUE;
}


#define MAXNAMELEN	256

char *expand_tilde (const char *path, const char *home) {
  char *res = NULL;
  char *slash;
  char name[MAXNAMELEN];
  int namelen;
  struct passwd *pwd;

  if (path) {
    if (path[0] == '~') {
      if (path[1] == '\0') {
	res = g_strdup (home);
      }
      else if (path[1] == '/') {
	res = file_in_dir (home, path + 2);
      }
      else {
	slash = strchr (&path[1], '/');
	if (slash)
	  namelen = slash - &path[1];
	else
	  namelen = strlen (&path[1]);

	if (namelen > MAXNAMELEN)
	  namelen = MAXNAMELEN - 1;

	strncpy (name, &path[1], namelen);
	name[namelen] = '\0';

	pwd = getpwnam (name);
	if (pwd)
	  res = file_in_dir (pwd->pw_dir, path + 2 + namelen);
	else
	  res = g_strdup (path);
      }
    }
    else {
      res = g_strdup (path);
    }
  }

#ifdef DEBUG
  fprintf (stderr, "Tilde expansion: %s -> %s\n", path, res);
#endif

  return res;
}


/*
 *  User Info
 */


char *get_user_name (void) {
  char *str;
  struct passwd *pwd;
  char *user = NULL;

  if ((str = getenv ("USER")) != NULL || (str = getenv ("LOGNAME")) != NULL)
    user = g_strdup (str);
  else {
    pwd = getpwuid (getuid ());
    if (pwd)
      user = g_strdup (pwd->pw_name);
  }
  return user;
}


char *get_user_home (void) {
  char *str;
  struct passwd *pwd;
  char *home = NULL;

  if ((str = getenv ("HOME")) != NULL || (str = getenv ("LOGDIR")) != NULL)
    home = g_strdup (str);
  else {
    pwd = getpwuid (getuid ());
    if (pwd)
      home = g_strdup (pwd->pw_dir);
  }
  return home;
}


/*
 *  Directory Scaning
 */


static int qsort_helper (const void *a, const void *b) {
  return strcmp (* (char **) a, * (char **) b);
}


GList *dir_to_list (const char *dirname, 
                              char * (*filter) (const char *, const char *)) {
  DIR *directory;
  struct dirent *dirent_ptr;
  GList *list = NULL;
  char *str;

  directory = opendir (dirname);
  if (directory == NULL)
    return NULL;

  while ((dirent_ptr = readdir (directory)) != NULL) {
    if (filter)
      str = filter (dirname, dirent_ptr->d_name);
    else
      str = g_strdup (dirent_ptr->d_name);

    if (str)
      list = g_list_prepend (list, str);
  }

  closedir (directory);

  return qsort_list (list, qsort_helper);
}


/*
 *  Lists
 */


GList *merge_sorted_string_lists (GList *list1, GList *list2) {
  GList *res = NULL;
  GList *ptr1;
  GList *ptr2;
  int cmpres;

  ptr1 = list1;
  ptr2 = list2;

  while (ptr1 && ptr2) {
    cmpres = strcmp ((char *) ptr1->data, (char *) ptr2->data);

    if (cmpres == 0) {
      res = g_list_prepend (res, ptr1->data);
      g_free (ptr2->data);
      ptr1 = ptr1->next;
      ptr2 = ptr2->next;
    } else {
      if (cmpres < 0) {
	res = g_list_prepend (res, ptr1->data);
	ptr1 = ptr1->next;
      }
      else {
	res = g_list_prepend (res, ptr2->data);
	ptr2 = ptr2->next;
      }
    }
  }

  while (ptr1) {
    res = g_list_prepend (res, ptr1->data);
    ptr1 = ptr1->next;
  }

  while (ptr2) {
    res = g_list_prepend (res, ptr2->data);
    ptr2 = ptr2->next;
  }

  res = g_list_reverse (res);
  g_list_free (list1);
  g_list_free (list2);
  return res;
}


GSList *qsort_slist (GSList *list, int (*func) (const void *, const void *)) {
  gpointer *array;
  gpointer *ptr;
  GSList *tmp;
  int len;

  len = g_slist_length (list);

  switch (len) {

  case 0:
  case 1:
    break;

  case 2:
    if (func (&list->data, &list->next->data) > 0) {
      tmp = list->data;
      list->data = list->next->data;
      list->next->data = tmp;
    }
    break;

  default:
    ptr = array = g_malloc (sizeof (gpointer) * len);
    for (tmp = list; tmp; tmp = tmp->next) {
      *ptr++ = tmp->data;
    }
    
    qsort (array, len, sizeof (gpointer), func);

    ptr = array;
    for (tmp = list; tmp; tmp = tmp->next)
      tmp->data = *ptr++;
    
    g_free (array);
    break;
  }

  return list;
}


GList *qsort_list (GList *list, int (*func) (const void *, const void *)) {
  gpointer *array;
  gpointer *ptr;
  GList *tmp;
  int len;

  len = g_list_length (list);

  switch (len) {

  case 0:
  case 1:
    break;

  case 2:
    if (func (&list->data, &list->next->data) > 0) {
      tmp = list->data;
      list->data = list->next->data;
      list->next->data = tmp;
    }
    break;

  default:
    ptr = array = g_malloc (sizeof (gpointer) * len);
    for (tmp = list; tmp; tmp = tmp->next)
      *ptr++ = tmp->data;
    
    qsort (array, len, sizeof (gpointer), func);

    ptr = array;
    for (tmp = list; tmp; tmp = tmp->next)
      tmp->data = *ptr++;
    
    g_free (array);
    break;
  }

  return list;
}


int slist_index (GSList *list, gpointer ptr) {
  int i;

  for (i = 0; list; i++) {
    if (list->data == ptr) {
      return i;
    }
    list = list->next;
  }
  return -1;
}


int list_index (GList *list, gpointer ptr) {
  int i;

  for (i = 0; list; i++) {
    if (list->data == ptr) {
      return i;
    }
    list = list->next;
  }
  return -1;
}


GSList *slist_replicate (GSList *list) {
  GSList *replicant = NULL;

  while (list) {
    replicant = g_slist_prepend (replicant, list->data);
    list = list->next;
  }
  replicant = g_slist_reverse (replicant);
  return replicant;
}


GSList *list_replicate (GList *list) {
  GSList *replicant = NULL;

  while (list) {
    replicant = g_slist_prepend (replicant, list->data);
    list = list->next;
  }
  replicant = g_slist_reverse (replicant);
  return replicant;
}


/*
 *  Signals 
 */


void on_sig (int signum, void (*func) (int signum)) {
  struct sigaction action;

  action.sa_handler = func;
  sigemptyset (&action.sa_mask);
  action.sa_flags = (signum == SIGCHLD)? 
                                       SA_RESTART | SA_NOCLDSTOP : SA_RESTART;
  sigaction (signum, &action, NULL);
}


void ignore_sigpipe (void) {
  on_sig (SIGPIPE, SIG_IGN);
}

