/* $Id: conf.c,v 1.1 1998/10/24 04:04:10 marcus Exp $
***************************************************************************
  
   LibGG - Configuration handling
  
   Copyright (C) 1997 Jason McMullan	[jmcc@ggi-project.org]
   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
  
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <ggi/gg.h>
#include "plat.h"

/* Private structures */
typedef struct gg_string_list {
	const char *str;
	struct gg_string_list *next;
} gg_string_list;

typedef struct gg_dlconfig {
	const char *name;
	const char *filename;
	gg_string_list *version;
	struct gg_dlconfig *next;
} gg_dlconfig;

static int _ggDoLoad(const char *file, gg_dlconfig **list, int incl)
{
	FILE *inf;
	char root[2048], line[2048], buff[2048], *cp, *bp;
	char *name, *fname;
	gg_dlconfig *dltmp, *dlconf;
	root[0] = 0;

	inf = fopen(file,"r");
	if (inf == NULL)
		return GGI_NOFILE;

	dlconf = *list;

	while (!feof(inf)) {
		fgets(line, 2048, inf);
		cp = line;

		while (isspace((int)*cp)) cp++;
		if (*cp == 0 || *cp == '#')
			continue;
	
		if (strncmp(cp, ".root:", 6) == 0) {
			cp += 6;
			while (isspace((int)*cp)) cp++;
			name = cp;
			while (*cp && !isspace((int)*cp)) cp++;
			strncpy(root, name, cp-name);
			root[cp-name] = 0;
			continue;
		}
		if (strncmp(cp, ".include", 8) == 0) {
			cp += 8;
			while (isspace((int)*cp)) cp++;
			name = cp;
			while (*cp && !isspace((int)*cp)) cp++;
			strncpy(buff, name, cp-name);
			buff[cp-name] = 0;
			if (incl+1 < 20) {
				*list = dlconf;
				_ggDoLoad(buff, list, incl+1);
				dlconf = *list;
			}
			else {
				fprintf(stderr,"libgg: .include recursion "
						"too deep\n");
			}
		}	
		bp = buff;
		while (!isspace((int)*cp)) *(bp++) = *(cp++);
		*(bp) = 0;
		if ((name = strdup(buff)) == NULL) {
			return GGI_NOMEM;
		}
		while (isspace((int)*cp)) cp++;
		if (*cp == 0 || *cp == '#') {
			free(name);
			continue;
		}
		bp=buff;
		while (!isspace((int)*cp)) {
			*(bp++) =* (cp++);
		}
		*(bp) = 0;
		if (buff[0] != CHAR_DIRDELIM) {
			if ((fname = (char *) malloc(strlen(buff)
						     +strlen(root)+2))
			    == NULL) {
				return GGI_NOMEM;
			}
			strcpy(fname,root);
			strcat(fname,STRING_DIRDELIM);
			strcat(fname,buff);
		} else {
			if ((fname = strdup(buff)) == NULL) {
				return GGI_NOMEM;
			}
		}
		/* Add the config entry....
		 */	
		if ((dltmp = (gg_dlconfig *)malloc(sizeof(gg_dlconfig)))
		    == NULL) {
			return GGI_NOMEM;
		}
		dltmp->name = name;
		dltmp->filename = fname;
		dltmp->version = NULL;
		dltmp->next = dlconf;
		dlconf = dltmp;

		do {	
			gg_string_list *vtmp;

			while (isspace((int)*cp)) cp++;
			if (*cp == 0 || *cp == '#') 
				break;
			bp = buff;
			while (!isspace((int)*cp)) *(bp++) = *(cp++);
			*(bp) = 0;
			if ((vtmp = (gg_string_list *)malloc(sizeof(gg_string_list)))
			    == NULL 
			    || (vtmp->str = strdup(buff)) == NULL) {
				return GGI_NOMEM;
			}
			vtmp->next = dltmp->version;
			dltmp->version = vtmp;
		} while (*cp != 0);
			
	}
	fclose(inf);

	*list = dlconf;

	return 0;
}

int ggLoadConfig(const char *file, void **ret)
{
	*ret = NULL;
	return _ggDoLoad(file, (gg_dlconfig **)ret, 0);
}


/*
 * ggFreeConfig()
 *
 * Free the entire config list
 */

void ggFreeConfig(void *conf)
{
	gg_dlconfig *conftmp, *confptr = (gg_dlconfig*) conf;
	
	while (confptr != NULL) {
		gg_string_list *strlist = confptr->version, *strtmp;
		free((char*)confptr->name);
		free((char*)confptr->filename);
		while (strlist != NULL) {
			strtmp = strlist->next;
			free((char*)strlist->str);
			free(strlist);
			strlist = strtmp;
		}
		conftmp = confptr->next;
		free(confptr);
		confptr = conftmp;
	}
}


/* Takes a gg_dlconfig, lib name, and a version. Returns the
 * closest match (Ha!)
 */
const char *ggMatchConfig(void *conf, const char *name,const char *ver)
{
	gg_dlconfig *dlc,*dtmp,*best = NULL;
	gg_string_list *vtmp;

	dlc = (gg_dlconfig*) conf;

	for (dtmp = dlc; dtmp != NULL; dtmp = dtmp->next) {
		if (strcasecmp(name, dtmp->name)!=0)
			continue;

		if (dtmp->version == NULL) {
			best = dtmp;
			continue;
		}

		for (vtmp = dtmp->version; vtmp != NULL; vtmp = vtmp->next) 
			if (strcmp(ver, vtmp->str) == 0) {
				return dtmp->filename;
			}
	}

	return (best == NULL) ? NULL : best->filename;
}
