/*
 * ratDSN.c --
 *
 *	This file handles the delivery status notifications.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * Postilion software and its included text and images
 * Copyright (C) 1998 Nic Bernstein
 *
 * The full text of the legal notices is contained in the files called
 * COPYING and COPYRIGHT.TkRat, included with this distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ratFolder.h"
#include <unistd.h>

typedef struct {
    char *envid;		/* Original envelope ID                       */
    Tcl_DString msgFields;	/* Contains tcl list of msg fields            */
    int numRecipients;		/* Number of recipients mentioned in this DSN */
    char **actionPtrPtr;	/* The action                                 */
    char **recTypePtrPtr;	/* Type of recipient addresses                */
    char **recipientPtrPtr;	/* Recipient addresses                        */
    Tcl_DString *rListPtrPtr;   /* Recipient fields                           */
} RatDeliveryStatus;

/*
 * Static data
 */
static Tcl_HashTable seenTable;

/*
 * Local functions
 */
static Tcl_Channel OpenIndex(Tcl_Interp *interp, char *mode);
static int RatDSNList(ClientData clientData, Tcl_Interp *interp,
	int argc, char **argv);
static int RatDSNGet(ClientData clientData, Tcl_Interp *interp,
	int argc, char **argv);
static int RatDSNExpire(Tcl_Interp *interp, char *line);
static char *RatParseDSNLine(char *buf, char *name, char *value, int *length);
static RatDeliveryStatus *RatParseDS(Tcl_Interp *interp, char *body,
	int length);
static void RatFreeDeliveryStatus(RatDeliveryStatus *statusPtr);


/*
 *----------------------------------------------------------------------
 *
 * RatDSNInit --
 *
 *      Initializes the DSN system. That is adds the apropriate commands
 *	to the interpreter.
 *
 * Results:
 *	A standard tcl resutl.
 *
 * Side effects:
 *	Commands are created in the interpreter.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatDSNInit(Tcl_Interp *interp)
{
    Tcl_InitHashTable(&seenTable, TCL_STRING_KEYS);
    Tcl_CreateCommand(interp, "RatDSNList", RatDSNList, NULL, NULL);
    Tcl_CreateCommand(interp, "RatDSNGet", RatDSNGet, NULL, NULL);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNStartMessage --
 *
 *      Start recording a new DSN message.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	A new current message is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

DSNhandle
RatDSNStartMessage(Tcl_Interp *interp, char *id, char *subject)
{
    Tcl_DString *dsPtr = (Tcl_DString*)ckalloc(sizeof(Tcl_DString));
    char buf[32], *header, *cPtr;
    time_t seconds;

    Tcl_DStringInit(dsPtr);
    Tcl_DStringAppendElement(dsPtr, id);
    seconds = time(NULL);
    sprintf(buf, "%d", (int)seconds);
    Tcl_DStringAppendElement(dsPtr, buf);
    cPtr = header = RatDecodeHeader(interp, subject);
    while ((cPtr = strchr(cPtr, '\n'))) {
	*cPtr++ = ' ';
    }
    Tcl_DStringAppendElement(dsPtr, header);
    Tcl_DStringStartSublist(dsPtr);

    return (DSNhandle)dsPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNAddRecipient --
 *
 *      Add a recipient to the currently recording DSN message.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current message is modified.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatDSNAddRecipient(Tcl_Interp *interp, DSNhandle handle, char *recipient)
{
    Tcl_DString *dsPtr = (Tcl_DString*)handle;

    Tcl_DStringStartSublist(dsPtr);
    Tcl_DStringAppendElement(dsPtr, "none");
    Tcl_DStringAppendElement(dsPtr, recipient);
    Tcl_DStringAppendElement(dsPtr, "");
    Tcl_DStringEndSublist(dsPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNAbort --
 *
 *	Aborts the composition of the indicated DSN message.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The handle becomes invalid.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatDSNAbort(Tcl_Interp *interp, DSNhandle handle)
{
    Tcl_DString *dsPtr = (Tcl_DString*)handle;

    if (dsPtr) {
	Tcl_DStringFree(dsPtr);
	ckfree(dsPtr);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNFinish --
 *
 *      Moves the message under construction to the list of outstanding
 *	DSN's.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current message is cleared.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatDSNFinish(Tcl_Interp *interp, DSNhandle handle)
{
    Tcl_Channel channel = OpenIndex(interp, "a");
    Tcl_DString *dsPtr = (Tcl_DString*)handle;

    if (!channel) {
	Tcl_BackgroundError(interp);
	return;
    }
    Tcl_DStringEndSublist(dsPtr);
    Tcl_Write(channel, Tcl_DStringValue(dsPtr), Tcl_DStringLength(dsPtr));
    Tcl_Write(channel, "\n", 1);
    Tcl_Close(interp, channel);
    Tcl_DStringFree(dsPtr);
    free(dsPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNHandle --
 *
 *      Handle an incoming DSN.
 *
 * Results:
 *	Returns true if the given DSN matched one of those in our list.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatDSNHandle (Tcl_Interp *interp, char *msg)
{
    int i, j, new, match = 0, changed = 0, argc, rfound, perm, rargc;
    RatDeliveryStatus *statusPtr;
    Tcl_HashEntry *entryPtr;
    Tcl_Channel channel;
    char buf[1024], id[1024], **argv, *oldId, *dir, *msgFile = NULL;
    char **rargv, *file;
    Tcl_DString ds, dirDS, line;
    Tcl_CmdInfo cmdInfo;

    /*
     * Avoid processing the same DSN twice
     */
    entryPtr = Tcl_CreateHashEntry(&seenTable, msg, &new);
    if (!new) {
	return (int)Tcl_GetHashValue(entryPtr);
    }
    Tcl_SetHashValue(entryPtr, 0);

    sprintf(buf, "[lindex [[%s body] children] 1] data 0", msg);
    if (TCL_OK != Tcl_Eval(interp, buf)) {
	return 0;
    }
    statusPtr = RatParseDS(interp, interp->result, strlen(interp->result));
    if (!statusPtr->envid) {
	RatFreeDeliveryStatus(statusPtr);
	return 0;
    }

    Tcl_DStringInit(&ds);
    Tcl_DStringInit(&line);
    Tcl_DStringInit(&dirDS);
    if (NULL == (channel = OpenIndex(interp, "r"))) {
	RatFreeDeliveryStatus(statusPtr);
	return 0;
    }
    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
	    TCL_GLOBAL_ONLY), &perm);
    dir = Tcl_TranslateFileName(interp, Tcl_GetVar2(interp, "option",
	    "dsn_directory", TCL_GLOBAL_ONLY), &dirDS);
    while (Tcl_DStringSetLength(&line, 0), -1 != Tcl_Gets(channel, &line)) {
	if (RatDSNExpire(interp, Tcl_DStringValue(&line))) {
	    /*
	     * This DSN has expired so we should remove all associated files
	     */
	    file = RatLindex(interp, Tcl_DStringValue(&line), 0);
	    sprintf(buf, "%s/%s", dir, file);
	    (void)unlink(buf);
	    Tcl_SplitList(interp, RatLindex(interp, Tcl_DStringValue(&line), 3),
		    &argc, &argv);
	    for (i=0; i < argc; i++) {
		file = RatLindex(interp, argv[i], 2);
		if (strlen(file)) {
		    sprintf(buf, "%s/%s", dir, file);
		    (void)unlink(buf);
		}
	    }
	    free(argv);
	    changed++;
	    continue;
	}
	if (strncmp(Tcl_DStringValue(&line), statusPtr->envid,
		strlen(statusPtr->envid))) {
	    Tcl_DStringAppendElement(&ds, Tcl_DStringValue(&line));
	    continue;
	}
	changed++;
	match = 1;
	Tcl_DStringStartSublist(&ds);
	Tcl_SplitList(interp, Tcl_DStringValue(&line), &argc, &argv);
	Tcl_DStringAppendElement(&ds, argv[0]);
	Tcl_DStringAppendElement(&ds, argv[1]);
	Tcl_DStringAppendElement(&ds, argv[2]);
	Tcl_SplitList(interp, argv[3], &rargc, &rargv);
	Tcl_DStringStartSublist(&ds);
	for (i=0; i<rargc; i++) {
	    for (j=rfound=0; !rfound && j<statusPtr->numRecipients; j++) {
		if (statusPtr->recTypePtrPtr[j]
			&& statusPtr->actionPtrPtr[j]
			&& !strcasecmp(statusPtr->recTypePtrPtr[j], "rfc822")
			&& !strcmp(statusPtr->recipientPtrPtr[j],
			        RatLindex(interp, rargv[i], 1))
			&& strcmp(statusPtr->actionPtrPtr[j],
			        RatLindex(interp, rargv[i], 0))) {
		    /*
		     * This DSN matched this recipient
		     * We start by saving the DSN message;
		     * then we add it to the index file.
		     * Finally we notify the user.
		     */
		    rfound = 1;
		    oldId = RatLindex(interp, rargv[i], 2);
		    RatGenId(NULL, interp, 0, NULL);
		    strcpy(id, interp->result);
		    if (strlen(oldId)) {
			sprintf(buf, "%s/%s", dir, oldId);
			(void)unlink(buf);
		    }

		    sprintf(buf, "%s/%s", dir, id);
		    if (!msgFile) {
			Tcl_DString msgDS;
			Tcl_Channel msgCh;

			msgFile = cpystr(buf);
			Tcl_DStringInit(&msgDS);
			Tcl_GetCommandInfo(interp, msg, &cmdInfo);
			RatMessageGet(interp,(MessageInfo*)cmdInfo.clientData,
				&msgDS, NULL, NULL);
			msgCh = Tcl_OpenFileChannel(interp, msgFile, "w", perm);
			Tcl_Write(msgCh, Tcl_DStringValue(&msgDS),
				Tcl_DStringLength(&msgDS));
			Tcl_Close(interp, msgCh);
			Tcl_DStringFree(&msgDS);
		    } else {
			link(msgFile, buf);
		    }
		    Tcl_DStringStartSublist(&ds);
		    Tcl_DStringAppendElement(&ds, statusPtr->actionPtrPtr[j]);
		    Tcl_DStringAppendElement(&ds,statusPtr->recipientPtrPtr[j]);
		    Tcl_DStringAppendElement(&ds, id);
		    Tcl_DStringEndSublist(&ds);
		    Tcl_VarEval(interp, "RatDSNRecieve {",
			    RatDecodeHeaderFull(interp, argv[2]), "} {",
			    statusPtr->actionPtrPtr[j], "} {",
			    statusPtr->recipientPtrPtr[j], "} {", id, "}",NULL);
		}
	    }
	    if (!rfound) {
		Tcl_DStringAppendElement(&ds, rargv[i]);
	    }
	}
	Tcl_DStringEndSublist(&ds);
	Tcl_DStringEndSublist(&ds);
	ckfree(argv);
	ckfree(rargv);
    }
    Tcl_Close(interp, channel);
    RatFreeDeliveryStatus(statusPtr);
    if (changed) {
	if (NULL == (channel = OpenIndex(interp, "w"))) {
	    return 0;
	}
	Tcl_SplitList(interp, Tcl_DStringValue(&ds), &argc, &argv);
	for (i=0; i<argc; i++) {
	    Tcl_Write(channel, argv[i], strlen(argv[i]));
	    Tcl_Write(channel, "\n", 1);
	}
	ckfree(argv);
	Tcl_Close(interp, channel);
    }
    Tcl_DStringFree(&ds);
    Tcl_DStringFree(&line);
    Tcl_SetHashValue(entryPtr, match);
    if (msgFile) {
	ckfree(msgFile);
    }
    return match;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNExtract --
 *
 *      Extract the DSN data from a dsn body part
 *
 * Results:
 *	A standard tcl result and the requested data in interp->result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatDSNExtract (Tcl_Interp *interp, char *data, int length)
{
    RatDeliveryStatus *sPtr = RatParseDS(interp, data, length);
    Tcl_DString ds;
    int i;

    Tcl_DStringInit(&ds);
    Tcl_DStringAppendElement(&ds, Tcl_DStringValue(&sPtr->msgFields));
    Tcl_DStringStartSublist(&ds);
    for (i=0; i<sPtr->numRecipients; i++) {
	Tcl_DStringAppendElement(&ds, Tcl_DStringValue(&sPtr->rListPtrPtr[i]));
    }
    Tcl_DStringEndSublist(&ds);
    Tcl_DStringResult(interp, &ds);
    RatFreeDeliveryStatus(sPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * OpenIndex --
 *
 *      Opens the DSN indexfile.
 *
 * Results:
 *	A Tcl channel handle. If an error occurs NULL is returned
 *	and an error message is left in interp->result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static Tcl_Channel
OpenIndex(Tcl_Interp *interp, char *mode)
{
    Tcl_DString ds;
    char *dir, buf[1024];
    struct stat sbuf;
    int perm;

    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
	    TCL_GLOBAL_ONLY), &perm);
    dir = Tcl_TranslateFileName(interp, Tcl_GetVar2(interp, "option",
	    "dsn_directory", TCL_GLOBAL_ONLY), &ds);
    if (stat(dir, &sbuf)) {
	if (mkdir(dir, perm|0100)) {
	    Tcl_AppendResult(interp, "Failed to create directory \"",
		    dir, "\" :", Tcl_PosixError(interp), NULL);
	    return NULL;
	}
    } else if (!S_ISDIR(sbuf.st_mode)) {
	Tcl_AppendResult(interp, "This is no directory \"", dir, "\"", NULL);
	return NULL;
    }
    sprintf(buf, "%s/index", dir);
    Tcl_DStringFree(&ds);

    return Tcl_OpenFileChannel(interp, buf, mode, perm);
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNList --
 *
 *      List the currently known DSN's
 *
 * Results:
 *	See ../doc/interface
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatDSNList (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    Tcl_Channel channel = OpenIndex(interp, "r");
    Tcl_DString line;

    if (!channel) {
	Tcl_ResetResult(interp);
	return TCL_OK;
    }

    Tcl_DStringInit(&line);

    while (-1 != Tcl_Gets(channel, &line)) {
	if (!RatDSNExpire(interp, Tcl_DStringValue(&line))) {
	    Tcl_AppendElement(interp, Tcl_DStringValue(&line));
	}
	Tcl_DStringSetLength(&line, 0);
    }
    Tcl_DStringFree(&line);
    if (!Tcl_Eof(channel)) {
	Tcl_Close(interp, channel);
	return TCL_ERROR;
    }
    Tcl_Close(interp, channel);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNGet --
 *
 *      Get information about a DSN.
 *
 * Results:
 *	See ../doc/interface
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatDSNGet (ClientData clientData, Tcl_Interp *interp, int argc, char **argv)
{
    char buf[1024], *dir, *msg, *data;
    RatDeliveryStatus *statusPtr;
    Tcl_Channel channel;
    Tcl_DString ds;
    int i, len;

    if (argc != 3 && argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" what id ?recipient?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (strcmp(argv[1], "msg") && strcmp(argv[1], "report")) {
	Tcl_AppendResult(interp, "Illegal 'what' argument; should be ",
		"'msg' or 'report'.", (char*) NULL);
	return TCL_ERROR;
    }
    if (!strlen(argv[2])) {
	Tcl_SetResult(interp, "Empty 'id' argument.", TCL_STATIC);
	return TCL_ERROR;
    }

    Tcl_DStringInit(&ds);
    dir = Tcl_TranslateFileName(interp, Tcl_GetVar2(interp, "option",
	    "dsn_directory", TCL_GLOBAL_ONLY), &ds);
    sprintf(buf, "%s/%s", dir, argv[2]);
    if (NULL == (channel = Tcl_OpenFileChannel(interp, buf, "r", 0))) {
       return TCL_ERROR;
    }
    Tcl_DStringFree(&ds);
    len = Tcl_Seek(channel, 0, SEEK_END);
    data = (char*) ckalloc(len+1);
    Tcl_Seek(channel, 0, SEEK_SET);
    len = Tcl_Read(channel, data, len);
    data[len] = '\0';
    Tcl_Close(interp, channel);
    msg = RatFrMessageCreate(interp, data, len, NULL);
    ckfree(data);

    if (!strcmp(argv[1], "msg")) {
	Tcl_SetResult(interp, msg, TCL_DYNAMIC);
    } else {
	sprintf(buf, "[lindex [[%s body] children] 1] data 0", msg);
	if (TCL_OK != Tcl_Eval(interp, buf)) {
	    return TCL_ERROR;
	}
	statusPtr = RatParseDS(interp, interp->result, strlen(interp->result));
	Tcl_DStringResult(interp, &statusPtr->msgFields);
	for (i=0; i < statusPtr->numRecipients; i++) {
	    if (!strcmp(statusPtr->recipientPtrPtr[i], argv[3])) {
		Tcl_AppendResult(interp, " ",
			Tcl_DStringValue(&statusPtr->rListPtrPtr[i]),
			(char *) NULL);
		break;
	    }
	}
	RatFreeDeliveryStatus(statusPtr);
	RatMessageDelete(interp, msg);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSNExpire --
 *
 *      Check if a given DSN line has expired.
 *
 * Results:
 *	Returns true if it has expired.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatDSNExpire(Tcl_Interp *interp, char *line)
{
    long intime;
    int days;

    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "dsn_expiration",
	    TCL_GLOBAL_ONLY), &days);
    sscanf(line, "%*s %ld", &intime);
    return (intime+days*24*60*60 < time(NULL));
}

/*
 *----------------------------------------------------------------------
 *
 * RatParseDSNLine --
 *
 *      Extract the next line of DSN information from a message/delivery-status
 *	message. The arguments name and place are expected to point to areas
 *	large enough to hold the results.
 *
 * Results:
 *	The name,value pair is written into the buffers given as arguments.
 *	The function returns a pointer to the chararacter after the last one
 *	parsed.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static char*
RatParseDSNLine(char *buf, char *name, char *value, int *length)
{
    int i, inName, preamble;
    char *dstPtr;

    *name = *value = '\0';
    for (i=0, dstPtr = name, inName = preamble = 1; buf[i]; i++) {
	switch(buf[i]) {
	case '\015':
	    if ('\n' == buf[i+1]) {
		break;
	    }
	    /* fallthrough */
	case '\012':
	    if (' ' != buf[i+1] && '\t' != buf[i+1]) {
		*dstPtr = '\0';
		*length -= i+1;
		return buf+i+1;
	    }
	    break;
	case ':':
	    if (inName) {
		*dstPtr = '\0';
		dstPtr = value;
		inName = 0;
		preamble = 1;
	    } else {
		*dstPtr++ = buf[i];
	    }
	    break;
	case ' ':	/* fallthrough */
	case '\t':
	    if (preamble) {
		break;
	    }
	default:
	    *dstPtr++ = buf[i];
	    preamble = 0;
	    break;
	}
    }
    *length -= i;
    return buf+i;
}

/*
 *----------------------------------------------------------------------
 *
 * RatParseDS --
 *
 *	Parse a message/delivery-status body.
 *
 * Results:
 *	A pointer to a RatDeliveryStatus structure. It is the callers
 *	responsibility to free this pointer later with a call to
 *	RatFreeDeliveryStatus().
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static RatDeliveryStatus*
RatParseDS(Tcl_Interp *interp, char *body, int length)
{
    RatDeliveryStatus *sPtr = (RatDeliveryStatus*)ckalloc(sizeof(*sPtr));
    char *name, *value, *cPtr;
    int allocated, i;

    name = (char*)ckalloc(strlen(body)+1);
    value = (char*)ckalloc(strlen(body)+1);

    /*
     * Parse the per message fields.
     */
    sPtr->envid = NULL;
    while (strchr(" \t\015\012", *body)) {
       body++;
       length--;
    }
    Tcl_DStringInit(&sPtr->msgFields);
    while (length > 0) {
	body = RatParseDSNLine(body, name, value, &length);
	if (!*name) {
	    break;
	}
	Tcl_DStringStartSublist(&sPtr->msgFields);
	Tcl_DStringAppendElement(&sPtr->msgFields, name);
	Tcl_DStringAppendElement(&sPtr->msgFields, value);
	Tcl_DStringEndSublist(&sPtr->msgFields);
	if (!strcasecmp("original-envelope-id", name)) {
	    sPtr->envid = cpystr(value);
	}
    }

    /*
     * Parse the per recipient fields
     */
    sPtr->numRecipients = 0;
    sPtr->actionPtrPtr = NULL;
    sPtr->recTypePtrPtr = NULL;
    sPtr->recipientPtrPtr = NULL;
    sPtr->rListPtrPtr = NULL;
    allocated = 0;
    while (length > 0) {
	if (allocated <= sPtr->numRecipients) {
	    allocated += 32;
	    sPtr->actionPtrPtr = (char**)REALLOC(sPtr->actionPtrPtr, 
		    allocated*sizeof(char*));
	    sPtr->recTypePtrPtr = (char**)REALLOC(sPtr->recTypePtrPtr, 
		    allocated*sizeof(char*));
	    sPtr->recipientPtrPtr = (char**)REALLOC(sPtr->recipientPtrPtr, 
		    allocated*sizeof(char*));
	    sPtr->rListPtrPtr = (Tcl_DString*)REALLOC(sPtr->rListPtrPtr, 
		    allocated*sizeof(Tcl_DString));
	}
	i = sPtr->numRecipients++;
	sPtr->actionPtrPtr[i] = NULL;
	sPtr->recTypePtrPtr[i] = NULL;
	Tcl_DStringInit(&sPtr->rListPtrPtr[i]);
	while (length > 0) {
	    body = RatParseDSNLine(body, name, value, &length);
	    if (!*name) {
		break;
	    }
	    Tcl_DStringStartSublist(&sPtr->rListPtrPtr[i]);
	    Tcl_DStringAppendElement(&sPtr->rListPtrPtr[i], name);
	    Tcl_DStringAppendElement(&sPtr->rListPtrPtr[i], value);
	    Tcl_DStringEndSublist(&sPtr->rListPtrPtr[i]);
	    if (!strcasecmp("original-recipient", name)) {
		sPtr->recTypePtrPtr[i] = cpystr(value);
		cPtr = strchr(sPtr->recTypePtrPtr[i], ';');
		*cPtr++ = '\0';
		sPtr->recipientPtrPtr[i] = cPtr;
	    }
	    if (!strcasecmp("action", name)) {
		sPtr->actionPtrPtr[i] = cpystr(value);
	    }
	}
    }
    ckfree(name);
    ckfree(value);

    return sPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatFreeDeliveryStatus --
 *
 *	Free a RatDeliveryStatus structure.
 *
 * Results:
 *	A pointer to a RatDeliveryStatus structure. It is the callers
 *	responsibility to free this pointer later with a call to
 *	RatFreeDeliveryStatus().
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatFreeDeliveryStatus(RatDeliveryStatus *statusPtr)
{
    int i;

    if (statusPtr->envid) {
	ckfree(statusPtr->envid);
    }
    Tcl_DStringFree(&statusPtr->msgFields);
    if (statusPtr->numRecipients) {
	for (i=0; i<statusPtr->numRecipients; i++) {
	    ckfree(statusPtr->actionPtrPtr[i]);
	    ckfree(statusPtr->recTypePtrPtr[i]);
	    Tcl_DStringFree(&statusPtr->rListPtrPtr[i]);
	}
	ckfree(statusPtr->actionPtrPtr);
	ckfree(statusPtr->recTypePtrPtr);
	ckfree(statusPtr->recipientPtrPtr);
	ckfree(statusPtr->rListPtrPtr);
    }

    ckfree(statusPtr);
}
