/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <stddef.h>
#include <stdlib.h>
#include "common.h"
#include "error.h"
#include "avl.h"
#include "str.h"
#include "lexer.h"
#include "var.h"
#include "do-ifP.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

int temporary;
dictionary *temp_dict;
int temp_trns;

#if 0
/* Displays all the value labels in TREE, with label S. */
void
display_tree (char *s, avl_tree *tree)
{
  value_label *iter;
  avl_traverser *trav = NULL;

  printf("%s tree:\n", s);
  fflush(stdout);
  while ((iter = avl_traverse (tree, &trav)) != NULL)
    printf (" %g: %s\n", iter->v.f, iter->s);
}
#endif

/* Parses the TEMPORARY command. */
int
cmd_temporary (void)
{
  /* Return value. */
  int code = 1;

  match_id (TEMPORARY);
  if (token != '.')
    {
      return syntax_error (_("expecting end of command"));
      code = -2;
    }

  /* TEMPORARY is not allowed inside DO IF or LOOP. */
  if (ctl_stack)
    {
      msg (SE, _("This command is not valid inside DO IF or LOOP."));
      return 0;
    }

  /* TEMPORARY can only appear once! */
  if (temporary)
    {
      msg (SE, _("This command may only appear once between "
	   "procedures and procedure-like commands."));
      return 0;
    }

  /* Everything is temporary, even if we think it'll last forever.
     Especially then. */
  temporary = 1;
  temp_dict = save_dictionary ();
  if (f_trns == n_trns)
    temp_trns = -1;
  else
    temp_trns = n_trns;
  debug_printf (("TEMPORARY: temp_trns=%d\n", temp_trns));

  return code;
}

/* Copies a variable structure. */
void
copy_variable (variable *dest, const variable *src)
{
  int i, n;

  assert (dest != src);
  dest->type = src->type;
  dest->left = src->left;
  dest->width = src->width;
  dest->fv = src->fv;
  dest->nv = src->nv;
  dest->miss_type = src->miss_type;
  
  switch (src->miss_type)
    {
    case MISSING_NONE:
      n = 0;
      break;
    case MISSING_1:
      n = 1;
      break;
    case MISSING_2:
    case MISSING_RANGE:
      n = 2;
      break;
    case MISSING_3:
    case MISSING_RANGE_1:
      n = 3;
      break;
    default:
      assert (0);
      break;
    }
  
  for (i = 0; i < n; i++)
    dest->missing[i] = src->missing[i];
  dest->print = src->print;
  dest->write = src->write;

  dest->val_lab = copy_value_labels (src->val_lab);
  dest->label = src->label ? xstrdup (src->label) : NULL;
}

/* Returns a newly created empty dictionary.  The file label and
   documents are copied from default_dict if COPY is nonzero. */
dictionary *
new_dictionary (int copy)
{
  dictionary *d = xmalloc (sizeof (dictionary));
  
  d->var = NULL;
  d->var_by_name = avl_create (NULL, cmp_variable, NULL);
  d->nvar = 0;

  d->N = 0;

  d->nval = 0;

  d->n_splits = 0;
  d->splits = NULL;

  if (default_dict.label && copy)
    d->label = xstrdup (default_dict.label);
  else
    d->label = NULL;

  if (default_dict.n_documents && copy)
    {
      d->n_documents = default_dict.n_documents;
      if (d->n_documents)
	{
	  d->documents = malloc (default_dict.n_documents * 80);
	  memcpy (d->documents, default_dict.documents,
		  default_dict.n_documents * 80);
	}
    }
  else
    {
      d->n_documents = 0;
      d->documents = NULL;
    }
  
  d->weight_index = -1;
  d->weight_var[0] = 0;

  d->filter_var[0] = 0;

  return d;
}
    
/* Copies the current dictionary info into a newly allocated
   dictionary structure, which is returned. */
dictionary *
save_dictionary (void)
{
  /* Dictionary being created. */
  dictionary *d = xmalloc (sizeof (dictionary));

  int i;

  /* First the easy stuff. */
  *d = default_dict;
  d->label = default_dict.label ? xstrdup (default_dict.label) : NULL;
  if (default_dict.n_documents)
    {
      d->documents = malloc (default_dict.n_documents * 80);
      memcpy (d->documents, default_dict.documents,
	      default_dict.n_documents * 80);
    }
  else d->documents = NULL;

  /* Then the variables. */
  d->var_by_name = NULL;
  d->var = xmalloc (default_dict.nvar * sizeof (variable *));
  for (i = 0; i < default_dict.nvar; i++)
    {
      d->var[i] = xmalloc (sizeof (variable));
      copy_variable (d->var[i], default_dict.var[i]);
      strcpy (d->var[i]->name, default_dict.var[i]->name);
      d->var[i]->index = i;
    }

  /* Then the SPLIT FILE variables. */
  if (default_dict.splits)
    {
      int i;

      d->n_splits = default_dict.n_splits;
      d->splits = xmalloc ((default_dict.n_splits + 1) * sizeof (variable *));
      for (i = 0; i < default_dict.n_splits; i++)
	d->splits[i] = d->var[default_dict.splits[i]->index];
      d->splits[default_dict.n_splits] = NULL;
    }
  else
    {
      d->n_splits = 0;
      d->splits = NULL;
    }
  
  return d;
}

/* Copies dictionary D into the active file dictionary.  Deletes
   dictionary D. */
void
restore_dictionary (dictionary * d)
{
  int i;

  /* 1. Delete the current dictionary. */
  default_dict.n_splits = 0;
  free (default_dict.splits);
  default_dict.splits = NULL;
  
  avl_destroy (default_dict.var_by_name, NULL);
  default_dict.var_by_name = NULL;
  
  for (i = 0; i < default_dict.nvar; i++)
    {
      clear_variable (&default_dict, default_dict.var[i]);
      free (default_dict.var[i]);
    }
  
  free (default_dict.var);
  free (default_dict.label);
  free (default_dict.documents);

  /* 2. Copy dictionary D into the active file dictionary. */
#if __CHECKER__
  strncpy (d->weight_var, d->weight_var, 9);
  strncpy (d->filter_var, d->filter_var, 9);
  memset (&((char *) d)[(offsetof (dictionary, filter_var)
			 + sizeof (d->filter_var))],
	  '*',
	  (sizeof (dictionary) - offsetof (dictionary, filter_var)
	   - sizeof (d->filter_var)));
#endif
  default_dict = *d;
  if (!default_dict.var_by_name)
    {
      default_dict.var_by_name = avl_create (NULL, cmp_variable, NULL);
      
      for (i = 0; i < default_dict.nvar; i++)
	avl_force_insert (default_dict.var_by_name, default_dict.var[i]);
    }

  /* 3. Destroy dictionary D. */
  free (d);
}

/* Destroys dictionary D. */
void
free_dictionary (dictionary * d)
{
  int i;

  d->n_splits = 0;
  free (d->splits);
  d->splits = NULL;
  
  if (d->var_by_name)
    avl_destroy (d->var_by_name, NULL);

  for (i = 0; i < d->nvar; i++)
    {
      variable *v = d->var[i];

      if (v->val_lab)
	{
	  avl_destroy (v->val_lab, free_val_lab);
	  v->val_lab = NULL;
	}
      if (v->label)
	{
	  free (v->label);
	  v->label = NULL;
	}
      free (d->var[i]);
    }
  free (d->var);

  free (d->label);
  free (d->documents);

  free (d);
}

/* Cancels the temporary transformation, if any. */
void
cancel_temporary (void)
{
  if (temporary)
    {
      if (temp_dict)
	free_dictionary (temp_dict);
      temporary = 0;
      temp_trns = 0;
    }
}
