/*
  $Id: conn.c,v 1.10 1997/01/17 01:50:24 luik Exp $

  conn.c - omirrd socket connections module.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#define NEED_ALLOCA
#define NEED_TIME
#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: conn.c,v 1.10 1997/01/17 01:50:24 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <assert.h>
#include <dirent.h> /* XXX portability check */
#include <errno.h>
#include <glob.h> /* XXX portability check */
#include <stdlib.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "libomirr.h"
#include "debug.h"
#include "error.h"
#include "cf.h"
#include "mp.h"
#include "bio.h"
#include "lio.h"
#include "request.h"
#include "sock.h"
#include "conn.h"
#include "op.h"
#include "omirrd.h"


/* XXX TODO: make the following configuration parameters configurable,
   e.g. using config file variables. */
/* CONNECT_RETRY_INTERVAL - minimum number of seconds to wait between
   connect retries for one specific host. These constants are used to
   initialize the `retry_time' and `retry_interval' fields of
   `ConnEntryRec'. The initalization is done in
   `connDoConnectionsToServers' and the values are reset in
   `connCloseMasterFd'.  The check whether the retry interval has
   expired is done immediately before sending a new notification to
   the remote machine in `connWriteMessage'.  This function also
   computes the new interval and retry time values.  */
#ifdef DEBUG
#define CONNECT_RETRY_INTERVAL		60
#define CONNECT_RETRY_MAX_INTERVAL	60
#else
#define CONNECT_RETRY_INTERVAL		(60 * 5) /* 5 mins */
#define CONNECT_RETRY_MAX_INTERVAL	(60 * 60 * 4) /* 4 hours */
#endif


/* The following two parameters specify the properties of the retry
   scheme used in `connAcceptClient' while trying to back-connect to the
   remote machine.  */
#define BACKCONNECT_RETRIES		3
#define BACKCONNECT_INITIALTIMO		2


/* Function forward declarations.  */
static ConnEntry connAllocEntry(char *name);
static void connFreeEntry(ConnEntry conn_entry);
static ConnEntry connFindByName(const char *name);
static ConnEntry connFindByHostname(const char *hostname);
static void connCloseMasterFd(ConnEntry e);
static void reject_client_msg(int s, char *msg);
static void connCloseSlaveFd(ConnEntry e);
static void process_daemon_message(char *line, void *data);
static int close_daemon_connection(int fd, void *data);


/* conn_entry_head.next points to linked list of connection entries */
static ConnEntryRec conn_entry_head;



/* connAllocEntry - allocates a new entry and adds it to the linked
   list. Sets name in struct and initializes other elements. Returns
   new allocated entry or NULL on failure (out of memory).  */
static ConnEntry connAllocEntry(char *name)
{
    ConnEntry e;

    e = malloc(sizeof(ConnEntryRec));
    if (!e)
	return NULL;
    e->name = strdup(name);
    if (!e->name) {
	free(e);
	return NULL;
    }
    e->hostname = NULL;
    e->masterfd = -1;
    e->slavefd = -1;
    e->slave_op = NULL;
    e->state = CONN_DEAD;
    e->retry_time = 0;
    e->retry_interval = 0;
    e->next = conn_entry_head.next;
    conn_entry_head.next = e;
    return (e);
}


/* connFreeEntry - removes the specified entry from the linked list
   and frees the name and hostname components (unless NULL) and the
   entry itself.  */
static void connFreeEntry(ConnEntry conn_entry)
{
    ConnEntry e;

    for (e = &conn_entry_head; e; e = e->next) {
	if (e->next == conn_entry) {
	    e->next = conn_entry->next;
	    break;
	}
    }

    assert(e);

    if (conn_entry->slave_op)
	opFreeSlaveOp(conn_entry->slave_op);
    if (conn_entry->hostname != conn_entry->name)
	xfree(conn_entry->hostname);
    xfree(conn_entry->name);
    free(conn_entry);
}


/* connFindByName - searches for the entry with the specified
   `name'. Returns the entry on success, NULL otherwise.  */
static ConnEntry connFindByName(const char *name)
{
    ConnEntry e;

    for (e = conn_entry_head.next; e; e = e->next)
	if (e->name && streq(e->name, name))
	    return e;
    return NULL;
}


/* connFindByHostname - searches for the entry with the specified
   `hostname'. Returns the entry on success, NULL otherwise.  */
static ConnEntry connFindByHostname(const char *hostname)
{
    ConnEntry e;

    for (e = conn_entry_head.next; e; e = e->next)
	if (e->hostname && streq(e->hostname, hostname))
	    return e;
    return NULL;
}


ConnEntry connAddEntry(char *name)
{
    ConnEntry e;
    struct hostent *hp;		/* from gethostbyname(3N) */
    unsigned long inaddr;	/* from inet_addr(3N) */

    if (!((e = connFindByName(name)))) { /* not found, need a new entry */
	if ((e = connAllocEntry(name))) {
	    if ((long) ((inaddr = inet_addr(name))) != -1)
		hp = gethostbyaddr((char *) &inaddr, sizeof(inaddr), AF_INET);
	    else
		hp = gethostbyname(name);
	    if (hp)
		e->hostname = strdup(hp->h_name);
	    if (!e->hostname)	/* gethostbyname or strdup failed */
		e->hostname = name;
	    e->state = CONN_DEAD;
	    e->retry_time = 0;
	    e->retry_interval = CONNECT_RETRY_INTERVAL;
	}
    }
    return (e);
}


void connRemoveEntry(ConnEntry e)
{
    /* XXX check file descriptors */
    connFreeEntry(e);
}



/*
 * Functions to connect to (other) server(s) (set masterfd).
 */

void connDoConnectionsToServers(CfCmd cmds)
{
    CfCmd c;
    ConnEntry e;
    ConnEntry next;
    struct hostent *hp;		/* from gethostbyname(3N) */
    unsigned long inaddr;	/* from inet_addr(3N) */

    /* First clear the `cmds' pointer in all connections. Afterwards,
       the connections which are still listed in `cmds' will get their
       pointer restored. The others will then be terminated.  */
    for (e = conn_entry_head.next; e; e = e->next)
	e->cmds = NULL;

    for (c = cmds; c; c = c->c_next) {
	if (c->c_conn == NULL) {
	    /* Check whether c->c_name is the IP address (dotted quad)
               or name of our host. Don't try to connect to ourselfs.  */
	    if ((long) ((inaddr = inet_addr(c->c_name))) != -1)
		hp = gethostbyaddr((char *) &inaddr, sizeof(inaddr), AF_INET);
	    else
		hp = gethostbyname(c->c_name);
	    if (hp && streq(hp->h_name, own_hostname))
		c->c_conn = CONN_OWN_HOST; /* dont try to connect to ourselfs */
	    else 
		c->c_conn = connAddEntry(c->c_name);
	}
	if (c->c_conn == NULL) { /* still NULL, assume no memory */
	    error("memory exhausted, connection `%s' could not be established\n",
		  c->c_name);
	}
	else if (c->c_conn != CONN_OWN_HOST) {
	    e = c->c_conn;
	    e->cmds = (void *) cmds; /* restore cmds pointer */
	    if (e->state == CONN_DEAD || e->state == CONN_CONNECTING) {
		connConnectToServer(e);
		e->retry_interval = CONNECT_RETRY_INTERVAL;
		e->retry_time = time(NULL) + e->retry_interval;
	    }
	}
    }

    for (e = conn_entry_head.next; e; e = next) {
	next = e->next;		/* e perhaps free'd below, therefore save next */
	if (e->cmds == NULL) {
	    connTerminateConnection(e);
	    connRemoveEntry(e);
	}
    }
}


void connConnectToServer(ConnEntry e)
{
    assert(e->state == CONN_DEAD || e->state == CONN_CONNECTING);

    debuglog(DEBUG_CONN, ("%s to %s\n",
			  (e->state == CONN_CONNECTING
			   ? "still connecting"
			   : (e->retry_time 
			      ? "retrying connect"
			      : "connecting")),
			  e->hostname));

    if (e->state == CONN_DEAD) {
	int s;

	assert(e->masterfd == -1);

	s = sockClientNonBlocking(e->hostname, SERVICE_OMIRR, IPPORT_OMIRR,
				  &e->mastersock_addr);
	if (s == -1) {
	    /* `error()' already called from `sockClientNonBlocking' */
	    return;		/* failed */
	}
	e->masterfd = s;
	e->state = CONN_CONNECTING;
    }

    assert(e->masterfd != -1);

    if (sockConnect(e->masterfd, &e->mastersock_addr) == -1) {
	if ((errno == EINPROGRESS || errno == EALREADY)) {
	    /* This or a previous connect attempt is still in
               progress, so keep state on CONN_CONNECTING and return.  */
	    assert(e->state == CONN_CONNECTING);
	    return;
	}
	else if (errno == EISCONN) {
	    /* A previous connect attempt succeeded in the meantime,
	       so change state from CONN_CONNECTING to CONN_CONNECTED
	       and continue.  */
	    e->state = CONN_CONNECTED;
	}
	else {			/* connect failed completely */
	    error("connect to %s failed%s: %s\n", e->hostname,
		  (e->retry_time ? " again" : ""), xstrerror(errno));
	    connCloseMasterFd(e);
	    return;
	}
    }
    else {
	/* Connect succeeded, so change state to CONN_CONNECTED.  */
	e->state = CONN_CONNECTED;
    }

    debuglog(DEBUG_CONN, ("connected to %s on fd %d\n",
			  e->hostname, e->masterfd));

    /* bio_context: allocate buffer for buffered, non-blocking I/O on
       `e->masterfd'. Write operations are multiplexed via `mp_context'.  */
    bioAddWriteBuffer(bio_context, e->masterfd, mp_context);

    /* mp_context: add `close_daemon_connection' as error callback
       to close socket connections in case of an error.  */
    mpAddErrorCallback(mp_context, e->masterfd, close_daemon_connection, e);

    /* XXX add lio/mp read callback for acknowledgement */

    connProcessRedoStore(e);
#if 0 /* replaced by 'Z' protocol request sent to slave */
    connSyncToServer(e);
#endif
}


static void connCloseMasterFd(ConnEntry e)
{
    if (e->state == CONN_ACTIVE)
	connCloseSlaveFd(e);

    assert(e->slavefd == -1);
    assert(e->state == CONN_CONNECTING || e->state == CONN_CONNECTED);

    if (e->state == CONN_CONNECTED) {
	/* Remove callbacks added in `connConnectToServer' for this socket.  */
	bioRemoveWriteBuffer(bio_context, e->masterfd);
	mpRemoveErrorCallback(mp_context, e->masterfd);

	/* Reset connect retry timers.  */
	e->retry_interval = CONNECT_RETRY_INTERVAL;
	e->retry_time = time(NULL) + e->retry_interval;
    }

    close(e->masterfd);
    e->masterfd = -1;
    e->state = CONN_DEAD;
}



void connTerminateConnection(ConnEntry e)
{
    if (e->state == CONN_DEAD)
	return;			/* nothing to do */

    debuglog(DEBUG_CONN, ("terminating connection to %s\n", e->hostname));

    /* XXX write termination message */
    /* XXX close should be delayed until everything is written */
    connCloseMasterFd(e);	/* closes slavefd, too */
}



void connDoTerminateConnectionsToServers(void)
{
    ConnEntry e;

    for (e = conn_entry_head.next; e; e = e->next)
	connTerminateConnection(e);
}



/*
 * Functions to accept (other) server(s) as clients (set slavefd).
 */

/* XXX reject_client is a hack. Should be fixed.  */
#include <sys/uio.h>
static void reject_client_msg(int s, char *msg)
{
    int i;
    struct iovec iov[10];

    i = 0;
    iov[i].iov_base = "\005";	iov[i++].iov_len = 1;
    iov[i].iov_base = msg;	iov[i++].iov_len = strlen(msg);
    writev(s, iov, i);	/* if this fails, never mind */
}



/* connAcceptClient - accept a new client on `sockfd'. Then check whether
   the connection is acceptable: reject it if either the client
   address is malformed, the client port is not a reserved one or the
   client hostname is not on our list of hosts to talk with.  If the
   connection is rejected, the socket descriptor of the client
   connection is immediately closed (after sending a error protocol
   message). Otherwise the welcome protocol message is send and the
   client socket fd is added to mp.  */
void connAcceptClient(int sockfd)
{
    int s;			/* accepted socket fd */
    struct sockaddr_in sock_addr; /* client sockaddr_in */
    int sock_addr_len;		/* length of client sockaddr_in */
    const char *hostname;	/* hostname of client (from sock_addr) */
    ConnEntry e;		/* matching connection entry for hostname */

    debuglog(DEBUG_CONN, ("accepting new connection from fd %d\n", sockfd));

    s = sockAccept(sockfd, &sock_addr, &sock_addr_len);
    if (s == -1) {
	/* XXX TODO: handle fatal accept errors.  */
	error("accept of new client failed: %s\n", xstrerror(errno));
	return;
    }

    /* Check format and address family of sock_addr.  */
    if (sock_addr_len != sizeof(sock_addr) || sock_addr.sin_family != AF_INET) {
	error("malformed from address, address family %d\n",
	      sock_addr.sin_family);
	reject_client_msg(s, "malformed from address");
	close(s);
	return;
    }

    hostname = sockHostname(&sock_addr); /* try to get client hostname */
    if (!hostname) {
	error("connection from unknown host `%s'\n",
	      inet_ntoa(sock_addr.sin_addr));
	reject_client_msg(s, "unknown host name");
	close(s);
	return;
    }

    /* Check port number of client, which must be a reserved port.  */
    if (!sockReservedPort(&sock_addr)) {
	error("connection from `%s' on illegal port %d\n",
	      hostname, ntohs(sock_addr.sin_port));
	reject_client_msg(s, "illegal port");
	close(s);
	return;
    }

    /* Check hostname of client whether it is on our list.  */
    e = connFindByHostname(hostname);
    if (!e) {
	error("connection from `%s' denied\n", hostname);
	reject_client_msg(s, "access denied");
	close(s);
	return;
    }

    if (e->state == CONN_ACTIVE) { /* Huh, already active??? */
	error("connection from `%s' rejected because already connected\n",
	      hostname);
	reject_client_msg(s, "already connected");
	close(s);
	return;
    }

    /* If we don't have a (back-)connection to the partner yet, try to
       get one. If that fails, we reject the connect attempt, too.  */
    if (e->state == CONN_DEAD || e->state == CONN_CONNECTING) {
	int retries;
	int timo = BACKCONNECT_INITIALTIMO;

	for (retries = BACKCONNECT_RETRIES; retries; timo *= 2, retries--) {
	    connConnectToServer(e);	/* try to connect */
	    if (e->state == CONN_CONNECTED)
		break;
	    if (retries > 1)	/* no need to sleep for last repetition */
		sleep(timo);
	}
    }

    if (e->state != CONN_CONNECTED) { /* still not connected */
	error("connection from `%s' rejected because back-connect failed\n",
	      hostname);
	reject_client_msg(s, "back-connect failed");
	close(s);
	return;
    }

    debuglog(DEBUG_CONN, ("accepted new connection from %s on fd %d\n",
			  hostname, s));

    assert(e->slavefd == -1);
    assert(e->state == CONN_CONNECTED);

    e->slavefd = s;
    e->state = CONN_ACTIVE;

    /* XXX add bioAddWriteBuffer(s) to write acknowledgement */

    /* lio_context: add `process_daemon_message' as callback.  */
    lioAddReadCallback(lio_context, s, process_daemon_message, e);
	
    /* mp_context: add `lioMpReadFunc' as callback, which will split
       the input stream into lines and call the callback function of
       the lio context.  */
    mpAddReadCallback(mp_context, s, lioMpReadFunc, (void *) lio_context);

    /* mp_context: add `close_daemon_connection' as error callback
       to close socket connections in case of an error.  */
    mpAddErrorCallback(mp_context, s, close_daemon_connection, e);

    { /* XXX hack */
	int i;
	struct iovec iov[10];
	
	i = 0;
	iov[i].iov_base = "\001";
	iov[i++].iov_len = 1;
	iov[i].iov_base = "omirrd version ";
	iov[i++].iov_len = strlen(iov[i].iov_base);
	iov[i].iov_base = program_version;
	iov[i++].iov_len = strlen(iov[i].iov_base);
	writev(s, iov, i);
    }
}


static void connCloseSlaveFd(ConnEntry e)
{
    assert(e->state == CONN_ACTIVE);

    /* Remove callbacks added in `connAcceptClient' for this socket.  */
    lioRemoveReadCallback(lio_context, e->slavefd);
    mpRemoveReadCallback(mp_context, e->slavefd);
    mpRemoveErrorCallback(mp_context, e->slavefd);

    close(e->slavefd);
    e->slavefd = -1;
    e->state = CONN_CONNECTED;
}





/*
 * I/O on connections
 */

void connWriteMessage(ReqRequest r)
{
    ConnEntry e;

    for (e = conn_entry_head.next; e; e = e->next) {
	/* If this connection is not functional and the retry timer
           has expired, try again to connect. Then recompute the retry
           timer values.  */
	if (((e->state == CONN_DEAD || e->state == CONN_CONNECTING)
	     && time(NULL) > e->retry_time)) {
	    connConnectToServer(e);
	    /* Recompute connect retry timers.  */
	    e->retry_interval *= 2;
	    if (e->retry_interval > CONNECT_RETRY_MAX_INTERVAL)
		e->retry_interval = CONNECT_RETRY_MAX_INTERVAL;
	    e->retry_time = time(NULL) + e->retry_interval;
	}
	if (e->state == CONN_CONNECTED || e->state == CONN_ACTIVE)
	    opMaster(r, e);
	else
	    opStoreRequest(r, e);
    }
}




/**********************************************************************
 * Resync
 **********************************************************************/

/* XXX very ugly, should be redone */
struct linkbuf {
    ino_t inum;
    dev_t devnum;
    int	count;
    char pathname[2048];
    struct linkbuf *nextp;
};

static struct linkbuf *ihead;	/* list of files with more than one link */

static struct linkbuf *savelink(char *filename, struct stat *stp)
{
    struct linkbuf *lp;

    for (lp = ihead; lp != NULL; lp = lp->nextp)
	if (lp->inum == stp->st_ino && lp->devnum == stp->st_dev) {
	    lp->count--;
	    return(lp);
	}

    lp = (struct linkbuf *) xmalloc(sizeof(*lp));
    lp->nextp = ihead;
    ihead = lp;
    lp->inum = stp->st_ino;
    lp->devnum = stp->st_dev;
    lp->count = stp->st_nlink - 1;
    strcpy(lp->pathname, filename);
    return(NULL);
}


/* XXX error handling, replace by ftw(3c)? */
void sync_to_server(ConnEntry e, char *filename)
{
    struct stat statbuf;
    DIR *dir;
    struct dirent *dirent;
    char *newname;
    int newnamelen;
    int allocated = 0;
    ReqRequestRec r;
    struct linkbuf *lp; /* XXX name? */

    /* process filename */
    if (lstat(filename, &statbuf) == -1) {
	if (errno != ENOENT)
	    /* XXX error handling */;
	return;
    }

    /* XXX Note that requests r built below are not queued and can
       therefore not be processed by ack/nack responses. This should
       be redone.  */

    /* XXX The following specific requests should be replaced by 'X'.  */

    r.path.path = filename;
    r.newpath.path = NULL;
    r.timestamp = statbuf.st_mtime;

    if (S_ISLNK(statbuf.st_mode)) {
	int result;
	char *buf;
	int buf_len = 512;

	for (;;) {
	    buf = alloca(buf_len);
	    result = Readlink(filename, buf, buf_len);
	    if (result == -1)
		return;
	    if (result < buf_len) {
		buf[result] = '\0';
		break;
	    }
	    buf_len *= 2;
	}
	r.op = 's';
	r.s = buf;
	opMaster(&r, e);
    }
    else if (S_ISDIR(statbuf.st_mode)) {
	r.op = 'd';
	r.i1 = statbuf.st_mode;
	r.i2 = 0;
	opMaster(&r, e);
	r.op = 'O';
	r.i1 = statbuf.st_uid;
	r.i2 = statbuf.st_gid;
	opMaster(&r, e);
    }
    else if (statbuf.st_nlink > 1 /* handle hard links */
	     && ((lp = savelink(filename, &statbuf)) != NULL)) {
	/* install link */
	r.op = 'l';
	r.path.path = lp->pathname;
	r.newpath.path = filename;
	opMaster(&r, e);
	r.newpath.path = NULL;
    }
    else if (S_ISBLK(statbuf.st_mode)
	     || S_ISCHR(statbuf.st_mode)
	     || S_ISFIFO(statbuf.st_mode)) {
	r.op = 'n';
	r.i1 = statbuf.st_mode;
	r.i2 = statbuf.st_rdev;
	opMaster(&r, e);
	r.op = 'O';
	r.i1 = statbuf.st_uid;
	r.i2 = statbuf.st_gid;
	opMaster(&r, e);
    }
    else if (S_ISREG(statbuf.st_mode)) {
#if 0 /* XXX not useful because this truncates file on remote host */
	r.op = 'c';
	r.i1 = statbuf.st_mode;
	r.i2 = 0;
	opMaster(&r, e);
	r.op = 'O';
	r.i1 = statbuf.st_uid;
	r.i2 = statbuf.st_gid;
	opMaster(&r, e);
#endif
	r.op = 'W';
	r.i1 = statbuf.st_mode;
	r.i2 = statbuf.st_size;
	opMaster(&r, e);
#if 1
	r.op = 'O';
	r.i1 = statbuf.st_uid;
	r.i2 = statbuf.st_gid;
	opMaster(&r, e);
#endif
    }
    else /* XXX error - ignore */;

    /* recursive subdirectory processing */
    dir = opendir(filename);
    if (dir) {
	while ((dirent = readdir(dir))) {
	    if (*dirent->d_name == '.'
		&& (dirent->d_name[1] == '\0'
		    || dirent->d_name[1] == '.' && dirent->d_name[2] == '\0'))
		continue;
	    newnamelen = strlen(filename) + 1 + strlen(dirent->d_name);
	    if (newnamelen + 1 > allocated) {
		if (allocated == 0)
		    newname = xmalloc(newnamelen + 1);
		else
		    newname = xrealloc(newname, newnamelen + 1);
		allocated = newnamelen + 1;
	    }
	    sprintf(newname, "%s/%s", filename, dirent->d_name);
	    sync_to_server(e, newname);
	}
	if (allocated)
	    free(newname);
	closedir(dir);
    }
}

void connSyncToServer(ConnEntry e)
{
    CfCmd c;
    CfNameList nl;
    glob_t globbuf;
    char **globv;

    for (c = (CfCmd) e->cmds; c; c = c->c_next) {
	if (c->c_conn != e)	/* wrong connection */
	    continue;
	if (c->c_type != C_TYPE_ARROW) /* wrong type */
	    continue;
	/* XXX c_label field ignored */
	for (nl = c->c_files; nl; nl = nl->n_next) {
	    glob(nl->n_name, 0, NULL, &globbuf); /* XXX error handling */
	    for (globv = globbuf.gl_pathv; globbuf.gl_pathc--; globv++)
		sync_to_server(e, *globv);
	}
    }
}



void connDoSyncsToServers(void)
{
    ConnEntry e;

    for (e = conn_entry_head.next; e; e = e->next)
	connSyncToServer(e);
}


void connProcessRedoStore(ConnEntry e)
{
    char filename[1024];
    char buf[1024];
    long start, end;
    FILE *f;
    int len;
    static ReqQueue req_queue;
    ReqRequest r;
    ReqState state;

    if (!req_queue)
	req_queue = reqAllocQueue();

    sprintf(filename, "%s/omirr-%s-%s", store_dir, own_hostname, e->name);

    f = fopen(filename, "r+");
    if (f) {
	while (start = ftell(f), fgets(buf, sizeof(buf), f) != NULL) {
	    if (*buf == '-')	/* already processed last time */
		continue;

	    len = strlen(buf);	/* remove trailing newline from fgets */
	    if (len && buf[len - 1] == '\n')
	        buf[len - 1] = '\0';

printf("connProcessRedoStore: %s\n", buf);

	    /* Add message to request queue and send it to slave.  */
	    if ((r = reqParseMessage(req_queue, buf))) {
	      state = reqFillRequest(r);
	      assert (state == REQ_COMPLETE);
	      opMaster(r, e);
	    }
	    reqDropFirstRequest(req_queue);

	    end = ftell(f);
	    fseek(f, start, SEEK_SET);
	    putc('-', f);
	    fseek(f, end, SEEK_SET);
	    fflush(f);
	}
	fclose(f);
	unlink(filename);
    }
    bioWrite(bio_context, e->masterfd, "Z", 2);
}



/*
 * Message processing.
 */

/*ARGSUSED*/
static void process_daemon_message(char *line, void *data)
{
    ConnEntry e = data;
    ConnEntry inactive_e;
    static ReqQueue req_queue;
    ReqRequest r;
    ReqState state;

    if (*line == 'D' && strlen(line) > 33) {
	debuglog(DEBUG_DPROT, ("from %s: %30.30s...\n", e->hostname, line));
    }
    else {
	debuglog(DEBUG_DPROT, ("from %s: %s\n", e->hostname, line));
    }

    if (!req_queue)
	req_queue = reqAllocQueue();

    /* Add message to request queue and process it.  */
    if ((r = reqParseMessage(req_queue, line))) {
	switch (r->op) {
	  case 'Z':
	    connSyncToServer(e);
printf("synchronizing with %s\n", e->name);
	    break;
	  default:
	    state = reqFillRequest(r);
	    assert(state == REQ_COMPLETE);
	    for (inactive_e = conn_entry_head.next; inactive_e; inactive_e = inactive_e->next) {
	        if (inactive_e != e
		    && (inactive_e->state == CONN_DEAD
			|| inactive_e->state == CONN_CONNECTING))
		    opStoreRequest(r, inactive_e);
	    }
	    opSlave(r, e);
	    /* Immediately remove request from queue again.  */
	    reqDropFirstRequest(req_queue);
	}
    }

#ifdef C_ALLOCA
    /* If C alloca is used, it is a good idea to call the alloca
       garbage collector in the main loop. Since this is the usual
       callback function from the main loop, we call it here.  */
    (void) alloca(0);
#endif
}


/*ARGSUSED*/
static int close_daemon_connection(int fd, void *data)
{
    ConnEntry e = data;

    assert(fd == e->slavefd || fd == e->masterfd);

    error("lost %s connection to %s: %s\n",
	  (fd == e->masterfd ? "master" : "slave"),
	  e->hostname, xstrerror(errno));
    connCloseMasterFd(e);	/* closes slavefd, too */

    return 0;
}



#ifdef DEBUG
/*
 * Debugging
 */

void connDebugConnections(FILE *f)
{
    ConnEntry e;
    int count;
    int retry_delta;

    for (count = 0, e = conn_entry_head.next; e; count++, e = e->next) {
	fprintf(f, "Connection [%d] to %s (%s)\n", count, e->hostname, e->name);
	switch (e->state) {
	  case CONN_DEAD:
	    fprintf(f, "\tDEAD");
	    /*FALLTHROUGH*/
	  case CONN_CONNECTING:
	    if (e->state == CONN_CONNECTING)
		fprintf(f, "\tCONNECTING");
	    if ((retry_delta = e->retry_time - time(NULL)) > 0)
		fprintf(f, ", retry in %d secs", retry_delta);
	    else
		fprintf(f, ", retry pending");
	    fprintf(f, ", interval %d secs\n", e->retry_interval);
	    break;
	  case CONN_CONNECTED:
	    fprintf(f, "\tCONNECTED\n");
	    break;
	  case CONN_ACTIVE:
	    fprintf(f, "\tACTIVE\n");
	    break;
	  default:
	    fprintf(f, "\t<unknown>\n");
	    break;
	}
	fprintf(f, "\tmasterfd %d, slavefd %d\n", e->masterfd, e->slavefd);
    }
    fflush(f);
}
#endif

