/* filename: rlpr-rlpq.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr-rlpq.c,v 1.1 1997/04/27 19:26:51 meem Exp meem $
 * contents: contains routines for the `rlpq' client of the rlpr package
 *
 * Time-stamp: <1997/05/04 14:37:03 cdt -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996, 1997 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <pwd.h>

#include "getopt.h"
#include "rlpr-rlpq.h"
#include "rlpr-msg.h"                 /* rlpr messaging utility */
#include "rlpr-util.h"                /* utility routines */
#include "rlpr-client.h"              /* common to parts of the rlpr client */
#include "rlpr-rfc1179.h"

static rlpr_rlpq_info rlpr_rlpq_props = { 0, /* all args */ };
extern struct option  rlpr_long_opts[]; 
extern const char *   rlpr_opts;

char * program_name;		      /* our name */

int main(int argc, char *argv[])
{
  int    sockfd;
  int    orig_argc = argc;            /* original # args */

  int    retval    = EXIT_SUCCESS;    /* assume an entry was in the queue  */

  program_name     = argv[0];
  
  toggle_euid();                      /* lose euid of root */

  argc -= rlpr_rlpq_args(orig_argc, argv);
  argv += orig_argc - argc;

  sockfd = rlpr_client_open();        /* server now connected */

  rlpq_send_queue_req(sockfd, argv);

  /* read from socket, pump it out only if --quiet or --silent isn't on */

  {
    ssize_t count;
    char buf[BUFSIZ * 4];
    char not_found[] = "no entries";
    
    char quiet_on    = *(char *)rlpr_msg_get(RLPR_MSG_QUIET);

    while ((count = read(sockfd, buf, sizeof buf)) > 0) 

      if (!quiet_on)
	safe_writen(STDOUT_FILENO, buf, count);
      else {
	if (count >= strlen(not_found) &&
	    strncmp(buf, strlower(buf), strlen(not_found)) == 0)
	  retval = EXIT_FAILURE;
      }
  }

  rlpr_client_close(sockfd);          /* server now disconnected */
  
  exit(retval);
}

int rlpr_rlpq_args(int argc, char *argv[])
{
  int c, num_args = 1;               /* the number of arguments read */
  extern int optind;

  optind = 0;                         /* reset to start of argv       */

  while ((c = getopt_long(argc, argv, rlpr_opts, rlpr_long_opts, NULL)) != EOF) {
    num_args++;
    switch(c) {
    case 'l':
      rlpr_rlpq_props.longlist = 1;
      break;
    case 'm':
      rlpr_rlpq_props.mine     = 1;
      break;
    case 'V':
      fprintf(stdout, "%s: version "VERSION" from "__DATE__" "__TIME__ 
              " -- meem@gnu.ai.mit.edu\n", program_name);
      exit(EXIT_SUCCESS);
    case -2:
      printf("usage: %s [--long] [--mine] [job # ... ] [ user ... ]"
             "\n\nplease see the manpage for detailed help.\n", program_name);
      exit(EXIT_SUCCESS);
    }
  }

  rlpr_msg_args(argc, argv);    /* initialize the messaging subsystem */
  rlpr_client_args(argc, argv); /* initialize the client subsystem    */

  return num_args;
}

void rlpq_send_queue_req(int sockfd, char *idlist[]) 
{
  char    c       = rlpr_rlpq_props.longlist ? SENDQL : SENDQS;
  char *  printer = rlpr_client_get(RLPR_CLIENT_PRINTER);
  char *  list[]  = { NULL, NULL };
  
  rlpr_msg(DEBUG, NO_ERRNO, "sending %s queue request to lpd for queue %s...",
           c == SENDQL ? "long" : "short", printer);

  safe_writen(sockfd, &c, 1);
  safe_writen(sockfd, printer, strlen(printer));

  if (rlpr_rlpq_props.mine) {
    struct passwd * id = getpwuid(getuid());
    list[0]            = id ? id->pw_name : "";
    idlist             = list;
  }
   
  for (; idlist && *idlist; idlist++) {
    safe_writen(sockfd, " ", 1);
    safe_writen(sockfd, *idlist, strlen(*idlist));
  }

   safe_writen(sockfd, "\n", 1);
}

struct option rlpr_long_opts[] = {
  { "help",      0, NULL, -2   },
  { "long",      0, NULL, 'l'  },
  { "mine",      0, NULL, 'm'  },
  { "version",   0, NULL, 'V'  },
  { "debug"   ,  0, NULL, -101 },
  { "quiet"   ,  0, NULL, 'q'  },
  { "silent"  ,  0, NULL, 'q'  },
  { "verbose" ,  0, NULL, -102 },
  { "no-bind" ,  0, NULL, 'N'  },
  { "port"    ,  1, NULL, -201 },
  { "printer" ,  1, NULL, 'P'  },
  { "queue"   ,  1, NULL, 'P'  },
  { "printhost", 1, NULL, 'H'  },
  { "proxy"   ,  1, NULL, 'X'  },
  { "proxyhost", 1, NULL, 'X'  },
  { 0, 0, 0, 0 }
};

const char * rlpr_opts = "lmVqP:Q:NH:X:";
