/*

 X-Chat
 
Copyright (c) 1998 Peter Zelezny.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#define BUFSIZE (prefs.bufsize)

#include "xchat.h"
#include <netdb.h>
#include <arpa/inet.h>
#include <gdk/gdkkeysyms.h>
#include "serverlist.h"
#include "urlgrab.h"


GdkFont *font_normal = 0;
//GdkFont *font_underline = 0;
GdkFont *font_bold = 0;

GdkFont *dialog_font_normal = 0;
//GdkFont *dialog_font_underline = 0;
GdkFont *dialog_font_bold = 0;

GSList *sess_list = 0;

extern GdkColor colors[];
struct server firstserver;
struct DCC firstDCC;
struct dialog firstdialog;
struct serverentry firstserverentry;
struct url firsturl;
struct xchatprefs prefs;

void xchat_cleanup(void);
void change_num_ops(struct session *sess, int change);
void change_num_total(struct session *sess, int change);
void add_name(struct session *sess, char *name, int sort);
void kill_session(GtkWidget *win, struct session *sess);
struct session *new_session(struct server *serv, char *channel);

// maingui.c

extern void create_window(struct session *sess, char *channel);

// rawlog.c

extern void add_rawlog(struct server *serv, char *text);
extern void close_rawlog(GtkWidget *wid, struct server *serv);

// outbound.c

extern void add_to_history(struct session *sess, char *cmd);
extern void handle_inputgad(GtkWidget *igad, struct session *sess);

// inbound.c

extern void new_topic(char *buf);
extern void names_list(char *buf);
extern void you_joined(struct session *sess, char *chan);
extern void user_joined(char *chan, char *user, char *ip);
extern void new_nick(struct session *sess, char *newnick);
extern void process_line(struct session *sess, struct server *serv, char *buf);

// dcc.c

extern void free_dccs(void);

// dialog.c

extern void free_dialogs(void);
extern struct dialog *new_dialog(char *nick);

// server.c

extern void connect_server(struct session *sess, char *server, int port, int quiet);
extern void disconnect_server(struct session *sess, int sendquit);

// serverlist.c

extern void free_serverentrys(void);
extern void load_serverentrys(void);
extern void open_server_list(GtkWidget *widd, struct session *sess);

// config.c

extern void save_config(void);
extern void load_config(void);

// userlist.c

extern void init_userlist_xpm(struct session *);
extern struct user *find_name(struct session *sess, char *name);


void cut_down_text(GtkWidget *textwidget)
{
   long n = gtk_text_get_length((GtkText*)textwidget);			 
   if(n > BUFSIZE && BUFSIZE > 0)
   {
      gtk_text_set_point((GtkText*)textwidget, n - BUFSIZE);
      gtk_text_backward_delete((GtkText*)textwidget, n - BUFSIZE);
      gtk_text_set_point((GtkText*)textwidget, gtk_text_get_length((GtkText*)textwidget));
   }
}

void PrintTextRaw(GtkWidget *textwidget, unsigned char *text,
		  char bg, char fg,
		  GdkFont *fontnorm,
		  GdkFont *fontbold)
{
   int esc = FALSE;
   int comma = FALSE;
   unsigned char buf[4096];
   unsigned char num[4];
   int bcol = bg, col = fg, j = 0, i = 0, k = 0;
   int scroll = FALSE;
   GtkAdjustment *adj;
   GdkFont *font = fontnorm;
   int bold = FALSE;

   adj = (GTK_TEXT(textwidget))->vadj;
   if(adj->value == adj->upper - adj->lower - adj->page_size) scroll = TRUE;

   gtk_text_freeze (GTK_TEXT(textwidget));
   while(1)
   {
      if(esc)
      {
	 if(text[i] == ' ')
	 {
	    esc = FALSE; bcol = bg; col = fg;
	 } else {
	    while(text[i] >= '0' && text[i] <= '9' && k < 2)
	    {
	       num[k] = text[i];
	       k++; i++;
	    }
	    num[k] = 0; k = 0;
	    switch(text[i])
	    {
	     case ',':
	       comma = TRUE;
	       col = atoi(num);
	       if(col < 0 || col > 15) col = 1;
	       break;
	     default:
	       if(comma)
	       {
		  comma = FALSE;
		  bcol = atoi(num);
		  if(bcol < 0 || bcol > 15) bcol = bg;
	       } else {
		  col = atoi(num);
		  if(col < 0 || col > 15) col = fg;
	       }
	       goto norm;
	    }
	 }
      } else {
	 switch(text[i])
	 {
	  case 0:
	    goto jump;
	  case 3: // CTRL-C
	    buf[j] = 0;
	    if(j>0)
	    {
	       gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol],  buf, -1);
	       j = 0;
	    }
	    esc = TRUE; k = 0;
	    break;
	  case '\026': // REVERSE
	    break;
	  case '\037': // UNDERLINE
	    break;
	  case '\002': // BOLD
	    buf[j] = 0;
	    if(j>0)
	    {
	       gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol],  buf, -1);
	       j = 0;
	    }
	    if(bold)
	    {
	       bold = FALSE;
	       font = fontnorm;
	    } else {
	       bold = TRUE;
	       font = fontbold;
	    }
	    break;
	  case '\017': // ALL OFF
	    buf[j] = 0;
	    if(j>0)
	    {
	       gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol],  buf, -1);
	       j = 0;
	    }
	    font = fontnorm;
	    break;
	  default:
	    norm:
	    esc = FALSE;
	    comma = FALSE;
	    buf[j] = text[i];
	    j++;
	    if(j == 4095) j = 4095;
	 }
      }
      i++;
   }
   jump:
   if(j)
   {
      GtkAdjustment *adj = (GTK_TEXT(textwidget))->vadj;
      buf[j] = 0;
      gtk_text_insert(GTK_TEXT(textwidget), font, &colors[col], &colors[bcol], buf, -1);
      cut_down_text(textwidget);
      gtk_text_thaw(GTK_TEXT(textwidget));
      if(scroll)
      	 gtk_adjustment_set_value(adj, adj->upper - adj->lower - adj->page_size);
   } else {
     cut_down_text(textwidget);
     gtk_text_thaw(GTK_TEXT(textwidget));
   }
}

void PrintTextServ(struct server *serv, unsigned char *text)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(sess->server == serv)
      {
	 PrintTextRaw(sess->textgad, text, prefs.bg_color, prefs.fg_color,
			   font_normal, font_bold);
	 return;
      }
      list = list->next;
   }
}

void PrintText(struct session *sess, unsigned char *text)
{
   if(!sess) sess = (struct session *)sess_list->data;
   PrintTextRaw(sess->textgad, text, prefs.bg_color, prefs.fg_color,
		font_normal, font_bold);
}

struct session *find_session_from_nick(char *nick, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(find_name(sess, nick)) return sess;
      list = list->next;
   }
   return 0;
}

struct session *find_session_from_channel(char *chan, struct server *serv)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(!strcasecmp(chan, sess->channel))
      {
	 if(!serv) return sess;
	 if(serv == sess->server) return sess;
      }
      list = list->next;
   }
   return 0;
}

struct session *find_session_from_waitchannel(char *chan)
{
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      if(!strcasecmp(chan, sess->waitchannel)) return sess;
      list = list->next;
   }
   return 0;
}

void show_generic_channel(struct server *serv, char *chan, char *msg)
{
   struct session *sess = find_session_from_channel(chan, serv);
   if(sess) PrintText(sess, msg);
}

struct server *find_serv_from_sok(gint sok)
{
   struct server *serv = &firstserver;
   while(serv)
   {
      if(serv->used)
      {
	 if(serv->sok == sok) return(serv);
      }
      serv = serv->next;
   }
   return 0;
}

void read_data(struct session *sess, gint sok)
{
   int len;
   char tbuf[2050];
   struct server *serv;

   if(!sess)
     serv = find_serv_from_sok(sok);
   else
     serv = sess->server;

   if(!serv) return;

   if((len = recv(sok, tbuf, sizeof tbuf-2, 0)) < 1)
   {
      disconnect_server(sess, FALSE);
      if(prefs.flags&(1<<PREFS_AUTORECONNECT))
	connect_server(sess, sess->server->servername, sess->server->port, FALSE);	
   } else {
      int i = 0;
      tbuf[len] = 0;

      while(i < len)
      {
	 switch(tbuf[i])
	 {
	  case '\r':
	    break;
	  case '\n':
	    serv->linebuf[serv->pos] = 0;
	    process_line(sess, serv, serv->linebuf);
	    serv->pos = 0;
	    break;
	  default:
	    serv->linebuf[serv->pos] = tbuf[i];
	    serv->pos++;
	    if(serv->pos == 2047) serv->pos = 2046;
	 }
	 i++; 
      }
   }
}

void kill_session(GtkWidget *win, struct session *sess)
{
   int willquit = TRUE;
   struct server *serv = sess->server;
   struct session *killsess = sess;
   struct session *s;
   GSList *list = sess_list;

   while(list)
   {
      s = (struct session *)list->data;
      if(s != killsess)
      {
	 willquit = FALSE;
	 break;
      }
      list = list->next;
   }

   if(sess->server->front_session == sess) sess->server->front_session = 0;

   gtk_widget_destroy(win);

   if(sess->server && sess->server->used && (sess->server->flags&(1<<0)) && willquit)
   {
      char tbuf[128];
      sprintf(tbuf, "QUIT :%s\r\n", prefs.quitreason);
      send(sess->server->sok, tbuf, strlen(tbuf), 0);
   } else {
      if(sess->server && sess->server->used && (sess->server->flags&(1<<0)) && sess->channel[0])
      {
	 char tbuf[128];
	 sprintf(tbuf, "PART %s\r\n", sess->channel);
	 send(sess->server->sok, tbuf, strlen(tbuf), 0);
      }
   }

   sess_list = g_slist_remove(sess_list, sess);
   free(sess);

   if(willquit)
   {
       if(prefs.flags&(1<<PREFS_AUTOSAVE)) save_config();
   }

   if(sess_list)
   {
         list = sess_list;
	 while(list)
	 {
	    sess = (struct session *)list->data;
	    if(sess->server == serv) return;
	    list = list->next;
	 }
	 if(serv->flags&(1<<0)) // if CONNECTED
	 {
	    char tbuf[128];
	    sprintf(tbuf, "QUIT :%s\r\n", prefs.quitreason);
	    send(serv->sok, tbuf, strlen(tbuf), 0);
	    gdk_input_remove(serv->iotag);
	    shutdown(serv->sok, 2);
	    close(serv->sok);
	 }
	 serv->sok = -1;
	 serv->used = 0;
	 serv->pos = 0;
	 serv->flags = 0;
	 if(serv->rawlog_window) close_rawlog(0, serv);
	 return;
   }
   xchat_cleanup();
}

int handle_keypress(GtkWidget *widget, GdkEventKey *event, struct session *sess)
{  
   //gtk_widget_grab_focus(GTK_WIDGET(sess->inputgad));
   switch (event->keyval)
   {
    /*case 65471: f1
      break;
    case 65472: f2
      break;*/
      
    case GDK_Down:
      sess->pos++;
      if(sess->pos == 100) sess->pos = 0;
      if(sess->history[sess->pos])
	{
	   gtk_entry_set_text(GTK_ENTRY(sess->inputgad), sess->history[sess->pos]);
	} else {
	   if(sess->pos != 0)
	     sess->pos--;
	   else
	     sess->pos = 99;
	}
	break;
 
    case GDK_Up:
      if(sess->pos == 0)
	sess->pos = 99;
      else
	sess->pos--;
       if(sess->history[sess->pos])
	{
	   gtk_entry_set_text(GTK_ENTRY(sess->inputgad), sess->history[sess->pos]);
	} else {
	   if(sess->pos == 99)
	     sess->pos = 0;
	   else
	     sess->pos++;
	}
      break;
 
      /*case GDK_Page_Up:
        break;
	
      case GDK_Page_Down:
	break;*/
      
    default:
      return 0;
   }
   gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
   return 1;
}

struct server *new_server(void)
{
   struct server *serv = &firstserver;
   struct server *prev;
   while(1)
   {
      if(serv->used == 0)
      {
	 serv->used = 1;
	 serv->servername[0] = 0;
	 serv->password[0] = 0;
	 serv->nickcount = 1;
	 serv->front_session = 0;
	 serv->sok = -1;
	 load_config();
	 strcpy(serv->nick, prefs.nick1);
	 return(serv);
      }
      prev = serv;
      serv = serv->next;
      if(!serv)
      {
	 serv = malloc(sizeof(struct server));
	 if(serv)
	 {
	    memset(serv, 0, sizeof(struct server));
	    prev->next = serv;
	 } else
	   return 0;
      }
   }
}

struct session *new_session(struct server *serv, char *channel)
{
   struct session *sess = malloc(sizeof(struct session));
   if(!sess) return 0;
   memset(sess, 0, sizeof(struct session));
   sess->selected = -1;
   sess->server = serv;
   
   create_window(sess, channel);
   if(channel) strcpy(sess->channel, channel);

   if(!font_normal)
     font_normal = gdk_font_load(prefs.font_normal);
   if(!font_bold)
     font_bold = gdk_font_load(prefs.font_bold);
   
   if(!dialog_font_normal)
     dialog_font_normal = gdk_font_load(prefs.dialog_font_normal);
   if(!dialog_font_bold)
     dialog_font_bold = gdk_font_load(prefs.dialog_font_bold);

   sess_list = g_slist_append(sess_list, sess);
   
   return(sess);
}

void free_servers(void)
{
   struct server *next;
   struct server *serv = firstserver.next;
   while(serv)
   {
      next = serv->next;
      free(serv);
      serv = next;
   }
}

void free_sessions(void)
{
   int i;
   struct session *sess;
   GSList *list = sess_list;
   while(list)
   {
      sess = (struct session *)list->data;
      for(i=0; i<100; i++)
      {
	 if(sess->history[i]) free(sess->history[i]);
      }
      free(sess);
      list = list->next;
   }
}

void xchat_init(void)
{
   memset(&firstserver, 0, sizeof(struct server));
   memset(&firstDCC, 0, sizeof(struct DCC));
   memset(&firstserverentry, 0, sizeof(struct serverentry));
   memset(&firsturl, 0, sizeof(struct url));

   load_config();

   load_serverentrys();

   //open_server_list(0, 0);
   
   {
      struct session *sess;
      struct server *serv;
      serv = new_server();
      if(serv)
      {
	 sess = new_session(serv, 0);
	 init_userlist_xpm(sess);
	 open_server_list(0, sess);
      }
   }
}

void xchat_cleanup(void)
{
   free_sessions();
   free_servers();
   free_dccs();
   free_dialogs();
   free_serverentrys();
   gtk_exit(0);
}

gint main(int argc, gchar *argv[])
{
   gtk_init(&argc, &argv);

   xchat_init();

   gtk_main();

   xchat_cleanup();

   return 0;
}
