/*
libutil -- miscellaneous stuff

Written by Arnt Gulbrandsen <agulbra@troll.no> and copyright 1995
Troll Tech AS, Postboks 6133 Etterstad, 0602 Oslo, Norway, fax +47
22646949.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>
and Randolf Skerka <rskerka@metronet.de>.
Copyright of the modifications 1997.
Modified by Kent Robotti <robotti@erols.com>. Copyright of the
modifications 1998.
Modified by Markus Enzenberger <enz@cip.physik.uni-muenchen.de>.
Copyright of the modifications 1998.
Modified by Cornelius Krasel <krasel@wpxx02.toxi.uni-wuerzburg.de>.
Copyright of the modifications 1998.

See file COPYING for restrictions on the use of this software.
*/

#include <fcntl.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdlib.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>

#include "leafnode.h"

#define   STRNCMPEQ(s1, s2, n)    (*(s1) == *(s2) && strncmp((s1), (s2), n) == 0)

char fqdn[256];

char s[PATH_MAX+1024]; /* long string, here to cut memory usage */

extern int errno;
extern int h_errno;
extern struct state _res;

/* no good but this server isn't going to be scalable so shut up */
const char * lookup (const char *msgid) {
    static char * name = NULL;
    static int namelen = 0;
    unsigned int r;
    unsigned int i;

    if (!msgid || !*msgid)
	return NULL;

    i = strlen(msgid)+strlen(spooldir)+30;

    if (!name) {
	name = (char *)malloc(i);
	namelen = i;
    } else if (i > namelen) {
	name = (char *)realloc(name, i);
	namelen = i;
    }

    if (!name) {
	syslog( LOG_ERR, "malloc(%d) failed", i );
	exit(1);
    }

    strcpy( name, spooldir );
    strcat( name, "/message.id/000/" );
    i = strlen( name );
    strcat( name, msgid );

    r = 0;
    do {
	if ( name[i] == '/' )
	    name[i] = '@';
	else if ( name[i] == '>' )
	    name[i+1] = '\0';
	r += (int)(name[i]);
	r += ++i;
    } while ( name[i] );

    i = strlen( spooldir )+14; /* to the last digit */
    r = (r%999)+1;
    name[i--] = '0'+(char)(r%10); r /= 10;
    name[i--] = '0'+(char)(r%10); r /= 10;
    name[i] = '0'+(char)(r);
    return name;
}



/*
 * replacement for malloc, syslogs allocation failures
 * and exits with the error message
 */
char * critmalloc(size_t size, const char* message) {
    char * a;

    a = malloc(size);
    if (!a) {
	syslog(LOG_ERR, "malloc(%d) failed: %s", (int)size, message);
	fprintf(stderr, "malloc(%d) failed: %s\n", (int)size, message);
	exit(1);
    }
    return a;
}

/*
 * replacement for realloc, syslogs allocation failures
 * and exits with the error message
 */
char * critrealloc(char *a, size_t size, const char* message) {
    a = realloc(a, size);
    if (!a) {
	syslog(LOG_ERR, "realloc(%d) failed: %s", (int)size, message);
	fprintf(stderr, "realloc(%d) failed: %s\n", (int)size, message);
	exit(1);
    }
    return a;
}

#define LM_SIZE 65536

static void makedir( char * d ) {
    char * p;
    char * q;

    if (!d || *d != '/' || chdir("/"))
	return;
    q = d;
    do {
	*q = '/';
	p = q;
	q = strchr( ++p, '/' );
	if (q)
	    *q = '\0';
	if (!chdir(p))
	    continue; /* ok, I do use it sometimes :) */
	if (errno==ENOENT)
	    if (mkdir(p, 0775)) {
		syslog( LOG_ERR, "mkdir %s: %m", d );
		exit( 1 );
	    }
	if (chdir(p)) {
	    syslog( LOG_ERR, "chdir %s: %m", d );
	    exit( 1 );
	}
    } while ( q );
}
	

/* chdir to the directory of the argument if it's a valid group */
void chdirgroup(const char *group) {
    char *p;

    if (group && *group) {
	strcpy(s, spooldir);
	p = s + strlen(s);
	*p++ = '/';
	strcpy(p, group);
	while(*p) {
	    if (*p=='.')
		*p = '/';
	    else
		*p = tolower(*p);
	    p++;
	}
	if (!chdir(s))
	    return;
	makedir( s );
	return;
    }
}


/* get the fully qualified domain name of this box into fqdn */

void whoami( void ) {
    struct hostent * he;

    if (!gethostname(fqdn, 255) && (he = gethostbyname(fqdn))!=NULL) {
	strncpy( fqdn, he->h_name, 255 );
	if (strchr(fqdn, '.') == NULL) {
	    char ** alias;
	    alias = he->h_aliases;
	    while( alias && *alias )
		if (strchr(*alias, '.') && (strlen(*alias)>strlen(fqdn)))
		    strncpy( fqdn, *alias, 255 );
		else
		    alias++;
	    }
    } else
	*fqdn = '\0';
}

/* next few routines implement a mapping from message-id to article
   number, and clearing the entire space */

struct msgidtree {
    struct msgidtree * left;
    struct msgidtree * right;
    int art;
    char msgid[1];
};

static struct msgidtree * head; /* starts as NULL */

void insertmsgid( const char * msgid, int art ) {
    struct msgidtree ** a;
    int c;

    if ( strchr( msgid, '@' ) == 0 )
	return;

    a = &head;
    while (a) {
	if (*a) {
	    /* comparing only by msgid is uncool because the tree becomes
	       very unbalanced */
	    c = strcmp(strchr((*a)->msgid, '@'), strchr(msgid, '@'));
	    if ( c == 0 )
		c = strcmp((*a)->msgid, msgid);
	    if (c<0)
		a = &((*a)->left);
	    else if (c>0)
		a = &((*a)->right);
	    else {
		return;
	    }
	} else {
	    *a = (struct msgidtree *)
		 critmalloc(sizeof(struct msgidtree) + strlen(msgid),
			    "Building expiry database");
	    (*a)->left = (*a)->right = NULL;
	    strcpy((*a)->msgid, msgid);
	    (*a)->art = art;
	    return;
	}
    }
}

int findmsgid( const char* msgid ) {
    struct msgidtree * a;
    int c;
    char * domainpart;

    /* domain part differs more than local-part, so try it first */

    domainpart = strchr( msgid, '@' );
    if ( domainpart == NULL )
	return 0;

    a = head;
    while (a) {
	c = strcmp(strchr(a->msgid, '@'), domainpart);
	if ( c == 0 )
	    c = strcmp(a->msgid, msgid);
	if ( c < 0 )
	    a = a->left;
	else if ( c > 0 )
	    a = a->right;
	else
	    return a->art;
    }
    return 0;
}

static void begone( struct msgidtree * m ) {
    if ( m ) {
	begone( m->right ) ;
	begone( m->left );
	free( (char *) m );
    }
}

void clearidtree( void ) {
    if ( head ) {
	begone( head->right ) ;
	begone( head->left );
	free( (char *) head );
    }
    head = NULL;
}


/* build and return an open fd to pseudoart in group */
FILE * buildpseudoart(const char * grp) {
    FILE * f;

    f = tmpfile();
    if (!f) {
	syslog( LOG_ERR, "Could not create pseudoarticle" );
	return f;
    }

    fprintf(f, "Path: %s\n", fqdn);
    fprintf(f, "Newsgroups: %s\n", grp);
    fprintf(f, "From: Leafnode <nobody@%s>\n", fqdn);
    fprintf(f, "Subject: Leafnode placeholder for group %s\n", grp);
    fprintf(f, "Date: %s\n", rfctime() );
    fprintf(f, "Message-ID: <leafnode:placeholder:%s@%s>\n", grp, fqdn);
    fprintf(f, "\n");
    fprintf(f,
	  "This server is running leafnode, which is a dynamic NNTP proxy.\n"
	  "This means that it does not retrieve newsgroups unless someone is\n"
	  "actively reading them.\n"
	  "\n"
	  "If you do an operation on a group - such as reading an article,\n"
	  "looking at the group table of contents or similar, then leafnode\n"
	  "will go and fetch articles from that group when it next updates.\n"
	  "\n"
	  "Since you have read this dummy article, leafnode will retrieve\n"
	  "the newsgroup %s when fetch is run\n"
	  "the next time. If you'll look into this group a little later, you\n"
	  "will see real articles.\n"
	  "\n"
	  "If you see articles in groups you do not read, that is almost\n"
	  "always because of cross-posting.  These articles do not occupy any\n"
	  "more space - they are hard-linked into each newsgroup directory.\n"
	  "\n"
	  "If you do not understand this, please talk to your newsmaster.\n"
	  "\n"
	  "Leafnode can be found at\n"
	  "\thttp://www.troll.no/freebies/leafnode.html\n\n\n", grp );

    rewind(f);
    return f;
}


const char* rfctime(void) {
    static char date[128];
    const char * months[] = { "Jan", "Feb", "Mar", "Apr",
	"May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
    const char * days[] = { "Sun", "Mon", "Tue", "Wed",
	"Thu", "Fri", "Sat" };
    time_t now;
    struct tm gmt;
    struct tm local;
    struct tm conv;
    signed int diff;

    /* get local and Greenwich times */
    now = time(0);
    gmt	  = *(gmtime(&now));
    local = *(localtime(&now));
    conv  = *(localtime(&now));

    /* if GMT were local, what would it be? */
    conv.tm_year = gmt.tm_year;
    conv.tm_mon	 = gmt.tm_mon;
    conv.tm_mday = gmt.tm_mday;
    conv.tm_hour = gmt.tm_hour;
    conv.tm_min	 = gmt.tm_min;
    conv.tm_sec	 = gmt.tm_sec;
    conv.tm_wday = gmt.tm_wday;
    conv.tm_yday = gmt.tm_yday;

    /* see how far the real local is from GMT-as-local */
    /* should convert to signed int from double properly */
    diff = (mktime(&local) - mktime(&conv)) / 60.0;

    /* finally print the string */
    sprintf(date, "%3s, %d %3s %4d %02d:%02d:%02d %+03d%02d",
	    days[local.tm_wday], local.tm_mday, months[local.tm_mon],
	    local.tm_year+1900, local.tm_hour, local.tm_min, local.tm_sec,
	    diff/60, diff%60);

    return (date);
}


#ifndef HAVE_STRDUP
char * strdup(const char *s) {
    char * s1;
    s1 = malloc(strlen(s)+1);
    if (!s1)
	return 0;
    strcpy (s1,s);
    return s1;
}
#endif
