#ifndef _ActiveDB_h_
#define _ActiveDB_h_

#include <ctype.h>

#include <iostream.h>

#include "NVcontainer.h"
#include "NSError.h"

class GroupInfo {
  nvtime_t _ctime;
  nvtime_t _mtime;
  unsigned int _fst,_lst,_n;
  char _flag;
  char _name[513];

public:
  GroupInfo() { clear(); }
  GroupInfo(const char *gd) { _name[sizeof(_name)-1]='\0'; set(gd); }

  void clear() { _ctime=_mtime=0; _name[0]=_name[sizeof(_name)-1]='\0'; }

  // All data-types stored in NVcontainers must have a set- and getraw
  // method
  void setraw(const char *buf, unsigned int bufsz) {
    if(bufsz>sizeof(_ctime)+sizeof(_mtime)+
       sizeof(_fst)+sizeof(_lst)+sizeof(_n)+
       sizeof(_flag)+sizeof(_name))
      throw Error("GroupInfo(24230): setraw buf too big");
    memcpy(this,buf,bufsz);
  }
  
  void getraw(const char **buf, unsigned int *bufsz) {
    *buf=(const char*)this; 
    *bufsz=sizeof(_ctime)+sizeof(_mtime)+
      sizeof(_fst)+sizeof(_lst)+sizeof(_n)+
      sizeof(_flag)+strlen(_name)+1;
  }

  void set(const char *gd) {
    const char *p=gd,*e;
    char *q=_name;
    unsigned int i=0;
    nvtime_t tm;

    while(i<sizeof(_name) && !isspace(*p)) { *q++=*p++; }
    if(i==sizeof(_name)) throw Error("Name of newsgroup too long");
    *q='\0';
    _lst=strtol(p,(char**)&p,10);
    _fst=strtol(p,(char**)&e,10); 
    if(_fst>_lst) _n=0;
    else _n=_lst-_fst+1;
    if(e==p) throw Error("gd not in form name int int flag");
    p=e;
    while(isspace(*p)) p++;
    _flag=*p;
    nvtime(&tm);
    _mtime=tm;
  }

  void set(const char *group, int fst, int lst, int n, char fl=0) {
    strncpy(_name,group,sizeof(_name));
    if(_name[sizeof(_name)-1]!='\0')
      throw Error("Name of newsgroup too long");
    setfln(fst,lst,n,fl);
  }

  void setfln(int first, int last, int n, char fl=0) {
    nvtime_t tm;
    _fst=first; _lst=last; _n=n; if(fl) _flag=fl;
    nvtime(&tm);
    _mtime=tm;
  }
  void setflag(char fl) {
    _flag=fl;
  }

  void setctime(nvtime_t ctime) { _ctime=ctime; }

  nvtime_t ctime() { return _ctime; }
  nvtime_t mtime() { return _mtime; }
  const char *name() { return _name; }
  unsigned int first() { return _fst; }
  unsigned int last() { return _lst; }
  unsigned int n() { return _n; }
  char flags() { return _flag; }

  friend ostream &operator <<(ostream &os, GroupInfo &gd) { 
    os << gd._name << ' ' 
       << gd._lst << ' ' << gd._fst << ' ' << gd._flag;
    return os;
  }
};

class ActiveDB {
protected:
  ActiveDB();
public:
  virtual ~ActiveDB();
  // Used for set
  enum { 
    update_only=0x01 
  };

  enum {
    m_active, m_times
  };

  enum { 
    F_STORE_READONLY=0x1,
    F_CLEAR=0x2
  };

  virtual void clear(void)=0;
  virtual int is_empty(void)=0;

  virtual void add(GroupInfo &gi)=0;
  virtual void set(GroupInfo &gi, int flags=0)=0;

  /* get(group,gi)
   * Returns information of the newsgroup >group<.
   * The space for gi must be allocated by the caller.
   */
  virtual int get(const char *group, GroupInfo *gi)=0;
  virtual int hasgroup(const char *group)=0;

  virtual void read(istream &is, const char *filter, int flags=0)=0;
  virtual void write(ostream &os, nvtime_t ctime=0, int mode=m_active, 
		     const char *filter=NULL)=0;

  virtual void setmtime(unsigned long tm, int force=0)=0;
  virtual void getmtime(unsigned long *tm)=0;
};

#endif
