#ifndef _Article_h_
#define _Article_h_

#include <ctype.h>

#include <string>

#include "NSError.h"
#include "Debug.h"
#include "readline.h"

/* class Article
 */
struct Article {
private:
  /* Article::find_field
   * Description:
   *   Return the position of a given field in the article's header.
   * Parameters:
   *   ifld ... The name of the field to be searched for including `:'
   * Exceptions:
   *   (none)
   */
  const char *find_field(const char *ifld) const {
    char hdrbuf[513],*p;
    const char *q,*fld;
    const char *ifp;
    unsigned int i;
    
    i=1; p=hdrbuf; ifp=ifld;
    while(i<sizeof(hdrbuf) && (*p=tolower(*ifp))) { i++; p++; ifp++; }
    if(i==sizeof(hdrbuf)) {
      p--;
      VERB(slog.p(Logger::Notice) << "Article::find_field: field exceeds 512 characters! Contact maintainer!\n");
    }
    *p='\0';

    q=_ctext;
    for(;;) {
      p=hdrbuf;
      fld=q;
      while(*p && *p==tolower(*q)) { p++; q++; }
      if(*p=='\0') return fld;
      // OK, bad luck find the next line
      while(*q!='\r' && *q!='\n') q++;
      
      // Skip first CRLF
      if(*q=='\n' && *(q+1)=='\n') return q+1;
      if(*q=='\r' && *(q+1)=='\n' && *(q+2)=='\r' && *(q+3)=='\n') return q+2;
      if(*q=='\r') q++;
      if(*q=='\n') q++;
      ASSERT(if(*q=='\r' || *q=='\n') {
	slog.p(Logger::Error) << "Line of article did not end with <CR-LF> or <LF>.\n";
	while(*q=='\r' || *q=='\n') q--;
	return q+1;
      });
    }
  }

protected:
  int _nbr;
  string _text;
  const char *_ctext;
  
public:
  enum {
    Head = 0x1,
    Body = 0x2
  };

  Article() {}
  Article(Article *a) {
    _nbr=a->getnbr();
    _text=a->gettext();
    _ctext=_text.c_str();
  }
  Article(int artnbr) { setnbr(artnbr); }
  Article(int artnbr, const char *text, int textlen=0) { 
    setnbr(artnbr);
    _text.assign(text,textlen);
    _ctext=_text.c_str();
  }
  ~Article() {}
  
  void read(istream &is) {
    DEBUG(slog.p(Logger::Debug) << "Article::read(&is)\n");
    _text="";
    readtext(is,_text);
    _ctext=_text.c_str();
  }

  void setnbr(int nbr) { _nbr=nbr; }
  void clear() { _nbr=0; _text=""; _ctext=_text.c_str(); }
  void settext(const string &text) { _text=text; _ctext=_text.c_str(); }

  int getnbr() const { return _nbr; }
  string gettext() { return _text; }
  const char *c_str() { return _ctext; }
  int length() const { return _text.length(); }

  int has_field(const char *ifld) {
    char c;

    c=*find_field(ifld);
    if(!c || c=='\r' || c=='\n') return 0;
    return 1;
  }

  /* Article::getfield
   * Description:
   *   Return a field of the article.
   * Parameters:
   *   ifld ... The name of the field to be extracted including `:'
   *   full ...  0 => strip fieldname, 
   *             1 => add fieldname
   * Exceptions:
   *   NotFoundError 
   *     raised, if fld is not part of the overview database
   */
  string getfield(const char *ifld, int Full=0) const {
    string rfld;
    const char *p,*fld;

    fld=find_field(ifld);
    if(*fld=='\r' || *fld=='\n') {
      // Bad luck, did not find the header
      throw NotFoundError(ifld);
    }

    // We have found the header
    p=fld+strlen(ifld);
    while(isspace(*p)) p++;
    if(!Full) fld=p;
    while(*p && *p!='\r' && *p!='\n') p++;
    rfld.assign(fld,p-fld);
    ASSERT(
      if(!(*p)) {
        slog.p(Logger::Error) << "Article without body.\n";
      }
      if(*p=='\r' && !(*(p+1))) {
	slog.p(Logger::Error) << "Article ended with <CR><NULL>.\n";
      });
    if(*p=='\n') p++;
    else if(*p=='\r' || *(p+1)=='\n') p+=2;

    // Multiline header?
    while(isspace(*p) && *p!='\n' && *p!='\r') {
      do { p++; } while(isspace(*p));
      fld=p;
      while(*p!='\r' && *p!='\n') p++;
      rfld.append(fld,p-fld);
      ASSERT(
	if(!(*p)) {
          slog.p(Logger::Error) << "Article without body.\n";
        }
        if(*p=='\r' && !(*(p+1))) {
	  slog.p(Logger::Error) << "Article ended with <CR><NULL>.\n";
	});
      if(*p=='\n') p++;
      else if(*p=='\r' || *(p+1)=='\n') p+=2;
    }
    return rfld;
  }

  void setfield(const char *ifld, const char *field_data) {
    const char *p,*fld;

    fld=find_field(ifld);
    if(*fld=='\r' || *fld=='\n') {
      // Bad luck, did not find the header
      _text.insert(fld-_ctext,field_data);
      _ctext=_text.c_str();
      return;
    }

    // We have found the header
    p=fld+strlen(ifld);
    while(isspace(*p)) p++;
    while(*p && *p!='\r' && *p!='\n') p++;
    ASSERT(
      if(!(*p)) {
        slog.p(Logger::Error) << "Article without body.\n";
      }
      if(*p=='\r' && !(*(p+1))) {
	slog.p(Logger::Error) << "Article ended with <CR><NULL>.\n";
      });
    if(*p=='\n') p++;
    if(*p=='\r' && *(p+1)=='\n') p+=2;

    // Multiline header?
    while(isspace(*p) && *p!='\n' && *p!='\r') {
      while(isspace(*p)) p++;
      while(*p!='\r' && *p!='\n') p++;
      ASSERT(
	if(!(*p)) {
          slog.p(Logger::Error) << "Article without body.\n";
        }
        if(*p=='\r' && !(*(p+1))) {
	  slog.p(Logger::Error) << "Article ended with <CR><NULL>.\n";
	});
      if(*p=='\n') p++;
      if(*p=='\r' && *(p+1)=='\n') p+=2;
    }
    _text.replace(fld-_ctext,p-fld,field_data);
    _ctext=_text.c_str();
  }

  ostream& write(ostream& os, int flags=Head|Body) {
    const char *p, *q;

    if((flags&(Head|Body))==(Head|Body)) {
      os << _text;
      return os;
    }

    p=q=_ctext;
    for(;;) {
      while(*q && *q!='\r' && *q!='\n') q++;
      if(!*q) {
        VERB(slog.p(Logger::Notice) << "Article::write: Article without body!\n");
        break;
      }
      if(*q=='\r' && *(q+1)=='\n' && *(q+2)=='\r' && *(q+3)=='\n') {
        q+=2;
        break;
      }
      if(*q=='\n' && *(q+1)=='\n') {
        q++;
        break;
      }
      while(*q=='\n' || *q=='\r') q++;
    }
    if(flags&Head) {
      os.write(p,q-p);
    } else if(flags&Body) {
      if(*q=='\n') p++;
      else if(*q=='\r' && *(q+1)=='\n') q+=2;
      
      os.write(q,_text.length()-(q-p));
    }
    return os;
  }
  friend ostream& operator<<(ostream& os, Article &art) {
    os << art._text;
    return os;
  }
};

#endif
