#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif
#include <fcntl.h>
#include <unistd.h>
#include <netdb.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include <iostream.h>
#include <fstream.h>
#include <string>

#include "NServer.h"
#include "readline.h"

//********************
//***   NServer
//********************
// protected:

/* NServer::NServer()
 * Description:
 *   Initialize the a new NServer class.
 * Parameters:
 *   none
 * Exceptions:
 *   none
 */
NServer::NServer()
{
  VERB(slog.p(Logger::Debug) << "NServer::NServer()\n"); 
  _OverviewFormat=NULL;
  _ActiveDB=NULL;
}

/* NServer::~NServer()
 * Description:
 *   Free allocated data. Virtual because it is possible, that
 *   a news server instance is destructed via the abstract 
 *   parent class.
 * Parameters:
 *   none
 * Exceptions:
 *   none
 */
NServer::~NServer()
{
  VERB(slog.p(Logger::Debug)<<"NServer::~NServer()\n");
  if(_OverviewFormat) delete _OverviewFormat;
  if(_ActiveDB) delete _ActiveDB;
  _OverviewFormat=NULL;
  _ActiveDB=NULL;
}

/* NServer::freegroup
 * Description:
 *   Free the newsgroup instance allocated by getgroup.
 * Parameters:
 *   group ... the newsgroup to be deleted
 * Return:
 *   void
 * Exceptions:
 *   none
 */
void NServer::freegroup(Newsgroup *group)
{
  VERB(slog.p(Logger::Debug) <<"NServer::freegroup(*group)\n");
  delete group;
}

//********************
//***   LServer
//********************

// construction of LServer class

/* LServer::LServer
 * Description:
 *   Construct an LServer class
 * Parameters:
 *   spooldir ... Name of spool-directory
 * Exceptions:
 *   none
 */
LServer::LServer(const char *spooldir) 
{
  VERB(slog.p(Logger::Debug) << "LServer::LServer(" << spooldir << ")\n"); 
  init(spooldir);
}

/* LServer::~LServer()
 * Description:
 *   Free allocated data.
 * Parameters:
 *   none
 * Exceptions:
 *   none
 */
LServer::~LServer()
{
  VERB(slog.p(Logger::Debug)<<"LServer::~LServer()\n");
}

void LServer::init(const char *spooldir) 
{
  VERB(slog.p(Logger::Debug) << "LServer::init(" << spooldir << ")\n"); 
  char buf[1024];

  // -16:
  // 4c .art
  // 11c number of article
  // 1c Trailing \0
  ASSERT(if(strlen(spooldir)>sizeof(_SpoolDirectory)-MAXGROUPNAMELEN-16) {
    throw AssertionError("name of spooldirectory too long. longest groupname+spooldirlength does not fit into maxpathlen");
  });
  strcpy(_SpoolDirectory,spooldir);

  if(!_OverviewFormat) _OverviewFormat=new OverviewFmt();
  if(!_ActiveDB) _ActiveDB=NULL;
}

// public methods

/* LServer::active
 * Description:
 *   Return a pointer to the active database.
 * Parameters:
 *   none
 * Return:
 *   Pointer to the active database. The database pointed to by
 *   the return vlue is maintained by the LServer class. Hence,
 *   it should not be modified by the user and must not be
 *   deleted!
 * Exceptions:
 *   none
 */
ActiveDB *LServer::active()
{
  return _ActiveDB;
}

/* LServer::groupinfo
 * Description:
 *   Returns information onto the newsgroup >name< from the
 *   active database.
 * Misfeatures:
 *   This method possibly should be moved to the NServer-class?
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   Returns informations of the newsgroup in a statically
 *   allocated GroupInfo structure.
 * Exceptions:
 *   NoSuchGroupError ... if the group does not exist
 */
GroupInfo *LServer::groupinfo(const char *name)
{
  VERB(slog.p(Logger::Debug) <<"LServer::groupinfo(" << name << ")\n");
  static GroupInfo agroup;

  ASSERT(if(strlen(name)>512) {
    throw AssertionError("LServer::groupinfo: name of newsgroup too long\n");
  });
  
  if(_ActiveDB->get(name,&agroup)<0) 
    throw NoSuchGroupError("no such group");
  return &agroup;
}

/* LServer::getgroup
 * Description:
 *   Return the newsgroup with name >name<.
 *   If the newsgroup does not exist, NULL will be returned.
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   Pointer to a Newsgroup structure. Free it by calling 
 *   the freegroup method. The newsgroup must be destructed by 
 *   the same Server-instance it was constructed from.
 * Exceptions:
 *   NoSuchGroupError ... if the newsgroup does not exist
 */
Newsgroup *LServer::getgroup(const char *name)
{
  VERB(slog.p(Logger::Debug) <<"LServer::getgroup(" << name << ")\n");
  if(!_ActiveDB->hasgroup(name)) 
    throw NoSuchGroupError("no such group");
  return new NVNewsgroup(_OverviewFormat,_SpoolDirectory,name);
}

/* LServer::post
 * Description:
 *   Post an article to the news server.
 * Misfeatures:
 *   Not yet supported.
 * Parameters:
 *   article ... The article to be posted
 * Return:
 *   0 on success, otherwise <0
 * Exceptions:
 *   none yet
 */
int LServer::post(Article *article)
{
//   TRACE(ctrace << "LServer::post(Article*)\n");
//   char fn[1024];
//   fstream fs;
//   int qnbr;
  
//   sprintf(fn,".outgoing/.qnbr");
//   fs.open(fn,ios::in|ios::out);
//   fs >> qnbr;
//   if(!fs.good()) {
//     qnbr=0;
//     fs.clear();
//   }
//   fs.seekp(0,ios::beg);
//   fs << qnbr+1 << endl;
//   if(!fs.good()) {
//     DEBUG(cdbg << "Failure on .outgoing/.qnbr\n");
//   }
//   fs.close();

//   DEBUG(cdbg << "LServer: qnbr=" << qnbr << endl << article << endl);

//   sprintf(fn,".outgoing/a%d",qnbr);
//   fs.open(fn,ios::out);
//   fs.unsetf(ios::skipws);
//   fs << article;
//   fs.close();
  return -1;
}

//********************
//***   RServer
//********************

// construction of RServer class

/* RServer::RServer
 * Description:
 *   Initialize the new RServer class. Sets up the server list
 *   and the default number of retries.
 * Misfeatures:
 *   ActiveDB for RServer not yet implemented.
 * Parameters:
 *   serverlist ... list of news servers and their newsgroups
 * Exceptions:
 *   none
 */
RServer::RServer(MPList *serverlist)
{ 
  VERB(slog.p(Logger::Debug) << "RServer::RServer/1\n"); 
  init(serverlist);
}

/* RServer::~RServer()
 * Description:
 *   Free allocated data.
 * Parameters:
 *   none
 * Exceptions:
 *   none
 */
RServer::~RServer()
{
  VERB(slog.p(Logger::Debug)<<"RServer::~RServer()\n");
  setserverlist(NULL);
}

void RServer::init(MPList *serverlist)
{ 
  VERB(slog.p(Logger::Debug) << "RServer::init/1\n"); 
  
  _ServerList=serverlist;
  _CurrentServer=NULL;
  _Retries=3;

  if(!_OverviewFormat) _OverviewFormat=new OverviewFmt();
  if(!_ActiveDB) _ActiveDB=NULL;
}

// protected methods

/* RServer::connect
 * Description:
 *   Connect to the news server. Stores whether posting
 *   is allowed in _CurrentServer->flags.
 * Parameters:
 *   none
 * Return:
 *   void
 * Exceptions:
 *   SystemError ... Thrown, if a system error (eg. cannot connect)
 *                      occurs after _Retries successive 
 *                      connection-failures.
 *   ResponseError ... Thrown, if the news server returns a response
 *                      code different from the allowed connection
 *                      setup return codes.
 *                      200 ... reading and posting allowed
 *                      201 ... reading allowed
 *   NoNewsServerError ... if no news server is configured for this 
 *                      group (in theory, this should be impossible)
 *   NoSuchGroupError ... if the group does not exist
 */
#define RSERVER_CONNECT_CHECK_CONNECTION(errstrg) \
    if(!_ServerStream.good()) { \
      if(i) { \
	i--; \
	sleep(1); \
	continue; \
      } \
      sprintf(buf,"Connection to %s:%s failed", \
	      _CurrentServer->hostname,_CurrentServer->servicename); \
      throw SystemError(buf,errno); \
    }
void RServer::connect()
{
  VERB(slog.p(Logger::Debug) << "RServer::connect()\n"); 
  if(is_connected()) return;

  string resp;
  char buf[1024];
  int i=_Retries;

  for(;;) {
    // Open connection
    _ServerStream.connect(_CurrentServer->hostname,
			  _CurrentServer->servicename);
    if(!_ServerStream.good()) {
      throw SystemError("cannot connect to news server",errno);
    }
    _ServerStream.unsetf(ios::skipws);
    nlreadline(_ServerStream,resp,0);
    slog.p(Logger::Info) << resp << "\n";

    RSERVER_CONNECT_CHECK_CONNECTION("Connection to %s:%s failed");
    if(resp[0]!='2' || resp[1]!='0' || (resp[2]!='0' && resp[2]!='1')) {
      string c("_connect_"),e("20[01]");
      throw ResponseError(c,e,resp);
    }

    // mode reader
    resp=issue("mode reader\r\n",NULL);
    RSERVER_CONNECT_CHECK_CONNECTION("Cannot read data from %s:%s");
    if(resp[0]!='2' || resp[1]!='0' || (resp[2]!='0' && resp[2]!='1')) {
      slog.p(Logger::Warning) << "mode reader failed; assuming that nnrp mode is already selected\n";
//       string c("mode reader"),e("20[01]");
//       throw ResponseError(c,e,resp);
    }
    if(resp[2]=='1') {
      _CurrentServer->nntpflags&=~MPListEntry::F_POST;
    }

    issue("list overview.fmt\r\n","215");
    _OverviewFormat->readxoin(_ServerStream);
    
    if(_CurrentGroup.name()[0]!='\0') {
      selectgroup(_CurrentGroup.name(),1);
    }
    return;
  }
}
#undef RSERVER_CONNECT_CHECK_CONNECTION

/* RServer::disconnect
 * Description:
 *   Disconnect from the current news server.
 * Parameters:
 *   none
 * Return:
 *   void
 * Exceptions:
 *   none
 */
void RServer::disconnect() {
  if(_ServerStream.is_connected()) {
    _ServerStream << "quit\r\n";
    // Who is interested in this return code? 
    _ServerStream.disconnect();  
  }
}

/* RServer::issue
 * Description:
 *   Send a command to the remote news server and return its
 *   response.
 * Parameters:
 *   command ... Command that should be sent to the news server
 *   expresp ... Expected response code. If omitted, any response
 *               code will do. Should be checked by the caller in
 *               this case.
 * Return:
 *   static string holding the news server's response.
 * Exceptions:
 *   SystemError ... if the connection fails
 *   ResponseError ... if the server returns an illegal response code
 *   NoNewsServerError ... if no news server is configured for the 
 *                      current group
 *   NoSuchGroupError ... if the group does not exist
 */
string RServer::issue(const char *command, const char *expresp=NULL) {
  string req,resp;
  int rs;
  int i=_Retries+1;

  req="issue ";
  req.append(command,strlen(command)-2);
  req.append("\n");
  // Send command
  for(;;) {
    slog.p(Logger::Debug) << req.c_str();
    _ServerStream << command;
    rs=nlreadline(_ServerStream,resp,0);
    // If the remote server closed the connection (400), reconnect
    if(!_ServerStream.good()) {
      slog.p(Logger::Warning) << "no connection to news server\n";
    } else if(strncmp(resp.c_str(),"400",3)==0) {
      slog.p(Logger::Warning) << "server closed connection---reconnecting\n";
    } else if(strncmp(resp.c_str(),"503",3)==0) {
      slog.p(Logger::Warning) << "server closed connection---reconnecting\n";
    } else {
      break;
    }
    
    // If we have lost the connection to the news server, we try
    // to reconnect to the server and reissue the command.
    for(;;) {
      i--;
      if(i==0) throw SystemError("maximum number of retries reached");
      disconnect();
      try {
	connect();
	break;
      } catch(SystemError &se) {
      }
    }
  }

  // Check the response code
  if((rs<3) || (expresp && strncmp(expresp,resp.data(),strlen(expresp)))) {
    string c(command),e(expresp);
    throw ResponseError(c,e,resp);
  }

  return resp;
}

/* RServer::setserver
 * Description:
 *   Disconnect from the current news server and set a new news 
 *   server as the current one, if the new one is different to
 *   the current.
 * Parameters:
 *   server ... A multiplexinglist entry pointing to the new news 
 *              server. This parameter must not be freed as long as 
 *              RServer uses this news server, since only the pointer 
 *              is stored.
 *              As soon as RServer connects to this news server,
 *              it will store whether posting is allowed in 
 *              server->flags.
 * Return:
 *   void
 * Exceptions:
 *   none
 */
void RServer::setserver(MPListEntry *server)
{
  VERB(slog.p(Logger::Debug) << "RServer::setserver({"
                            << server->hostname << "," 
                            << server->servicename << "})\n");
  char buf[MAXPATHLEN];

  if(_CurrentServer==server) return;

  if(_CurrentServer) disconnect();
  _CurrentGroup.clear();
  _CurrentServer=server;
}

/* RServer::selectgroup
 * Description:
 *   Select a newsgroup on the remote news server.
 * Remarks:
 *   group: 211 nflsg selected 411 no group
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   void
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 */
void RServer::selectgroup(const char *name, int force)
{
  VERB(slog.p(Logger::Debug) <<"RServer::selectgroup(" << name << ")\n");
  ASSERT(if(strlen(name)>512) {
    throw AssertionError("RServer::selectgroup: name of newsgroup too long\n");
  });

  if(!force && strcmp(_CurrentGroup.name(),name)==0) return;

  char buf[MAXPATHLEN];
  const char *ep,*p;
  int nbr, fst, lst;
  MPListEntry *mpe;
  string resp;

  // Find the server responsible for this newsgroup
  if((mpe=_ServerList->server(name))==NULL) {
    // No server configured for this group
    sprintf(buf,"no news server configured for %s",name);
    slog.p(Logger::Warning) << buf << "\n";
    throw NoNewsServerError(buf);
  }

  setserver(mpe);
  sprintf(buf,"group %s\r\n",name);
  resp=issue(buf,NULL);

  p=resp.c_str();
  if(p[0]!='2' || p[1]!='1' || p[2]!='1') {
    if(strncmp("411",p,3)==0) throw NoSuchGroupError("no such group");
    string c(buf),e("211");
    throw ResponseError(c,e,resp);
  }

  p+=3;
  nbr=strtol(p,(char**)&p,10);
  fst=strtol(p,(char**)&p,10);
  lst=strtol(p,(char**)&ep,10);
  
  if(ep==resp) { 
    //THROW!?! _Response_should_be_
    // 211 $n $f $l $g selected
    // $n Nbr. of articles in group
    // $f Nbr. of first article
    // $l Nbr. of last article
    // $g Name of newsgroup
    slog.p(Logger::Error) 
      << "response from news server not in >>211 n f l g ...<< format\n";
    // If we cannot parse the response code, use the first/last number 
    // and number of articles from the active database
    if(_ActiveDB->get(name,&_CurrentGroup)<0) {
      _CurrentGroup.set(name,0,0,0);
      _ActiveDB->set(_CurrentGroup);
    }
  } else {
    _CurrentGroup.set(name,fst,lst,nbr);
    _ActiveDB->set(_CurrentGroup);
  }
}

// public methods

/* RServer::setserverlist
 * Description:
 *   Delete all previously allocated data and install a new list of 
 *   news servers and newsgroups. 
 * Parameters:
 *   serverlist ... The list of news servers. This parameter must not 
 *                  be freed as long as RServer uses this server-list, 
 *                  since only the pointer is stored.
 * Return:
 *   void
 * Exceptions:
 *   none
 */
void RServer::setserverlist(MPList *serverlist)
{
  // Disconnect
  disconnect();
  
  // Clean up
  _CurrentGroup.clear();
  if(_ActiveDB) {
    delete _ActiveDB;
    _ActiveDB=NULL;
  }
  _CurrentServer=NULL;
  _ServerList=serverlist;
}

/* RServer::active
 * Description:
 *   Retrieves the active database from the news servers and
 *   returns a pointer to it.
 * Remarks:
 *   The offline stuff does not belong to the RServer. Thus it should 
 *   be moved into the CServer.
 *   Setting up all of this filter expressions can be done
 *   at the server-list's creation time. However, at the
 *   moment this is neglectable. The active database is
 *   retrieved rather seldom and takes a long time nevertheless.
 * Parameters:
 *   none
 * Return:
 *   Pointer to the active database.
 * Exceptions:
 *   none ... if an error occurs while retrieving the database from
 *                      a server, the active database will not be
 *                      retrieved from this server.
 */
ActiveDB *RServer::active()
{
  VERB(slog.p(Logger::Debug) <<"RServer::active()\n");
  int i,flags;
  char cgroup[MAXGROUPNAMELEN+1], *cgp ,buf[1024];
  const char *filter, *sp;
  char c,d;
  
  ASSERT(if(!_ServerList) {
    throw AssertionError("RServer::active: _ServerList is a null-pointer");
  });
  
  for(i=0;i<_ServerList->e_used;i++) {
    if(!_ServerList->entries[i].hostname[0]) continue;
    // Connect to ith news server
    try {
      setserver(_ServerList->entries+i);
      if(_CurrentServer->flags&(MPListEntry::F_OFFLINE|MPListEntry::F_SEMIOFFLINE)) continue;
      connect();
      flags=((_CurrentServer->nntpflags&MPListEntry::F_POST) ||
	     !(_CurrentServer->flags&MPListEntry::F_SETPOSTFLAG))?
	0 : ActiveDB::F_STORE_READONLY;
      
      if(_CurrentServer->nntpflags&MPListEntry::F_LIST_ACTIVE_WILDMAT) {
	try {
	  sp=_CurrentServer->read;
	  do {
	    // Extract a newsgroup-expression
	    cgp=cgroup;
	    while((c=*sp++)!=',' && c) *cgp++=c;
	    *cgp='\0';
	    // Create filter expression
	    
	    sprintf(buf,"list active %s\r\n",cgroup);
	    issue(buf,"215");
	    filter=_ServerList->makeFilter(i,cgroup);
	    _ActiveDB->read(_ServerStream,filter,flags);
	  } while(c);
	} catch(ResponseError &re) {
	  slog.p(Logger::Notice) 
	    << "list active [wildmat] failed for "
	    << _ServerList->entries[i].hostname << ":" 
	    << _ServerList->entries[i].servicename<< "\n";
	  _CurrentServer->nntpflags&=~MPListEntry::F_LIST_ACTIVE_WILDMAT;
	}
      }
      if(!(_CurrentServer->nntpflags&MPListEntry::F_LIST_ACTIVE_WILDMAT)) {
	issue("list active\r\n", "215");	    
	filter=_ServerList->makeFilter(i,"*");
	_ActiveDB->read(_ServerStream,filter,flags);
      }
    } catch(ResponseError &re) {
      slog.p(Logger::Error) 
	<< "retrieval of activedb failed for "
	<< _ServerList->entries[i].hostname << ":" 
	<< _ServerList->entries[i].servicename<< "\n";
    } catch(SystemError &se) {
      slog.p(Logger::Warning) 
	<< "connection to "
	<< _ServerList->entries[i].hostname << ":" 
	<< _ServerList->entries[i].servicename<< " failed\n";
    } catch(Error &e) {
      slog.p(Logger::Alert)
	<< "UNEXPECTED EXCEPTION WHILE RETRIEVING ACTIVEDB FROM "
	<< _ServerList->entries[i].hostname << ":" 
	<< _ServerList->entries[i].servicename 
	<< "\nPLEASE REPORT TO tom@infosys.tuwien.ac.at\n";
      e.print();
    }
  }
  return _ActiveDB;
}

/* RServer::groupinfo
 * Description:
 *   Returns information onto the newsgroup >name< from the
 *   active database.
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   Returns informations of the newsgroup in a statically
 *   allocated GroupInfo structure.
 * Exceptions:
 *   NoSuchGroupError ... if the group does not exist
 */
GroupInfo *RServer::groupinfo(const char *name)
{
  VERB(slog.p(Logger::Debug) <<"RServer::groupinfo(" << name << ")\n");
  static GroupInfo agroup;

  ASSERT(if(strlen(name)>512) {
    throw AssertionError("RServer::groupinfo: name of newsgroup too long\n");
  });
  
  // If the newsgroup cannot be found within the active database
  // reload the database from the news server and try again.
  if(_ActiveDB->get(name,&agroup)<0) {
    active();
    if(_ActiveDB->get(name,&agroup)<0) 
      throw NoSuchGroupError("no such group");
  }
  return &agroup;
}

/* RServer::getgroup
 * Description:
 *   Return the newsgroup with name >name<.
 *   If the newsgroup does not exist, NULL will be returned.
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   Pointer to a Newsgroup structure. Free it by calling 
 *   the freegroup method. The newsgroup must destructed by 
 *   the same Server-instance it was constructed from.
 * Exceptions:
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 *   SystemError ... if the connection to the news server fails
 */
Newsgroup *RServer::getgroup(const char *name)
{
  VERB(slog.p(Logger::Debug) <<"RServer::getgroup(" << name << ")\n");

  selectgroup(name);

#ifdef ENABLE_NOTCACHED
  RNewsgroup *grp=new RNewsgroup(this,_OverviewFormat,name);
#else
  NVNewsgroup *grp=new NVNewsgroup(_OverviewFormat,"/tmp",name);
#endif
  grp->setsize(_CurrentGroup.first(),_CurrentGroup.last());
  return grp;
}

/* RServer::listgroup
 * Description:
 *   Return a list of article available in a newsgroup
 * Parameters:
 *   gname ... Name of the newsgroup
 *   lstgrp ... Pointer to a preallocated array, where the list is stored
 *   lstgrpsz ... Size of the array above
 * Return:
 *   void
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 */
void RServer::listgroup(const char *gname, char *lstgrp, 
			unsigned int f, unsigned int l)
{
  VERB(slog.p(Logger::Debug) << "RServer::listgroup("  << gname << ",...)\n");    
  char buf[513];
  unsigned int i;
  string resp,line;

  selectgroup(gname);
  resp=issue("listgroup\r\n",NULL);
  if(strncmp("211",resp.c_str(),3)) {
    string c("listgroup\r\n"),e("211");
    throw ResponseError(c,e,resp);
  }

  for(;;) {
    nlreadline(_ServerStream,line);
    if(line==".\r\n") break;
    if(!_ServerStream.good()) 
      throw SystemError("error while reading from server",errno);
    i=atoi(line.data());
    if(f<=i && i<=l) lstgrp[i-f]=1;
  }
}

/* RServer::overviewdb
 * Description:
 *   Return the overviewdatabase of the newsgroup >ng<.
 *   This function should not be called by the user. Instead it
 *   should be called via a Newsgroup object.
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   void
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 *   NotAllowedError ... if retrieving the overviewdb is prohibited
 *   NSError ... if news server does not implement over and xover
 */
void RServer::overviewdb(Newsgroup *ng, unsigned int fst, unsigned int lst)
{
  VERB(slog.p(Logger::Debug) << "RServer::overviewdb(*ng," 
                            << fst << "-" << lst << ")\n");    
  ASSERT(if(!ng) {
    throw AssertionError("RServer::overviewdb: ng parameter is a null-pointer");
  });

  char buf[513];
  string resp;
  const char *p;
  const char *fmt="over %d-%d\r\n";

  for(;;) {
    if(!(_CurrentServer->nntpflags&MPListEntry::F_OVER)) fmt="xover %d-%d\r\n";
    if(!(_CurrentServer->nntpflags&MPListEntry::F_XOVER)) {
      slog.p(Logger::Critical) << "remote news server does not implement OVER and XOVER\n"; 
      throw NSError("news server does not implement OVER and XOVER");
    }
    
    selectgroup(ng->name());
    sprintf(buf,fmt,fst,lst);
    resp=issue(buf,NULL);
    p=resp.c_str();
    if(strncmp(p,"224",3)==0) {
      ng->readoverdb(_ServerStream);
      return;
    }
    if(strncmp(p,"502",3)==0) {
      throw NotAllowedError(resp);
    }
    if(strncmp(p,"412",3)==0) {
      slog.p(Logger::Warning) << "newsgroup disappeared while working on it (=p(LotteryWin)?\n"; 
      throw NoSuchGroupError("no such group");
    }
    if(strncmp(p,"420",3)==0) {
      slog.p(Logger::Error) << "NEWS SERVER RETURNED THEORETICALLY IMPOSSIBLE RESPONSE CODE\n";
      slog.p(Logger::Error) << "PLEASE REPORT TO tom@infosys.tuwien.ac.at\n";
      throw NSError(resp);
    }
    if(fmt[0]=='o') _CurrentServer->nntpflags&=~MPListEntry::F_OVER;
    else _CurrentServer->nntpflags&=~MPListEntry::F_XOVER;
  }
}

/* RServer::article
 * Description:
 *   Return a given article of a given newsgroup
 * Remarks:
 *   ARTICLE nbr
 *   => 220 success
 *   => 412 no group sel
 *   => 420 no art sel
 *   => 423 art nbr not found in grp
 *   => 430 no such art found
 * Parameters:
 *   gname ... Name of the newsgroup
 *   nbr ... Number of the article
 * Return:
 *   Pointer to an article, this has to be freed by the user
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   NoSuchArticleError ... if the requested article does not exist
 *   ResponseError ... if the news server replies an invalid response code
 */
void RServer::article(const char *gname, unsigned int nbr, Article *art)
{
  VERB(slog.p(Logger::Debug) << "RServer::article("  << nbr << ")\n");    
  char buf[513];
  const char *p;
  string resp;

  selectgroup(gname);

  sprintf(buf,"article %u\r\n",nbr);
  resp=issue(buf,NULL);
  p=resp.c_str();
  if(strncmp(p,"220",3)!=0) {
    // 412 cannot happen since we have already selected the newsgroup
    // 420 -"- since we specified the article nbr
    // 430 -"- since we specified the article nbr, not the art-id
    if(strncmp(p,"423",3)==0)
      throw NoSuchArticleError(resp);
    string c(buf),e("220");
    throw ResponseError(c,e,resp);
  }
  art->read(_ServerStream);
  art->setnbr(nbr);
}

/* RServer::post
 * Description:
 *   ***
 * Remarks:
 *   The news server, where the article should be posted, should
 *   be taken from the article's newsgroup field. An article posted
 *   to comp.os.linux should be posted to the news server from where
 *   we receive this newsgroup.
 *   POST
 *   => 240 post ok
 *   => 340 send art
 *   => 440 not allowed
 *   => 441 post failed
 * Parameters:
 *   srvr ... news server the article should be posted to
 *   article ... The article to be posted
 * Return:
 *   -1 duplicate article
 *   0 if the article has been posted successfully
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   ResponseError ... if the news server replies an invalid response code
 *   NotAllowedError ... if posting is prohibited
 *   NoNewsServerError ... if no news server is configured for this 
 *                      group (in theory, this should be impossible)
 *   NoSuchGroupError ... if the group does not exist (in theory, this 
 *                      should be impossible)
 */
int RServer::post(MPListEntry *srvr, Article *article)
{
  string resp;
  const char *p;
  int i=_Retries+1;

  setserver(srvr);
  connect();

  for(;;) {
    resp=issue("post\r\n",NULL);
    p=resp.c_str();
    if(strncmp(p,"340",3)!=0) {
      if(strncmp(p,"440",3)==0) throw NotAllowedError(resp);
      throw ResponseError("post\r\n","[34]40",resp);
    }
    _ServerStream << *article << ".\r\n";
    nlreadline(_ServerStream,resp);
    if(_ServerStream.good()) break;
    
    // If we have lost the connection to the news server, we try
    // to reconnect to the server and reissue the command.
    slog.p(Logger::Warning) << "lost connection to news server unexpectedly\n";
    for(;;) {
      i--;
      if(i==0) throw SystemError("maximum number of retries reached while trying to post an article");
      disconnect();
      try {
	connect();
	break;
      } catch(SystemError &se) {
      }
    }
  }

  p=resp.c_str();
  if(strncmp("240",p,3)==0) {
    return 0;
  } else if(strncmp("440",p,3)==0) {
    throw NotAllowedError(resp);
  } else if(strncmp("441",p,3)==0) {
    const char *p,*q,*str="duplicate";
    p=resp.c_str();
    q=str;
    do {
      while(*p!='\n' && tolower(*p)==*q) { p++; q++; }
      if(!*q) {
	// match
	return -1;
      }
      p++;
    } while(*p!='\n');
    throw PostingFailedError(resp);
  } else {
    throw ResponseError("post\r\n","[23]40,44[01]",resp);
  }
}

/* RServer::post
 * Description:
 *   ***
 * Misfeatures:
 *   A per server result code should be returned.
 * Parameters:
 *   article ... The article to be posted
 * Return:
 *   -n number of news hosts that failed
 *   0 if the article has been posted successfully
 * Exceptions:
 *   InvalidArticleError ... If article is not correctly formated
 */
int RServer::post(Article *article)
{
  VERB(slog.p(Logger::Debug) << "RServer::post(Article *article)\n");    

  const char *p, *q;
  string newsgroups, cgroup;
  char msgid[1024];
  MPListEntry **mpe;
  MPListEntry *c;
  int sc=0,i,err,res;

  Article art(article);

  if((mpe=(MPListEntry**)malloc(_ServerList->e_used*sizeof(MPListEntry*)))==NULL) {
    throw SystemError("cannot allocate buffer for post servers",errno);
  }
  try {
    newsgroups=art.getfield("newsgroups:");
    q=newsgroups.c_str();
  } catch (NotFoundError &n) {
    free(mpe);
    throw InvalidArticleError("no newsgroups field");
  }
  while(isspace(*q) || *q==',') q++;
  do {
    p=q;
    while(*q && *q!=',' && !isspace(*q)) q++;
    cgroup.assign(p,q-p);
    c=_ServerList->postserver(cgroup.c_str());
    if(c) {
      i=0; while(i<sc && c!=mpe[i]) i++;
      if(i==sc) { 
	mpe[sc]=c;
	sc++;
      }
    } else {
      slog.p(Logger::Info) 
	<< "posting to unknown newsgroup: " << cgroup << "\n";
    }
    while(isspace(*q) || *q==',') q++;
  } while(*q);
  
  if(!art.has_field("message-id:")) {
    struct timeval tv;
    int j;
    char hname[512],mid[768];
    char *p;
    p=mid;
    gettimeofday(&tv,NULL);
    while(tv.tv_sec) {
      i=tv.tv_sec%36;
      tv.tv_sec/=36;
      if(i<10) *p++='0'+i;
      else *p++='a'+i-10;
    }
    *p++='$';
    j=getpid();
    while(j) {
      i=j%36;
      j/=36;
      if(i<10) *p++='0'+i;
      else *p++='a'+i-10;
    }
    *p++='@';
    gethostname(hname,sizeof(hname)-1);
    hname[sizeof(hname)-1]='\0';
    strcpy(p,hname);

    sprintf(msgid,"Message-ID: <newscache$%s>\r\n",mid);
    art.setfield("Message-ID:",msgid);
  }

  if(!sc) {
    slog.p(Logger::Info) << "no news server configured for posting\n";
    free(mpe);
    throw NoNewsServerError("no news server configured for posting");
  }

  err=0;
  i=0;
  do {
    try {
      if(mpe[i]->flags&MPListEntry::F_OFFLINE) err--;
      else res=post(mpe[i],&art);
    } catch(Error &e) {
      err--;
    }
    i++;
  } while(i<sc);

  free(mpe);
  return err;
}

//********************
//***   CServer
//********************

// construction of CServer class

/* CServer::CServer
 * Description:
 *   Construct an CServer class
 * Parameters:
 *   spooldir ... Name of spool-directory
 *   serverlist ... list of news servers and their newsgroups
 * Exceptions:
 *   none
 */
CServer::CServer(const char *spooldir, MPList *serverlist) 
{
  VERB(slog.p(Logger::Debug)<<"CServer::CServer("<<spooldir<<",*serverlist)\n");
  char buf[MAXPATHLEN];

  ASSERT(if(!serverlist) {
    slog.p(Logger::Notice) << "CServer::CServer: serverlist is a null-pointer\n";
  });
  if(!_OverviewFormat) {
    _OverviewFormat=new OverviewFmt;
  }
  if(!_ActiveDB) {
    sprintf(buf,"%s/.active",spooldir);
    _ActiveDB=_NVActiveDB=new NVActiveDB(buf);
  }

  ASSERT(if(!serverlist) {
    slog.p(Logger::Notice) << "RServer::RServer: serverlist is a null-pointer\n";
  });
  // other constructors
  LServer::init(spooldir);
  RServer::init(serverlist);

  ASSERT(if(!_ServerList) {
    slog.p(Logger::Notice) << "RServer::RServer: serverlist is a null-pointer\n";
    exit(9);
  });
  _TTLActive=300;
  _TTLDesc=500;
}
  
/* CServer::~CServer()
 * Description:
 *   Free allocated data.
 * Parameters:
 *   none
 * Exceptions:
 *   none
 */
CServer::~CServer()
{
  VERB(slog.p(Logger::Debug)<<"CServer::~CServer()\n");
  if(_OverviewFormat) {
    delete _OverviewFormat;
    _OverviewFormat=NULL;
  }
  if(_NVActiveDB) {
    delete _NVActiveDB;
    _NVActiveDB=NULL;
    _ActiveDB=NULL;
  }
}

// public methods

/* CServer::active
 * Description:
 *   Retrieves the active database (list of newsgroup and the number 
 *   of their first and last article) from the news servers, if the
 *   local copy is older than _TTLActive seconds.
 * Parameters:
 *   none
 * Return:
 *   ActiveDB* ... pointer to the active database
 * Exceptions:
 *   none
 */
ActiveDB *CServer::active()
{
  // The validity of the active database has to be tested twice,
  // because it is possible that several processes can reach
  // this point and in this case all these processes would
  // transfer the active database
  // Locking the database before the first active_valid call
  // is no good idea, since this requires to set a lock
  // even in cases where this is not necessary at all

  if(!active_valid()) {
    slog.p(Logger::Info) << "active database timed out\n";
    _NVActiveDB->lock(NVcontainer::ExclLock);
    try {
      if(!active_valid()) {
	RServer::active();
      }
    } catch(Error &e) {
      slog.p(Logger::Alert)
	<< "CServer::active: UNEXPECTED EXCEPTION CAUGHT, WHILE IN CRITICAL REGION!\n"
	<< "CServer::active: PLEASE REPORT TO tom@infosys.tuwien.ac.at\n";
      e.print();
    } catch(...) {
      slog.p(Logger::Alert)
	<< "CServer::active: UNEXPECTED EXCEPTION CAUGHT, WHILE IN CRITICAL REGION!\n"
	<< "CServer::active: PLEASE REPORT TO tom@infosys.tuwien.ac.at\n";
    }
    _NVActiveDB->lock(NVcontainer::UnLock);
  }
  return _NVActiveDB;
}

/* CServer::groupinfo
 * Description:
 *   Returns information onto the newsgroup >name< from the
 *   active database. If the last information obtained for 
 *   the newsgroup is older than timeOut a group request
 *   is issued to the news server.
 * Misfeatures:
 *   Does not use the USE_EXCEPTIONS-define
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   Returns informations of the newsgroup in a statically
 *   allocated GroupInfo structure.
 * Exceptions:
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 */
GroupInfo *CServer::groupinfo(const char *name)
{
  VERB(slog.p(Logger::Debug) <<"CServer::groupinfo(" << name << ")\n");
  ASSERT(if(strlen(name)>512) {
    throw AssertionError("CServer::groupinfo: Name of newsgroup too long\n");
  });

  static GroupInfo agroup;
  MPListEntry *mpe;
  
  mpe=_ServerList->server(name);
  if(mpe->flags&MPListEntry::F_OFFLINE) 
    return LServer::groupinfo(name);

  if(_NVActiveDB->get(name,&agroup)<0 || 
     agroup.mtime()+mpe->groupTimeout<nvtime(NULL)) {
    try {
      RServer::selectgroup(name,1);
    } catch(SystemError &r) {
      if(_NVActiveDB->get(name,&agroup)<0) 
	throw NoSuchGroupError("no such group");
      return &agroup;
    }
    if(_NVActiveDB->get(name,&agroup)<0) {
      slog.p(Logger::Warning) << "newsgroup disappeared while working on it (=p(LotteryWin)?)\n";
      throw NoSuchGroupError("no such group");
    }
  }
  return &agroup;
}

/* CServer::getgroup
 * Description:
 *   Return the newsgroup with name >name<.
 *   If the newsgroup does not exist, NULL will be returned.
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   Pointer to a Newsgroup structure. Free it by calling 
 *   the freegroup method. The newsgroup must destructed by 
 *   the same Server-instance it was constructed from.
 * Exceptions:
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 */
Newsgroup *CServer::getgroup(const char *name)
{
  VERB(slog.p(Logger::Debug) <<"CServer::getgroup(" << name << ")\n");
  CNewsgroup *grp;
  GroupInfo grpinfo;
  MPListEntry *mpe;

  mpe=_ServerList->server(name);
  if(mpe->flags&MPListEntry::F_OFFLINE) {
    if(_NVActiveDB->get(name,&grpinfo)<0) throw NoSuchGroupError("no such group");
    NVNewsgroup *grp=new NVNewsgroup(_OverviewFormat,_SpoolDirectory,name);
    grp->setsize(grpinfo.first(),grpinfo.last());
    return grp;
  }
  if(!(mpe->flags&MPListEntry::F_CACHED)) {
    if(_NVActiveDB->get(name,&grpinfo)<0) throw NoSuchGroupError("no such group");
#ifdef ENABLE_NOTCACHED
    RNewsgroup *grp=new RNewsgroup(this,_OverviewFormat,name);
#else
    NVNewsgroup *grp=new NVNewsgroup(_OverviewFormat,"/tmp",name);
#endif
    grp->setsize(grpinfo.first(),grpinfo.last());
    return grp;
  }

  if(_NVActiveDB->get(name,&grpinfo)<0 ||
     grpinfo.mtime()+mpe->groupTimeout<nvtime(NULL)) {
    try {
      RServer::selectgroup(name,1);
    } catch(SystemError &r) {
      if(_NVActiveDB->get(name,&grpinfo)<0) 
	throw NoSuchGroupError("no such group");
      grp=new CNewsgroup(this,
			 _OverviewFormat,
			 _SpoolDirectory,name);
      grp->setsize(grpinfo.first(),grpinfo.last());
      grp->setttl(mpe->groupTimeout);
      return grp;
    }
    if(_NVActiveDB->get(name,&grpinfo)<0) {
      slog.p(Logger::Warning) << "newsgroup disappeared while working on it (=p(LotteryWin)?)\n";
      throw NoSuchGroupError("no such group");
    }
  }

  grp=new CNewsgroup(this,
		     _OverviewFormat,
		     _SpoolDirectory,name);
  grp->setsize(grpinfo.first(),grpinfo.last());
  grp->setttl(mpe->groupTimeout);
  return grp;
}

/* CServer::post
 * Description:
 *   Post an article to the primary news server. The primary news server
 *   is the news server being listed first in the MPList. If the
 *   article cannot be submitted immediately, it is kept for later 
 *   transmission.
 * Parameters:
 *   article ... The article to be posted
 * Return:
 *   0 if the article has been posted successfully
 *   1 if the article has been spooled for later submission
 * Exceptions:
 *   InvalidArticleError ... If article is not correctly formated
 *   ResponseError ... if posting fails
 *   SystemError ... if spooling of the article fails
 */
int CServer::post(Article *article)
{
  VERB(slog.p(Logger::Debug) << "CServer::post(Article*)\n");
  int r;

  if(RServer::post(article)<0) {
    spoolarticle(article);
    return 1;
  }

  return 0;
}

/* CServer::spoolarticle
 * Description:
 *   Spool a news article for later transmission
 * Misfeatures:
 *   Uses exceptions - exceptions are not well supported under gcc
 *   for alpha. :(
 * Parameters:
 *   article ... The article to be spooled
 * Return:
 *   void
 * Exceptions:
 *   SystemError
 */
void CServer::spoolarticle(Article *article)
{
  char fn[MAXPATHLEN];
  char buf[MAXPATHLEN+256];
  fstream fs;
  int fd,err=0,seqf=0,seqt=0;
  struct flock l;
  struct stat s;

  l.l_type=F_WRLCK;
  l.l_whence=SEEK_SET;
  l.l_start=l.l_len=0;
  sprintf(fn,"%s/.outgoing/status",_SpoolDirectory);
  if((fd=open(fn,O_RDWR))<0) {
    if(errno==ENOENT) {
      sprintf(buf,"%s/.outgoing",_SpoolDirectory);
      mkpdir(buf,0755);
      if((fd=open(fn,O_RDWR|O_CREAT,0644))<0) err=1;
    } else err=1;
  }
  if(err) {
    sprintf(buf,"cannot open %s\n",fn);
    throw SystemError(buf,errno);
  }
  
  if(fcntl(fd,F_SETLKW,&l)<0) {
    sprintf(buf,"cannot lock %s\n",fn);
    throw SystemError(buf,errno);
  }
  if(fstat(fd,&s)<0) {
    sprintf(buf,"cannot fstat %s\n",fn);
    throw SystemError(buf,errno);
  }
  fs.attach(fd);
  if(s.st_size) {
    fs >> seqf >> seqt;
    if(fs.bad()) {
      sprintf(buf,"cannot read from %s\n",fn);
      throw SystemError(buf,errno);
    }
    fs.seekg(0,ios::beg);
  }
  fs << seqf << " " << seqt+1 << "\n";
  if(!fs.good()) {
    sprintf(buf,"cannot write to %s\n",fn);
    throw SystemError(buf,errno);
  }
  l.l_type=F_UNLCK;
  l.l_whence=SEEK_SET;
  l.l_start=l.l_len=0;
  if(fcntl(fd,F_SETLKW,&l)<0) {
    sprintf(buf,"cannot unlock %s\n",fn);
    throw SystemError(buf,errno);
  }
  fs.close();
  
  sprintf(fn,"%s/.outgoing/art%d",_SpoolDirectory,seqt);
  fs.open(fn,ios::out);
  fs << *article;
  if(!fs.good()) {
    sprintf(buf,"cannot write to %s\n",fn);
    throw SystemError(buf,errno);
  }
  fs.close();
}

/* CServer::postspooled
 * Description:
 *   Post all those articles that have been queued for later
 *   transmission.
 * Misfeatures:
 *   Uses exceptions - exceptions are not well supported under gcc
 *   for alpha. :(
 * Parameters:
 *   none
 * Return:
 *   void
 * Exceptions:
 *   SystemError
 */
void CServer::postspooled(void)
{
  VERB(slog.p(Logger::Debug) << "CServer::postspooled()\n");
  char fn[MAXPATHLEN];
  char buf[MAXPATHLEN+256];
  fstream fs;
  int fd;
  int i,seqf,seqt,err=0;
  struct flock l;
  struct stat s;

  l.l_type=F_WRLCK;
  l.l_whence=SEEK_SET;
  l.l_start=l.l_len=0;
  sprintf(fn,"%s/.outgoing/status",_SpoolDirectory);

  if((fd=open(fn,O_RDWR))<0) {
    if(errno==ENOENT) {
      sprintf(buf,"%s/.outgoing",_SpoolDirectory);
      mkpdir(buf,0755);
      if((fd=open(fn,O_RDWR|O_CREAT,0644))<0) err=1;
    } else err=1;
  }
  if(err) {
    sprintf(buf,"cannot open %s\n",fn);
    throw SystemError(buf,errno);
  }
  VERB(slog.p(Logger::Debug) << "Waiting for lock\n");
  if(fcntl(fd,F_SETLKW,&l)<0) {
    sprintf(buf,"cannot lock %s\n",fn);
    throw SystemError(buf,errno);
  }
  VERB(slog.p(Logger::Debug) << "Locked\n");
  if(fstat(fd,&s)<0) {
    sprintf(buf,"cannot fstat %s\n",fn);
    throw SystemError(buf,errno);
  }
  fs.attach(fd);
  if(s.st_size) {
    fs >> seqf >> seqt;
    if(fs.bad()) {
      if(!fs.eof()) {
	sprintf(buf,"cannot read from %s\n",fn);
	throw SystemError(buf,errno);
      }    
      seqf=seqt=0;
      fs.clear();
    }
    
    try {
      for(i=seqf;i<seqt;i++) {
	ifstream ifs;
	Article a;
	sprintf(fn,"%s/.outgoing/art%d",_SpoolDirectory,i);
	ifs.open(fn);
	a.read(ifs);
	if(fs.bad() && !fs.eof()) {
	  sprintf(buf,"cannot read from %s\n",fn);
	  throw SystemError(buf,errno);
	}
	ifs.close();
	RServer::post(&a);
	if(unlink(fn)<0) {
	  VERB(slog.p(Logger::Error) << "Cannot unlink " << fn 
	                            << ". Please remove it manually.\n");
	}
      }
    }
    catch(Error &r) {
      fs.seekg(0,ios::beg);
      fs << i << " " << seqt;
      if(!fs.good()) {
	sprintf(buf,"cannot write to %s\n",fn);
	throw SystemError(buf,errno);
      }
      l.l_type=F_UNLCK;
      l.l_whence=SEEK_SET;
      l.l_start=l.l_len=0;
      if(fcntl(fd,F_SETLKW,&l)<0) {
	sprintf(buf,"cannot unlock %s\n",fn);
	throw SystemError(buf,errno);
      }
      fs.close();
      throw r;
    }
    fs.seekg(0);
    fs << i << " " << seqt << "\n";
    if(!fs.good()) {
      sprintf(buf,"cannot write to %s\n",fn);
      throw SystemError(buf,errno);
    }
  } else {
    fs << "0 0\n";
    if(fs.bad()) {
      sprintf(buf,"cannot write to %s\n",fn);
      throw SystemError(buf,errno);
    }
  }
  
  l.l_type=F_UNLCK;
  l.l_whence=SEEK_SET;
  l.l_start=l.l_len=0;
  if(fcntl(fd,F_SETLKW,&l)<0) {
    sprintf(buf,"cannot unlock %s\n",fn);
    throw SystemError(buf,errno);
  }
  fs.close();
}

/* CServer::listgroup
 * Description:
 *   Return a list of article available in a newsgroup
 * Parameters:
 *   gname ... Name of the newsgroup
 *   lstgrp ... Pointer to a preallocated array, where the list is stored
 *   lstgrpsz ... Size of the array above
 * Return:
 *   void
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 *   NSError ... if in offline mode
 */
void CServer::listgroup(const char *gname, char *lstgrp, 
			    unsigned int f, unsigned int l)
{
  VERB(slog.p(Logger::Debug) << "CServer::listgroup("<<gname<<",...)\n");
  ASSERT(if(!gname) {
    throw AssertionError("CServer::listgroup: gname parameter is a null-pointer");
  }
  if(!lstgrp) {
    throw AssertionError("CServer::listgroup: lstgrp parameter is a null-pointer");
  });
  MPListEntry *mpe;

  mpe=_ServerList->server(gname);
  if(mpe->flags&MPListEntry::F_OFFLINE) 
    throw NSError("cannot listgroup in offline mode");

  RServer::listgroup(gname,lstgrp,f,l);
}

/* CServer::overviewdb
 * Description:
 *   Return the overviewdatabase of the newsgroup >ng<.
 *   This function should not be called by the user. Instead it
 *   should be called via a Newsgroup object.
 * Parameters:
 *   name ... name of the newsgroup
 * Return:
 *   void
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   ResponseError ... if the news server replies an invalid response code
 */
void CServer::overviewdb(Newsgroup *ng, unsigned int fst, unsigned int lst)
{
  VERB(slog.p(Logger::Debug) << "CServer::overviewdb(*ng," 
                             << fst << "-" << lst << ")\n");    
  MPListEntry *mpe;

  mpe=_ServerList->server(ng->name());
  if(!(mpe->flags&MPListEntry::F_OFFLINE))
    RServer::overviewdb(ng,fst,lst);
}

/* CServer::article
 * Description:
 *   Return a given article of a given newsgroup
 * Parameters:
 *   gname ... Name of the newsgroup
 *   nbr ... Number of the article
 * Return:
 *   Pointer to a statically preallocated article
 * Exceptions:
 *   SystemError ... if the connection to the news server fails
 *   NoNewsServerError ... if no news server is configured for this group
 *   NoSuchGroupError ... if the group does not exist
 *   NoSuchArticleError ... if the requested article does not exist
 *   NSError ... if in offline mode
 */
void CServer::article(const char *gname, unsigned int nbr, Article *art)
{
  VERB(slog.p(Logger::Debug) << "CServer::article(" << nbr << ")\n");    
  MPListEntry *mpe;

  mpe=_ServerList->server(gname);
  if(mpe->flags&MPListEntry::F_OFFLINE)
    throw NSError("cannot retrieve articles in offline mode");

  RServer::article(gname,nbr,art);
}
