#ifndef __NServer_h__
#define __NServer_h__

#include <sys/param.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#include <iostream.h>
#include <fstream.h>

class NServer;
class LServer;
class RServer;
class CServer;

#include "config.h"
#include "Debug.h"

#include "sstream.h"
#include "util.h"

#include "OverviewFmt.h"
#include "ActiveDB.h"
#include "NVActiveDB.h"
#include "NVNewsgroup.h"
#include "RNewsgroup.h"
#include "CNewsgroup.h"
#include "Article.h"
#include "MPList.h"
#include "NSError.h"

/* NServer parent-class, RServer and LServer child-classes.
 */
class NServer {
protected:
  OverviewFmt *_OverviewFormat;
  ActiveDB *_ActiveDB;

  NServer();
  virtual ~NServer();
public:
  virtual OverviewFmt *overviewfmt() { return _OverviewFormat; }
  virtual ActiveDB *active()=0;
  virtual GroupInfo *groupinfo(const char *name)=0;
  virtual Newsgroup *getgroup(const char *name)=0;
  virtual void freegroup(Newsgroup *group);
  virtual int post(Article *article)=0;
}; //NServer

/* LServer
 * Provides an interface to access news locally
 */
class LServer: virtual public NServer {
protected:
  char _SpoolDirectory[MAXPATHLEN];
  
  LServer() : NServer() {
    _SpoolDirectory[0]='\0';
  }
  void init(const char *spooldir);

public:
  LServer(const char *spooldir);
  virtual ~LServer();

  // News retrieval functions
  virtual ActiveDB *active();
  virtual GroupInfo *groupinfo(const char *name);
  virtual Newsgroup *getgroup(const char *name);
  virtual int post(Article *article);
};

/* RServer
 * Provides an interface to access news on a remote site
 */
class RServer: virtual public NServer {
private:
  int post(MPListEntry *srvr, Article *article);

protected:
  MPList *_ServerList;
  MPListEntry *_CurrentServer;
  GroupInfo _CurrentGroup;

  sstream _ServerStream;
  int _Retries;

  RServer() : NServer() {
    _ServerList=NULL;
    _CurrentServer=NULL;
    _Retries=3;
  }
  void init(MPList *serverlist);

  void connect();
  void disconnect();

  string issue(const char *command, const char *expresp=NULL);
  void setserver(MPListEntry *server);
  int is_connected() { 
    return _ServerStream.is_connected(); 
  }
  void selectgroup(const char *name, int force=0);
public:
  RServer(MPList *serverlist);
  virtual ~RServer();

  // Administrative functions
  void setserverlist(MPList *serverlist);
  
  /* RServer::getserverlist
   * Description:
   *   Return the currently installed list of news servers.
   * Parameters:
   *   none
   * Return:
   *   The server-list.
   * Exceptions:
   *   none
   */
  MPList *getserverlist(void) { return _ServerList; }

  // News retrieval functions
  virtual ActiveDB *active();
  virtual GroupInfo *groupinfo(const char *name);
  virtual Newsgroup *getgroup(const char *name);
  virtual int post(Article *article);

  // RServer specific functions
  virtual void listgroup(const char *gname, char *lstgrp, unsigned int f, unsigned int l);
  virtual void overviewdb(Newsgroup *ng, unsigned int fst, unsigned int lst);
  virtual void article(const char *gname, unsigned int nb, Article *artr);
};

/* CServer
 * If the news server cannot be reached, the CServer class
 * stores the articles in its spool directory. However, this
 * class does not post these articles to the news server as soon
 * as the link is up again! Locking is done by setting an exclusive
 * lock onto the status file located which is located in the
 * spool directory.
 */
class CServer: virtual public LServer, public RServer {
protected:
  NVActiveDB *_NVActiveDB;

  nvtime_t _TTLActive;
  nvtime_t _TTLDesc;
  int active_valid() { 
    unsigned long tm;
    _NVActiveDB->getmtime(&tm);
    return (tm+_TTLActive)>nvtime(NULL); 
  }
  int desc_valid() { 
    return 1; 
  }
  int group_valid();

  // This variable indicates, whether a newsgroup has been 
  // already selected on the remote news server
  // int _RSGroup;
  // Select group on remote server, if not already selected
  // int RSGroup(const char *name);

public:
  CServer(const char *spooldir, MPList *serverlist);
  ~CServer();
   
  void setttl(time_t ttl_list, 
	      time_t ttl_desc) {
    _TTLActive=ttl_list;
    _TTLDesc=ttl_desc;
  }
  
  // News retrieval functions
  virtual OverviewFmt *overviewfmt() { return _OverviewFormat; }
  virtual ActiveDB *active();
  virtual GroupInfo *groupinfo(const char *name);
  virtual Newsgroup *getgroup(const char *name);
  virtual int post(Article *article);

  // CServer specific functions
  virtual void spoolarticle(Article *article);
  virtual void postspooled(void);
  virtual void listgroup(const char *gname, char *lstgrp, unsigned int f, unsigned int l);
  virtual void overviewdb(Newsgroup *ng, unsigned int fst, unsigned int lst);
  virtual void article(const char *gname, unsigned int nbr, Article *art);
};

#endif
