/* updatenews.cc (c) 1996 by Thomas GSCHWIND 
 * -- prefetch a set of newsgroups
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>

#include <iostream.h>

#include "config.h"
#include "Debug.h"
#include "Config.h"
#include "NServer.h"
#include "Logger.h"
#include "setugid.h"

//Notes {
// Config berarbeiten. Bei Syntaxfehler beenden. 
// Switch ?-d? der nur Config einliest und ausgibt.
//} Notes

enum {
  F_PREFETCH_ACTIVE=0x01,
  F_PREFETCH_GROUP=0x02,
  F_PREFETCH_OVER=0x04,
  F_PREFETCH_LOCKGROUP=0x08
};

Logger slog;
const char *cmnd;
Config Cfg;
int Flags;
int Workers;
int Xsignal;

void catchsignal(int num)
{
  Xsignal=num;
  slog.p(Logger::Warning) << "received signal " << num << "\n";
  signal(num,catchsignal);
}

void doprefetch(CServer *cs, const char *group)
{
  Newsgroup *grp;
  try {
    slog.p(Logger::Info) << "prefetching " << group << "\n";
    grp=cs->getgroup(group);
    if(Flags&F_PREFETCH_GROUP) 
      grp->prefetchGroup(Flags&F_PREFETCH_LOCKGROUP);
    else 
      if(Flags&F_PREFETCH_OVER) grp->prefetchOverview();
    cs->freegroup(grp);
  } catch(...) {
    slog.p(Logger::Error) << "prefetching " << group << " failed\n";
  }
}

int create_worker(CServer *cs, const char *group) 
{
  int c;
  switch(c=fork()) {
  case -1:
    // failed
    slog.p(Logger::Error) << "fork failed, using this process\n";
    doprefetch(cs, group);
    return 0;
  case 0:
    // child
    doprefetch(cs, group);
    exit(0);
  default:
    // parent
    return 1;
    break;
  }
}

void update(void)
{
  char group[MAXGROUPNAMELEN+1];
  ifstream ifs;
  CServer cs(Cfg.SpoolDirectory,&(Cfg.srvrs));
  int wi=Workers?0:-1;

  cs.setttl(Cfg.ttl_list,Cfg.ttl_desc);

  slog.p(Logger::Info) << "posting spooled articles\n";
  cs.postspooled();
  slog.p(Logger::Info) << "prefetching active database\n";
  cs.active();

  if(!(Flags&(F_PREFETCH_GROUP|F_PREFETCH_OVER))) return; 

  // Open the newsgroups-file and retrieve all the unread articles
  ifs.open(Cfg.PrefetchFile);
  for(;Xsignal<0;) {
    while (wi==Workers) {
      // maximum workers spawned, wait until a worker exits
      int cstat;
      int cid;
      cid=wait(&cstat);
      if(WIFSIGNALED(cstat)) {
	slog.p(Logger::Error) << "worker terminated unexpectedly\n";
      } else if(WIFEXITED(cstat)) {
	--wi;
      }
    }

    ifs >> group;
    if(ifs.eof()) break;
    if(group[0]=='!') continue;
    if(ifs.bad()) {
      slog.p(Logger::Error) << "cannot read from " << Cfg.PrefetchFile;
      break;
    }
    if (Workers==0) doprefetch(&cs, group);
    else wi+=create_worker(&cs, group);
  }
  ifs.close();
}

main(int argc, char **argv)
{
#ifndef WITH_SYSLOG
  char logfile[MAXPATHLEN];
#endif
  char conffile[MAXPATHLEN];
  time_t t;
  pid_t p;
  int argidx;
  int optset=0,opt_file=1;

//   strcpy(ServerRoot,SYSCONFDIR);
//   chdir(ServerRoot);

  time(&t);
  p=getpid();
  cmnd=argv[0];
  Workers=0;
  for(argidx=1;argidx<argc;argidx++) {
    if(strcmp("--over",argv[argidx])==0) {
      Flags|=F_PREFETCH_OVER;
      optset=1;
    } else if(strcmp("--group",argv[argidx])==0) {
      Flags|=F_PREFETCH_GROUP;
      optset=1;
    } else if(strcmp("--active",argv[argidx])==0) {
      Flags|=F_PREFETCH_ACTIVE;
      optset=1;
    } else if(strcmp("--lock-group",argv[argidx])==0) {
      Flags|=F_PREFETCH_LOCKGROUP;
      optset=1;
    } else if(strcmp("--workers",argv[argidx])==0) {
      ++argidx;
      Workers=atoi(argv[argidx]);
    } else if(strcmp("--help",argv[argidx])==0) {
      cerr << "Usage: " << cmnd << "[--help]|([--active] [[--workers #] [--over] [[--group] [--lock-group]]])\n";
      exit(1);
    } else if(strcmp("-v",argv[argidx])==0) {
      opt_file=0;
    }
  }
  if(!optset) Flags=F_PREFETCH_GROUP|F_PREFETCH_ACTIVE;

  if(argc-argidx>1) {
    cerr << "Usage: " << cmnd << " [Config-File]\n";
    exit(1);
  }
  
  if(argidx<=argc) {
    sprintf(conffile,"%s/newscache.conf",SYSCONFDIR);
    Cfg.read(conffile);
  } else {
    Cfg.read(argv[argidx]);
  }
  for(int i=0; i<Cfg.srvrs.e_used; i++) {
    // Switch off offline mode
    Cfg.srvrs.entries[i].flags&=~(MPListEntry::F_OFFLINE|MPListEntry::F_SEMIOFFLINE);
    // Set timeout to 0
    Cfg.srvrs.entries[i].groupTimeout=0;
  }

  // signal
  Xsignal=-1;
  signal(SIGHUP,catchsignal);
  signal(SIGPIPE,catchsignal);
  signal(SIGINT,catchsignal);
  signal(SIGPIPE,catchsignal);
  signal(SIGALRM,catchsignal);
  signal(SIGTERM,catchsignal);
  signal(SIGUSR1,catchsignal);
  signal(SIGUSR2,catchsignal);

#ifdef WITH_SYSLOG
  slog.open(PACKAGE,LOG_NDELAY|LOG_PID,LOG_NEWS);
#else
  if(opt_file) {
    sprintf(logfile,"%s/Trace-%ld-%d-ud",Cfg.LogDirectory,t,p);
    slog.open(logfile);
  } else {
    slog.attach(2);
  }
  slog.p(Logger::Info) << "sendq started\n";
#endif

  // Check the queue in each server-directory and send it off to the 
  // news server
  if(getuid()==CONF_UIDROOT) setugid(Cfg.Username,Cfg.Groupname);
  update();
  cout << "Done\n";
}
