##
# Generic pilot sychronizing routine, by Alan Harder.
#
# Assumptions-
# 1. File side records have some sort of unique Id.
# 2. Pilot records don't need any translation (may add hooks for this later).
# 3. Records can contain scalars, hash refs and array refs, to any depth.
#    AppInfo record can contain _only_ scalars and arrays of scalars.
#
# Required parameters
#	Dlp	  = The $dlp parameter passed into the conduitSync method.
#
#	DbInfo    = Hash ref containing info about the pilot database.  Must
#		    have the following fields: name, creator, type, flags,
#		    version.  Ex: 'MemoDB, 'memo', 'DATA', 0, 0
#
#	ReqFields = Array ref listing the key fields in the records.
#		    These are the fields that will be compared to determine
#		    if two records are the same.
#
#	InfoFields= Array ref listing the key fields in the appinfo
#		    structure.  These fields will be compared and updated
#		    between the pilot and file sides.
#
#	IdField   = Name of the field used to store unique Id in file records.
#
#	MasterFile= Full path filename of conduit db file
#
#	Datafile  = Full path filename of datafile
#
#	NameHook  = Used for PilotMgr log output describing actions.
#		    This method takes a record hash as parameter and
#		    returns a "pretty name" string for the output.
#
#	ReadHook  = Code ref for method to read data from the file side.
#		    Subroutine should take data filename as a parameter and
#		    return a hash ref in the following format:
#			'__RECORDS' => Array ref containing all records
#				       Order will be maintained in case it is
#				       important to your app.
#
#			'__APPINFO' => Array ref containing appinfo data.
#				       (can be undef if no appinfo exists)
#
#			<REC_ID> => <index>   Mappings of your IdField values
#					      to the index that record can be
#					      found at in the __RECORDS list.
#
#	WriteHook = Code ref for method to write data back to file side.
#		    Subroutine parameters are data filename and Hash ref
#		    containing records (hash returned from ReadHook).
#
#	IdHook    = Code ref to generate a new unique record id for the file
#		    side.  Entire hash from ReadHook is passed in as param.
#
#	TranslateHook= (optional)
#		    A translate hook is require for conduits not using one
#		    of the five builtin databases (MemoDB, DatebookDB,
#		    ToDoDB, AddressDB, ExpenseDB).  Records with be returned
#		    from the pilot in 'raw' format.  This hook is used to
#		    read the raw data and fill in the hash with the relevant
#		    fields.  The parameters to this hook are the hash
#		    containing the raw data (key=='raw') and 0 for
#		    raw->fields, 1 for fields->raw.
#
#	AppInfoHook= (optional)Code ref to subroutine which translates to
#		    and from a 'raw' appinfo from pilot and the expanded
#		    format.  Gets called if appinfo block returned from
#		    getAppBlock() call returns a hash with a single
#		    key=='raw'.  Parameters are appinfo hash and 0 for
#		    raw->expanded, 1 for expanded->raw.
#
package PilotSync;

use Data::Dumper;

my $VERSION = '0.91 BETA2';
my ($master_db, $file_db, $pilot_db);
my ($gIdField, $gIdHook, $gReqFields, $gTranslateHook, $gNameHook);
my ($FILECHANGE, $PILOTCHANGE, $FILEDELETE, $PILOTDELETE) = (1,4,2,8);

sub doSync
{
    my ($dlp, $dbinfo, $reqFields, $infoFields, $idField,
	$masterFile, $dataFile, $nameHook, $readHook, $writeHook, $idHook,
	$translateHook, $appinfoHook) = @_;

    $gIdField = $idField;
    $gIdHook = $idHook;
    $gReqFields = $reqFields;
    $gTranslateHook = $translateHook;
    $gNameHook = $nameHook;

    # Open or create database
    #
    my ($pilot_dbhandle, $appinfo);
    $pilot_dbhandle = &openDB($dlp, $dbinfo);

    if (!defined($pilot_dbhandle))
    {
	PilotMgr::msg("Unable to open '$dbinfo->{name}'.  Aborting!");
	return;
    }

    # Set display on pilot
    #
    $dlp->getStatus();

    # Read inputs
    #
    $dlp->tickle;
    $dlp->watchdog(20);

    $master_db = &readMaster($masterFile);
    $file_db = &$readHook($dataFile);

    $dlp->watchdog(0);
    $dlp->tickle;

    $pilot_db = %$master_db
		    ? &readPilotChanges($dlp, $pilot_dbhandle, $translateHook)
		    : &readPilotAll($dlp, $pilot_dbhandle, $translateHook);

    # Do sync
    #
    &syncAppInfo($pilot_dbhandle, $infoFields, $appinfoHook);
    &doFastSync($dlp, $pilot_dbhandle);

    # Write dbs
    #
    $dlp->tickle;
    $dlp->watchdog(20);

    &writeMaster($masterFile, $master_db);
    &$writeHook($dataFile, $file_db);

    $dlp->watchdog(0);
    $dlp->tickle;

    # Clear flags and close db
    #
    $pilot_dbhandle->purge();
    $pilot_dbhandle->resetFlags();
    $pilot_dbhandle->close();
}

sub openDB
{
    my ($dlp, $dbinfo) = @_;
    my ($dbname, $dbh) = ($dbinfo->{'name'});

    eval
    {
	$dbh = $dlp->open($dbname);
    };
    if ($@ =~ /read-only value/)
    {
	PilotMgr::msg("Pilot database '$dbname' does not exist.\n" .
		      "Creating it...");

	$dbh = $dlp->create($dbname, $dbinfo->{'creator'},
			    $dbinfo->{'type'}, $dbinfo->{'flags'},
			    $dbinfo->{'version'});
    }
    elsif ($@)
    {
	croak($@);
    }

    return $dbh;
}

sub readMaster
{
    my ($fname) = @_;
    my ($lines, $masterdb);

    open(FD, "<$fname") || return {};
    $lines = join('', <FD>);
    close(FD);
    eval $lines;

    return $masterdb;
}

sub writeMaster
{
    my ($fname, $masterdb) = @_;

    $Data::Dumper::Purity = 1;
    $Data::Dumper::Deepcopy = 1;
    $Data::Dumper::Indent = 0;

    unless (open(FD, ">$fname"))
    {
	PilotMgr::msg("Unable to write to $fname.  Help!");
	return;
    }

    print FD Data::Dumper->Dumpxs([$masterdb], ['masterdb']), "1;\n";
    close(FD);
}

sub readPilotChanges
{
    my ($dlp, $dbh, $translateHook) = @_;
    my ($db, $i, $msg, $rec) = ({}, 1, "Reading Pilot Changes");

    PilotMgr::status($msg, 0);
    while (1)
    {
	$rec = $dbh->getNextModRecord();
	last unless (defined($rec));

	&$translateHook($rec) if (defined $translateHook);

	$db->{$rec->{id}} = $rec;

	PilotMgr::status($msg, $i++);
	$i = 96 if ($i > 100);
    }
    PilotMgr::status($msg, 100);

    return $db;
}

sub readPilotAll
{
    my ($dlp, $dbh, $translateHook) = @_;
    my ($db, $i, $msg, $rec, $max) = ({}, 0, "Reading All Pilot Records");

    PilotMgr::status($msg, 0);
    $max = $dbh->getRecords();
    while (1)
    {
	$rec = $dbh->getRecord($i++);
	last unless (defined $rec);

	&$translateHook($rec) if (defined $translateHook);
	$db->{$rec->{id}} = $rec;

	PilotMgr::status($msg, int(100 * $i / $max));
    }
    PilotMgr::status($msg, 100);

    return $db;
}

################################

sub doFastSync
{
    my ($dlp, $dbh) = @_;
    my ($frec, $mrec, $prec, $id, $i, $tcl);

    # Check file records first
    #
    $tcl = time;
    for ($i=$[; $i < @{$file_db->{'__RECORDS'}}; $i++)
    {
	$frec = $file_db->{'__RECORDS'}->[$i];

	# Find matching records in master and pilot, if any
	#
	$id = $master_db->{$gIdField . '_' . $frec->{$gIdField}};
	$mrec = defined($id) ? $master_db->{$id} : undef;
	$prec = defined($id) ? $pilot_db->{$id} : undef;

	unless (defined $mrec)
	{
	    # Try to match to existing pilot record
	    #
	    $prec = &findPilotMatch($frec);
	    $id = $prec->{id} if (defined $prec);
	}

	# Sync record - method returns any change in size to
	# $file_db->{records} array so we can keep this loop straight..
	#
	$i += &syncRecord($mrec, $frec, $prec, $dlp, $dbh);

	delete $pilot_db->{$id} if (defined $prec);
	$id = $master_db->{$gIdField . '_' . $frec->{$gIdField}};
	$master_db->{$id}->{'__GOT_SYNCED'} = 1 if (defined $id);

	if (time - $tcl >= 20)
	{
	    $dlp->tickle;
	    $tcl = time;
	}
    }

    # find deleted file records
    #
    $frec = undef;
    foreach $mrec (values %$master_db)
    {
	next unless (ref($mrec) eq 'HASH');

	if (defined $mrec->{'__GOT_SYNCED'})
	{
	    delete $mrec->{'__GOT_SYNCED'};
	    next;
	}

	$id = $master_db->{$gIdField . '_' . $mrec->{$gIdField}};
	$prec = defined($id) ? $pilot_db->{$id} : undef;

	&syncRecord($mrec, $frec, $prec, $dlp, $dbh, $FILEDELETE);
    }

    # remaining pilot changes
    $tcl = time;
    foreach $prec (values %$pilot_db)
    {
	# Find matching records in master and file, if any
	#
	$mrec = $master_db->{$prec->{id}};
	$frec = defined($mrec) ?
	    $file_db->{'__RECORDS'}->[$file_db->{$mrec->{$gIdField}}] : undef;

	&syncRecord($mrec, $frec, $prec, $dlp, $dbh);

	if (time - $tcl >= 20)
	{
	    $dlp->tickle;
	    $tcl = time;
	}
    }
}

sub syncRecord
{
    my ($mrec, $frec, $prec, $dlp, $dbh, $rule) = @_;
    $rule = 0 unless (defined $rule);

    $rule |= $FILECHANGE
	if (defined($frec) && (!defined($mrec) || &recsDiffer($frec, $mrec)));

    $rule |= $prec->{deleted} ? $PILOTDELETE : $PILOTCHANGE
	if (defined($prec));

    ##
    # Apply rule...
    #
    # Rules:
    #   0: Pilot=unchanged    File=unchanged    * No action
    #   1: Pilot=unchanged    File=changed/new  * Update Pilot
    #   2: Pilot=unchanged    File=deleted      * Delete Pilot
    #   4: Pilot=changed/new  File=unchanged    * Update File
    #   8: Pilot=deleted      File=unchanged    * Delete File
    #  10: Pilot=deleted      File=deleted      * Just remove master rec
    #
    #   5: Pilot=changed/new  File=changed/new  * Merge changes, Update Both
    #   6: Pilot=changed/new  File=deleted      * Restore File ?
    #   9: Pilot=deleted      File=changed/new  * Restore Pilot ?
    #
    my ($file_sizechange, $id) = (0);
    return 0 if ($rule == 0);

    if ($rule == 10)
    {
	delete $master_db->{$prec->{id}};
	delete $master_db->{$gIdField . '_' . $mrec->{$gIdField}};
    }

    if ($rule == 5)
    {
	unless (&recsDiffer($frec, $prec))
	{
	    # both changed, but they're still the same- just update master
	    unless (defined $mrec)
	    {
		$master_db->{$prec->{id}} = $frec;
		$master_db->{$gIdField . '_' . $frec->{$gIdField}} =
		    $prec->{id};
		PilotMgr::msg("Matching '" . &$gNameHook($prec) . "' " .
			      "(pilot $prec->{id}, file $frec->{$gIdField})");
	    }
	    else
	    {
		&copyRec($frec, $mrec);
	    }
	    return 0;
	}

	# XXX: Merge!
	#
	PilotMgr::msg('Both Pilot and File records changed: ' .
		      &$gNameHook($prec) . ' / ' . &$gNameHook($frec) .
		      "\n(Merging not yet implemented!  Leaving both sides " .
		      "unchanged for now.. please make them the same)");
	return 0;
    }

    if ($rule == 1 || $rule == 5 || $rule == 9)
    {
	# Update Pilot
	#
	$prec = &copyRec($frec, (defined($prec) ? $prec : $dbh->newRecord()));
	$prec->{'id'} = $master_db->{$gIdField . '_' . $frec->{$gIdField}};
	$prec->{'id'} ||= 0;
	$prec->{'category'} ||= 0;

	# going to restore a deleted pilot rec with file changes:
	$prec->{'deleted'} = 0 if ($rule ==9);

	&$gTranslateHook($prec, 1) if (defined $gTranslateHook);

	$id = $dbh->setRecord($prec);
	PilotMgr::msg('Update Pilot: ' . &$gNameHook($frec));

	$master_db->{$id} = $frec;
	$master_db->{$gIdField . '_' . $frec->{$gIdField}} = $id
	    unless (defined $mrec);
    }

    if ($rule == 4 && defined $frec && !&recsDiffer($frec, $prec))
    {
	# Pilot reports record modified, but it's really still the same..
	# (this can happen after using Installer to restore a db)
	#
	$rule = -1;
    }

    if ($rule == 4 || $rule == 5 || $rule == 6)
    {
	# Update File
	#
	unless (defined($frec))
	{
	    $frec = {};
	    $frec->{$gIdField} = &$gIdHook($file_db);
	    push(@{$file_db->{'__RECORDS'}}, $frec);
	    $file_db->{$frec->{$gIdField}} = $#{$file_db->{'__RECORDS'}};
	    $file_sizechange = 1;
	}

	&copyRec($prec, $frec);
	PilotMgr::msg('Update File: ' . &$gNameHook($prec));

	$master_db->{$prec->{id}} = $frec;
	$master_db->{$gIdField . '_' . $frec->{$gIdField}} = $prec->{id}
	    unless (defined $mrec);
    }

    if ($rule == 2)
    {
	# Delete Pilot
	#
	$id = $master_db->{$gIdField . '_' . $mrec->{$gIdField}};

	$dbh->deleteRecord($id);
	PilotMgr::msg('Delete Pilot: ' . &$gNameHook($mrec));

	delete $master_db->{$id};
	delete $master_db->{$gIdField . '_' . $mrec->{$gIdField}};
    }

    if ($rule == 8 && defined($mrec))
    {
	# Delete File
	#
	$id = $file_db->{$mrec->{$gIdField}};
	splice(@{$file_db->{'__RECORDS'}}, $id, 1);
	delete $file_db->{$mrec->{$gIdField}};
	$file_sizechange = -1;
	PilotMgr::msg('Delete File: ' . &$gNameHook($mrec));

	for (; $id < @{$file_db->{'__RECORDS'}}; $id++)
	{
	    $file_db->{ $file_db->{'__RECORDS'}->[$id]->{$gIdField} }--;
	}

	delete $master_db->{$gIdField . '_' . $mrec->{$gIdField}};
	delete $master_db->{$prec->{id}};
    }

    return $file_sizechange;
}

sub copyRec
{
    my ($src, $dst) = @_;
    my ($fld, $val);

    foreach $fld (@$gReqFields)
    {
	$val = $src->{$fld};
	if (ref($val) eq 'HASH')	{ $dst->{$fld} = &copyHash($val);   }
	elsif (ref($val) eq 'ARRAY')	{ $dst->{$fld} = &copyArray($val);  }
	else				{ $dst->{$fld} = $val;		    }
    }
    return $dst;
}

sub copyHash
{
    my ($src) = @_;
    my ($h, $fld, $val) = ({});

    foreach $fld (keys %$src)
    {
	$val = $src->{$fld};
	if (ref($val) eq 'HASH')	{ $h->{$fld} = &copyHash($val);    }
	elsif (ref($val) eq 'ARRAY')	{ $h->{$fld} = &copyArray($val);   }
	else				{ $h->{$fld} = $val;		   }
    }
    return $h;
}

sub copyArray
{
    my ($src) = @_;
    my ($a, $val) = ([]);

    foreach $val (@$src)
    {
	if (ref($val) eq 'HASH')	{ push(@$a, &copyHash($val));	}
	elsif (ref($val) eq 'ARRAY')	{ push(@$a, &copyArray($val));	}
	else				{ push(@$a, $val);		}
    }
    return $a;
}

sub recsDiffer
{
    my ($rec1, $rec2) = @_;
    my ($fld);

    foreach $fld (@$gReqFields)
    {
	return 1 if (defined $rec1->{$fld} ^ defined $rec2->{$fld});
	next unless (defined $rec1->{$fld});
	return 1 if ( ref($rec1->{$fld}) ne ref($rec2->{$fld})  ||
		     (ref($rec1->{$fld}) eq 'HASH' &&
				&hashDiffer($rec1->{$fld}, $rec2->{$fld}) ) ||
		     (ref($rec1->{$fld}) eq 'ARRAY' &&
				&arrayDiffer($rec1->{$fld}, $rec2->{$fld}) ) ||
		     (ref($rec1->{$fld}) eq '' &&
				$rec1->{$fld} ne $rec2->{$fld}) );
    }
    return 0;
}

sub hashDiffer
{
    my ($h1, $h2) = @_;
    my ($fld);

    foreach $fld (keys %$h1)
    {
	return 1 if ( !defined $h2->{$fld} ||
		      ref($h1->{$fld}) ne ref($h2->{$fld}) ||
		      (ref($h1->{$fld}) eq 'HASH' &&
			&hashDiffer($h1->{$fld}, $h2->{$fld})) ||
		      (ref($h1->{$fld}) eq 'ARRAY' &&
			&arrayDiffer($h1->{$fld}, $h2->{$fld})) ||
		      (ref($h1->{$fld}) eq '' && $h1->{$fld} ne $h2->{$fld}) );
    }
    return 0;
}

sub arrayDiffer
{
    my ($a1, $a2) = @_;
    my ($i);

    return 1 if ($#$a1 ne $#$a2);
    for ($i=0; $i < @$a1; $i++)
    {
	return 1 if (defined $a1->[$i] ^ defined $a2->[$i]);
	next unless (defined $a1->[$i]);
	return 1 if ( ref($a1->[$i]) ne ref($a2->[$i])  ||
		     (ref($a1->[$i]) eq 'HASH' &&
			&hashDiffer($a1->[$i], $a2->[$i])) ||
		     (ref($a1->[$i]) eq 'ARRAY' &&
			&arrayDiffer($a1->[$i], $a2->[$i])) ||
		     (ref($a1->[$i]) eq '' && $a1->[$i] ne $a2->[$i]) );
    }
    return 0;
}

sub findPilotMatch
{
    my ($frec) = @_;
    my ($prec);

    foreach $prec (values %$pilot_db)
    {
	return $prec unless (&recsDiffer($frec, $prec));
    }
    return undef;
}

sub syncAppInfo
{
    my ($dbh, $InfoFields, $translateHook) = @_;
    my ($writePi, $transl, $pappi, $fappi, $mappi, $fld) = (0, 0);

    $pappi = $dbh->getAppBlock();
    $fappi = $file_db->{'__APPINFO'};

    return unless (defined $pappi || defined $fappi);

    if (keys %$pappi == 1 && defined $pappi->{'raw'})
    {
	&$translateHook($pappi, 0);
	$transl = 1;
    }

    $mappi = $master_db->{'__APPINFO'};
    $master_db->{'__APPINFO'} = $mappi = {} unless (defined $mappi);
    $mappi->{'__GOT_SYNCED'} = 1;

    unless (defined $pappi) { $pappi = {}; $writePi=1; }
    $file_db->{'__APPINFO'} = $fappi = {} unless (defined $fappi);

    foreach $fld (@$InfoFields)
    {
	if (ref($pappi->{$fld}) eq 'ARRAY' || ref($fappi->{$fld}) eq 'ARRAY')
	{
	    # for array values
	    #
	    my ($size, $i);
	    $size = @{$fappi->{$fld}} if defined($fappi->{$fld});
	    $size = @{$pappi->{$fld}} if defined($pappi->{$fld});

	    $pappi->{$fld} = [] unless (defined $pappi->{$fld});
	    $fappi->{$fld} = [] unless (defined $fappi->{$fld});
	    $mappi->{$fld} = [] unless (defined $mappi->{$fld});

	    for ($i=0; $i < $size; $i++)
	    {
		$writePi |= &aiCheck($pappi, $fappi, $mappi, $fld, $i);
	    }
	}
	else
	{
	    # for scalar values
	    #
	    $writePi |= &aiCheck($pappi, $fappi, $mappi, $fld);
	}
    }

    if ($writePi)
    {
	&$translateHook($pappi, 1) if ($transl);
	PilotMgr::msg("Updating Pilot AppInfo..");
	$dbh->setAppBlock($pappi);
    }
}

sub aiCheck
{
    my ($pappi, $fappi, $mappi, $fld, $i) = @_;
    my ($pichange, $pval, $fval, $mval) = (0);

    $pval = $pappi->{$fld};
    $fval = $fappi->{$fld};
    $mval = $mappi->{$fld};
    if (defined $i)
    {
	$pval = $pval->[$i];
	$fval = $fval->[$i];
	$mval = $mval->[$i];
    }

    # Pilot   Master  File
    # undef   any     undef	-> huh?
    # def \\  any  // def	-> copy to master (pilot==file)
    # undef   any     def	-> use file
    # def     any     undef	-> use pilot
    # def  == def     def	-> use file
    # def     def ==  def	-> use pilot
    # def  != any !=  def	-> merge? use pilot for now

    return unless (defined $pval || defined $fval);

    if (defined $pval && defined $fval && $pval eq $fval)
    {
	if (defined $i) { $mappi->{$fld}->[$i] = $fval; }
	else		{ $mappi->{$fld} = $fval;	}
    }
    elsif (!defined $fval || (defined $mval && $fval eq $mval))
    {
	if (defined $i) { $mappi->{$fld}->[$i] = $fappi->{$fld}->[$i] = $pval;}
	else		{ $mappi->{$fld} = $fappi->{$fld} = $pval;	      }
    }
    elsif (!defined $pval || (defined $mval && $pval eq $mval))
    {
	if (defined $i) { $mappi->{$fld}->[$i] = $pappi->{$fld}->[$i] = $fval;}
	else		{ $mappi->{$fld} = $pappi->{$fld} = $fval;	      }
	$pichange = 1;
    }
    else
    {
	PilotMgr::msg("AppInfo field $fld" .
	    ((defined $i) ? "[$i]" : '') .
	    " changed on both pilot and file!\n" .
	    "Not updating either side.. please change them to be the same.");
    }

    return $pichange;
}

1;

