/*
 *  Timers abstract layer
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "timer.h"
#include "info.h"

snd_timer_t *snd_timer_devices[ SND_TIMER_DEVICES ];

snd_mutex_define_static( register );
snd_mutex_define_static( open );

/*
 *
 */

snd_timer_t *snd_timer_open( char *owner, unsigned int resolution )
{
  int idx;
  snd_timer_t *timer;
  
  snd_mutex_down_static( open );
  for ( idx = SND_TIMER_DEVICES - 1; idx >= 0; idx-- ) {
    timer = snd_timer_devices[ idx ];
    if ( timer == NULL ) continue;
    if ( !(timer -> flags & SND_TIMER_FLG_USED) &&
         timer -> hw.resolution <= resolution ) {
      if ( timer -> hw.open ) {
        if ( !timer -> hw.open( timer ) ) break;
      } else {
        break;
      }
    }
    timer = NULL;
  }
  if ( timer == NULL ) {
    for ( idx = 0; idx < SND_TIMER_DEVICES; idx++ ) {
      timer = snd_timer_devices[ idx ];
      if ( timer == NULL ) continue;
      if ( !(timer -> flags & SND_TIMER_FLG_USED) ) {
        if ( timer -> hw.open ) {
          if ( !timer -> hw.open( timer ) ) break;
        } else {
          break;
        }
      }
      timer = NULL;
    }
    if ( timer == NULL ) {
      snd_mutex_up_static( open );
      return NULL;
    }
  }
  timer -> flags |= SND_TIMER_FLG_USED;
  timer -> owner = snd_malloc_strdup( owner );
  MOD_INC_USE_COUNT;
  timer -> card -> use_inc( timer -> card );
  snd_mutex_up_static( open );
  return timer;
}

int snd_timer_close( snd_timer_t *timer )
{
  if ( !timer ) return -EINVAL;
  snd_mutex_down_static( open );
  if ( timer -> hw.close )
    timer -> hw.close( timer );
  timer -> flags &= ~SND_TIMER_FLG_USED;
  snd_free_str( timer -> owner );
  timer -> owner = NULL;
  MOD_DEC_USE_COUNT;
  snd_mutex_up_static( open );
  timer -> card -> use_dec( timer -> card );
  return 0;
}

unsigned snd_timer_resolution( snd_timer_t *timer )
{
  if ( timer -> hw.c_resolution )
    return timer -> hw.c_resolution( timer );
  return timer -> hw.resolution;
}

void snd_timer_start( snd_timer_t *timer, unsigned int ticks )
{
  if ( !timer ) return;
  if ( timer -> hw.low_ticks > ticks ) ticks = timer -> hw.low_ticks;
  timer -> ticks = timer -> cticks = ticks;
  timer -> hw.start( timer );
}

void snd_timer_stop( snd_timer_t *timer )
{
  if ( !timer ) return;
  timer -> hw.stop( timer );
}

void snd_timer_continue( snd_timer_t *timer )
{
  if ( timer -> hw.t_continue ) {
    timer -> hw.t_continue( timer );
    return;
  }
  if ( !timer -> cticks ) timer -> cticks = 1;
  timer -> hw.start( timer );
}

snd_timer_t *snd_timer_open_always( char *owner, unsigned int resolution )
{
  snd_timer_t *timer;
  
  timer = snd_timer_open( owner, resolution );
  if ( timer != NULL ) return timer;
  return snd_timer_open_system( owner );
}

int snd_timer_close_always( snd_timer_t *timer )
{
  if ( timer -> flags & SND_TIMER_FLG_SYSTEM )
    return snd_timer_close_system( timer );
  return snd_timer_close( timer );
}

/*
 *
 */

snd_timer_t *snd_timer_new_device( snd_card_t *card, char *id )
{
  snd_timer_t *timer;

  timer = (snd_timer_t *)snd_calloc( sizeof( snd_timer_t ) );
  if ( !timer ) return NULL;
  timer -> card = card;
  if ( id ) {
    strncpy( timer -> id, id, sizeof( timer -> id ) - 1 );
  }
  return timer;
}

int snd_timer_free( snd_timer_t *timer )
{
  if ( !timer ) return -EINVAL;
  if ( timer -> private_free )
    timer -> private_free( timer -> private_data );
  snd_free( timer, sizeof( snd_timer_t ) );
  return 0;  
}

int snd_timer_register( snd_timer_t *timer )
{
  int idx, idx1;
  snd_timer_t *timer1;

  if ( !timer ) return -EINVAL;
  if ( snd_timer_devices[ SND_TIMER_DEVICES - 1 ] != NULL ) return -EBUSY;
  snd_mutex_down_static( register );
  for ( idx = 0; idx < SND_TIMER_DEVICES; idx++ ) {
    if ( (timer1 = snd_timer_devices[ idx ]) == NULL ) {
      snd_timer_devices[ idx ] = timer;
      break;
    }
    if ( timer1 -> hw.resolution > timer -> hw.resolution ) {
      for ( idx1 = SND_TIMER_DEVICES - 1; idx1 > idx; idx1-- ) {
        snd_timer_devices[ idx1 ] = snd_timer_devices[ idx1 - 1 ];
        if ( snd_timer_devices[ idx1 ] && idx1 < SND_CARDS - 1 ) {
          snd_oss_info_unregister( SND_OSS_INFO_DEV_TIMERS, idx1 );
          snd_oss_info_register( SND_OSS_INFO_DEV_TIMERS, idx1, snd_timer_devices[ idx1 ] -> name );
        }
      }
      snd_oss_info_unregister( SND_OSS_INFO_DEV_TIMERS, idx );
      snd_timer_devices[ idx ] = timer;
      break;
    }
  }
  if ( idx < SND_CARDS - 1 )
    snd_oss_info_register( SND_OSS_INFO_DEV_TIMERS, idx, timer -> name );  
  snd_mutex_up_static( register );
  return 0;
}

int snd_timer_unregister( snd_timer_t *timer )
{
  int idx;

  if ( !timer ) return -EINVAL;
  snd_mutex_down_static( register );
  for ( idx = 0; idx < SND_TIMER_DEVICES; idx++ ) {
    if ( snd_timer_devices[ idx ] == timer ) {
      snd_timer_devices[ idx ] = NULL;
      if ( idx < SND_CARDS - 1 )
        snd_oss_info_unregister( SND_OSS_INFO_DEV_TIMERS, idx );  
      snd_mutex_up_static( register );
      return snd_timer_free( timer );
    }
  }
  snd_mutex_up_static( register );
  return -EINVAL;
}

/*
 *  System timer
 */

unsigned int snd_timer_system_resolution( void )
{
  return 1000000000L / HZ;
}

static void snd_timer_s_function( unsigned long data )
{
  snd_timer_t *timer;
  
  timer = (snd_timer_t *)data;
  timer -> flags &= ~SND_TIMER_FLG_RUNNING;
  if ( timer -> callback ) 
    timer -> callback( timer, timer -> callback_data );
}

static void snd_timer_s_start( snd_timer_t *timer )
{
  struct timer_list *tlist;
  
  tlist = (struct timer_list *)timer -> private_data;
  tlist -> expires = jiffies + timer -> cticks;
  timer -> flags |= SND_TIMER_FLG_RUNNING;
  add_timer( tlist );
}

static void snd_timer_s_stop( snd_timer_t *timer )
{
  struct timer_list *tlist;
  
  tlist = (struct timer_list *)timer -> private_data;
  del_timer( tlist );  
  timer -> flags &= ~SND_TIMER_FLG_RUNNING;
  timer -> cticks = tlist -> expires - jiffies;
}

static void snd_timer_s_continue( snd_timer_t *timer )
{
  struct timer_list *tlist;
  
  tlist = (struct timer_list *)timer -> private_data;
  timer -> flags |= SND_TIMER_FLG_RUNNING;
  tlist -> expires = jiffies + timer -> cticks;
  add_timer( tlist );
}

static struct snd_stru_timer_hardware snd_timer_system = {
  1000000000L / HZ,		/* resolution in us */
  1,				/* low ticks */
  10000000L,			/* high ticks */
  NULL,				/* open */
  NULL,				/* close */
  NULL,				/* resolution */
  snd_timer_s_start,		/* start */
  snd_timer_s_stop,		/* stop */
  snd_timer_s_continue,		/* continue */
};

snd_timer_t *snd_timer_open_system( char *owner )
{
  snd_timer_t *timer;
  struct timer_list *tlist;
  
  timer = snd_timer_new_device( NULL, "system" );
  if ( !timer ) return NULL;
  strcpy( timer -> name, "system timer" );
  memcpy( &timer -> hw, &snd_timer_system, sizeof( snd_timer_system ) );
  timer -> flags |= SND_TIMER_FLG_USED | SND_TIMER_FLG_SYSTEM;
  tlist = (struct timer_list *)snd_calloc( sizeof( struct timer_list ) );
  if ( !tlist ) {
    snd_timer_free( timer );
    return NULL;
  }
  tlist -> function = snd_timer_s_function;
  tlist -> data = (unsigned long)timer;
  timer -> private_data = tlist;
  MOD_INC_USE_COUNT;
  return timer;
}

int snd_timer_close_system( snd_timer_t *timer )
{
  snd_free( timer -> private_data, sizeof( struct timer_list ) );
  snd_timer_free( timer );
  MOD_DEC_USE_COUNT;
  return 0;
}

/*
 *  Info interface
 */

static void snd_timer_proc_read( snd_info_buffer_t *buffer, void *private_data )
{
  int idx;
  unsigned int tmp;
  snd_timer_t *timer;

  snd_mutex_down_static( register );
  snd_mutex_down_static( open );
  for ( idx = 0; idx < SND_TIMER_DEVICES; idx++ ) {
    timer = snd_timer_devices[ idx ];
    if ( timer == NULL ) continue;
    snd_iprintf( buffer, "%02i-", idx );
    if ( timer -> card )
      snd_iprintf( buffer, "%02i: ", timer -> card -> number );
     else
      snd_iprintf( buffer, "XX: " );
    snd_iprintf( buffer, timer -> name );
    if ( timer -> hw.resolution )
      snd_iprintf( buffer, " : %u.%uus (%u-%u)", timer -> hw.resolution / 1000, timer -> hw.resolution % 1000, timer -> hw.low_ticks, timer -> hw.high_ticks );
    if ( timer -> flags & SND_TIMER_FLG_USED )
      snd_iprintf( buffer, " : %s", timer -> owner ? timer -> owner : "uknown" );
    snd_iprintf( buffer, "\n" );
  }
  tmp = snd_timer_system_resolution();
  snd_iprintf( buffer, "XX-XX: unlimited system timer : %i.%ius (%u-%u)\n", tmp / 1000, tmp % 1000, snd_timer_system.low_ticks, snd_timer_system.high_ticks );
  snd_mutex_up_static( open );
  snd_mutex_up_static( register );
}

/*
 *  ENTRY functions
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_timer_export;
#endif

static snd_info_entry_t *snd_timer_proc_entry = NULL;

int init_module( void )
{
  snd_info_entry_t *entry;

#ifndef LINUX_2_1
  if ( register_symtab( &snd_symbol_table_timer_export ) < 0 )
    return -ENOMEM;
#endif
  memset( snd_timer_devices, 0, sizeof( snd_timer_devices ) );
  snd_oss_info_register( SND_OSS_INFO_DEV_TIMERS, SND_CARDS - 1, "system timer" );  
  if ( (entry = snd_info_create_entry( NULL, "timers" )) != NULL ) {
    entry -> t.text.read_size = SND_TIMER_DEVICES * 128;
    entry -> t.text.read = snd_timer_proc_read;
    if ( snd_info_register( entry ) < 0 ) {
      snd_info_free_entry( entry );
      entry = NULL;
    }
  }
  snd_timer_proc_entry = entry;
  return 0;
}

void cleanup_module( void )
{
  if ( snd_timer_proc_entry ) {
    snd_info_unregister( snd_timer_proc_entry );
    snd_timer_proc_entry = NULL;
  }
  snd_oss_info_unregister( SND_OSS_INFO_DEV_TIMERS, SND_CARDS - 1 );
}
